using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model
{
  
  
  public class DebugContextActionsHostModel : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public IRdProperty<DebugSessionInfo> CurrentSessionInfo { get { return _CurrentSessionInfo; }}
    [NotNull] public IRdSignal<ShowClassInstancesArg> ShowClassInstances { get { return _ShowClassInstances; }}
    [NotNull] public IRdSignal<RunToStatementArg> RunToCursor { get { return _RunToCursor; }}
    [NotNull] public IRdSignal<RunToStatementArg> ForceRunToCursor { get { return _ForceRunToCursor; }}
    [NotNull] public IRdSignal<RunToStatementArg> SetNextStatement { get { return _SetNextStatement; }}
    
    //private fields
    [NotNull] private readonly RdProperty<DebugSessionInfo> _CurrentSessionInfo;
    [NotNull] private readonly RdSignal<ShowClassInstancesArg> _ShowClassInstances;
    [NotNull] private readonly RdSignal<RunToStatementArg> _RunToCursor;
    [NotNull] private readonly RdSignal<RunToStatementArg> _ForceRunToCursor;
    [NotNull] private readonly RdSignal<RunToStatementArg> _SetNextStatement;
    
    //primary constructor
    private DebugContextActionsHostModel(
      [NotNull] RdProperty<DebugSessionInfo> currentSessionInfo,
      [NotNull] RdSignal<ShowClassInstancesArg> showClassInstances,
      [NotNull] RdSignal<RunToStatementArg> runToCursor,
      [NotNull] RdSignal<RunToStatementArg> forceRunToCursor,
      [NotNull] RdSignal<RunToStatementArg> setNextStatement
    )
    {
      if (currentSessionInfo == null) throw new ArgumentNullException("currentSessionInfo");
      if (showClassInstances == null) throw new ArgumentNullException("showClassInstances");
      if (runToCursor == null) throw new ArgumentNullException("runToCursor");
      if (forceRunToCursor == null) throw new ArgumentNullException("forceRunToCursor");
      if (setNextStatement == null) throw new ArgumentNullException("setNextStatement");
      
      _CurrentSessionInfo = currentSessionInfo;
      _ShowClassInstances = showClassInstances;
      _RunToCursor = runToCursor;
      _ForceRunToCursor = forceRunToCursor;
      _SetNextStatement = setNextStatement;
      _CurrentSessionInfo.OptimizeNested = true;
      _CurrentSessionInfo.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("currentSessionInfo", _CurrentSessionInfo));
      BindableChildren.Add(new KeyValuePair<string, object>("showClassInstances", _ShowClassInstances));
      BindableChildren.Add(new KeyValuePair<string, object>("runToCursor", _RunToCursor));
      BindableChildren.Add(new KeyValuePair<string, object>("forceRunToCursor", _ForceRunToCursor));
      BindableChildren.Add(new KeyValuePair<string, object>("setNextStatement", _SetNextStatement));
    }
    //secondary constructor
    private DebugContextActionsHostModel (
    ) : this (
      new RdProperty<DebugSessionInfo>(ReadDebugSessionInfoNullable, WriteDebugSessionInfoNullable),
      new RdSignal<ShowClassInstancesArg>(ShowClassInstancesArg.Read, ShowClassInstancesArg.Write),
      new RdSignal<RunToStatementArg>(RunToStatementArg.Read, RunToStatementArg.Write),
      new RdSignal<RunToStatementArg>(RunToStatementArg.Read, RunToStatementArg.Write),
      new RdSignal<RunToStatementArg>(RunToStatementArg.Read, RunToStatementArg.Write)
    ) {}
    //statics
    
    public static CtxReadDelegate<DebugSessionInfo> ReadDebugSessionInfoNullable = DebugSessionInfo.Read.NullableClass();
    
    public static CtxWriteDelegate<DebugSessionInfo> WriteDebugSessionInfoNullable = DebugSessionInfo.Write.NullableClass();
    
    protected override long SerializationHash => -4694178867660785522L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(SourcePosition.Read, SourcePosition.Write);
      serializers.Register(DebugSessionInfo.Read, DebugSessionInfo.Write);
      serializers.Register(ShowClassInstancesArg.Read, ShowClassInstancesArg.Write);
      serializers.Register(RunToStatementArg.Read, RunToStatementArg.Write);
      
      serializers.RegisterToplevelOnce(typeof(IdeRoot), IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    public DebugContextActionsHostModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DebugContextActionsHostModel (");
      using (printer.IndentCookie()) {
        printer.Print("currentSessionInfo = "); _CurrentSessionInfo.PrintEx(printer); printer.Println();
        printer.Print("showClassInstances = "); _ShowClassInstances.PrintEx(printer); printer.Println();
        printer.Print("runToCursor = "); _RunToCursor.PrintEx(printer); printer.Println();
        printer.Print("forceRunToCursor = "); _ForceRunToCursor.PrintEx(printer); printer.Println();
        printer.Print("setNextStatement = "); _SetNextStatement.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DebugSessionInfo : IPrintable, IEquatable<DebugSessionInfo>
  {
    //fields
    //public fields
    public bool IsSuspended {get; private set;}
    public bool IsMemoryViewEnabled {get; private set;}
    public bool IsSetNextStatementEnabled {get; private set;}
    [CanBeNull] public SourcePosition CurrentTopFramePosition {get; private set;}
    
    //private fields
    //primary constructor
    public DebugSessionInfo(
      bool isSuspended,
      bool isMemoryViewEnabled,
      bool isSetNextStatementEnabled,
      [CanBeNull] SourcePosition currentTopFramePosition
    )
    {
      IsSuspended = isSuspended;
      IsMemoryViewEnabled = isMemoryViewEnabled;
      IsSetNextStatementEnabled = isSetNextStatementEnabled;
      CurrentTopFramePosition = currentTopFramePosition;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<DebugSessionInfo> Read = (ctx, reader) => 
    {
      var isSuspended = reader.ReadBool();
      var isMemoryViewEnabled = reader.ReadBool();
      var isSetNextStatementEnabled = reader.ReadBool();
      var currentTopFramePosition = ReadSourcePositionNullable(ctx, reader);
      return new DebugSessionInfo(isSuspended, isMemoryViewEnabled, isSetNextStatementEnabled, currentTopFramePosition);
    };
    public static CtxReadDelegate<SourcePosition> ReadSourcePositionNullable = SourcePosition.Read.NullableClass();
    
    public static CtxWriteDelegate<DebugSessionInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.IsSuspended);
      writer.Write(value.IsMemoryViewEnabled);
      writer.Write(value.IsSetNextStatementEnabled);
      WriteSourcePositionNullable(ctx, writer, value.CurrentTopFramePosition);
    };
    public static CtxWriteDelegate<SourcePosition> WriteSourcePositionNullable = SourcePosition.Write.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DebugSessionInfo) obj);
    }
    public bool Equals(DebugSessionInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return IsSuspended == other.IsSuspended && IsMemoryViewEnabled == other.IsMemoryViewEnabled && IsSetNextStatementEnabled == other.IsSetNextStatementEnabled && Equals(CurrentTopFramePosition, other.CurrentTopFramePosition);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + IsSuspended.GetHashCode();
        hash = hash * 31 + IsMemoryViewEnabled.GetHashCode();
        hash = hash * 31 + IsSetNextStatementEnabled.GetHashCode();
        hash = hash * 31 + (CurrentTopFramePosition != null ?CurrentTopFramePosition.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DebugSessionInfo (");
      using (printer.IndentCookie()) {
        printer.Print("isSuspended = "); IsSuspended.PrintEx(printer); printer.Println();
        printer.Print("isMemoryViewEnabled = "); IsMemoryViewEnabled.PrintEx(printer); printer.Println();
        printer.Print("isSetNextStatementEnabled = "); IsSetNextStatementEnabled.PrintEx(printer); printer.Println();
        printer.Print("currentTopFramePosition = "); CurrentTopFramePosition.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RunToStatementArg : IPrintable, IEquatable<RunToStatementArg>
  {
    //fields
    //public fields
    [NotNull] public string FilePath {get; private set;}
    public int Offset {get; private set;}
    
    //private fields
    //primary constructor
    public RunToStatementArg(
      [NotNull] string filePath,
      int offset
    )
    {
      if (filePath == null) throw new ArgumentNullException("filePath");
      
      FilePath = filePath;
      Offset = offset;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RunToStatementArg> Read = (ctx, reader) => 
    {
      var filePath = reader.ReadString();
      var offset = reader.ReadInt();
      return new RunToStatementArg(filePath, offset);
    };
    
    public static CtxWriteDelegate<RunToStatementArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.FilePath);
      writer.Write(value.Offset);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RunToStatementArg) obj);
    }
    public bool Equals(RunToStatementArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return FilePath == other.FilePath && Offset == other.Offset;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + FilePath.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RunToStatementArg (");
      using (printer.IndentCookie()) {
        printer.Print("filePath = "); FilePath.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ShowClassInstancesArg : IPrintable, IEquatable<ShowClassInstancesArg>
  {
    //fields
    //public fields
    [NotNull] public string ClassName {get; private set;}
    [NotNull] public JetBrains.Rider.Model.TextControlId TextControlId {get; private set;}
    
    //private fields
    //primary constructor
    public ShowClassInstancesArg(
      [NotNull] string className,
      [NotNull] JetBrains.Rider.Model.TextControlId textControlId
    )
    {
      if (className == null) throw new ArgumentNullException("className");
      if (textControlId == null) throw new ArgumentNullException("textControlId");
      
      ClassName = className;
      TextControlId = textControlId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ShowClassInstancesArg> Read = (ctx, reader) => 
    {
      var className = reader.ReadString();
      var textControlId = JetBrains.Rider.Model.TextControlId.Read(ctx, reader);
      return new ShowClassInstancesArg(className, textControlId);
    };
    
    public static CtxWriteDelegate<ShowClassInstancesArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ClassName);
      JetBrains.Rider.Model.TextControlId.Write(ctx, writer, value.TextControlId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ShowClassInstancesArg) obj);
    }
    public bool Equals(ShowClassInstancesArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ClassName == other.ClassName && Equals(TextControlId, other.TextControlId);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ClassName.GetHashCode();
        hash = hash * 31 + TextControlId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ShowClassInstancesArg (");
      using (printer.IndentCookie()) {
        printer.Print("className = "); ClassName.PrintEx(printer); printer.Println();
        printer.Print("textControlId = "); TextControlId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SourcePosition : IPrintable, IEquatable<SourcePosition>
  {
    //fields
    //public fields
    [NotNull] public string File {get; private set;}
    public int Offset {get; private set;}
    
    //private fields
    //primary constructor
    public SourcePosition(
      [NotNull] string file,
      int offset
    )
    {
      if (file == null) throw new ArgumentNullException("file");
      
      File = file;
      Offset = offset;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SourcePosition> Read = (ctx, reader) => 
    {
      var file = reader.ReadString();
      var offset = reader.ReadInt();
      return new SourcePosition(file, offset);
    };
    
    public static CtxWriteDelegate<SourcePosition> Write = (ctx, writer, value) => 
    {
      writer.Write(value.File);
      writer.Write(value.Offset);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SourcePosition) obj);
    }
    public bool Equals(SourcePosition other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return File == other.File && Offset == other.Offset;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + File.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SourcePosition (");
      using (printer.IndentCookie()) {
        printer.Print("file = "); File.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
