using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.debuggerHelper
{
  
  
  public class DebuggerHelperHostModel : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public RdEndpoint<int, bool> IsProcess64Bit { get { return _IsProcess64Bit; }}
    [NotNull] public RdEndpoint<string, bool> IsExeWillRunAs64Bit { get { return _IsExeWillRunAs64Bit; }}
    [NotNull] public RdEndpoint<RdVoid, List<string>> GetDebuggerAssemblies { get { return _GetDebuggerAssemblies; }}
    [NotNull] public RdEndpoint<GetDumpInfoArg, GetDumpInfoResult> GetDebugInfo { get { return _GetDebugInfo; }}
    
    //private fields
    [NotNull] private readonly RdEndpoint<int, bool> _IsProcess64Bit;
    [NotNull] private readonly RdEndpoint<string, bool> _IsExeWillRunAs64Bit;
    [NotNull] private readonly RdEndpoint<RdVoid, List<string>> _GetDebuggerAssemblies;
    [NotNull] private readonly RdEndpoint<GetDumpInfoArg, GetDumpInfoResult> _GetDebugInfo;
    
    //primary constructor
    private DebuggerHelperHostModel(
      [NotNull] RdEndpoint<int, bool> isProcess64Bit,
      [NotNull] RdEndpoint<string, bool> isExeWillRunAs64Bit,
      [NotNull] RdEndpoint<RdVoid, List<string>> getDebuggerAssemblies,
      [NotNull] RdEndpoint<GetDumpInfoArg, GetDumpInfoResult> getDebugInfo
    )
    {
      if (isProcess64Bit == null) throw new ArgumentNullException("isProcess64Bit");
      if (isExeWillRunAs64Bit == null) throw new ArgumentNullException("isExeWillRunAs64Bit");
      if (getDebuggerAssemblies == null) throw new ArgumentNullException("getDebuggerAssemblies");
      if (getDebugInfo == null) throw new ArgumentNullException("getDebugInfo");
      
      _IsProcess64Bit = isProcess64Bit;
      _IsExeWillRunAs64Bit = isExeWillRunAs64Bit;
      _GetDebuggerAssemblies = getDebuggerAssemblies;
      _GetDebugInfo = getDebugInfo;
      _GetDebuggerAssemblies.Async = true;
      _GetDebugInfo.Async = true;
      BindableChildren.Add(new KeyValuePair<string, object>("isProcess64Bit", _IsProcess64Bit));
      BindableChildren.Add(new KeyValuePair<string, object>("isExeWillRunAs64Bit", _IsExeWillRunAs64Bit));
      BindableChildren.Add(new KeyValuePair<string, object>("getDebuggerAssemblies", _GetDebuggerAssemblies));
      BindableChildren.Add(new KeyValuePair<string, object>("getDebugInfo", _GetDebugInfo));
    }
    //secondary constructor
    private DebuggerHelperHostModel (
    ) : this (
      new RdEndpoint<int, bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdEndpoint<string, bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdEndpoint<RdVoid, List<string>>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid, ReadStringList, WriteStringList),
      new RdEndpoint<GetDumpInfoArg, GetDumpInfoResult>(GetDumpInfoArg.Read, GetDumpInfoArg.Write, GetDumpInfoResult.Read, GetDumpInfoResult.Write)
    ) {}
    //statics
    
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    
    protected override long SerializationHash => -2040047082782704530L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(GetDumpInfoArg.Read, GetDumpInfoArg.Write);
      serializers.Register(GetDumpInfoResult.Read, GetDumpInfoResult.Write);
      serializers.RegisterEnum<RuntimeType>();
      serializers.RegisterEnum<DumpPlatformKind>();
      
      serializers.RegisterToplevelOnce(typeof(JetBrains.Rider.Model.IdeRoot), JetBrains.Rider.Model.IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    public DebuggerHelperHostModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerHelperHostModel (");
      using (printer.IndentCookie()) {
        printer.Print("isProcess64Bit = "); _IsProcess64Bit.PrintEx(printer); printer.Println();
        printer.Print("isExeWillRunAs64Bit = "); _IsExeWillRunAs64Bit.PrintEx(printer); printer.Println();
        printer.Print("getDebuggerAssemblies = "); _GetDebuggerAssemblies.PrintEx(printer); printer.Println();
        printer.Print("getDebugInfo = "); _GetDebugInfo.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum DumpPlatformKind {
    x64,
    x86
  }
  
  
  public class GetDumpInfoArg : IPrintable, IEquatable<GetDumpInfoArg>
  {
    //fields
    //public fields
    [NotNull] public string Path {get; private set;}
    
    //private fields
    //primary constructor
    public GetDumpInfoArg(
      [NotNull] string path
    )
    {
      if (path == null) throw new ArgumentNullException("path");
      
      Path = path;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetDumpInfoArg> Read = (ctx, reader) => 
    {
      var path = reader.ReadString();
      return new GetDumpInfoArg(path);
    };
    
    public static CtxWriteDelegate<GetDumpInfoArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Path);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetDumpInfoArg) obj);
    }
    public bool Equals(GetDumpInfoArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Path == other.Path;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Path.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetDumpInfoArg (");
      using (printer.IndentCookie()) {
        printer.Print("path = "); Path.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class GetDumpInfoResult : IPrintable, IEquatable<GetDumpInfoResult>
  {
    //fields
    //public fields
    [NotNull] public string DmpPath {get; private set;}
    [NotNull] public List<string> Symbols {get; private set;}
    public JetBrains.Rider.Model.debuggerHelper.RuntimeType RuntimeType {get; private set;}
    public DumpPlatformKind Platform {get; private set;}
    [CanBeNull] public string DbgShimDirectory {get; private set;}
    
    //private fields
    //primary constructor
    public GetDumpInfoResult(
      [NotNull] string dmpPath,
      [NotNull] List<string> symbols,
      JetBrains.Rider.Model.debuggerHelper.RuntimeType runtimeType,
      DumpPlatformKind platform,
      [CanBeNull] string dbgShimDirectory
    )
    {
      if (dmpPath == null) throw new ArgumentNullException("dmpPath");
      if (symbols == null) throw new ArgumentNullException("symbols");
      
      DmpPath = dmpPath;
      Symbols = symbols;
      RuntimeType = runtimeType;
      Platform = platform;
      DbgShimDirectory = dbgShimDirectory;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetDumpInfoResult> Read = (ctx, reader) => 
    {
      var dmpPath = reader.ReadString();
      var symbols = ReadStringList(ctx, reader);
      var runtimeType = (JetBrains.Rider.Model.debuggerHelper.RuntimeType)reader.ReadInt();
      var platform = (DumpPlatformKind)reader.ReadInt();
      var dbgShimDirectory = ReadStringNullable(ctx, reader);
      return new GetDumpInfoResult(dmpPath, symbols, runtimeType, platform, dbgShimDirectory);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<GetDumpInfoResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.DmpPath);
      WriteStringList(ctx, writer, value.Symbols);
      writer.Write((int)value.RuntimeType);
      writer.Write((int)value.Platform);
      WriteStringNullable(ctx, writer, value.DbgShimDirectory);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetDumpInfoResult) obj);
    }
    public bool Equals(GetDumpInfoResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return DmpPath == other.DmpPath && Symbols.SequenceEqual(other.Symbols) && RuntimeType == other.RuntimeType && Platform == other.Platform && Equals(DbgShimDirectory, other.DbgShimDirectory);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DmpPath.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Symbols);
        hash = hash * 31 + (int) RuntimeType;
        hash = hash * 31 + (int) Platform;
        hash = hash * 31 + (DbgShimDirectory != null ?DbgShimDirectory.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetDumpInfoResult (");
      using (printer.IndentCookie()) {
        printer.Print("dmpPath = "); DmpPath.PrintEx(printer); printer.Println();
        printer.Print("symbols = "); Symbols.PrintEx(printer); printer.Println();
        printer.Print("runtimeType = "); RuntimeType.PrintEx(printer); printer.Println();
        printer.Print("platform = "); Platform.PrintEx(printer); printer.Println();
        printer.Print("dbgShimDirectory = "); DbgShimDirectory.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RuntimeType {
    DesktopClr,
    CoreClr
  }
}
