using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.DebuggerWorker
{
  
  
  public class DebuggerWorkerModel : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public IRdProperty<int> HostPort { get { return _HostPort; }}
    [NotNull] public IRdProperty<bool> Initialized { get { return _Initialized; }}
    [NotNull] public IRdProperty<DotNetDebuggerSessionModel> ActiveSession { get { return _ActiveSession; }}
    [NotNull] public ISource<TargetExitedArgs> TargetExited { get { return _TargetExited; }}
    [NotNull] public ISink<RdVoid> DetachTarget { get { return _DetachTarget; }}
    [NotNull] public ISink<RdVoid> DestroyTarget { get { return _DestroyTarget; }}
    [NotNull] public ISource<string> LogError { get { return _LogError; }}
    [NotNull] public IRdProperty<List<string>> TraceCategories { get { return _TraceCategories; }}
    
    //private fields
    [NotNull] private readonly RdProperty<int> _HostPort;
    [NotNull] private readonly RdProperty<bool> _Initialized;
    [NotNull] private readonly RdProperty<DotNetDebuggerSessionModel> _ActiveSession;
    [NotNull] private readonly RdSignal<TargetExitedArgs> _TargetExited;
    [NotNull] private readonly RdSignal<RdVoid> _DetachTarget;
    [NotNull] private readonly RdSignal<RdVoid> _DestroyTarget;
    [NotNull] private readonly RdSignal<string> _LogError;
    [NotNull] private readonly RdProperty<List<string>> _TraceCategories;
    
    //primary constructor
    private DebuggerWorkerModel(
      [NotNull] RdProperty<int> hostPort,
      [NotNull] RdProperty<bool> initialized,
      [NotNull] RdProperty<DotNetDebuggerSessionModel> activeSession,
      [NotNull] RdSignal<TargetExitedArgs> targetExited,
      [NotNull] RdSignal<RdVoid> detachTarget,
      [NotNull] RdSignal<RdVoid> destroyTarget,
      [NotNull] RdSignal<string> logError,
      [NotNull] RdProperty<List<string>> traceCategories
    )
    {
      if (hostPort == null) throw new ArgumentNullException("hostPort");
      if (initialized == null) throw new ArgumentNullException("initialized");
      if (activeSession == null) throw new ArgumentNullException("activeSession");
      if (targetExited == null) throw new ArgumentNullException("targetExited");
      if (detachTarget == null) throw new ArgumentNullException("detachTarget");
      if (destroyTarget == null) throw new ArgumentNullException("destroyTarget");
      if (logError == null) throw new ArgumentNullException("logError");
      if (traceCategories == null) throw new ArgumentNullException("traceCategories");
      
      _HostPort = hostPort;
      _Initialized = initialized;
      _ActiveSession = activeSession;
      _TargetExited = targetExited;
      _DetachTarget = detachTarget;
      _DestroyTarget = destroyTarget;
      _LogError = logError;
      _TraceCategories = traceCategories;
      _HostPort.OptimizeNested = true;
      _Initialized.OptimizeNested = true;
      _TraceCategories.OptimizeNested = true;
      _TraceCategories.Async = true;
      _ActiveSession.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("hostPort", _HostPort));
      BindableChildren.Add(new KeyValuePair<string, object>("initialized", _Initialized));
      BindableChildren.Add(new KeyValuePair<string, object>("activeSession", _ActiveSession));
      BindableChildren.Add(new KeyValuePair<string, object>("targetExited", _TargetExited));
      BindableChildren.Add(new KeyValuePair<string, object>("detachTarget", _DetachTarget));
      BindableChildren.Add(new KeyValuePair<string, object>("destroyTarget", _DestroyTarget));
      BindableChildren.Add(new KeyValuePair<string, object>("logError", _LogError));
      BindableChildren.Add(new KeyValuePair<string, object>("traceCategories", _TraceCategories));
    }
    //secondary constructor
    private DebuggerWorkerModel (
    ) : this (
      new RdProperty<int>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<DotNetDebuggerSessionModel>(ReadDotNetDebuggerSessionModelNullable, WriteDotNetDebuggerSessionModelNullable),
      new RdSignal<TargetExitedArgs>(TargetExitedArgs.Read, TargetExitedArgs.Write),
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdSignal<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<List<string>>(ReadStringList, WriteStringList)
    ) {}
    //statics
    
    public static CtxReadDelegate<DotNetDebuggerSessionModel> ReadDotNetDebuggerSessionModelNullable = DotNetDebuggerSessionModel.Read.NullableClass();
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<DotNetDebuggerSessionModel> WriteDotNetDebuggerSessionModelNullable = DotNetDebuggerSessionModel.Write.NullableClass();
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    
    protected override long SerializationHash => -138643581601636242L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.RegisterEnum<BreakpointStatusKind>();
      serializers.Register(BreakpointStatus.Read, BreakpointStatus.Write);
      serializers.Register(BreakpointId.Read, BreakpointId.Write);
      serializers.RegisterEnum<BreakpointHitCountModeModel>();
      serializers.Register(EvaluationExpressionModel.Read, EvaluationExpressionModel.Write);
      serializers.Register(BreakpointElement.Read, BreakpointElement.Write);
      serializers.Register(LineBreakpointModel.Read, LineBreakpointModel.Write);
      serializers.Register(ExceptionBreakpointModel.Read, ExceptionBreakpointModel.Write);
      serializers.Register(ObjectPropertiesProxy.Read, ObjectPropertiesProxy.Write);
      serializers.Register(ObjectProxy.Read, ObjectProxy.Write);
      serializers.Register(SourcePositionProxy.Read, SourcePositionProxy.Write);
      serializers.Register(StackFrameProxy.Read, StackFrameProxy.Write);
      serializers.Register(StackProxy.Read, StackProxy.Write);
      serializers.RegisterEnum<ExecutionCommandKind>();
      serializers.Register(ExecutionCommandProxy.Read, ExecutionCommandProxy.Write);
      serializers.Register(ExecutionCommandProxyWithPosition.Read, ExecutionCommandProxyWithPosition.Write);
      serializers.Register(ExceptionModel.Read, ExceptionModel.Write);
      serializers.Register(SmartStepIntoElementProxy.Read, SmartStepIntoElementProxy.Write);
      serializers.Register(SmartStepIntoInfoProxy.Read, SmartStepIntoInfoProxy.Write);
      serializers.Register(SmartStepIntoCommandProxy.Read, SmartStepIntoCommandProxy.Write);
      serializers.Register(HeapClassInfoModel.Read, HeapClassInfoModel.Write);
      serializers.Register(FetchMemoryViewResult.Read, FetchMemoryViewResult.Write);
      serializers.Register(GetClassInstancesArg.Read, GetClassInstancesArg.Write);
      serializers.Register(GetFilesSequencePointsArg.Read, GetFilesSequencePointsArg.Write);
      serializers.Register(SequencePointProxy.Read, SequencePointProxy.Write);
      serializers.Register(SequencePointsAndFilePair.Read, SequencePointsAndFilePair.Write);
      serializers.Register(GetFilesSequencePointsResult.Read, GetFilesSequencePointsResult.Write);
      serializers.RegisterEnum<ExecutionContextType>();
      serializers.Register(ExecutionContextProxy.Read, ExecutionContextProxy.Write);
      serializers.Register(SessionProperties.Read, SessionProperties.Write);
      serializers.Register(StringPair.Read, StringPair.Write);
      serializers.Register(DumpMsDotNetStartInfo.Read, DumpMsDotNetStartInfo.Write);
      serializers.Register(DumpDotNetCoreStartInfo.Read, DumpDotNetCoreStartInfo.Write);
      serializers.Register(MonoExeStartInfo.Read, MonoExeStartInfo.Write);
      serializers.Register(DotNetClrExeStartInfo.Read, DotNetClrExeStartInfo.Write);
      serializers.Register(MonoAttachStartInfo.Read, MonoAttachStartInfo.Write);
      serializers.Register(DotNetClrAttachStartInfo.Read, DotNetClrAttachStartInfo.Write);
      serializers.Register(DotNetCoreExeStartInfo.Read, DotNetCoreExeStartInfo.Write);
      serializers.Register(DotNetCoreInfo.Read, DotNetCoreInfo.Write);
      serializers.Register(DotNetCoreAttachStartInfo.Read, DotNetCoreAttachStartInfo.Write);
      serializers.Register(XamarinIOSWiFiStartInfo.Read, XamarinIOSWiFiStartInfo.Write);
      serializers.Register(XamarinIOSUsbStartInfo.Read, XamarinIOSUsbStartInfo.Write);
      serializers.Register(XamarinMacStartInfo.Read, XamarinMacStartInfo.Write);
      serializers.Register(XamarinAndroidStartInfo.Read, XamarinAndroidStartInfo.Write);
      serializers.Register(OutputMessage.Read, OutputMessage.Write);
      serializers.Register(DebuggerSessionInfo.Read, DebuggerSessionInfo.Write);
      serializers.Register(DotNetDebuggerSessionModel.Read, DotNetDebuggerSessionModel.Write);
      serializers.Register(TargetExitedArgs.Read, TargetExitedArgs.Write);
      serializers.RegisterEnum<FrameKind>();
      serializers.RegisterEnum<ExceptionModelType>();
      serializers.Register(ComputeFrameLocalsArg.Read, ComputeFrameLocalsArg.Write);
      serializers.Register(ComputeFrameLocalsResult.Read, ComputeFrameLocalsResult.Write);
      serializers.Register(ComputeValueChildrenArg.Read, ComputeValueChildrenArg.Write);
      serializers.Register(ComputeValueChildrenResult.Read, ComputeValueChildrenResult.Write);
      serializers.Register(RefreshValueArg.Read, RefreshValueArg.Write);
      serializers.Register(RefreshValueResult.Read, RefreshValueResult.Write);
      serializers.Register(MakeObjectIdArg.Read, MakeObjectIdArg.Write);
      serializers.Register(MakeObjectIdResult.Read, MakeObjectIdResult.Write);
      serializers.Register(RemoveObjectIdArg.Read, RemoveObjectIdArg.Write);
      serializers.Register(ComputeFramesArg.Read, ComputeFramesArg.Write);
      serializers.Register(ComputeFramesResult.Read, ComputeFramesResult.Write);
      serializers.Register(GetEvaluationExpressionArg.Read, GetEvaluationExpressionArg.Write);
      serializers.Register(EvaluateArg.Read, EvaluateArg.Write);
      serializers.Register(EvaluateResult.Read, EvaluateResult.Write);
      serializers.Register(ComputeObjectPropertiesArg.Read, ComputeObjectPropertiesArg.Write);
      serializers.Register(SetValueArg.Read, SetValueArg.Write);
      serializers.Register(ComputeFieldsArg.Read, ComputeFieldsArg.Write);
      serializers.Register(ComputeFieldsResult.Read, ComputeFieldsResult.Write);
      serializers.Register(RefreshStackFrameArg.Read, RefreshStackFrameArg.Write);
      serializers.Register(EvalPair.Read, EvalPair.Write);
      serializers.RegisterEnum<DebugKind>();
      serializers.RegisterEnum<OutputType>();
      serializers.RegisterEnum<OutputSubject>();
      serializers.Register(ShowNotification.Read, ShowNotification.Write);
      serializers.Register(BreakpointModel_Unknown.Read, BreakpointModel_Unknown.Write);
      serializers.Register(ExecutionCommandBase_Unknown.Read, ExecutionCommandBase_Unknown.Write);
      serializers.Register(ExceptionModelBase_Unknown.Read, ExceptionModelBase_Unknown.Write);
      serializers.Register(DebuggerStartInfoBase_Unknown.Read, DebuggerStartInfoBase_Unknown.Write);
      serializers.Register(DumpStartInfo_Unknown.Read, DumpStartInfo_Unknown.Write);
      serializers.Register(ExeStartInfo_Unknown.Read, ExeStartInfo_Unknown.Write);
      serializers.Register(ProcessIdAttachInfo_Unknown.Read, ProcessIdAttachInfo_Unknown.Write);
      serializers.Register(XamarinStartInfoBase_Unknown.Read, XamarinStartInfoBase_Unknown.Write);
      
      serializers.RegisterToplevelOnce(typeof(DebuggerWorkerModel), DebuggerWorkerModel.RegisterDeclaredTypesSerializers);
    }
    
    public DebuggerWorkerModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerWorkerModel (");
      using (printer.IndentCookie()) {
        printer.Print("hostPort = "); _HostPort.PrintEx(printer); printer.Println();
        printer.Print("initialized = "); _Initialized.PrintEx(printer); printer.Println();
        printer.Print("activeSession = "); _ActiveSession.PrintEx(printer); printer.Println();
        printer.Print("targetExited = "); _TargetExited.PrintEx(printer); printer.Println();
        printer.Print("detachTarget = "); _DetachTarget.PrintEx(printer); printer.Println();
        printer.Print("destroyTarget = "); _DestroyTarget.PrintEx(printer); printer.Println();
        printer.Print("logError = "); _LogError.PrintEx(printer); printer.Println();
        printer.Print("traceCategories = "); _TraceCategories.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class BreakpointElement : IPrintable, IEquatable<BreakpointElement>
  {
    //fields
    //public fields
    [NotNull] public string Filename {get; private set;}
    public int Line {get; private set;}
    public int Column {get; private set;}
    public bool IsMultiMethodBreakpoint {get; private set;}
    
    //private fields
    //primary constructor
    public BreakpointElement(
      [NotNull] string filename,
      int line,
      int column,
      bool isMultiMethodBreakpoint
    )
    {
      if (filename == null) throw new ArgumentNullException("filename");
      
      Filename = filename;
      Line = line;
      Column = column;
      IsMultiMethodBreakpoint = isMultiMethodBreakpoint;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<BreakpointElement> Read = (ctx, reader) => 
    {
      var filename = reader.ReadString();
      var line = reader.ReadInt();
      var column = reader.ReadInt();
      var isMultiMethodBreakpoint = reader.ReadBool();
      return new BreakpointElement(filename, line, column, isMultiMethodBreakpoint);
    };
    
    public static CtxWriteDelegate<BreakpointElement> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Filename);
      writer.Write(value.Line);
      writer.Write(value.Column);
      writer.Write(value.IsMultiMethodBreakpoint);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((BreakpointElement) obj);
    }
    public bool Equals(BreakpointElement other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Filename == other.Filename && Line == other.Line && Column == other.Column && IsMultiMethodBreakpoint == other.IsMultiMethodBreakpoint;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Filename.GetHashCode();
        hash = hash * 31 + Line.GetHashCode();
        hash = hash * 31 + Column.GetHashCode();
        hash = hash * 31 + IsMultiMethodBreakpoint.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("BreakpointElement (");
      using (printer.IndentCookie()) {
        printer.Print("filename = "); Filename.PrintEx(printer); printer.Println();
        printer.Print("line = "); Line.PrintEx(printer); printer.Println();
        printer.Print("column = "); Column.PrintEx(printer); printer.Println();
        printer.Print("isMultiMethodBreakpoint = "); IsMultiMethodBreakpoint.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum BreakpointHitCountModeModel {
    None,
    LessThan,
    LessThanOrEqualTo,
    EqualTo,
    GreaterThan,
    GreaterThanOrEqualTo,
    MultipleOf
  }
  
  
  public class BreakpointId : IPrintable, IEquatable<BreakpointId>
  {
    //fields
    //public fields
    public RdId Id {get; private set;}
    
    //private fields
    //primary constructor
    public BreakpointId(
      RdId id
    )
    {
      Id = id;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<BreakpointId> Read = (ctx, reader) => 
    {
      var id = reader.ReadRdId();
      return new BreakpointId(id);
    };
    
    public static CtxWriteDelegate<BreakpointId> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((BreakpointId) obj);
    }
    public bool Equals(BreakpointId other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("BreakpointId (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class BreakpointModel : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public BreakpointId Id {get; private set;}
    [NotNull] public IRdProperty<bool> ShouldNotifyHitWhenDisabled { get { return _ShouldNotifyHitWhenDisabled; }}
    [NotNull] public ISource<BreakpointStatus> StatusChanged { get { return _StatusChanged; }}
    [CanBeNull] public EvaluationExpressionModel Condition {get; private set;}
    public bool Suspended {get; private set;}
    [CanBeNull] public EvaluationExpressionModel TraceExpression {get; private set;}
    public bool TraceBreakpoint {get; private set;}
    public int HitCount {get; private set;}
    public BreakpointHitCountModeModel HitCountMode {get; private set;}
    
    //private fields
    [NotNull] protected readonly RdProperty<bool> _ShouldNotifyHitWhenDisabled;
    [NotNull] protected readonly RdSignal<BreakpointStatus> _StatusChanged;
    
    //primary constructor
    protected BreakpointModel(
      [NotNull] BreakpointId id,
      [NotNull] RdProperty<bool> shouldNotifyHitWhenDisabled,
      [NotNull] RdSignal<BreakpointStatus> statusChanged,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    )
    {
      if (id == null) throw new ArgumentNullException("id");
      if (shouldNotifyHitWhenDisabled == null) throw new ArgumentNullException("shouldNotifyHitWhenDisabled");
      if (statusChanged == null) throw new ArgumentNullException("statusChanged");
      
      Id = id;
      _ShouldNotifyHitWhenDisabled = shouldNotifyHitWhenDisabled;
      _StatusChanged = statusChanged;
      Condition = condition;
      Suspended = suspended;
      TraceExpression = traceExpression;
      TraceBreakpoint = traceBreakpoint;
      HitCount = hitCount;
      HitCountMode = hitCountMode;
      _ShouldNotifyHitWhenDisabled.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("shouldNotifyHitWhenDisabled", _ShouldNotifyHitWhenDisabled));
      BindableChildren.Add(new KeyValuePair<string, object>("statusChanged", _StatusChanged));
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<BreakpointModel> Read = Polymorphic<BreakpointModel>.ReadAbstract(BreakpointModel_Unknown.Read);
    
    public static CtxWriteDelegate<BreakpointModel> Write = Polymorphic<BreakpointModel>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class BreakpointModel_Unknown : BreakpointModel
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    private BreakpointModel_Unknown(
      [NotNull] BreakpointId id,
      [NotNull] RdProperty<bool> shouldNotifyHitWhenDisabled,
      [NotNull] RdSignal<BreakpointStatus> statusChanged,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : base (
      id,
      shouldNotifyHitWhenDisabled,
      statusChanged,
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
     ) 
    {
    }
    //secondary constructor
    public BreakpointModel_Unknown (
      [NotNull] BreakpointId id,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : this (
      id,
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<BreakpointStatus>(BreakpointStatus.Read, BreakpointStatus.Write),
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
    ) {}
    //statics
    
    public static new CtxReadDelegate<BreakpointModel_Unknown> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var id = BreakpointId.Read(ctx, reader);
      var shouldNotifyHitWhenDisabled = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var statusChanged = RdSignal<BreakpointStatus>.Read(ctx, reader, BreakpointStatus.Read, BreakpointStatus.Write);
      var condition = ReadEvaluationExpressionModelNullable(ctx, reader);
      var suspended = reader.ReadBool();
      var traceExpression = ReadEvaluationExpressionModelNullable(ctx, reader);
      var traceBreakpoint = reader.ReadBool();
      var hitCount = reader.ReadInt();
      var hitCountMode = (BreakpointHitCountModeModel)reader.ReadInt();
      return new BreakpointModel_Unknown(id, shouldNotifyHitWhenDisabled, statusChanged, condition, suspended, traceExpression, traceBreakpoint, hitCount, hitCountMode).WithId(_id);
    };
    public static CtxReadDelegate<EvaluationExpressionModel> ReadEvaluationExpressionModelNullable = EvaluationExpressionModel.Read.NullableClass();
    
    public static new CtxWriteDelegate<BreakpointModel_Unknown> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      BreakpointId.Write(ctx, writer, value.Id);
      RdProperty<bool>.Write(ctx, writer, value._ShouldNotifyHitWhenDisabled);
      RdSignal<BreakpointStatus>.Write(ctx, writer, value._StatusChanged);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.Condition);
      writer.Write(value.Suspended);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.TraceExpression);
      writer.Write(value.TraceBreakpoint);
      writer.Write(value.HitCount);
      writer.Write((int)value.HitCountMode);
    };
    public static CtxWriteDelegate<EvaluationExpressionModel> WriteEvaluationExpressionModelNullable = EvaluationExpressionModel.Write.NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("BreakpointModel_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("shouldNotifyHitWhenDisabled = "); _ShouldNotifyHitWhenDisabled.PrintEx(printer); printer.Println();
        printer.Print("statusChanged = "); _StatusChanged.PrintEx(printer); printer.Println();
        printer.Print("condition = "); Condition.PrintEx(printer); printer.Println();
        printer.Print("suspended = "); Suspended.PrintEx(printer); printer.Println();
        printer.Print("traceExpression = "); TraceExpression.PrintEx(printer); printer.Println();
        printer.Print("traceBreakpoint = "); TraceBreakpoint.PrintEx(printer); printer.Println();
        printer.Print("hitCount = "); HitCount.PrintEx(printer); printer.Println();
        printer.Print("hitCountMode = "); HitCountMode.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class BreakpointStatus : IPrintable, IEquatable<BreakpointStatus>
  {
    //fields
    //public fields
    public BreakpointStatusKind Status {get; private set;}
    [CanBeNull] public string Message {get; private set;}
    
    //private fields
    //primary constructor
    public BreakpointStatus(
      BreakpointStatusKind status,
      [CanBeNull] string message
    )
    {
      Status = status;
      Message = message;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<BreakpointStatus> Read = (ctx, reader) => 
    {
      var status = (BreakpointStatusKind)reader.ReadInt();
      var message = ReadStringNullable(ctx, reader);
      return new BreakpointStatus(status, message);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<BreakpointStatus> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Status);
      WriteStringNullable(ctx, writer, value.Message);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((BreakpointStatus) obj);
    }
    public bool Equals(BreakpointStatus other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Status == other.Status && Equals(Message, other.Message);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (int) Status;
        hash = hash * 31 + (Message != null ?Message.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("BreakpointStatus (");
      using (printer.IndentCookie()) {
        printer.Print("status = "); Status.PrintEx(printer); printer.Println();
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum BreakpointStatusKind {
    Disconnected,
    NotBound,
    Bound,
    Invalid,
    BindError,
    ExternalCodeDisabled
  }
  
  
  public class ComputeFieldsArg : IPrintable, IEquatable<ComputeFieldsArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFieldsArg(
      int valueId
    )
    {
      ValueId = valueId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFieldsArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      return new ComputeFieldsArg(valueId);
    };
    
    public static CtxWriteDelegate<ComputeFieldsArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFieldsArg) obj);
    }
    public bool Equals(ComputeFieldsArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFieldsArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeFieldsResult : IPrintable, IEquatable<ComputeFieldsResult>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    [NotNull] public List<ObjectProxy> Fields {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFieldsResult(
      int valueId,
      [NotNull] List<ObjectProxy> fields
    )
    {
      if (fields == null) throw new ArgumentNullException("fields");
      
      ValueId = valueId;
      Fields = fields;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFieldsResult> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var fields = ReadObjectProxyList(ctx, reader);
      return new ComputeFieldsResult(valueId, fields);
    };
    public static CtxReadDelegate<List<ObjectProxy>> ReadObjectProxyList = ObjectProxy.Read.List();
    
    public static CtxWriteDelegate<ComputeFieldsResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      WriteObjectProxyList(ctx, writer, value.Fields);
    };
    public static CtxWriteDelegate<List<ObjectProxy>> WriteObjectProxyList = ObjectProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFieldsResult) obj);
    }
    public bool Equals(ComputeFieldsResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && Fields.SequenceEqual(other.Fields);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Fields);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFieldsResult (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("fields = "); Fields.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeFrameLocalsArg : IPrintable, IEquatable<ComputeFrameLocalsArg>
  {
    //fields
    //public fields
    public int FrameId {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFrameLocalsArg(
      int frameId
    )
    {
      FrameId = frameId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFrameLocalsArg> Read = (ctx, reader) => 
    {
      var frameId = reader.ReadInt();
      return new ComputeFrameLocalsArg(frameId);
    };
    
    public static CtxWriteDelegate<ComputeFrameLocalsArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.FrameId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFrameLocalsArg) obj);
    }
    public bool Equals(ComputeFrameLocalsArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return FrameId == other.FrameId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + FrameId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFrameLocalsArg (");
      using (printer.IndentCookie()) {
        printer.Print("frameId = "); FrameId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeFrameLocalsResult : IPrintable, IEquatable<ComputeFrameLocalsResult>
  {
    //fields
    //public fields
    public int FrameId {get; private set;}
    [NotNull] public List<ObjectProxy> Locals {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFrameLocalsResult(
      int frameId,
      [NotNull] List<ObjectProxy> locals
    )
    {
      if (locals == null) throw new ArgumentNullException("locals");
      
      FrameId = frameId;
      Locals = locals;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFrameLocalsResult> Read = (ctx, reader) => 
    {
      var frameId = reader.ReadInt();
      var locals = ReadObjectProxyList(ctx, reader);
      return new ComputeFrameLocalsResult(frameId, locals);
    };
    public static CtxReadDelegate<List<ObjectProxy>> ReadObjectProxyList = ObjectProxy.Read.List();
    
    public static CtxWriteDelegate<ComputeFrameLocalsResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.FrameId);
      WriteObjectProxyList(ctx, writer, value.Locals);
    };
    public static CtxWriteDelegate<List<ObjectProxy>> WriteObjectProxyList = ObjectProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFrameLocalsResult) obj);
    }
    public bool Equals(ComputeFrameLocalsResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return FrameId == other.FrameId && Locals.SequenceEqual(other.Locals);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + FrameId.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Locals);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFrameLocalsResult (");
      using (printer.IndentCookie()) {
        printer.Print("frameId = "); FrameId.PrintEx(printer); printer.Println();
        printer.Print("locals = "); Locals.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeFramesArg : IPrintable, IEquatable<ComputeFramesArg>
  {
    //fields
    //public fields
    public int StackId {get; private set;}
    public int FirstFrameIndex {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFramesArg(
      int stackId,
      int firstFrameIndex
    )
    {
      StackId = stackId;
      FirstFrameIndex = firstFrameIndex;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFramesArg> Read = (ctx, reader) => 
    {
      var stackId = reader.ReadInt();
      var firstFrameIndex = reader.ReadInt();
      return new ComputeFramesArg(stackId, firstFrameIndex);
    };
    
    public static CtxWriteDelegate<ComputeFramesArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.StackId);
      writer.Write(value.FirstFrameIndex);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFramesArg) obj);
    }
    public bool Equals(ComputeFramesArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return StackId == other.StackId && FirstFrameIndex == other.FirstFrameIndex;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + StackId.GetHashCode();
        hash = hash * 31 + FirstFrameIndex.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFramesArg (");
      using (printer.IndentCookie()) {
        printer.Print("stackId = "); StackId.PrintEx(printer); printer.Println();
        printer.Print("firstFrameIndex = "); FirstFrameIndex.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeFramesResult : IPrintable, IEquatable<ComputeFramesResult>
  {
    //fields
    //public fields
    public int StackId {get; private set;}
    public bool Last {get; private set;}
    [NotNull] public List<StackFrameProxy> Frames {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeFramesResult(
      int stackId,
      bool last,
      [NotNull] List<StackFrameProxy> frames
    )
    {
      if (frames == null) throw new ArgumentNullException("frames");
      
      StackId = stackId;
      Last = last;
      Frames = frames;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeFramesResult> Read = (ctx, reader) => 
    {
      var stackId = reader.ReadInt();
      var last = reader.ReadBool();
      var frames = ReadStackFrameProxyList(ctx, reader);
      return new ComputeFramesResult(stackId, last, frames);
    };
    public static CtxReadDelegate<List<StackFrameProxy>> ReadStackFrameProxyList = StackFrameProxy.Read.List();
    
    public static CtxWriteDelegate<ComputeFramesResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.StackId);
      writer.Write(value.Last);
      WriteStackFrameProxyList(ctx, writer, value.Frames);
    };
    public static CtxWriteDelegate<List<StackFrameProxy>> WriteStackFrameProxyList = StackFrameProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeFramesResult) obj);
    }
    public bool Equals(ComputeFramesResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return StackId == other.StackId && Last == other.Last && Frames.SequenceEqual(other.Frames);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + StackId.GetHashCode();
        hash = hash * 31 + Last.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Frames);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeFramesResult (");
      using (printer.IndentCookie()) {
        printer.Print("stackId = "); StackId.PrintEx(printer); printer.Println();
        printer.Print("last = "); Last.PrintEx(printer); printer.Println();
        printer.Print("frames = "); Frames.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeObjectPropertiesArg : IPrintable, IEquatable<ComputeObjectPropertiesArg>
  {
    //fields
    //public fields
    public int ObjectId {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeObjectPropertiesArg(
      int objectId
    )
    {
      ObjectId = objectId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeObjectPropertiesArg> Read = (ctx, reader) => 
    {
      var objectId = reader.ReadInt();
      return new ComputeObjectPropertiesArg(objectId);
    };
    
    public static CtxWriteDelegate<ComputeObjectPropertiesArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ObjectId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeObjectPropertiesArg) obj);
    }
    public bool Equals(ComputeObjectPropertiesArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ObjectId == other.ObjectId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ObjectId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeObjectPropertiesArg (");
      using (printer.IndentCookie()) {
        printer.Print("objectId = "); ObjectId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeValueChildrenArg : IPrintable, IEquatable<ComputeValueChildrenArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    public bool AllowTargetInvoke {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeValueChildrenArg(
      int valueId,
      bool allowTargetInvoke
    )
    {
      ValueId = valueId;
      AllowTargetInvoke = allowTargetInvoke;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeValueChildrenArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var allowTargetInvoke = reader.ReadBool();
      return new ComputeValueChildrenArg(valueId, allowTargetInvoke);
    };
    
    public static CtxWriteDelegate<ComputeValueChildrenArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      writer.Write(value.AllowTargetInvoke);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeValueChildrenArg) obj);
    }
    public bool Equals(ComputeValueChildrenArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && AllowTargetInvoke == other.AllowTargetInvoke;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + AllowTargetInvoke.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeValueChildrenArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("allowTargetInvoke = "); AllowTargetInvoke.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ComputeValueChildrenResult : IPrintable, IEquatable<ComputeValueChildrenResult>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    [NotNull] public List<ObjectProxy> Children {get; private set;}
    
    //private fields
    //primary constructor
    public ComputeValueChildrenResult(
      int valueId,
      [NotNull] List<ObjectProxy> children
    )
    {
      if (children == null) throw new ArgumentNullException("children");
      
      ValueId = valueId;
      Children = children;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ComputeValueChildrenResult> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var children = ReadObjectProxyList(ctx, reader);
      return new ComputeValueChildrenResult(valueId, children);
    };
    public static CtxReadDelegate<List<ObjectProxy>> ReadObjectProxyList = ObjectProxy.Read.List();
    
    public static CtxWriteDelegate<ComputeValueChildrenResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      WriteObjectProxyList(ctx, writer, value.Children);
    };
    public static CtxWriteDelegate<List<ObjectProxy>> WriteObjectProxyList = ObjectProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ComputeValueChildrenResult) obj);
    }
    public bool Equals(ComputeValueChildrenResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && Children.SequenceEqual(other.Children);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Children);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ComputeValueChildrenResult (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("children = "); Children.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum DebugKind {
    Live,
    Dump
  }
  
  
  public class DebuggerSessionInfo : IPrintable, IEquatable<DebuggerSessionInfo>
  {
    //fields
    //public fields
    public int ProcessId {get; private set;}
    public bool CanSetNextStatement {get; private set;}
    public bool CanFetchHeapValues {get; private set;}
    public bool CanMakeObjectIds {get; private set;}
    
    //private fields
    //primary constructor
    public DebuggerSessionInfo(
      int processId,
      bool canSetNextStatement,
      bool canFetchHeapValues,
      bool canMakeObjectIds
    )
    {
      ProcessId = processId;
      CanSetNextStatement = canSetNextStatement;
      CanFetchHeapValues = canFetchHeapValues;
      CanMakeObjectIds = canMakeObjectIds;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<DebuggerSessionInfo> Read = (ctx, reader) => 
    {
      var processId = reader.ReadInt();
      var canSetNextStatement = reader.ReadBool();
      var canFetchHeapValues = reader.ReadBool();
      var canMakeObjectIds = reader.ReadBool();
      return new DebuggerSessionInfo(processId, canSetNextStatement, canFetchHeapValues, canMakeObjectIds);
    };
    
    public static CtxWriteDelegate<DebuggerSessionInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProcessId);
      writer.Write(value.CanSetNextStatement);
      writer.Write(value.CanFetchHeapValues);
      writer.Write(value.CanMakeObjectIds);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DebuggerSessionInfo) obj);
    }
    public bool Equals(DebuggerSessionInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProcessId == other.ProcessId && CanSetNextStatement == other.CanSetNextStatement && CanFetchHeapValues == other.CanFetchHeapValues && CanMakeObjectIds == other.CanMakeObjectIds;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProcessId.GetHashCode();
        hash = hash * 31 + CanSetNextStatement.GetHashCode();
        hash = hash * 31 + CanFetchHeapValues.GetHashCode();
        hash = hash * 31 + CanMakeObjectIds.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerSessionInfo (");
      using (printer.IndentCookie()) {
        printer.Print("processId = "); ProcessId.PrintEx(printer); printer.Println();
        printer.Print("canSetNextStatement = "); CanSetNextStatement.PrintEx(printer); printer.Println();
        printer.Print("canFetchHeapValues = "); CanFetchHeapValues.PrintEx(printer); printer.Println();
        printer.Print("canMakeObjectIds = "); CanMakeObjectIds.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class DebuggerStartInfoBase{
    //fields
    //public fields
    public bool NeedToBeInitializedImmediately {get; private set;}
    
    //private fields
    //primary constructor
    protected DebuggerStartInfoBase(
      bool needToBeInitializedImmediately
    )
    {
      NeedToBeInitializedImmediately = needToBeInitializedImmediately;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<DebuggerStartInfoBase> Read = Polymorphic<DebuggerStartInfoBase>.ReadAbstract(DebuggerStartInfoBase_Unknown.Read);
    
    public static CtxWriteDelegate<DebuggerStartInfoBase> Write = Polymorphic<DebuggerStartInfoBase>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class DebuggerStartInfoBase_Unknown : DebuggerStartInfoBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DebuggerStartInfoBase_Unknown(
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DebuggerStartInfoBase_Unknown> Read = (ctx, reader) => 
    {
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DebuggerStartInfoBase_Unknown(needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<DebuggerStartInfoBase_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DebuggerStartInfoBase_Unknown) obj);
    }
    public bool Equals(DebuggerStartInfoBase_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerStartInfoBase_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetClrAttachStartInfo : ProcessIdAttachInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DotNetClrAttachStartInfo(
      int processId,
      bool needToBeInitializedImmediately
    ) : base (
      processId,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DotNetClrAttachStartInfo> Read = (ctx, reader) => 
    {
      var processId = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DotNetClrAttachStartInfo(processId, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<DotNetClrAttachStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProcessId);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DotNetClrAttachStartInfo) obj);
    }
    public bool Equals(DotNetClrAttachStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProcessId == other.ProcessId && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProcessId.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetClrAttachStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("processId = "); ProcessId.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetClrExeStartInfo : ExeStartInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DotNetClrExeStartInfo(
      [NotNull] string exePath,
      [NotNull] string workingDirectory,
      [NotNull] string arguments,
      [NotNull] List<StringPair> environmentVariables,
      [CanBeNull] string runtimeArguments,
      bool executeAsIs,
      bool useExternalConsole,
      bool needToBeInitializedImmediately
    ) : base (
      exePath,
      workingDirectory,
      arguments,
      environmentVariables,
      runtimeArguments,
      executeAsIs,
      useExternalConsole,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DotNetClrExeStartInfo> Read = (ctx, reader) => 
    {
      var exePath = reader.ReadString();
      var workingDirectory = reader.ReadString();
      var arguments = reader.ReadString();
      var environmentVariables = ReadStringPairList(ctx, reader);
      var runtimeArguments = ReadStringNullable(ctx, reader);
      var executeAsIs = reader.ReadBool();
      var useExternalConsole = reader.ReadBool();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DotNetClrExeStartInfo(exePath, workingDirectory, arguments, environmentVariables, runtimeArguments, executeAsIs, useExternalConsole, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<StringPair>> ReadStringPairList = StringPair.Read.List();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static new CtxWriteDelegate<DotNetClrExeStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ExePath);
      writer.Write(value.WorkingDirectory);
      writer.Write(value.Arguments);
      WriteStringPairList(ctx, writer, value.EnvironmentVariables);
      WriteStringNullable(ctx, writer, value.RuntimeArguments);
      writer.Write(value.ExecuteAsIs);
      writer.Write(value.UseExternalConsole);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    public static CtxWriteDelegate<List<StringPair>> WriteStringPairList = StringPair.Write.List();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DotNetClrExeStartInfo) obj);
    }
    public bool Equals(DotNetClrExeStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ExePath == other.ExePath && WorkingDirectory == other.WorkingDirectory && Arguments == other.Arguments && EnvironmentVariables.SequenceEqual(other.EnvironmentVariables) && Equals(RuntimeArguments, other.RuntimeArguments) && ExecuteAsIs == other.ExecuteAsIs && UseExternalConsole == other.UseExternalConsole && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ExePath.GetHashCode();
        hash = hash * 31 + WorkingDirectory.GetHashCode();
        hash = hash * 31 + Arguments.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(EnvironmentVariables);
        hash = hash * 31 + (RuntimeArguments != null ?RuntimeArguments.GetHashCode() : 0);
        hash = hash * 31 + ExecuteAsIs.GetHashCode();
        hash = hash * 31 + UseExternalConsole.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetClrExeStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("exePath = "); ExePath.PrintEx(printer); printer.Println();
        printer.Print("workingDirectory = "); WorkingDirectory.PrintEx(printer); printer.Println();
        printer.Print("arguments = "); Arguments.PrintEx(printer); printer.Println();
        printer.Print("environmentVariables = "); EnvironmentVariables.PrintEx(printer); printer.Println();
        printer.Print("runtimeArguments = "); RuntimeArguments.PrintEx(printer); printer.Println();
        printer.Print("executeAsIs = "); ExecuteAsIs.PrintEx(printer); printer.Println();
        printer.Print("useExternalConsole = "); UseExternalConsole.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetCoreAttachStartInfo : ProcessIdAttachInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DotNetCoreAttachStartInfo(
      int processId,
      bool needToBeInitializedImmediately
    ) : base (
      processId,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DotNetCoreAttachStartInfo> Read = (ctx, reader) => 
    {
      var processId = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DotNetCoreAttachStartInfo(processId, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<DotNetCoreAttachStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProcessId);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DotNetCoreAttachStartInfo) obj);
    }
    public bool Equals(DotNetCoreAttachStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProcessId == other.ProcessId && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProcessId.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetCoreAttachStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("processId = "); ProcessId.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetCoreExeStartInfo : ExeStartInfo
  {
    //fields
    //public fields
    [NotNull] public JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo DotNetCoreInfo {get; private set;}
    
    //private fields
    //primary constructor
    public DotNetCoreExeStartInfo(
      [NotNull] JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo dotNetCoreInfo,
      [NotNull] string exePath,
      [NotNull] string workingDirectory,
      [NotNull] string arguments,
      [NotNull] List<StringPair> environmentVariables,
      [CanBeNull] string runtimeArguments,
      bool executeAsIs,
      bool useExternalConsole,
      bool needToBeInitializedImmediately
    ) : base (
      exePath,
      workingDirectory,
      arguments,
      environmentVariables,
      runtimeArguments,
      executeAsIs,
      useExternalConsole,
      needToBeInitializedImmediately
     ) 
    {
      if (dotNetCoreInfo == null) throw new ArgumentNullException("dotNetCoreInfo");
      
      DotNetCoreInfo = dotNetCoreInfo;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DotNetCoreExeStartInfo> Read = (ctx, reader) => 
    {
      var exePath = reader.ReadString();
      var workingDirectory = reader.ReadString();
      var arguments = reader.ReadString();
      var environmentVariables = ReadStringPairList(ctx, reader);
      var runtimeArguments = ReadStringNullable(ctx, reader);
      var executeAsIs = reader.ReadBool();
      var useExternalConsole = reader.ReadBool();
      var needToBeInitializedImmediately = reader.ReadBool();
      var dotNetCoreInfo = JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo.Read(ctx, reader);
      return new DotNetCoreExeStartInfo(dotNetCoreInfo, exePath, workingDirectory, arguments, environmentVariables, runtimeArguments, executeAsIs, useExternalConsole, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<StringPair>> ReadStringPairList = StringPair.Read.List();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static new CtxWriteDelegate<DotNetCoreExeStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ExePath);
      writer.Write(value.WorkingDirectory);
      writer.Write(value.Arguments);
      WriteStringPairList(ctx, writer, value.EnvironmentVariables);
      WriteStringNullable(ctx, writer, value.RuntimeArguments);
      writer.Write(value.ExecuteAsIs);
      writer.Write(value.UseExternalConsole);
      writer.Write(value.NeedToBeInitializedImmediately);
      JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo.Write(ctx, writer, value.DotNetCoreInfo);
    };
    public static CtxWriteDelegate<List<StringPair>> WriteStringPairList = StringPair.Write.List();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DotNetCoreExeStartInfo) obj);
    }
    public bool Equals(DotNetCoreExeStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(DotNetCoreInfo, other.DotNetCoreInfo) && ExePath == other.ExePath && WorkingDirectory == other.WorkingDirectory && Arguments == other.Arguments && EnvironmentVariables.SequenceEqual(other.EnvironmentVariables) && Equals(RuntimeArguments, other.RuntimeArguments) && ExecuteAsIs == other.ExecuteAsIs && UseExternalConsole == other.UseExternalConsole && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DotNetCoreInfo.GetHashCode();
        hash = hash * 31 + ExePath.GetHashCode();
        hash = hash * 31 + WorkingDirectory.GetHashCode();
        hash = hash * 31 + Arguments.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(EnvironmentVariables);
        hash = hash * 31 + (RuntimeArguments != null ?RuntimeArguments.GetHashCode() : 0);
        hash = hash * 31 + ExecuteAsIs.GetHashCode();
        hash = hash * 31 + UseExternalConsole.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetCoreExeStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("dotNetCoreInfo = "); DotNetCoreInfo.PrintEx(printer); printer.Println();
        printer.Print("exePath = "); ExePath.PrintEx(printer); printer.Println();
        printer.Print("workingDirectory = "); WorkingDirectory.PrintEx(printer); printer.Println();
        printer.Print("arguments = "); Arguments.PrintEx(printer); printer.Println();
        printer.Print("environmentVariables = "); EnvironmentVariables.PrintEx(printer); printer.Println();
        printer.Print("runtimeArguments = "); RuntimeArguments.PrintEx(printer); printer.Println();
        printer.Print("executeAsIs = "); ExecuteAsIs.PrintEx(printer); printer.Println();
        printer.Print("useExternalConsole = "); UseExternalConsole.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetCoreInfo : IPrintable, IEquatable<DotNetCoreInfo>
  {
    //fields
    //public fields
    [CanBeNull] public string DotNetCliExePath {get; private set;}
    [CanBeNull] public string DbgShimDirectory {get; private set;}
    
    //private fields
    //primary constructor
    public DotNetCoreInfo(
      [CanBeNull] string dotNetCliExePath,
      [CanBeNull] string dbgShimDirectory
    )
    {
      DotNetCliExePath = dotNetCliExePath;
      DbgShimDirectory = dbgShimDirectory;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<DotNetCoreInfo> Read = (ctx, reader) => 
    {
      var dotNetCliExePath = ReadStringNullable(ctx, reader);
      var dbgShimDirectory = ReadStringNullable(ctx, reader);
      return new DotNetCoreInfo(dotNetCliExePath, dbgShimDirectory);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<DotNetCoreInfo> Write = (ctx, writer, value) => 
    {
      WriteStringNullable(ctx, writer, value.DotNetCliExePath);
      WriteStringNullable(ctx, writer, value.DbgShimDirectory);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DotNetCoreInfo) obj);
    }
    public bool Equals(DotNetCoreInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(DotNetCliExePath, other.DotNetCliExePath) && Equals(DbgShimDirectory, other.DbgShimDirectory);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (DotNetCliExePath != null ?DotNetCliExePath.GetHashCode() : 0);
        hash = hash * 31 + (DbgShimDirectory != null ?DbgShimDirectory.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetCoreInfo (");
      using (printer.IndentCookie()) {
        printer.Print("dotNetCliExePath = "); DotNetCliExePath.PrintEx(printer); printer.Println();
        printer.Print("dbgShimDirectory = "); DbgShimDirectory.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DotNetDebuggerSessionModel : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public DebuggerStartInfoBase StartInfo {get; private set;}
    [NotNull] public JetBrains.Rider.Model.DebuggerWorker.SessionProperties SessionProperties {get; private set;}
    [NotNull] public ISink<RdVoid> Initialized { get { return _Initialized; }}
    [NotNull] public ISource<DebuggerSessionInfo> TargetReady { get { return _TargetReady; }}
    [NotNull] public IViewableMap<BreakpointId, BreakpointModel> Breakpoints { get { return _Breakpoints; }}
    [NotNull] public IViewableMap<string, string> RemappingLocationRoots { get { return _RemappingLocationRoots; }}
    [NotNull] public IRdProperty<ExecutionContextProxy> ExecutionContext { get { return _ExecutionContext; }}
    [NotNull] public IRdProperty<bool> InitializedDebugConnection { get { return _InitializedDebugConnection; }}
    [NotNull] public IAsyncSink<ExecutionCommandBase> Command { get { return _Command; }}
    [NotNull] public ISource<OutputMessage> TargetOutput { get { return _TargetOutput; }}
    [NotNull] public ISource<OutputMessage> TargetDebug { get { return _TargetDebug; }}
    [NotNull] public ISource<OutputMessage> DebuggerOutput { get { return _DebuggerOutput; }}
    [NotNull] public ISource<JetBrains.Rider.Model.DebuggerWorker.ShowNotification> ShowNotification { get { return _ShowNotification; }}
    [NotNull] public IRdProperty<int> Pid { get { return _Pid; }}
    [NotNull] public IRdProperty<bool> IgnoreMissingSourceLocations { get { return _IgnoreMissingSourceLocations; }}
    [NotNull] public IRdProperty<int> ActiveThread { get { return _ActiveThread; }}
    [NotNull] public IRdCall<string, string> TranslateFilePath { get { return _TranslateFilePath; }}
    [NotNull] public IRdCall<BreakpointId, RdVoid> NotifyDisabledBreakpointHit { get { return _NotifyDisabledBreakpointHit; }}
    
    //private fields
    [NotNull] private readonly RdSignal<RdVoid> _Initialized;
    [NotNull] private readonly RdSignal<DebuggerSessionInfo> _TargetReady;
    [NotNull] private readonly RdMap<BreakpointId, BreakpointModel> _Breakpoints;
    [NotNull] private readonly RdMap<string, string> _RemappingLocationRoots;
    [NotNull] private readonly RdProperty<ExecutionContextProxy> _ExecutionContext;
    [NotNull] private readonly RdProperty<bool> _InitializedDebugConnection;
    [NotNull] private readonly RdSignal<ExecutionCommandBase> _Command;
    [NotNull] private readonly RdSignal<OutputMessage> _TargetOutput;
    [NotNull] private readonly RdSignal<OutputMessage> _TargetDebug;
    [NotNull] private readonly RdSignal<OutputMessage> _DebuggerOutput;
    [NotNull] private readonly RdSignal<JetBrains.Rider.Model.DebuggerWorker.ShowNotification> _ShowNotification;
    [NotNull] private readonly RdProperty<int> _Pid;
    [NotNull] private readonly RdProperty<bool> _IgnoreMissingSourceLocations;
    [NotNull] private readonly RdProperty<int> _ActiveThread;
    [NotNull] private readonly RdCall<string, string> _TranslateFilePath;
    [NotNull] private readonly RdCall<BreakpointId, RdVoid> _NotifyDisabledBreakpointHit;
    
    //primary constructor
    private DotNetDebuggerSessionModel(
      [NotNull] DebuggerStartInfoBase startInfo,
      [NotNull] JetBrains.Rider.Model.DebuggerWorker.SessionProperties sessionProperties,
      [NotNull] RdSignal<RdVoid> initialized,
      [NotNull] RdSignal<DebuggerSessionInfo> targetReady,
      [NotNull] RdMap<BreakpointId, BreakpointModel> breakpoints,
      [NotNull] RdMap<string, string> remappingLocationRoots,
      [NotNull] RdProperty<ExecutionContextProxy> executionContext,
      [NotNull] RdProperty<bool> initializedDebugConnection,
      [NotNull] RdSignal<ExecutionCommandBase> command,
      [NotNull] RdSignal<OutputMessage> targetOutput,
      [NotNull] RdSignal<OutputMessage> targetDebug,
      [NotNull] RdSignal<OutputMessage> debuggerOutput,
      [NotNull] RdSignal<JetBrains.Rider.Model.DebuggerWorker.ShowNotification> showNotification,
      [NotNull] RdProperty<int> pid,
      [NotNull] RdProperty<bool> ignoreMissingSourceLocations,
      [NotNull] RdProperty<int> activeThread,
      [NotNull] RdCall<string, string> translateFilePath,
      [NotNull] RdCall<BreakpointId, RdVoid> notifyDisabledBreakpointHit
    )
    {
      if (startInfo == null) throw new ArgumentNullException("startInfo");
      if (sessionProperties == null) throw new ArgumentNullException("sessionProperties");
      if (initialized == null) throw new ArgumentNullException("initialized");
      if (targetReady == null) throw new ArgumentNullException("targetReady");
      if (breakpoints == null) throw new ArgumentNullException("breakpoints");
      if (remappingLocationRoots == null) throw new ArgumentNullException("remappingLocationRoots");
      if (executionContext == null) throw new ArgumentNullException("executionContext");
      if (initializedDebugConnection == null) throw new ArgumentNullException("initializedDebugConnection");
      if (command == null) throw new ArgumentNullException("command");
      if (targetOutput == null) throw new ArgumentNullException("targetOutput");
      if (targetDebug == null) throw new ArgumentNullException("targetDebug");
      if (debuggerOutput == null) throw new ArgumentNullException("debuggerOutput");
      if (showNotification == null) throw new ArgumentNullException("showNotification");
      if (pid == null) throw new ArgumentNullException("pid");
      if (ignoreMissingSourceLocations == null) throw new ArgumentNullException("ignoreMissingSourceLocations");
      if (activeThread == null) throw new ArgumentNullException("activeThread");
      if (translateFilePath == null) throw new ArgumentNullException("translateFilePath");
      if (notifyDisabledBreakpointHit == null) throw new ArgumentNullException("notifyDisabledBreakpointHit");
      
      StartInfo = startInfo;
      SessionProperties = sessionProperties;
      _Initialized = initialized;
      _TargetReady = targetReady;
      _Breakpoints = breakpoints;
      _RemappingLocationRoots = remappingLocationRoots;
      _ExecutionContext = executionContext;
      _InitializedDebugConnection = initializedDebugConnection;
      _Command = command;
      _TargetOutput = targetOutput;
      _TargetDebug = targetDebug;
      _DebuggerOutput = debuggerOutput;
      _ShowNotification = showNotification;
      _Pid = pid;
      _IgnoreMissingSourceLocations = ignoreMissingSourceLocations;
      _ActiveThread = activeThread;
      _TranslateFilePath = translateFilePath;
      _NotifyDisabledBreakpointHit = notifyDisabledBreakpointHit;
      _RemappingLocationRoots.OptimizeNested = true;
      _InitializedDebugConnection.OptimizeNested = true;
      _Pid.OptimizeNested = true;
      _IgnoreMissingSourceLocations.OptimizeNested = true;
      _ActiveThread.OptimizeNested = true;
      _Command.Async = true;
      _TranslateFilePath.Async = true;
      _NotifyDisabledBreakpointHit.Async = true;
      _ExecutionContext.ValueCanBeNull = true;
      _TranslateFilePath.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("sessionProperties", SessionProperties));
      BindableChildren.Add(new KeyValuePair<string, object>("initialized", _Initialized));
      BindableChildren.Add(new KeyValuePair<string, object>("targetReady", _TargetReady));
      BindableChildren.Add(new KeyValuePair<string, object>("breakpoints", _Breakpoints));
      BindableChildren.Add(new KeyValuePair<string, object>("remappingLocationRoots", _RemappingLocationRoots));
      BindableChildren.Add(new KeyValuePair<string, object>("executionContext", _ExecutionContext));
      BindableChildren.Add(new KeyValuePair<string, object>("initializedDebugConnection", _InitializedDebugConnection));
      BindableChildren.Add(new KeyValuePair<string, object>("command", _Command));
      BindableChildren.Add(new KeyValuePair<string, object>("targetOutput", _TargetOutput));
      BindableChildren.Add(new KeyValuePair<string, object>("targetDebug", _TargetDebug));
      BindableChildren.Add(new KeyValuePair<string, object>("debuggerOutput", _DebuggerOutput));
      BindableChildren.Add(new KeyValuePair<string, object>("showNotification", _ShowNotification));
      BindableChildren.Add(new KeyValuePair<string, object>("pid", _Pid));
      BindableChildren.Add(new KeyValuePair<string, object>("ignoreMissingSourceLocations", _IgnoreMissingSourceLocations));
      BindableChildren.Add(new KeyValuePair<string, object>("activeThread", _ActiveThread));
      BindableChildren.Add(new KeyValuePair<string, object>("translateFilePath", _TranslateFilePath));
      BindableChildren.Add(new KeyValuePair<string, object>("notifyDisabledBreakpointHit", _NotifyDisabledBreakpointHit));
    }
    //secondary constructor
    public DotNetDebuggerSessionModel (
      [NotNull] DebuggerStartInfoBase startInfo
    ) : this (
      startInfo,
      new JetBrains.Rider.Model.DebuggerWorker.SessionProperties(),
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdSignal<DebuggerSessionInfo>(DebuggerSessionInfo.Read, DebuggerSessionInfo.Write),
      new RdMap<BreakpointId, BreakpointModel>(BreakpointId.Read, BreakpointId.Write, BreakpointModel.Read, BreakpointModel.Write),
      new RdMap<string, string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<ExecutionContextProxy>(ReadExecutionContextProxyNullable, WriteExecutionContextProxyNullable),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<ExecutionCommandBase>(ExecutionCommandBase.Read, ExecutionCommandBase.Write),
      new RdSignal<OutputMessage>(OutputMessage.Read, OutputMessage.Write),
      new RdSignal<OutputMessage>(OutputMessage.Read, OutputMessage.Write),
      new RdSignal<OutputMessage>(OutputMessage.Read, OutputMessage.Write),
      new RdSignal<JetBrains.Rider.Model.DebuggerWorker.ShowNotification>(JetBrains.Rider.Model.DebuggerWorker.ShowNotification.Read, JetBrains.Rider.Model.DebuggerWorker.ShowNotification.Write),
      new RdProperty<int>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<int>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt),
      new RdCall<string, string>(ReadStringNullable, WriteStringNullable, ReadStringNullable, WriteStringNullable),
      new RdCall<BreakpointId, RdVoid>(BreakpointId.Read, BreakpointId.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid)
    ) {}
    //statics
    
    public static CtxReadDelegate<DotNetDebuggerSessionModel> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var startInfo = DebuggerStartInfoBase.Read(ctx, reader);
      var sessionProperties = JetBrains.Rider.Model.DebuggerWorker.SessionProperties.Read(ctx, reader);
      var initialized = RdSignal<RdVoid>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var targetReady = RdSignal<DebuggerSessionInfo>.Read(ctx, reader, DebuggerSessionInfo.Read, DebuggerSessionInfo.Write);
      var breakpoints = RdMap<BreakpointId, BreakpointModel>.Read(ctx, reader, BreakpointId.Read, BreakpointId.Write, BreakpointModel.Read, BreakpointModel.Write);
      var remappingLocationRoots = RdMap<string, string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var executionContext = RdProperty<ExecutionContextProxy>.Read(ctx, reader, ReadExecutionContextProxyNullable, WriteExecutionContextProxyNullable);
      var initializedDebugConnection = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var command = RdSignal<ExecutionCommandBase>.Read(ctx, reader, ExecutionCommandBase.Read, ExecutionCommandBase.Write);
      var targetOutput = RdSignal<OutputMessage>.Read(ctx, reader, OutputMessage.Read, OutputMessage.Write);
      var targetDebug = RdSignal<OutputMessage>.Read(ctx, reader, OutputMessage.Read, OutputMessage.Write);
      var debuggerOutput = RdSignal<OutputMessage>.Read(ctx, reader, OutputMessage.Read, OutputMessage.Write);
      var showNotification = RdSignal<JetBrains.Rider.Model.DebuggerWorker.ShowNotification>.Read(ctx, reader, JetBrains.Rider.Model.DebuggerWorker.ShowNotification.Read, JetBrains.Rider.Model.DebuggerWorker.ShowNotification.Write);
      var pid = RdProperty<int>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt);
      var ignoreMissingSourceLocations = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var activeThread = RdProperty<int>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt);
      var translateFilePath = RdCall<string, string>.Read(ctx, reader, ReadStringNullable, WriteStringNullable, ReadStringNullable, WriteStringNullable);
      var notifyDisabledBreakpointHit = RdCall<BreakpointId, RdVoid>.Read(ctx, reader, BreakpointId.Read, BreakpointId.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      return new DotNetDebuggerSessionModel(startInfo, sessionProperties, initialized, targetReady, breakpoints, remappingLocationRoots, executionContext, initializedDebugConnection, command, targetOutput, targetDebug, debuggerOutput, showNotification, pid, ignoreMissingSourceLocations, activeThread, translateFilePath, notifyDisabledBreakpointHit).WithId(_id);
    };
    public static CtxReadDelegate<ExecutionContextProxy> ReadExecutionContextProxyNullable = ExecutionContextProxy.Read.NullableClass();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<DotNetDebuggerSessionModel> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      DebuggerStartInfoBase.Write(ctx, writer, value.StartInfo);
      JetBrains.Rider.Model.DebuggerWorker.SessionProperties.Write(ctx, writer, value.SessionProperties);
      RdSignal<RdVoid>.Write(ctx, writer, value._Initialized);
      RdSignal<DebuggerSessionInfo>.Write(ctx, writer, value._TargetReady);
      RdMap<BreakpointId, BreakpointModel>.Write(ctx, writer, value._Breakpoints);
      RdMap<string, string>.Write(ctx, writer, value._RemappingLocationRoots);
      RdProperty<ExecutionContextProxy>.Write(ctx, writer, value._ExecutionContext);
      RdProperty<bool>.Write(ctx, writer, value._InitializedDebugConnection);
      RdSignal<ExecutionCommandBase>.Write(ctx, writer, value._Command);
      RdSignal<OutputMessage>.Write(ctx, writer, value._TargetOutput);
      RdSignal<OutputMessage>.Write(ctx, writer, value._TargetDebug);
      RdSignal<OutputMessage>.Write(ctx, writer, value._DebuggerOutput);
      RdSignal<JetBrains.Rider.Model.DebuggerWorker.ShowNotification>.Write(ctx, writer, value._ShowNotification);
      RdProperty<int>.Write(ctx, writer, value._Pid);
      RdProperty<bool>.Write(ctx, writer, value._IgnoreMissingSourceLocations);
      RdProperty<int>.Write(ctx, writer, value._ActiveThread);
      RdCall<string, string>.Write(ctx, writer, value._TranslateFilePath);
      RdCall<BreakpointId, RdVoid>.Write(ctx, writer, value._NotifyDisabledBreakpointHit);
    };
    public static CtxWriteDelegate<ExecutionContextProxy> WriteExecutionContextProxyNullable = ExecutionContextProxy.Write.NullableClass();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DotNetDebuggerSessionModel (");
      using (printer.IndentCookie()) {
        printer.Print("startInfo = "); StartInfo.PrintEx(printer); printer.Println();
        printer.Print("sessionProperties = "); SessionProperties.PrintEx(printer); printer.Println();
        printer.Print("initialized = "); _Initialized.PrintEx(printer); printer.Println();
        printer.Print("targetReady = "); _TargetReady.PrintEx(printer); printer.Println();
        printer.Print("breakpoints = "); _Breakpoints.PrintEx(printer); printer.Println();
        printer.Print("remappingLocationRoots = "); _RemappingLocationRoots.PrintEx(printer); printer.Println();
        printer.Print("executionContext = "); _ExecutionContext.PrintEx(printer); printer.Println();
        printer.Print("initializedDebugConnection = "); _InitializedDebugConnection.PrintEx(printer); printer.Println();
        printer.Print("command = "); _Command.PrintEx(printer); printer.Println();
        printer.Print("targetOutput = "); _TargetOutput.PrintEx(printer); printer.Println();
        printer.Print("targetDebug = "); _TargetDebug.PrintEx(printer); printer.Println();
        printer.Print("debuggerOutput = "); _DebuggerOutput.PrintEx(printer); printer.Println();
        printer.Print("showNotification = "); _ShowNotification.PrintEx(printer); printer.Println();
        printer.Print("pid = "); _Pid.PrintEx(printer); printer.Println();
        printer.Print("ignoreMissingSourceLocations = "); _IgnoreMissingSourceLocations.PrintEx(printer); printer.Println();
        printer.Print("activeThread = "); _ActiveThread.PrintEx(printer); printer.Println();
        printer.Print("translateFilePath = "); _TranslateFilePath.PrintEx(printer); printer.Println();
        printer.Print("notifyDisabledBreakpointHit = "); _NotifyDisabledBreakpointHit.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DumpDotNetCoreStartInfo : DumpStartInfo
  {
    //fields
    //public fields
    [NotNull] public JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo DotNetCoreInfo {get; private set;}
    
    //private fields
    //primary constructor
    public DumpDotNetCoreStartInfo(
      [NotNull] JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo dotNetCoreInfo,
      [NotNull] string dumpFilePath,
      [NotNull] List<string> symbols,
      bool needToBeInitializedImmediately
    ) : base (
      dumpFilePath,
      symbols,
      needToBeInitializedImmediately
     ) 
    {
      if (dotNetCoreInfo == null) throw new ArgumentNullException("dotNetCoreInfo");
      
      DotNetCoreInfo = dotNetCoreInfo;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DumpDotNetCoreStartInfo> Read = (ctx, reader) => 
    {
      var dumpFilePath = reader.ReadString();
      var symbols = ReadStringList(ctx, reader);
      var needToBeInitializedImmediately = reader.ReadBool();
      var dotNetCoreInfo = JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo.Read(ctx, reader);
      return new DumpDotNetCoreStartInfo(dotNetCoreInfo, dumpFilePath, symbols, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static new CtxWriteDelegate<DumpDotNetCoreStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.DumpFilePath);
      WriteStringList(ctx, writer, value.Symbols);
      writer.Write(value.NeedToBeInitializedImmediately);
      JetBrains.Rider.Model.DebuggerWorker.DotNetCoreInfo.Write(ctx, writer, value.DotNetCoreInfo);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DumpDotNetCoreStartInfo) obj);
    }
    public bool Equals(DumpDotNetCoreStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(DotNetCoreInfo, other.DotNetCoreInfo) && DumpFilePath == other.DumpFilePath && Symbols.SequenceEqual(other.Symbols) && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DotNetCoreInfo.GetHashCode();
        hash = hash * 31 + DumpFilePath.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Symbols);
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DumpDotNetCoreStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("dotNetCoreInfo = "); DotNetCoreInfo.PrintEx(printer); printer.Println();
        printer.Print("dumpFilePath = "); DumpFilePath.PrintEx(printer); printer.Println();
        printer.Print("symbols = "); Symbols.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DumpMsDotNetStartInfo : DumpStartInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DumpMsDotNetStartInfo(
      [NotNull] string dumpFilePath,
      [NotNull] List<string> symbols,
      bool needToBeInitializedImmediately
    ) : base (
      dumpFilePath,
      symbols,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DumpMsDotNetStartInfo> Read = (ctx, reader) => 
    {
      var dumpFilePath = reader.ReadString();
      var symbols = ReadStringList(ctx, reader);
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DumpMsDotNetStartInfo(dumpFilePath, symbols, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static new CtxWriteDelegate<DumpMsDotNetStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.DumpFilePath);
      WriteStringList(ctx, writer, value.Symbols);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DumpMsDotNetStartInfo) obj);
    }
    public bool Equals(DumpMsDotNetStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return DumpFilePath == other.DumpFilePath && Symbols.SequenceEqual(other.Symbols) && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DumpFilePath.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Symbols);
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DumpMsDotNetStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("dumpFilePath = "); DumpFilePath.PrintEx(printer); printer.Println();
        printer.Print("symbols = "); Symbols.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class DumpStartInfo : DebuggerStartInfoBase
  {
    //fields
    //public fields
    [NotNull] public string DumpFilePath {get; private set;}
    [NotNull] public List<string> Symbols {get; private set;}
    
    //private fields
    //primary constructor
    protected DumpStartInfo(
      [NotNull] string dumpFilePath,
      [NotNull] List<string> symbols,
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
      if (dumpFilePath == null) throw new ArgumentNullException("dumpFilePath");
      if (symbols == null) throw new ArgumentNullException("symbols");
      
      DumpFilePath = dumpFilePath;
      Symbols = symbols;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DumpStartInfo> Read = Polymorphic<DumpStartInfo>.ReadAbstract(DumpStartInfo_Unknown.Read);
    
    public static new CtxWriteDelegate<DumpStartInfo> Write = Polymorphic<DumpStartInfo>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class DumpStartInfo_Unknown : DumpStartInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public DumpStartInfo_Unknown(
      [NotNull] string dumpFilePath,
      [NotNull] List<string> symbols,
      bool needToBeInitializedImmediately
    ) : base (
      dumpFilePath,
      symbols,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DumpStartInfo_Unknown> Read = (ctx, reader) => 
    {
      var dumpFilePath = reader.ReadString();
      var symbols = ReadStringList(ctx, reader);
      var needToBeInitializedImmediately = reader.ReadBool();
      return new DumpStartInfo_Unknown(dumpFilePath, symbols, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static new CtxWriteDelegate<DumpStartInfo_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.DumpFilePath);
      WriteStringList(ctx, writer, value.Symbols);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DumpStartInfo_Unknown) obj);
    }
    public bool Equals(DumpStartInfo_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return DumpFilePath == other.DumpFilePath && Symbols.SequenceEqual(other.Symbols) && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DumpFilePath.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Symbols);
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DumpStartInfo_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("dumpFilePath = "); DumpFilePath.PrintEx(printer); printer.Println();
        printer.Print("symbols = "); Symbols.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class EvalPair : IPrintable, IEquatable<EvalPair>
  {
    //fields
    //public fields
    [NotNull] public string Expression {get; private set;}
    [NotNull] public ObjectProxy Value {get; private set;}
    
    //private fields
    //primary constructor
    public EvalPair(
      [NotNull] string expression,
      [NotNull] ObjectProxy value
    )
    {
      if (expression == null) throw new ArgumentNullException("expression");
      if (value == null) throw new ArgumentNullException("value");
      
      Expression = expression;
      Value = value;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<EvalPair> Read = (ctx, reader) => 
    {
      var expression = reader.ReadString();
      var value = ObjectProxy.Read(ctx, reader);
      return new EvalPair(expression, value);
    };
    
    public static CtxWriteDelegate<EvalPair> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Expression);
      ObjectProxy.Write(ctx, writer, value.Value);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((EvalPair) obj);
    }
    public bool Equals(EvalPair other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Expression == other.Expression && Equals(Value, other.Value);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Expression.GetHashCode();
        hash = hash * 31 + Value.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("EvalPair (");
      using (printer.IndentCookie()) {
        printer.Print("expression = "); Expression.PrintEx(printer); printer.Println();
        printer.Print("value = "); Value.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class EvaluateArg : IPrintable, IEquatable<EvaluateArg>
  {
    //fields
    //public fields
    public int FrameId {get; private set;}
    [NotNull] public List<string> Expressions {get; private set;}
    public int NumberOfExpressionsBefore {get; private set;}
    public bool EvaluationInEditor {get; private set;}
    [NotNull] public List<string> NamespaceImports {get; private set;}
    
    //private fields
    //primary constructor
    public EvaluateArg(
      int frameId,
      [NotNull] List<string> expressions,
      int numberOfExpressionsBefore,
      bool evaluationInEditor,
      [NotNull] List<string> namespaceImports
    )
    {
      if (expressions == null) throw new ArgumentNullException("expressions");
      if (namespaceImports == null) throw new ArgumentNullException("namespaceImports");
      
      FrameId = frameId;
      Expressions = expressions;
      NumberOfExpressionsBefore = numberOfExpressionsBefore;
      EvaluationInEditor = evaluationInEditor;
      NamespaceImports = namespaceImports;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<EvaluateArg> Read = (ctx, reader) => 
    {
      var frameId = reader.ReadInt();
      var expressions = ReadStringList(ctx, reader);
      var numberOfExpressionsBefore = reader.ReadInt();
      var evaluationInEditor = reader.ReadBool();
      var namespaceImports = ReadStringList(ctx, reader);
      return new EvaluateArg(frameId, expressions, numberOfExpressionsBefore, evaluationInEditor, namespaceImports);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<EvaluateArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.FrameId);
      WriteStringList(ctx, writer, value.Expressions);
      writer.Write(value.NumberOfExpressionsBefore);
      writer.Write(value.EvaluationInEditor);
      WriteStringList(ctx, writer, value.NamespaceImports);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((EvaluateArg) obj);
    }
    public bool Equals(EvaluateArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return FrameId == other.FrameId && Expressions.SequenceEqual(other.Expressions) && NumberOfExpressionsBefore == other.NumberOfExpressionsBefore && EvaluationInEditor == other.EvaluationInEditor && NamespaceImports.SequenceEqual(other.NamespaceImports);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + FrameId.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Expressions);
        hash = hash * 31 + NumberOfExpressionsBefore.GetHashCode();
        hash = hash * 31 + EvaluationInEditor.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(NamespaceImports);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("EvaluateArg (");
      using (printer.IndentCookie()) {
        printer.Print("frameId = "); FrameId.PrintEx(printer); printer.Println();
        printer.Print("expressions = "); Expressions.PrintEx(printer); printer.Println();
        printer.Print("numberOfExpressionsBefore = "); NumberOfExpressionsBefore.PrintEx(printer); printer.Println();
        printer.Print("evaluationInEditor = "); EvaluationInEditor.PrintEx(printer); printer.Println();
        printer.Print("namespaceImports = "); NamespaceImports.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class EvaluateResult : IPrintable, IEquatable<EvaluateResult>
  {
    //fields
    //public fields
    public int FrameId {get; private set;}
    [NotNull] public List<EvalPair> Expressions {get; private set;}
    
    //private fields
    //primary constructor
    public EvaluateResult(
      int frameId,
      [NotNull] List<EvalPair> expressions
    )
    {
      if (expressions == null) throw new ArgumentNullException("expressions");
      
      FrameId = frameId;
      Expressions = expressions;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<EvaluateResult> Read = (ctx, reader) => 
    {
      var frameId = reader.ReadInt();
      var expressions = ReadEvalPairList(ctx, reader);
      return new EvaluateResult(frameId, expressions);
    };
    public static CtxReadDelegate<List<EvalPair>> ReadEvalPairList = EvalPair.Read.List();
    
    public static CtxWriteDelegate<EvaluateResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.FrameId);
      WriteEvalPairList(ctx, writer, value.Expressions);
    };
    public static CtxWriteDelegate<List<EvalPair>> WriteEvalPairList = EvalPair.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((EvaluateResult) obj);
    }
    public bool Equals(EvaluateResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return FrameId == other.FrameId && Expressions.SequenceEqual(other.Expressions);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + FrameId.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Expressions);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("EvaluateResult (");
      using (printer.IndentCookie()) {
        printer.Print("frameId = "); FrameId.PrintEx(printer); printer.Println();
        printer.Print("expressions = "); Expressions.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class EvaluationExpressionModel : IPrintable, IEquatable<EvaluationExpressionModel>
  {
    //fields
    //public fields
    [NotNull] public string Expression {get; private set;}
    [NotNull] public List<string> ImportsList {get; private set;}
    
    //private fields
    //primary constructor
    public EvaluationExpressionModel(
      [NotNull] string expression,
      [NotNull] List<string> importsList
    )
    {
      if (expression == null) throw new ArgumentNullException("expression");
      if (importsList == null) throw new ArgumentNullException("importsList");
      
      Expression = expression;
      ImportsList = importsList;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<EvaluationExpressionModel> Read = (ctx, reader) => 
    {
      var expression = reader.ReadString();
      var importsList = ReadStringList(ctx, reader);
      return new EvaluationExpressionModel(expression, importsList);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<EvaluationExpressionModel> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Expression);
      WriteStringList(ctx, writer, value.ImportsList);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((EvaluationExpressionModel) obj);
    }
    public bool Equals(EvaluationExpressionModel other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Expression == other.Expression && ImportsList.SequenceEqual(other.ImportsList);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Expression.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(ImportsList);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("EvaluationExpressionModel (");
      using (printer.IndentCookie()) {
        printer.Print("expression = "); Expression.PrintEx(printer); printer.Println();
        printer.Print("importsList = "); ImportsList.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ExceptionBreakpointModel : BreakpointModel
  {
    //fields
    //public fields
    [NotNull] public string Exception {get; private set;}
    public bool JustMyCode {get; private set;}
    
    //private fields
    //primary constructor
    private ExceptionBreakpointModel(
      [NotNull] string exception,
      bool justMyCode,
      [NotNull] BreakpointId id,
      [NotNull] RdProperty<bool> shouldNotifyHitWhenDisabled,
      [NotNull] RdSignal<BreakpointStatus> statusChanged,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : base (
      id,
      shouldNotifyHitWhenDisabled,
      statusChanged,
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
     ) 
    {
      if (exception == null) throw new ArgumentNullException("exception");
      
      Exception = exception;
      JustMyCode = justMyCode;
    }
    //secondary constructor
    public ExceptionBreakpointModel (
      [NotNull] string exception,
      bool justMyCode,
      [NotNull] BreakpointId id,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : this (
      exception,
      justMyCode,
      id,
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<BreakpointStatus>(BreakpointStatus.Read, BreakpointStatus.Write),
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
    ) {}
    //statics
    
    public static new CtxReadDelegate<ExceptionBreakpointModel> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var id = BreakpointId.Read(ctx, reader);
      var shouldNotifyHitWhenDisabled = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var statusChanged = RdSignal<BreakpointStatus>.Read(ctx, reader, BreakpointStatus.Read, BreakpointStatus.Write);
      var condition = ReadEvaluationExpressionModelNullable(ctx, reader);
      var suspended = reader.ReadBool();
      var traceExpression = ReadEvaluationExpressionModelNullable(ctx, reader);
      var traceBreakpoint = reader.ReadBool();
      var hitCount = reader.ReadInt();
      var hitCountMode = (BreakpointHitCountModeModel)reader.ReadInt();
      var exception = reader.ReadString();
      var justMyCode = reader.ReadBool();
      return new ExceptionBreakpointModel(exception, justMyCode, id, shouldNotifyHitWhenDisabled, statusChanged, condition, suspended, traceExpression, traceBreakpoint, hitCount, hitCountMode).WithId(_id);
    };
    public static CtxReadDelegate<EvaluationExpressionModel> ReadEvaluationExpressionModelNullable = EvaluationExpressionModel.Read.NullableClass();
    
    public static new CtxWriteDelegate<ExceptionBreakpointModel> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      BreakpointId.Write(ctx, writer, value.Id);
      RdProperty<bool>.Write(ctx, writer, value._ShouldNotifyHitWhenDisabled);
      RdSignal<BreakpointStatus>.Write(ctx, writer, value._StatusChanged);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.Condition);
      writer.Write(value.Suspended);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.TraceExpression);
      writer.Write(value.TraceBreakpoint);
      writer.Write(value.HitCount);
      writer.Write((int)value.HitCountMode);
      writer.Write(value.Exception);
      writer.Write(value.JustMyCode);
    };
    public static CtxWriteDelegate<EvaluationExpressionModel> WriteEvaluationExpressionModelNullable = EvaluationExpressionModel.Write.NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("ExceptionBreakpointModel (");
      using (printer.IndentCookie()) {
        printer.Print("exception = "); Exception.PrintEx(printer); printer.Println();
        printer.Print("justMyCode = "); JustMyCode.PrintEx(printer); printer.Println();
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("shouldNotifyHitWhenDisabled = "); _ShouldNotifyHitWhenDisabled.PrintEx(printer); printer.Println();
        printer.Print("statusChanged = "); _StatusChanged.PrintEx(printer); printer.Println();
        printer.Print("condition = "); Condition.PrintEx(printer); printer.Println();
        printer.Print("suspended = "); Suspended.PrintEx(printer); printer.Println();
        printer.Print("traceExpression = "); TraceExpression.PrintEx(printer); printer.Println();
        printer.Print("traceBreakpoint = "); TraceBreakpoint.PrintEx(printer); printer.Println();
        printer.Print("hitCount = "); HitCount.PrintEx(printer); printer.Println();
        printer.Print("hitCountMode = "); HitCountMode.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ExceptionModel : ExceptionModelBase
  {
    //fields
    //public fields
    [NotNull] public string Message {get; private set;}
    [NotNull] public string Type {get; private set;}
    [NotNull] public string FullInfo {get; private set;}
    public ExceptionModelType ExceptionType {get; private set;}
    [CanBeNull] public ExceptionModelBase InnerException {get; private set;}
    [NotNull] public List<ExceptionModelBase> InnerExceptions {get; private set;}
    
    //private fields
    //primary constructor
    public ExceptionModel(
      [NotNull] string message,
      [NotNull] string type,
      [NotNull] string fullInfo,
      ExceptionModelType exceptionType,
      [CanBeNull] ExceptionModelBase innerException,
      [NotNull] List<ExceptionModelBase> innerExceptions
    )
    {
      if (message == null) throw new ArgumentNullException("message");
      if (type == null) throw new ArgumentNullException("type");
      if (fullInfo == null) throw new ArgumentNullException("fullInfo");
      if (innerExceptions == null) throw new ArgumentNullException("innerExceptions");
      
      Message = message;
      Type = type;
      FullInfo = fullInfo;
      ExceptionType = exceptionType;
      InnerException = innerException;
      InnerExceptions = innerExceptions;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExceptionModel> Read = (ctx, reader) => 
    {
      var message = reader.ReadString();
      var type = reader.ReadString();
      var fullInfo = reader.ReadString();
      var exceptionType = (ExceptionModelType)reader.ReadInt();
      var innerException = ReadExceptionModelBaseNullable(ctx, reader);
      var innerExceptions = ReadExceptionModelBaseList(ctx, reader);
      return new ExceptionModel(message, type, fullInfo, exceptionType, innerException, innerExceptions);
    };
    public static CtxReadDelegate<ExceptionModelBase> ReadExceptionModelBaseNullable = ExceptionModelBase.Read.NullableClass();
    public static CtxReadDelegate<List<ExceptionModelBase>> ReadExceptionModelBaseList = ExceptionModelBase.Read.List();
    
    public static new CtxWriteDelegate<ExceptionModel> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Message);
      writer.Write(value.Type);
      writer.Write(value.FullInfo);
      writer.Write((int)value.ExceptionType);
      WriteExceptionModelBaseNullable(ctx, writer, value.InnerException);
      WriteExceptionModelBaseList(ctx, writer, value.InnerExceptions);
    };
    public static CtxWriteDelegate<ExceptionModelBase> WriteExceptionModelBaseNullable = ExceptionModelBase.Write.NullableClass();
    public static CtxWriteDelegate<List<ExceptionModelBase>> WriteExceptionModelBaseList = ExceptionModelBase.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExceptionModel) obj);
    }
    public bool Equals(ExceptionModel other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Message == other.Message && Type == other.Type && FullInfo == other.FullInfo && ExceptionType == other.ExceptionType && Equals(InnerException, other.InnerException) && InnerExceptions.SequenceEqual(other.InnerExceptions);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Message.GetHashCode();
        hash = hash * 31 + Type.GetHashCode();
        hash = hash * 31 + FullInfo.GetHashCode();
        hash = hash * 31 + (int) ExceptionType;
        hash = hash * 31 + (InnerException != null ?InnerException.GetHashCode() : 0);
        hash = hash * 31 + Collections.GetHashCode(InnerExceptions);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExceptionModel (");
      using (printer.IndentCookie()) {
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
        printer.Print("fullInfo = "); FullInfo.PrintEx(printer); printer.Println();
        printer.Print("exceptionType = "); ExceptionType.PrintEx(printer); printer.Println();
        printer.Print("innerException = "); InnerException.PrintEx(printer); printer.Println();
        printer.Print("innerExceptions = "); InnerExceptions.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class ExceptionModelBase{
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ExceptionModelBase> Read = Polymorphic<ExceptionModelBase>.ReadAbstract(ExceptionModelBase_Unknown.Read);
    
    public static CtxWriteDelegate<ExceptionModelBase> Write = Polymorphic<ExceptionModelBase>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class ExceptionModelBase_Unknown : ExceptionModelBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExceptionModelBase_Unknown> Read = (ctx, reader) => 
    {
      return new ExceptionModelBase_Unknown();
    };
    
    public static new CtxWriteDelegate<ExceptionModelBase_Unknown> Write = (ctx, writer, value) => 
    {
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExceptionModelBase_Unknown) obj);
    }
    public bool Equals(ExceptionModelBase_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return true;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExceptionModelBase_Unknown (");
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum ExceptionModelType {
    Unknown,
    Handled,
    Unhandled,
    UserUnhandled
  }
  
  
  public abstract class ExeStartInfo : DebuggerStartInfoBase
  {
    //fields
    //public fields
    [NotNull] public string ExePath {get; private set;}
    [NotNull] public string WorkingDirectory {get; private set;}
    [NotNull] public string Arguments {get; private set;}
    [NotNull] public List<StringPair> EnvironmentVariables {get; private set;}
    [CanBeNull] public string RuntimeArguments {get; private set;}
    public bool ExecuteAsIs {get; private set;}
    public bool UseExternalConsole {get; private set;}
    
    //private fields
    //primary constructor
    protected ExeStartInfo(
      [NotNull] string exePath,
      [NotNull] string workingDirectory,
      [NotNull] string arguments,
      [NotNull] List<StringPair> environmentVariables,
      [CanBeNull] string runtimeArguments,
      bool executeAsIs,
      bool useExternalConsole,
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
      if (exePath == null) throw new ArgumentNullException("exePath");
      if (workingDirectory == null) throw new ArgumentNullException("workingDirectory");
      if (arguments == null) throw new ArgumentNullException("arguments");
      if (environmentVariables == null) throw new ArgumentNullException("environmentVariables");
      
      ExePath = exePath;
      WorkingDirectory = workingDirectory;
      Arguments = arguments;
      EnvironmentVariables = environmentVariables;
      RuntimeArguments = runtimeArguments;
      ExecuteAsIs = executeAsIs;
      UseExternalConsole = useExternalConsole;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExeStartInfo> Read = Polymorphic<ExeStartInfo>.ReadAbstract(ExeStartInfo_Unknown.Read);
    
    public static new CtxWriteDelegate<ExeStartInfo> Write = Polymorphic<ExeStartInfo>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class ExeStartInfo_Unknown : ExeStartInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public ExeStartInfo_Unknown(
      [NotNull] string exePath,
      [NotNull] string workingDirectory,
      [NotNull] string arguments,
      [NotNull] List<StringPair> environmentVariables,
      [CanBeNull] string runtimeArguments,
      bool executeAsIs,
      bool useExternalConsole,
      bool needToBeInitializedImmediately
    ) : base (
      exePath,
      workingDirectory,
      arguments,
      environmentVariables,
      runtimeArguments,
      executeAsIs,
      useExternalConsole,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExeStartInfo_Unknown> Read = (ctx, reader) => 
    {
      var exePath = reader.ReadString();
      var workingDirectory = reader.ReadString();
      var arguments = reader.ReadString();
      var environmentVariables = ReadStringPairList(ctx, reader);
      var runtimeArguments = ReadStringNullable(ctx, reader);
      var executeAsIs = reader.ReadBool();
      var useExternalConsole = reader.ReadBool();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new ExeStartInfo_Unknown(exePath, workingDirectory, arguments, environmentVariables, runtimeArguments, executeAsIs, useExternalConsole, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<StringPair>> ReadStringPairList = StringPair.Read.List();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static new CtxWriteDelegate<ExeStartInfo_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ExePath);
      writer.Write(value.WorkingDirectory);
      writer.Write(value.Arguments);
      WriteStringPairList(ctx, writer, value.EnvironmentVariables);
      WriteStringNullable(ctx, writer, value.RuntimeArguments);
      writer.Write(value.ExecuteAsIs);
      writer.Write(value.UseExternalConsole);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    public static CtxWriteDelegate<List<StringPair>> WriteStringPairList = StringPair.Write.List();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExeStartInfo_Unknown) obj);
    }
    public bool Equals(ExeStartInfo_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ExePath == other.ExePath && WorkingDirectory == other.WorkingDirectory && Arguments == other.Arguments && EnvironmentVariables.SequenceEqual(other.EnvironmentVariables) && Equals(RuntimeArguments, other.RuntimeArguments) && ExecuteAsIs == other.ExecuteAsIs && UseExternalConsole == other.UseExternalConsole && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ExePath.GetHashCode();
        hash = hash * 31 + WorkingDirectory.GetHashCode();
        hash = hash * 31 + Arguments.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(EnvironmentVariables);
        hash = hash * 31 + (RuntimeArguments != null ?RuntimeArguments.GetHashCode() : 0);
        hash = hash * 31 + ExecuteAsIs.GetHashCode();
        hash = hash * 31 + UseExternalConsole.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExeStartInfo_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("exePath = "); ExePath.PrintEx(printer); printer.Println();
        printer.Print("workingDirectory = "); WorkingDirectory.PrintEx(printer); printer.Println();
        printer.Print("arguments = "); Arguments.PrintEx(printer); printer.Println();
        printer.Print("environmentVariables = "); EnvironmentVariables.PrintEx(printer); printer.Println();
        printer.Print("runtimeArguments = "); RuntimeArguments.PrintEx(printer); printer.Println();
        printer.Print("executeAsIs = "); ExecuteAsIs.PrintEx(printer); printer.Println();
        printer.Print("useExternalConsole = "); UseExternalConsole.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class ExecutionCommandBase{
    //fields
    //public fields
    public ExecutionCommandKind Kind {get; private set;}
    
    //private fields
    //primary constructor
    protected ExecutionCommandBase(
      ExecutionCommandKind kind
    )
    {
      Kind = kind;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ExecutionCommandBase> Read = Polymorphic<ExecutionCommandBase>.ReadAbstract(ExecutionCommandBase_Unknown.Read);
    
    public static CtxWriteDelegate<ExecutionCommandBase> Write = Polymorphic<ExecutionCommandBase>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class ExecutionCommandBase_Unknown : ExecutionCommandBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public ExecutionCommandBase_Unknown(
      ExecutionCommandKind kind
    ) : base (
      kind
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExecutionCommandBase_Unknown> Read = (ctx, reader) => 
    {
      var kind = (ExecutionCommandKind)reader.ReadInt();
      return new ExecutionCommandBase_Unknown(kind);
    };
    
    public static new CtxWriteDelegate<ExecutionCommandBase_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Kind);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExecutionCommandBase_Unknown) obj);
    }
    public bool Equals(ExecutionCommandBase_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Kind == other.Kind;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (int) Kind;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExecutionCommandBase_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("kind = "); Kind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum ExecutionCommandKind {
    Resume,
    Pause,
    StepOver,
    StepInto,
    SmartStepInto,
    StepOut,
    RunToCursor,
    SetNextStatement
  }
  
  
  public class ExecutionCommandProxy : ExecutionCommandBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public ExecutionCommandProxy(
      ExecutionCommandKind kind
    ) : base (
      kind
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExecutionCommandProxy> Read = (ctx, reader) => 
    {
      var kind = (ExecutionCommandKind)reader.ReadInt();
      return new ExecutionCommandProxy(kind);
    };
    
    public static new CtxWriteDelegate<ExecutionCommandProxy> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Kind);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExecutionCommandProxy) obj);
    }
    public bool Equals(ExecutionCommandProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Kind == other.Kind;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (int) Kind;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExecutionCommandProxy (");
      using (printer.IndentCookie()) {
        printer.Print("kind = "); Kind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ExecutionCommandProxyWithPosition : ExecutionCommandBase
  {
    //fields
    //public fields
    [NotNull] public SourcePositionProxy SourcePosition {get; private set;}
    
    //private fields
    //primary constructor
    public ExecutionCommandProxyWithPosition(
      [NotNull] SourcePositionProxy sourcePosition,
      ExecutionCommandKind kind
    ) : base (
      kind
     ) 
    {
      if (sourcePosition == null) throw new ArgumentNullException("sourcePosition");
      
      SourcePosition = sourcePosition;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ExecutionCommandProxyWithPosition> Read = (ctx, reader) => 
    {
      var kind = (ExecutionCommandKind)reader.ReadInt();
      var sourcePosition = SourcePositionProxy.Read(ctx, reader);
      return new ExecutionCommandProxyWithPosition(sourcePosition, kind);
    };
    
    public static new CtxWriteDelegate<ExecutionCommandProxyWithPosition> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Kind);
      SourcePositionProxy.Write(ctx, writer, value.SourcePosition);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ExecutionCommandProxyWithPosition) obj);
    }
    public bool Equals(ExecutionCommandProxyWithPosition other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(SourcePosition, other.SourcePosition) && Kind == other.Kind;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + SourcePosition.GetHashCode();
        hash = hash * 31 + (int) Kind;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ExecutionCommandProxyWithPosition (");
      using (printer.IndentCookie()) {
        printer.Print("sourcePosition = "); SourcePosition.PrintEx(printer); printer.Println();
        printer.Print("kind = "); Kind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ExecutionContextProxy : RdBindableBase
  {
    //fields
    //public fields
    public int ActiveThreadId {get; private set;}
    [CanBeNull] public BreakpointId Breakpoint {get; private set;}
    [CanBeNull] public ExceptionModel Exception {get; private set;}
    [NotNull] public List<StackProxy> Stacks {get; private set;}
    public ExecutionContextType Type {get; private set;}
    [NotNull] public RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult> ComputeFrameLocals { get { return _ComputeFrameLocals; }}
    [NotNull] public RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult> ComputeValueChildren { get { return _ComputeValueChildren; }}
    [NotNull] public RdEndpoint<RefreshValueArg, RefreshValueResult> RefreshValue { get { return _RefreshValue; }}
    [NotNull] public RdEndpoint<MakeObjectIdArg, MakeObjectIdResult> MakeObjectId { get { return _MakeObjectId; }}
    [NotNull] public RdEndpoint<RemoveObjectIdArg, RdVoid> RemoveObjectId { get { return _RemoveObjectId; }}
    [NotNull] public RdEndpoint<ComputeFramesArg, ComputeFramesResult> ComputeFrames { get { return _ComputeFrames; }}
    [NotNull] public RdEndpoint<GetEvaluationExpressionArg, string> GetEvaluationExpression { get { return _GetEvaluationExpression; }}
    [NotNull] public RdEndpoint<EvaluateArg, EvaluateResult> EvaluateExpressions { get { return _EvaluateExpressions; }}
    [NotNull] public RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy> ComputeObjectProperties { get { return _ComputeObjectProperties; }}
    [NotNull] public RdEndpoint<SetValueArg, RdVoid> SetValue { get { return _SetValue; }}
    [NotNull] public RdEndpoint<ComputeFieldsArg, ComputeFieldsResult> ComputeFields { get { return _ComputeFields; }}
    [NotNull] public RdEndpoint<RefreshStackFrameArg, StackFrameProxy> RefreshStackFrame { get { return _RefreshStackFrame; }}
    [NotNull] public RdEndpoint<RdVoid, SmartStepIntoInfoProxy> GetSmartStepIntoInfo { get { return _GetSmartStepIntoInfo; }}
    [NotNull] public RdEndpoint<RdVoid, FetchMemoryViewResult> FetchMemoryView { get { return _FetchMemoryView; }}
    [NotNull] public RdEndpoint<GetClassInstancesArg, List<ObjectProxy>> GetClassInstances { get { return _GetClassInstances; }}
    [NotNull] public RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult> GetFilesSequencePoints { get { return _GetFilesSequencePoints; }}
    
    //private fields
    [NotNull] private readonly RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult> _ComputeFrameLocals;
    [NotNull] private readonly RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult> _ComputeValueChildren;
    [NotNull] private readonly RdEndpoint<RefreshValueArg, RefreshValueResult> _RefreshValue;
    [NotNull] private readonly RdEndpoint<MakeObjectIdArg, MakeObjectIdResult> _MakeObjectId;
    [NotNull] private readonly RdEndpoint<RemoveObjectIdArg, RdVoid> _RemoveObjectId;
    [NotNull] private readonly RdEndpoint<ComputeFramesArg, ComputeFramesResult> _ComputeFrames;
    [NotNull] private readonly RdEndpoint<GetEvaluationExpressionArg, string> _GetEvaluationExpression;
    [NotNull] private readonly RdEndpoint<EvaluateArg, EvaluateResult> _EvaluateExpressions;
    [NotNull] private readonly RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy> _ComputeObjectProperties;
    [NotNull] private readonly RdEndpoint<SetValueArg, RdVoid> _SetValue;
    [NotNull] private readonly RdEndpoint<ComputeFieldsArg, ComputeFieldsResult> _ComputeFields;
    [NotNull] private readonly RdEndpoint<RefreshStackFrameArg, StackFrameProxy> _RefreshStackFrame;
    [NotNull] private readonly RdEndpoint<RdVoid, SmartStepIntoInfoProxy> _GetSmartStepIntoInfo;
    [NotNull] private readonly RdEndpoint<RdVoid, FetchMemoryViewResult> _FetchMemoryView;
    [NotNull] private readonly RdEndpoint<GetClassInstancesArg, List<ObjectProxy>> _GetClassInstances;
    [NotNull] private readonly RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult> _GetFilesSequencePoints;
    
    //primary constructor
    private ExecutionContextProxy(
      int activeThreadId,
      [CanBeNull] BreakpointId breakpoint,
      [CanBeNull] ExceptionModel exception,
      [NotNull] List<StackProxy> stacks,
      ExecutionContextType type,
      [NotNull] RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult> computeFrameLocals,
      [NotNull] RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult> computeValueChildren,
      [NotNull] RdEndpoint<RefreshValueArg, RefreshValueResult> refreshValue,
      [NotNull] RdEndpoint<MakeObjectIdArg, MakeObjectIdResult> makeObjectId,
      [NotNull] RdEndpoint<RemoveObjectIdArg, RdVoid> removeObjectId,
      [NotNull] RdEndpoint<ComputeFramesArg, ComputeFramesResult> computeFrames,
      [NotNull] RdEndpoint<GetEvaluationExpressionArg, string> getEvaluationExpression,
      [NotNull] RdEndpoint<EvaluateArg, EvaluateResult> evaluateExpressions,
      [NotNull] RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy> computeObjectProperties,
      [NotNull] RdEndpoint<SetValueArg, RdVoid> setValue,
      [NotNull] RdEndpoint<ComputeFieldsArg, ComputeFieldsResult> computeFields,
      [NotNull] RdEndpoint<RefreshStackFrameArg, StackFrameProxy> refreshStackFrame,
      [NotNull] RdEndpoint<RdVoid, SmartStepIntoInfoProxy> getSmartStepIntoInfo,
      [NotNull] RdEndpoint<RdVoid, FetchMemoryViewResult> fetchMemoryView,
      [NotNull] RdEndpoint<GetClassInstancesArg, List<ObjectProxy>> getClassInstances,
      [NotNull] RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult> getFilesSequencePoints
    )
    {
      if (stacks == null) throw new ArgumentNullException("stacks");
      if (computeFrameLocals == null) throw new ArgumentNullException("computeFrameLocals");
      if (computeValueChildren == null) throw new ArgumentNullException("computeValueChildren");
      if (refreshValue == null) throw new ArgumentNullException("refreshValue");
      if (makeObjectId == null) throw new ArgumentNullException("makeObjectId");
      if (removeObjectId == null) throw new ArgumentNullException("removeObjectId");
      if (computeFrames == null) throw new ArgumentNullException("computeFrames");
      if (getEvaluationExpression == null) throw new ArgumentNullException("getEvaluationExpression");
      if (evaluateExpressions == null) throw new ArgumentNullException("evaluateExpressions");
      if (computeObjectProperties == null) throw new ArgumentNullException("computeObjectProperties");
      if (setValue == null) throw new ArgumentNullException("setValue");
      if (computeFields == null) throw new ArgumentNullException("computeFields");
      if (refreshStackFrame == null) throw new ArgumentNullException("refreshStackFrame");
      if (getSmartStepIntoInfo == null) throw new ArgumentNullException("getSmartStepIntoInfo");
      if (fetchMemoryView == null) throw new ArgumentNullException("fetchMemoryView");
      if (getClassInstances == null) throw new ArgumentNullException("getClassInstances");
      if (getFilesSequencePoints == null) throw new ArgumentNullException("getFilesSequencePoints");
      
      ActiveThreadId = activeThreadId;
      Breakpoint = breakpoint;
      Exception = exception;
      Stacks = stacks;
      Type = type;
      _ComputeFrameLocals = computeFrameLocals;
      _ComputeValueChildren = computeValueChildren;
      _RefreshValue = refreshValue;
      _MakeObjectId = makeObjectId;
      _RemoveObjectId = removeObjectId;
      _ComputeFrames = computeFrames;
      _GetEvaluationExpression = getEvaluationExpression;
      _EvaluateExpressions = evaluateExpressions;
      _ComputeObjectProperties = computeObjectProperties;
      _SetValue = setValue;
      _ComputeFields = computeFields;
      _RefreshStackFrame = refreshStackFrame;
      _GetSmartStepIntoInfo = getSmartStepIntoInfo;
      _FetchMemoryView = fetchMemoryView;
      _GetClassInstances = getClassInstances;
      _GetFilesSequencePoints = getFilesSequencePoints;
      _ComputeFrameLocals.Async = true;
      _ComputeValueChildren.Async = true;
      _RefreshValue.Async = true;
      _ComputeFrames.Async = true;
      _EvaluateExpressions.Async = true;
      _ComputeObjectProperties.Async = true;
      _SetValue.Async = true;
      _RefreshStackFrame.ValueCanBeNull = true;
      _GetSmartStepIntoInfo.ValueCanBeNull = true;
      _GetClassInstances.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("computeFrameLocals", _ComputeFrameLocals));
      BindableChildren.Add(new KeyValuePair<string, object>("computeValueChildren", _ComputeValueChildren));
      BindableChildren.Add(new KeyValuePair<string, object>("refreshValue", _RefreshValue));
      BindableChildren.Add(new KeyValuePair<string, object>("makeObjectId", _MakeObjectId));
      BindableChildren.Add(new KeyValuePair<string, object>("removeObjectId", _RemoveObjectId));
      BindableChildren.Add(new KeyValuePair<string, object>("computeFrames", _ComputeFrames));
      BindableChildren.Add(new KeyValuePair<string, object>("getEvaluationExpression", _GetEvaluationExpression));
      BindableChildren.Add(new KeyValuePair<string, object>("evaluateExpressions", _EvaluateExpressions));
      BindableChildren.Add(new KeyValuePair<string, object>("computeObjectProperties", _ComputeObjectProperties));
      BindableChildren.Add(new KeyValuePair<string, object>("setValue", _SetValue));
      BindableChildren.Add(new KeyValuePair<string, object>("computeFields", _ComputeFields));
      BindableChildren.Add(new KeyValuePair<string, object>("refreshStackFrame", _RefreshStackFrame));
      BindableChildren.Add(new KeyValuePair<string, object>("getSmartStepIntoInfo", _GetSmartStepIntoInfo));
      BindableChildren.Add(new KeyValuePair<string, object>("fetchMemoryView", _FetchMemoryView));
      BindableChildren.Add(new KeyValuePair<string, object>("getClassInstances", _GetClassInstances));
      BindableChildren.Add(new KeyValuePair<string, object>("getFilesSequencePoints", _GetFilesSequencePoints));
    }
    //secondary constructor
    public ExecutionContextProxy (
      int activeThreadId,
      [CanBeNull] BreakpointId breakpoint,
      [CanBeNull] ExceptionModel exception,
      [NotNull] List<StackProxy> stacks,
      ExecutionContextType type
    ) : this (
      activeThreadId,
      breakpoint,
      exception,
      stacks,
      type,
      new RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult>(ComputeFrameLocalsArg.Read, ComputeFrameLocalsArg.Write, ComputeFrameLocalsResult.Read, ComputeFrameLocalsResult.Write),
      new RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult>(ComputeValueChildrenArg.Read, ComputeValueChildrenArg.Write, ComputeValueChildrenResult.Read, ComputeValueChildrenResult.Write),
      new RdEndpoint<RefreshValueArg, RefreshValueResult>(RefreshValueArg.Read, RefreshValueArg.Write, RefreshValueResult.Read, RefreshValueResult.Write),
      new RdEndpoint<MakeObjectIdArg, MakeObjectIdResult>(MakeObjectIdArg.Read, MakeObjectIdArg.Write, MakeObjectIdResult.Read, MakeObjectIdResult.Write),
      new RdEndpoint<RemoveObjectIdArg, RdVoid>(RemoveObjectIdArg.Read, RemoveObjectIdArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdEndpoint<ComputeFramesArg, ComputeFramesResult>(ComputeFramesArg.Read, ComputeFramesArg.Write, ComputeFramesResult.Read, ComputeFramesResult.Write),
      new RdEndpoint<GetEvaluationExpressionArg, string>(GetEvaluationExpressionArg.Read, GetEvaluationExpressionArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdEndpoint<EvaluateArg, EvaluateResult>(EvaluateArg.Read, EvaluateArg.Write, EvaluateResult.Read, EvaluateResult.Write),
      new RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy>(ComputeObjectPropertiesArg.Read, ComputeObjectPropertiesArg.Write, ObjectPropertiesProxy.Read, ObjectPropertiesProxy.Write),
      new RdEndpoint<SetValueArg, RdVoid>(SetValueArg.Read, SetValueArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdEndpoint<ComputeFieldsArg, ComputeFieldsResult>(ComputeFieldsArg.Read, ComputeFieldsArg.Write, ComputeFieldsResult.Read, ComputeFieldsResult.Write),
      new RdEndpoint<RefreshStackFrameArg, StackFrameProxy>(RefreshStackFrameArg.Read, RefreshStackFrameArg.Write, ReadStackFrameProxyNullable, WriteStackFrameProxyNullable),
      new RdEndpoint<RdVoid, SmartStepIntoInfoProxy>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid, ReadSmartStepIntoInfoProxyNullable, WriteSmartStepIntoInfoProxyNullable),
      new RdEndpoint<RdVoid, FetchMemoryViewResult>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid, FetchMemoryViewResult.Read, FetchMemoryViewResult.Write),
      new RdEndpoint<GetClassInstancesArg, List<ObjectProxy>>(GetClassInstancesArg.Read, GetClassInstancesArg.Write, ReadObjectProxyListNullable, WriteObjectProxyListNullable),
      new RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult>(GetFilesSequencePointsArg.Read, GetFilesSequencePointsArg.Write, GetFilesSequencePointsResult.Read, GetFilesSequencePointsResult.Write)
    ) {}
    //statics
    
    public static CtxReadDelegate<ExecutionContextProxy> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var activeThreadId = reader.ReadInt();
      var breakpoint = ReadBreakpointIdNullable(ctx, reader);
      var exception = ReadExceptionModelNullable(ctx, reader);
      var stacks = ReadStackProxyList(ctx, reader);
      var type = (ExecutionContextType)reader.ReadInt();
      var computeFrameLocals = RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult>.Read(ctx, reader, ComputeFrameLocalsArg.Read, ComputeFrameLocalsArg.Write, ComputeFrameLocalsResult.Read, ComputeFrameLocalsResult.Write);
      var computeValueChildren = RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult>.Read(ctx, reader, ComputeValueChildrenArg.Read, ComputeValueChildrenArg.Write, ComputeValueChildrenResult.Read, ComputeValueChildrenResult.Write);
      var refreshValue = RdEndpoint<RefreshValueArg, RefreshValueResult>.Read(ctx, reader, RefreshValueArg.Read, RefreshValueArg.Write, RefreshValueResult.Read, RefreshValueResult.Write);
      var makeObjectId = RdEndpoint<MakeObjectIdArg, MakeObjectIdResult>.Read(ctx, reader, MakeObjectIdArg.Read, MakeObjectIdArg.Write, MakeObjectIdResult.Read, MakeObjectIdResult.Write);
      var removeObjectId = RdEndpoint<RemoveObjectIdArg, RdVoid>.Read(ctx, reader, RemoveObjectIdArg.Read, RemoveObjectIdArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var computeFrames = RdEndpoint<ComputeFramesArg, ComputeFramesResult>.Read(ctx, reader, ComputeFramesArg.Read, ComputeFramesArg.Write, ComputeFramesResult.Read, ComputeFramesResult.Write);
      var getEvaluationExpression = RdEndpoint<GetEvaluationExpressionArg, string>.Read(ctx, reader, GetEvaluationExpressionArg.Read, GetEvaluationExpressionArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var evaluateExpressions = RdEndpoint<EvaluateArg, EvaluateResult>.Read(ctx, reader, EvaluateArg.Read, EvaluateArg.Write, EvaluateResult.Read, EvaluateResult.Write);
      var computeObjectProperties = RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy>.Read(ctx, reader, ComputeObjectPropertiesArg.Read, ComputeObjectPropertiesArg.Write, ObjectPropertiesProxy.Read, ObjectPropertiesProxy.Write);
      var setValue = RdEndpoint<SetValueArg, RdVoid>.Read(ctx, reader, SetValueArg.Read, SetValueArg.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var computeFields = RdEndpoint<ComputeFieldsArg, ComputeFieldsResult>.Read(ctx, reader, ComputeFieldsArg.Read, ComputeFieldsArg.Write, ComputeFieldsResult.Read, ComputeFieldsResult.Write);
      var refreshStackFrame = RdEndpoint<RefreshStackFrameArg, StackFrameProxy>.Read(ctx, reader, RefreshStackFrameArg.Read, RefreshStackFrameArg.Write, ReadStackFrameProxyNullable, WriteStackFrameProxyNullable);
      var getSmartStepIntoInfo = RdEndpoint<RdVoid, SmartStepIntoInfoProxy>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid, ReadSmartStepIntoInfoProxyNullable, WriteSmartStepIntoInfoProxyNullable);
      var fetchMemoryView = RdEndpoint<RdVoid, FetchMemoryViewResult>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid, FetchMemoryViewResult.Read, FetchMemoryViewResult.Write);
      var getClassInstances = RdEndpoint<GetClassInstancesArg, List<ObjectProxy>>.Read(ctx, reader, GetClassInstancesArg.Read, GetClassInstancesArg.Write, ReadObjectProxyListNullable, WriteObjectProxyListNullable);
      var getFilesSequencePoints = RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult>.Read(ctx, reader, GetFilesSequencePointsArg.Read, GetFilesSequencePointsArg.Write, GetFilesSequencePointsResult.Read, GetFilesSequencePointsResult.Write);
      return new ExecutionContextProxy(activeThreadId, breakpoint, exception, stacks, type, computeFrameLocals, computeValueChildren, refreshValue, makeObjectId, removeObjectId, computeFrames, getEvaluationExpression, evaluateExpressions, computeObjectProperties, setValue, computeFields, refreshStackFrame, getSmartStepIntoInfo, fetchMemoryView, getClassInstances, getFilesSequencePoints).WithId(_id);
    };
    public static CtxReadDelegate<BreakpointId> ReadBreakpointIdNullable = BreakpointId.Read.NullableClass();
    public static CtxReadDelegate<ExceptionModel> ReadExceptionModelNullable = ExceptionModel.Read.NullableClass();
    public static CtxReadDelegate<List<StackProxy>> ReadStackProxyList = StackProxy.Read.List();
    public static CtxReadDelegate<StackFrameProxy> ReadStackFrameProxyNullable = StackFrameProxy.Read.NullableClass();
    public static CtxReadDelegate<SmartStepIntoInfoProxy> ReadSmartStepIntoInfoProxyNullable = SmartStepIntoInfoProxy.Read.NullableClass();
    public static CtxReadDelegate<List<ObjectProxy>> ReadObjectProxyListNullable = ObjectProxy.Read.List().NullableClass();
    
    public static CtxWriteDelegate<ExecutionContextProxy> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.ActiveThreadId);
      WriteBreakpointIdNullable(ctx, writer, value.Breakpoint);
      WriteExceptionModelNullable(ctx, writer, value.Exception);
      WriteStackProxyList(ctx, writer, value.Stacks);
      writer.Write((int)value.Type);
      RdEndpoint<ComputeFrameLocalsArg, ComputeFrameLocalsResult>.Write(ctx, writer, value._ComputeFrameLocals);
      RdEndpoint<ComputeValueChildrenArg, ComputeValueChildrenResult>.Write(ctx, writer, value._ComputeValueChildren);
      RdEndpoint<RefreshValueArg, RefreshValueResult>.Write(ctx, writer, value._RefreshValue);
      RdEndpoint<MakeObjectIdArg, MakeObjectIdResult>.Write(ctx, writer, value._MakeObjectId);
      RdEndpoint<RemoveObjectIdArg, RdVoid>.Write(ctx, writer, value._RemoveObjectId);
      RdEndpoint<ComputeFramesArg, ComputeFramesResult>.Write(ctx, writer, value._ComputeFrames);
      RdEndpoint<GetEvaluationExpressionArg, string>.Write(ctx, writer, value._GetEvaluationExpression);
      RdEndpoint<EvaluateArg, EvaluateResult>.Write(ctx, writer, value._EvaluateExpressions);
      RdEndpoint<ComputeObjectPropertiesArg, ObjectPropertiesProxy>.Write(ctx, writer, value._ComputeObjectProperties);
      RdEndpoint<SetValueArg, RdVoid>.Write(ctx, writer, value._SetValue);
      RdEndpoint<ComputeFieldsArg, ComputeFieldsResult>.Write(ctx, writer, value._ComputeFields);
      RdEndpoint<RefreshStackFrameArg, StackFrameProxy>.Write(ctx, writer, value._RefreshStackFrame);
      RdEndpoint<RdVoid, SmartStepIntoInfoProxy>.Write(ctx, writer, value._GetSmartStepIntoInfo);
      RdEndpoint<RdVoid, FetchMemoryViewResult>.Write(ctx, writer, value._FetchMemoryView);
      RdEndpoint<GetClassInstancesArg, List<ObjectProxy>>.Write(ctx, writer, value._GetClassInstances);
      RdEndpoint<GetFilesSequencePointsArg, GetFilesSequencePointsResult>.Write(ctx, writer, value._GetFilesSequencePoints);
    };
    public static CtxWriteDelegate<BreakpointId> WriteBreakpointIdNullable = BreakpointId.Write.NullableClass();
    public static CtxWriteDelegate<ExceptionModel> WriteExceptionModelNullable = ExceptionModel.Write.NullableClass();
    public static CtxWriteDelegate<List<StackProxy>> WriteStackProxyList = StackProxy.Write.List();
    public static CtxWriteDelegate<StackFrameProxy> WriteStackFrameProxyNullable = StackFrameProxy.Write.NullableClass();
    public static CtxWriteDelegate<SmartStepIntoInfoProxy> WriteSmartStepIntoInfoProxyNullable = SmartStepIntoInfoProxy.Write.NullableClass();
    public static CtxWriteDelegate<List<ObjectProxy>> WriteObjectProxyListNullable = ObjectProxy.Write.List().NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("ExecutionContextProxy (");
      using (printer.IndentCookie()) {
        printer.Print("activeThreadId = "); ActiveThreadId.PrintEx(printer); printer.Println();
        printer.Print("breakpoint = "); Breakpoint.PrintEx(printer); printer.Println();
        printer.Print("exception = "); Exception.PrintEx(printer); printer.Println();
        printer.Print("stacks = "); Stacks.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
        printer.Print("computeFrameLocals = "); _ComputeFrameLocals.PrintEx(printer); printer.Println();
        printer.Print("computeValueChildren = "); _ComputeValueChildren.PrintEx(printer); printer.Println();
        printer.Print("refreshValue = "); _RefreshValue.PrintEx(printer); printer.Println();
        printer.Print("makeObjectId = "); _MakeObjectId.PrintEx(printer); printer.Println();
        printer.Print("removeObjectId = "); _RemoveObjectId.PrintEx(printer); printer.Println();
        printer.Print("computeFrames = "); _ComputeFrames.PrintEx(printer); printer.Println();
        printer.Print("getEvaluationExpression = "); _GetEvaluationExpression.PrintEx(printer); printer.Println();
        printer.Print("evaluateExpressions = "); _EvaluateExpressions.PrintEx(printer); printer.Println();
        printer.Print("computeObjectProperties = "); _ComputeObjectProperties.PrintEx(printer); printer.Println();
        printer.Print("setValue = "); _SetValue.PrintEx(printer); printer.Println();
        printer.Print("computeFields = "); _ComputeFields.PrintEx(printer); printer.Println();
        printer.Print("refreshStackFrame = "); _RefreshStackFrame.PrintEx(printer); printer.Println();
        printer.Print("getSmartStepIntoInfo = "); _GetSmartStepIntoInfo.PrintEx(printer); printer.Println();
        printer.Print("fetchMemoryView = "); _FetchMemoryView.PrintEx(printer); printer.Println();
        printer.Print("getClassInstances = "); _GetClassInstances.PrintEx(printer); printer.Println();
        printer.Print("getFilesSequencePoints = "); _GetFilesSequencePoints.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum ExecutionContextType {
    OnStop,
    OnBreakpoint,
    OnException,
    OnStepCompleted
  }
  
  
  public class FetchMemoryViewResult : IPrintable, IEquatable<FetchMemoryViewResult>
  {
    //fields
    //public fields
    [NotNull] public List<HeapClassInfoModel> Values {get; private set;}
    
    //private fields
    //primary constructor
    public FetchMemoryViewResult(
      [NotNull] List<HeapClassInfoModel> values
    )
    {
      if (values == null) throw new ArgumentNullException("values");
      
      Values = values;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<FetchMemoryViewResult> Read = (ctx, reader) => 
    {
      var values = ReadHeapClassInfoModelList(ctx, reader);
      return new FetchMemoryViewResult(values);
    };
    public static CtxReadDelegate<List<HeapClassInfoModel>> ReadHeapClassInfoModelList = HeapClassInfoModel.Read.List();
    
    public static CtxWriteDelegate<FetchMemoryViewResult> Write = (ctx, writer, value) => 
    {
      WriteHeapClassInfoModelList(ctx, writer, value.Values);
    };
    public static CtxWriteDelegate<List<HeapClassInfoModel>> WriteHeapClassInfoModelList = HeapClassInfoModel.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((FetchMemoryViewResult) obj);
    }
    public bool Equals(FetchMemoryViewResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Values.SequenceEqual(other.Values);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Values);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("FetchMemoryViewResult (");
      using (printer.IndentCookie()) {
        printer.Print("values = "); Values.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum FrameKind {
    ILFrame,
    NativeFrame
  }
  
  
  public class GetClassInstancesArg : IPrintable, IEquatable<GetClassInstancesArg>
  {
    //fields
    //public fields
    [NotNull] public string ClassName {get; private set;}
    public long From {get; private set;}
    public long Count {get; private set;}
    
    //private fields
    //primary constructor
    public GetClassInstancesArg(
      [NotNull] string className,
      long from,
      long count
    )
    {
      if (className == null) throw new ArgumentNullException("className");
      
      ClassName = className;
      From = from;
      Count = count;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetClassInstancesArg> Read = (ctx, reader) => 
    {
      var className = reader.ReadString();
      var from = reader.ReadLong();
      var count = reader.ReadLong();
      return new GetClassInstancesArg(className, from, count);
    };
    
    public static CtxWriteDelegate<GetClassInstancesArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ClassName);
      writer.Write(value.From);
      writer.Write(value.Count);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetClassInstancesArg) obj);
    }
    public bool Equals(GetClassInstancesArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ClassName == other.ClassName && From == other.From && Count == other.Count;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ClassName.GetHashCode();
        hash = hash * 31 + From.GetHashCode();
        hash = hash * 31 + Count.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetClassInstancesArg (");
      using (printer.IndentCookie()) {
        printer.Print("className = "); ClassName.PrintEx(printer); printer.Println();
        printer.Print("from = "); From.PrintEx(printer); printer.Println();
        printer.Print("count = "); Count.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class GetEvaluationExpressionArg : IPrintable, IEquatable<GetEvaluationExpressionArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    
    //private fields
    //primary constructor
    public GetEvaluationExpressionArg(
      int valueId
    )
    {
      ValueId = valueId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetEvaluationExpressionArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      return new GetEvaluationExpressionArg(valueId);
    };
    
    public static CtxWriteDelegate<GetEvaluationExpressionArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetEvaluationExpressionArg) obj);
    }
    public bool Equals(GetEvaluationExpressionArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetEvaluationExpressionArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class GetFilesSequencePointsArg : IPrintable, IEquatable<GetFilesSequencePointsArg>
  {
    //fields
    //public fields
    [NotNull] public List<string> Files {get; private set;}
    
    //private fields
    //primary constructor
    public GetFilesSequencePointsArg(
      [NotNull] List<string> files
    )
    {
      if (files == null) throw new ArgumentNullException("files");
      
      Files = files;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetFilesSequencePointsArg> Read = (ctx, reader) => 
    {
      var files = ReadStringList(ctx, reader);
      return new GetFilesSequencePointsArg(files);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<GetFilesSequencePointsArg> Write = (ctx, writer, value) => 
    {
      WriteStringList(ctx, writer, value.Files);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetFilesSequencePointsArg) obj);
    }
    public bool Equals(GetFilesSequencePointsArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Files.SequenceEqual(other.Files);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Files);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetFilesSequencePointsArg (");
      using (printer.IndentCookie()) {
        printer.Print("files = "); Files.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class GetFilesSequencePointsResult : IPrintable, IEquatable<GetFilesSequencePointsResult>
  {
    //fields
    //public fields
    [NotNull] public List<SequencePointsAndFilePair> Result {get; private set;}
    
    //private fields
    //primary constructor
    public GetFilesSequencePointsResult(
      [NotNull] List<SequencePointsAndFilePair> result
    )
    {
      if (result == null) throw new ArgumentNullException("result");
      
      Result = result;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<GetFilesSequencePointsResult> Read = (ctx, reader) => 
    {
      var result = ReadSequencePointsAndFilePairList(ctx, reader);
      return new GetFilesSequencePointsResult(result);
    };
    public static CtxReadDelegate<List<SequencePointsAndFilePair>> ReadSequencePointsAndFilePairList = SequencePointsAndFilePair.Read.List();
    
    public static CtxWriteDelegate<GetFilesSequencePointsResult> Write = (ctx, writer, value) => 
    {
      WriteSequencePointsAndFilePairList(ctx, writer, value.Result);
    };
    public static CtxWriteDelegate<List<SequencePointsAndFilePair>> WriteSequencePointsAndFilePairList = SequencePointsAndFilePair.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((GetFilesSequencePointsResult) obj);
    }
    public bool Equals(GetFilesSequencePointsResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Result.SequenceEqual(other.Result);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Result);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("GetFilesSequencePointsResult (");
      using (printer.IndentCookie()) {
        printer.Print("result = "); Result.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class HeapClassInfoModel : IPrintable, IEquatable<HeapClassInfoModel>
  {
    //fields
    //public fields
    [NotNull] public string Type {get; private set;}
    public long Count {get; private set;}
    public long TotalSize {get; private set;}
    
    //private fields
    //primary constructor
    public HeapClassInfoModel(
      [NotNull] string type,
      long count,
      long totalSize
    )
    {
      if (type == null) throw new ArgumentNullException("type");
      
      Type = type;
      Count = count;
      TotalSize = totalSize;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<HeapClassInfoModel> Read = (ctx, reader) => 
    {
      var type = reader.ReadString();
      var count = reader.ReadLong();
      var totalSize = reader.ReadLong();
      return new HeapClassInfoModel(type, count, totalSize);
    };
    
    public static CtxWriteDelegate<HeapClassInfoModel> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Type);
      writer.Write(value.Count);
      writer.Write(value.TotalSize);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((HeapClassInfoModel) obj);
    }
    public bool Equals(HeapClassInfoModel other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Type == other.Type && Count == other.Count && TotalSize == other.TotalSize;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Type.GetHashCode();
        hash = hash * 31 + Count.GetHashCode();
        hash = hash * 31 + TotalSize.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("HeapClassInfoModel (");
      using (printer.IndentCookie()) {
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
        printer.Print("count = "); Count.PrintEx(printer); printer.Println();
        printer.Print("totalSize = "); TotalSize.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class LineBreakpointModel : BreakpointModel
  {
    //fields
    //public fields
    [NotNull] public List<BreakpointElement> BreakpointElements {get; private set;}
    public bool IsSolutionFile {get; private set;}
    public int ThreadIdToStop {get; private set;}
    
    //private fields
    //primary constructor
    private LineBreakpointModel(
      [NotNull] List<BreakpointElement> breakpointElements,
      bool isSolutionFile,
      int threadIdToStop,
      [NotNull] BreakpointId id,
      [NotNull] RdProperty<bool> shouldNotifyHitWhenDisabled,
      [NotNull] RdSignal<BreakpointStatus> statusChanged,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : base (
      id,
      shouldNotifyHitWhenDisabled,
      statusChanged,
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
     ) 
    {
      if (breakpointElements == null) throw new ArgumentNullException("breakpointElements");
      
      BreakpointElements = breakpointElements;
      IsSolutionFile = isSolutionFile;
      ThreadIdToStop = threadIdToStop;
    }
    //secondary constructor
    public LineBreakpointModel (
      [NotNull] List<BreakpointElement> breakpointElements,
      bool isSolutionFile,
      int threadIdToStop,
      [NotNull] BreakpointId id,
      [CanBeNull] EvaluationExpressionModel condition,
      bool suspended,
      [CanBeNull] EvaluationExpressionModel traceExpression,
      bool traceBreakpoint,
      int hitCount,
      BreakpointHitCountModeModel hitCountMode
    ) : this (
      breakpointElements,
      isSolutionFile,
      threadIdToStop,
      id,
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<BreakpointStatus>(BreakpointStatus.Read, BreakpointStatus.Write),
      condition,
      suspended,
      traceExpression,
      traceBreakpoint,
      hitCount,
      hitCountMode
    ) {}
    //statics
    
    public static new CtxReadDelegate<LineBreakpointModel> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var id = BreakpointId.Read(ctx, reader);
      var shouldNotifyHitWhenDisabled = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var statusChanged = RdSignal<BreakpointStatus>.Read(ctx, reader, BreakpointStatus.Read, BreakpointStatus.Write);
      var condition = ReadEvaluationExpressionModelNullable(ctx, reader);
      var suspended = reader.ReadBool();
      var traceExpression = ReadEvaluationExpressionModelNullable(ctx, reader);
      var traceBreakpoint = reader.ReadBool();
      var hitCount = reader.ReadInt();
      var hitCountMode = (BreakpointHitCountModeModel)reader.ReadInt();
      var breakpointElements = ReadBreakpointElementList(ctx, reader);
      var isSolutionFile = reader.ReadBool();
      var threadIdToStop = reader.ReadInt();
      return new LineBreakpointModel(breakpointElements, isSolutionFile, threadIdToStop, id, shouldNotifyHitWhenDisabled, statusChanged, condition, suspended, traceExpression, traceBreakpoint, hitCount, hitCountMode).WithId(_id);
    };
    public static CtxReadDelegate<List<BreakpointElement>> ReadBreakpointElementList = BreakpointElement.Read.List();
    public static CtxReadDelegate<EvaluationExpressionModel> ReadEvaluationExpressionModelNullable = EvaluationExpressionModel.Read.NullableClass();
    
    public static new CtxWriteDelegate<LineBreakpointModel> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      BreakpointId.Write(ctx, writer, value.Id);
      RdProperty<bool>.Write(ctx, writer, value._ShouldNotifyHitWhenDisabled);
      RdSignal<BreakpointStatus>.Write(ctx, writer, value._StatusChanged);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.Condition);
      writer.Write(value.Suspended);
      WriteEvaluationExpressionModelNullable(ctx, writer, value.TraceExpression);
      writer.Write(value.TraceBreakpoint);
      writer.Write(value.HitCount);
      writer.Write((int)value.HitCountMode);
      WriteBreakpointElementList(ctx, writer, value.BreakpointElements);
      writer.Write(value.IsSolutionFile);
      writer.Write(value.ThreadIdToStop);
    };
    public static CtxWriteDelegate<List<BreakpointElement>> WriteBreakpointElementList = BreakpointElement.Write.List();
    public static CtxWriteDelegate<EvaluationExpressionModel> WriteEvaluationExpressionModelNullable = EvaluationExpressionModel.Write.NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("LineBreakpointModel (");
      using (printer.IndentCookie()) {
        printer.Print("breakpointElements = "); BreakpointElements.PrintEx(printer); printer.Println();
        printer.Print("isSolutionFile = "); IsSolutionFile.PrintEx(printer); printer.Println();
        printer.Print("threadIdToStop = "); ThreadIdToStop.PrintEx(printer); printer.Println();
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("shouldNotifyHitWhenDisabled = "); _ShouldNotifyHitWhenDisabled.PrintEx(printer); printer.Println();
        printer.Print("statusChanged = "); _StatusChanged.PrintEx(printer); printer.Println();
        printer.Print("condition = "); Condition.PrintEx(printer); printer.Println();
        printer.Print("suspended = "); Suspended.PrintEx(printer); printer.Println();
        printer.Print("traceExpression = "); TraceExpression.PrintEx(printer); printer.Println();
        printer.Print("traceBreakpoint = "); TraceBreakpoint.PrintEx(printer); printer.Println();
        printer.Print("hitCount = "); HitCount.PrintEx(printer); printer.Println();
        printer.Print("hitCountMode = "); HitCountMode.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class MakeObjectIdArg : IPrintable, IEquatable<MakeObjectIdArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    [NotNull] public string LabelName {get; private set;}
    
    //private fields
    //primary constructor
    public MakeObjectIdArg(
      int valueId,
      [NotNull] string labelName
    )
    {
      if (labelName == null) throw new ArgumentNullException("labelName");
      
      ValueId = valueId;
      LabelName = labelName;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<MakeObjectIdArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var labelName = reader.ReadString();
      return new MakeObjectIdArg(valueId, labelName);
    };
    
    public static CtxWriteDelegate<MakeObjectIdArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      writer.Write(value.LabelName);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((MakeObjectIdArg) obj);
    }
    public bool Equals(MakeObjectIdArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && LabelName == other.LabelName;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + LabelName.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("MakeObjectIdArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("labelName = "); LabelName.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class MakeObjectIdResult : IPrintable, IEquatable<MakeObjectIdResult>
  {
    //fields
    //public fields
    [CanBeNull] public string ObjectId {get; private set;}
    
    //private fields
    //primary constructor
    public MakeObjectIdResult(
      [CanBeNull] string objectId
    )
    {
      ObjectId = objectId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<MakeObjectIdResult> Read = (ctx, reader) => 
    {
      var objectId = ReadStringNullable(ctx, reader);
      return new MakeObjectIdResult(objectId);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<MakeObjectIdResult> Write = (ctx, writer, value) => 
    {
      WriteStringNullable(ctx, writer, value.ObjectId);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((MakeObjectIdResult) obj);
    }
    public bool Equals(MakeObjectIdResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(ObjectId, other.ObjectId);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (ObjectId != null ?ObjectId.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("MakeObjectIdResult (");
      using (printer.IndentCookie()) {
        printer.Print("objectId = "); ObjectId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class MonoAttachStartInfo : DebuggerStartInfoBase
  {
    //fields
    //public fields
    [CanBeNull] public string MonoAddress {get; private set;}
    public int MonoPort {get; private set;}
    public bool ListenForConnections {get; private set;}
    
    //private fields
    //primary constructor
    public MonoAttachStartInfo(
      [CanBeNull] string monoAddress,
      int monoPort,
      bool listenForConnections,
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
      MonoAddress = monoAddress;
      MonoPort = monoPort;
      ListenForConnections = listenForConnections;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<MonoAttachStartInfo> Read = (ctx, reader) => 
    {
      var needToBeInitializedImmediately = reader.ReadBool();
      var monoAddress = ReadStringNullable(ctx, reader);
      var monoPort = reader.ReadInt();
      var listenForConnections = reader.ReadBool();
      return new MonoAttachStartInfo(monoAddress, monoPort, listenForConnections, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static new CtxWriteDelegate<MonoAttachStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.NeedToBeInitializedImmediately);
      WriteStringNullable(ctx, writer, value.MonoAddress);
      writer.Write(value.MonoPort);
      writer.Write(value.ListenForConnections);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((MonoAttachStartInfo) obj);
    }
    public bool Equals(MonoAttachStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(MonoAddress, other.MonoAddress) && MonoPort == other.MonoPort && ListenForConnections == other.ListenForConnections && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (MonoAddress != null ?MonoAddress.GetHashCode() : 0);
        hash = hash * 31 + MonoPort.GetHashCode();
        hash = hash * 31 + ListenForConnections.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("MonoAttachStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("monoAddress = "); MonoAddress.PrintEx(printer); printer.Println();
        printer.Print("monoPort = "); MonoPort.PrintEx(printer); printer.Println();
        printer.Print("listenForConnections = "); ListenForConnections.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class MonoExeStartInfo : ExeStartInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public MonoExeStartInfo(
      [NotNull] string exePath,
      [NotNull] string workingDirectory,
      [NotNull] string arguments,
      [NotNull] List<StringPair> environmentVariables,
      [CanBeNull] string runtimeArguments,
      bool executeAsIs,
      bool useExternalConsole,
      bool needToBeInitializedImmediately
    ) : base (
      exePath,
      workingDirectory,
      arguments,
      environmentVariables,
      runtimeArguments,
      executeAsIs,
      useExternalConsole,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<MonoExeStartInfo> Read = (ctx, reader) => 
    {
      var exePath = reader.ReadString();
      var workingDirectory = reader.ReadString();
      var arguments = reader.ReadString();
      var environmentVariables = ReadStringPairList(ctx, reader);
      var runtimeArguments = ReadStringNullable(ctx, reader);
      var executeAsIs = reader.ReadBool();
      var useExternalConsole = reader.ReadBool();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new MonoExeStartInfo(exePath, workingDirectory, arguments, environmentVariables, runtimeArguments, executeAsIs, useExternalConsole, needToBeInitializedImmediately);
    };
    public static CtxReadDelegate<List<StringPair>> ReadStringPairList = StringPair.Read.List();
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static new CtxWriteDelegate<MonoExeStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ExePath);
      writer.Write(value.WorkingDirectory);
      writer.Write(value.Arguments);
      WriteStringPairList(ctx, writer, value.EnvironmentVariables);
      WriteStringNullable(ctx, writer, value.RuntimeArguments);
      writer.Write(value.ExecuteAsIs);
      writer.Write(value.UseExternalConsole);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    public static CtxWriteDelegate<List<StringPair>> WriteStringPairList = StringPair.Write.List();
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((MonoExeStartInfo) obj);
    }
    public bool Equals(MonoExeStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ExePath == other.ExePath && WorkingDirectory == other.WorkingDirectory && Arguments == other.Arguments && EnvironmentVariables.SequenceEqual(other.EnvironmentVariables) && Equals(RuntimeArguments, other.RuntimeArguments) && ExecuteAsIs == other.ExecuteAsIs && UseExternalConsole == other.UseExternalConsole && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ExePath.GetHashCode();
        hash = hash * 31 + WorkingDirectory.GetHashCode();
        hash = hash * 31 + Arguments.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(EnvironmentVariables);
        hash = hash * 31 + (RuntimeArguments != null ?RuntimeArguments.GetHashCode() : 0);
        hash = hash * 31 + ExecuteAsIs.GetHashCode();
        hash = hash * 31 + UseExternalConsole.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("MonoExeStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("exePath = "); ExePath.PrintEx(printer); printer.Println();
        printer.Print("workingDirectory = "); WorkingDirectory.PrintEx(printer); printer.Println();
        printer.Print("arguments = "); Arguments.PrintEx(printer); printer.Println();
        printer.Print("environmentVariables = "); EnvironmentVariables.PrintEx(printer); printer.Println();
        printer.Print("runtimeArguments = "); RuntimeArguments.PrintEx(printer); printer.Println();
        printer.Print("executeAsIs = "); ExecuteAsIs.PrintEx(printer); printer.Println();
        printer.Print("useExternalConsole = "); UseExternalConsole.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ObjectPropertiesProxy : IPrintable, IEquatable<ObjectPropertiesProxy>
  {
    //fields
    //public fields
    public int ObjectId {get; private set;}
    [NotNull] public string Value {get; private set;}
    [NotNull] public string Type {get; private set;}
    public bool HasChildren {get; private set;}
    public bool IsStatic {get; private set;}
    public bool IsPrimitive {get; private set;}
    public bool IsArray {get; private set;}
    public bool IsGroup {get; private set;}
    public bool IsError {get; private set;}
    public bool IsImplicitEvaluationDisabled {get; private set;}
    public bool IsField {get; private set;}
    public bool IsVariable {get; private set;}
    public bool IsParameter {get; private set;}
    public bool IsReadonly {get; private set;}
    public bool HasRefresh {get; private set;}
    public bool HasRejectCrossThread {get; private set;}
    public bool IsString {get; private set;}
    public bool IsValueEllipsized {get; private set;}
    public bool IsNull {get; private set;}
    public bool IsObject {get; private set;}
    public bool CouldBeMarked {get; private set;}
    [CanBeNull] public string ObjectLabel {get; private set;}
    [CanBeNull] public string AdditionalInfo {get; private set;}
    
    //private fields
    //primary constructor
    public ObjectPropertiesProxy(
      int objectId,
      [NotNull] string value,
      [NotNull] string type,
      bool hasChildren,
      bool isStatic,
      bool isPrimitive,
      bool isArray,
      bool isGroup,
      bool isError,
      bool isImplicitEvaluationDisabled,
      bool isField,
      bool isVariable,
      bool isParameter,
      bool isReadonly,
      bool hasRefresh,
      bool hasRejectCrossThread,
      bool isString,
      bool isValueEllipsized,
      bool isNull,
      bool isObject,
      bool couldBeMarked,
      [CanBeNull] string objectLabel,
      [CanBeNull] string additionalInfo
    )
    {
      if (value == null) throw new ArgumentNullException("value");
      if (type == null) throw new ArgumentNullException("type");
      
      ObjectId = objectId;
      Value = value;
      Type = type;
      HasChildren = hasChildren;
      IsStatic = isStatic;
      IsPrimitive = isPrimitive;
      IsArray = isArray;
      IsGroup = isGroup;
      IsError = isError;
      IsImplicitEvaluationDisabled = isImplicitEvaluationDisabled;
      IsField = isField;
      IsVariable = isVariable;
      IsParameter = isParameter;
      IsReadonly = isReadonly;
      HasRefresh = hasRefresh;
      HasRejectCrossThread = hasRejectCrossThread;
      IsString = isString;
      IsValueEllipsized = isValueEllipsized;
      IsNull = isNull;
      IsObject = isObject;
      CouldBeMarked = couldBeMarked;
      ObjectLabel = objectLabel;
      AdditionalInfo = additionalInfo;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ObjectPropertiesProxy> Read = (ctx, reader) => 
    {
      var objectId = reader.ReadInt();
      var value = reader.ReadString();
      var type = reader.ReadString();
      var hasChildren = reader.ReadBool();
      var isStatic = reader.ReadBool();
      var isPrimitive = reader.ReadBool();
      var isArray = reader.ReadBool();
      var isGroup = reader.ReadBool();
      var isError = reader.ReadBool();
      var isImplicitEvaluationDisabled = reader.ReadBool();
      var isField = reader.ReadBool();
      var isVariable = reader.ReadBool();
      var isParameter = reader.ReadBool();
      var isReadonly = reader.ReadBool();
      var hasRefresh = reader.ReadBool();
      var hasRejectCrossThread = reader.ReadBool();
      var isString = reader.ReadBool();
      var isValueEllipsized = reader.ReadBool();
      var isNull = reader.ReadBool();
      var isObject = reader.ReadBool();
      var couldBeMarked = reader.ReadBool();
      var objectLabel = ReadStringNullable(ctx, reader);
      var additionalInfo = ReadStringNullable(ctx, reader);
      return new ObjectPropertiesProxy(objectId, value, type, hasChildren, isStatic, isPrimitive, isArray, isGroup, isError, isImplicitEvaluationDisabled, isField, isVariable, isParameter, isReadonly, hasRefresh, hasRejectCrossThread, isString, isValueEllipsized, isNull, isObject, couldBeMarked, objectLabel, additionalInfo);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<ObjectPropertiesProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ObjectId);
      writer.Write(value.Value);
      writer.Write(value.Type);
      writer.Write(value.HasChildren);
      writer.Write(value.IsStatic);
      writer.Write(value.IsPrimitive);
      writer.Write(value.IsArray);
      writer.Write(value.IsGroup);
      writer.Write(value.IsError);
      writer.Write(value.IsImplicitEvaluationDisabled);
      writer.Write(value.IsField);
      writer.Write(value.IsVariable);
      writer.Write(value.IsParameter);
      writer.Write(value.IsReadonly);
      writer.Write(value.HasRefresh);
      writer.Write(value.HasRejectCrossThread);
      writer.Write(value.IsString);
      writer.Write(value.IsValueEllipsized);
      writer.Write(value.IsNull);
      writer.Write(value.IsObject);
      writer.Write(value.CouldBeMarked);
      WriteStringNullable(ctx, writer, value.ObjectLabel);
      WriteStringNullable(ctx, writer, value.AdditionalInfo);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ObjectPropertiesProxy) obj);
    }
    public bool Equals(ObjectPropertiesProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ObjectId == other.ObjectId && Value == other.Value && Type == other.Type && HasChildren == other.HasChildren && IsStatic == other.IsStatic && IsPrimitive == other.IsPrimitive && IsArray == other.IsArray && IsGroup == other.IsGroup && IsError == other.IsError && IsImplicitEvaluationDisabled == other.IsImplicitEvaluationDisabled && IsField == other.IsField && IsVariable == other.IsVariable && IsParameter == other.IsParameter && IsReadonly == other.IsReadonly && HasRefresh == other.HasRefresh && HasRejectCrossThread == other.HasRejectCrossThread && IsString == other.IsString && IsValueEllipsized == other.IsValueEllipsized && IsNull == other.IsNull && IsObject == other.IsObject && CouldBeMarked == other.CouldBeMarked && Equals(ObjectLabel, other.ObjectLabel) && Equals(AdditionalInfo, other.AdditionalInfo);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ObjectId.GetHashCode();
        hash = hash * 31 + Value.GetHashCode();
        hash = hash * 31 + Type.GetHashCode();
        hash = hash * 31 + HasChildren.GetHashCode();
        hash = hash * 31 + IsStatic.GetHashCode();
        hash = hash * 31 + IsPrimitive.GetHashCode();
        hash = hash * 31 + IsArray.GetHashCode();
        hash = hash * 31 + IsGroup.GetHashCode();
        hash = hash * 31 + IsError.GetHashCode();
        hash = hash * 31 + IsImplicitEvaluationDisabled.GetHashCode();
        hash = hash * 31 + IsField.GetHashCode();
        hash = hash * 31 + IsVariable.GetHashCode();
        hash = hash * 31 + IsParameter.GetHashCode();
        hash = hash * 31 + IsReadonly.GetHashCode();
        hash = hash * 31 + HasRefresh.GetHashCode();
        hash = hash * 31 + HasRejectCrossThread.GetHashCode();
        hash = hash * 31 + IsString.GetHashCode();
        hash = hash * 31 + IsValueEllipsized.GetHashCode();
        hash = hash * 31 + IsNull.GetHashCode();
        hash = hash * 31 + IsObject.GetHashCode();
        hash = hash * 31 + CouldBeMarked.GetHashCode();
        hash = hash * 31 + (ObjectLabel != null ?ObjectLabel.GetHashCode() : 0);
        hash = hash * 31 + (AdditionalInfo != null ?AdditionalInfo.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ObjectPropertiesProxy (");
      using (printer.IndentCookie()) {
        printer.Print("objectId = "); ObjectId.PrintEx(printer); printer.Println();
        printer.Print("value = "); Value.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
        printer.Print("hasChildren = "); HasChildren.PrintEx(printer); printer.Println();
        printer.Print("isStatic = "); IsStatic.PrintEx(printer); printer.Println();
        printer.Print("isPrimitive = "); IsPrimitive.PrintEx(printer); printer.Println();
        printer.Print("isArray = "); IsArray.PrintEx(printer); printer.Println();
        printer.Print("isGroup = "); IsGroup.PrintEx(printer); printer.Println();
        printer.Print("isError = "); IsError.PrintEx(printer); printer.Println();
        printer.Print("isImplicitEvaluationDisabled = "); IsImplicitEvaluationDisabled.PrintEx(printer); printer.Println();
        printer.Print("isField = "); IsField.PrintEx(printer); printer.Println();
        printer.Print("isVariable = "); IsVariable.PrintEx(printer); printer.Println();
        printer.Print("isParameter = "); IsParameter.PrintEx(printer); printer.Println();
        printer.Print("isReadonly = "); IsReadonly.PrintEx(printer); printer.Println();
        printer.Print("hasRefresh = "); HasRefresh.PrintEx(printer); printer.Println();
        printer.Print("hasRejectCrossThread = "); HasRejectCrossThread.PrintEx(printer); printer.Println();
        printer.Print("isString = "); IsString.PrintEx(printer); printer.Println();
        printer.Print("isValueEllipsized = "); IsValueEllipsized.PrintEx(printer); printer.Println();
        printer.Print("isNull = "); IsNull.PrintEx(printer); printer.Println();
        printer.Print("isObject = "); IsObject.PrintEx(printer); printer.Println();
        printer.Print("couldBeMarked = "); CouldBeMarked.PrintEx(printer); printer.Println();
        printer.Print("objectLabel = "); ObjectLabel.PrintEx(printer); printer.Println();
        printer.Print("additionalInfo = "); AdditionalInfo.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ObjectProxy : IPrintable, IEquatable<ObjectProxy>
  {
    //fields
    //public fields
    public int Id {get; private set;}
    [NotNull] public string Name {get; private set;}
    [CanBeNull] public ObjectPropertiesProxy Properties {get; private set;}
    
    //private fields
    //primary constructor
    public ObjectProxy(
      int id,
      [NotNull] string name,
      [CanBeNull] ObjectPropertiesProxy properties
    )
    {
      if (name == null) throw new ArgumentNullException("name");
      
      Id = id;
      Name = name;
      Properties = properties;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ObjectProxy> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var name = reader.ReadString();
      var properties = ReadObjectPropertiesProxyNullable(ctx, reader);
      return new ObjectProxy(id, name, properties);
    };
    public static CtxReadDelegate<ObjectPropertiesProxy> ReadObjectPropertiesProxyNullable = ObjectPropertiesProxy.Read.NullableClass();
    
    public static CtxWriteDelegate<ObjectProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      writer.Write(value.Name);
      WriteObjectPropertiesProxyNullable(ctx, writer, value.Properties);
    };
    public static CtxWriteDelegate<ObjectPropertiesProxy> WriteObjectPropertiesProxyNullable = ObjectPropertiesProxy.Write.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ObjectProxy) obj);
    }
    public bool Equals(ObjectProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && Name == other.Name && Equals(Properties, other.Properties);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        hash = hash * 31 + (Properties != null ?Properties.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ObjectProxy (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
        printer.Print("properties = "); Properties.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class OutputMessage : IPrintable, IEquatable<OutputMessage>
  {
    //fields
    //public fields
    [NotNull] public string Output {get; private set;}
    public OutputType Type {get; private set;}
    public OutputSubject Subject {get; private set;}
    
    //private fields
    //primary constructor
    public OutputMessage(
      [NotNull] string output,
      OutputType type,
      OutputSubject subject
    )
    {
      if (output == null) throw new ArgumentNullException("output");
      
      Output = output;
      Type = type;
      Subject = subject;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<OutputMessage> Read = (ctx, reader) => 
    {
      var output = reader.ReadString();
      var type = (OutputType)reader.ReadInt();
      var subject = (OutputSubject)reader.ReadInt();
      return new OutputMessage(output, type, subject);
    };
    
    public static CtxWriteDelegate<OutputMessage> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Output);
      writer.Write((int)value.Type);
      writer.Write((int)value.Subject);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((OutputMessage) obj);
    }
    public bool Equals(OutputMessage other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Output == other.Output && Type == other.Type && Subject == other.Subject;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Output.GetHashCode();
        hash = hash * 31 + (int) Type;
        hash = hash * 31 + (int) Subject;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("OutputMessage (");
      using (printer.IndentCookie()) {
        printer.Print("output = "); Output.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
        printer.Print("subject = "); Subject.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum OutputSubject {
    Default,
    ConnectionError,
    StartupError,
    StartupElevationRequiredError
  }
  
  
  public enum OutputType {
    Info,
    Warning,
    Error
  }
  
  
  public abstract class ProcessIdAttachInfo : DebuggerStartInfoBase
  {
    //fields
    //public fields
    public int ProcessId {get; private set;}
    
    //private fields
    //primary constructor
    protected ProcessIdAttachInfo(
      int processId,
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
      ProcessId = processId;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ProcessIdAttachInfo> Read = Polymorphic<ProcessIdAttachInfo>.ReadAbstract(ProcessIdAttachInfo_Unknown.Read);
    
    public static new CtxWriteDelegate<ProcessIdAttachInfo> Write = Polymorphic<ProcessIdAttachInfo>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class ProcessIdAttachInfo_Unknown : ProcessIdAttachInfo
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public ProcessIdAttachInfo_Unknown(
      int processId,
      bool needToBeInitializedImmediately
    ) : base (
      processId,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<ProcessIdAttachInfo_Unknown> Read = (ctx, reader) => 
    {
      var processId = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new ProcessIdAttachInfo_Unknown(processId, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<ProcessIdAttachInfo_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProcessId);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ProcessIdAttachInfo_Unknown) obj);
    }
    public bool Equals(ProcessIdAttachInfo_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProcessId == other.ProcessId && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProcessId.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ProcessIdAttachInfo_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("processId = "); ProcessId.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RefreshStackFrameArg : IPrintable, IEquatable<RefreshStackFrameArg>
  {
    //fields
    //public fields
    public int StackId {get; private set;}
    public int FrameId {get; private set;}
    public bool CanGetDebugInfoFromHost {get; private set;}
    
    //private fields
    //primary constructor
    public RefreshStackFrameArg(
      int stackId,
      int frameId,
      bool canGetDebugInfoFromHost
    )
    {
      StackId = stackId;
      FrameId = frameId;
      CanGetDebugInfoFromHost = canGetDebugInfoFromHost;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RefreshStackFrameArg> Read = (ctx, reader) => 
    {
      var stackId = reader.ReadInt();
      var frameId = reader.ReadInt();
      var canGetDebugInfoFromHost = reader.ReadBool();
      return new RefreshStackFrameArg(stackId, frameId, canGetDebugInfoFromHost);
    };
    
    public static CtxWriteDelegate<RefreshStackFrameArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.StackId);
      writer.Write(value.FrameId);
      writer.Write(value.CanGetDebugInfoFromHost);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RefreshStackFrameArg) obj);
    }
    public bool Equals(RefreshStackFrameArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return StackId == other.StackId && FrameId == other.FrameId && CanGetDebugInfoFromHost == other.CanGetDebugInfoFromHost;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + StackId.GetHashCode();
        hash = hash * 31 + FrameId.GetHashCode();
        hash = hash * 31 + CanGetDebugInfoFromHost.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RefreshStackFrameArg (");
      using (printer.IndentCookie()) {
        printer.Print("stackId = "); StackId.PrintEx(printer); printer.Println();
        printer.Print("frameId = "); FrameId.PrintEx(printer); printer.Println();
        printer.Print("canGetDebugInfoFromHost = "); CanGetDebugInfoFromHost.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RefreshValueArg : IPrintable, IEquatable<RefreshValueArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    public bool ForceRefresh {get; private set;}
    [NotNull] public List<string> NamespaceImports {get; private set;}
    
    //private fields
    //primary constructor
    public RefreshValueArg(
      int valueId,
      bool forceRefresh,
      [NotNull] List<string> namespaceImports
    )
    {
      if (namespaceImports == null) throw new ArgumentNullException("namespaceImports");
      
      ValueId = valueId;
      ForceRefresh = forceRefresh;
      NamespaceImports = namespaceImports;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RefreshValueArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var forceRefresh = reader.ReadBool();
      var namespaceImports = ReadStringList(ctx, reader);
      return new RefreshValueArg(valueId, forceRefresh, namespaceImports);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<RefreshValueArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      writer.Write(value.ForceRefresh);
      WriteStringList(ctx, writer, value.NamespaceImports);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RefreshValueArg) obj);
    }
    public bool Equals(RefreshValueArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && ForceRefresh == other.ForceRefresh && NamespaceImports.SequenceEqual(other.NamespaceImports);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + ForceRefresh.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(NamespaceImports);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RefreshValueArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("forceRefresh = "); ForceRefresh.PrintEx(printer); printer.Println();
        printer.Print("namespaceImports = "); NamespaceImports.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RefreshValueResult : IPrintable, IEquatable<RefreshValueResult>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    [NotNull] public ObjectProxy Value {get; private set;}
    
    //private fields
    //primary constructor
    public RefreshValueResult(
      int valueId,
      [NotNull] ObjectProxy value
    )
    {
      if (value == null) throw new ArgumentNullException("value");
      
      ValueId = valueId;
      Value = value;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RefreshValueResult> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      var value = ObjectProxy.Read(ctx, reader);
      return new RefreshValueResult(valueId, value);
    };
    
    public static CtxWriteDelegate<RefreshValueResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
      ObjectProxy.Write(ctx, writer, value.Value);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RefreshValueResult) obj);
    }
    public bool Equals(RefreshValueResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId && Equals(Value, other.Value);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        hash = hash * 31 + Value.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RefreshValueResult (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
        printer.Print("value = "); Value.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RemoveObjectIdArg : IPrintable, IEquatable<RemoveObjectIdArg>
  {
    //fields
    //public fields
    public int ValueId {get; private set;}
    
    //private fields
    //primary constructor
    public RemoveObjectIdArg(
      int valueId
    )
    {
      ValueId = valueId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RemoveObjectIdArg> Read = (ctx, reader) => 
    {
      var valueId = reader.ReadInt();
      return new RemoveObjectIdArg(valueId);
    };
    
    public static CtxWriteDelegate<RemoveObjectIdArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ValueId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RemoveObjectIdArg) obj);
    }
    public bool Equals(RemoveObjectIdArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ValueId == other.ValueId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ValueId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RemoveObjectIdArg (");
      using (printer.IndentCookie()) {
        printer.Print("valueId = "); ValueId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SequencePointProxy : IPrintable, IEquatable<SequencePointProxy>
  {
    //fields
    //public fields
    public int MethodToken {get; private set;}
    public int Offset {get; private set;}
    public int StartLine {get; private set;}
    public int StartColumn {get; private set;}
    public int EndLine {get; private set;}
    public int EndColumn {get; private set;}
    
    //private fields
    //primary constructor
    public SequencePointProxy(
      int methodToken,
      int offset,
      int startLine,
      int startColumn,
      int endLine,
      int endColumn
    )
    {
      MethodToken = methodToken;
      Offset = offset;
      StartLine = startLine;
      StartColumn = startColumn;
      EndLine = endLine;
      EndColumn = endColumn;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SequencePointProxy> Read = (ctx, reader) => 
    {
      var methodToken = reader.ReadInt();
      var offset = reader.ReadInt();
      var startLine = reader.ReadInt();
      var startColumn = reader.ReadInt();
      var endLine = reader.ReadInt();
      var endColumn = reader.ReadInt();
      return new SequencePointProxy(methodToken, offset, startLine, startColumn, endLine, endColumn);
    };
    
    public static CtxWriteDelegate<SequencePointProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.MethodToken);
      writer.Write(value.Offset);
      writer.Write(value.StartLine);
      writer.Write(value.StartColumn);
      writer.Write(value.EndLine);
      writer.Write(value.EndColumn);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SequencePointProxy) obj);
    }
    public bool Equals(SequencePointProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return MethodToken == other.MethodToken && Offset == other.Offset && StartLine == other.StartLine && StartColumn == other.StartColumn && EndLine == other.EndLine && EndColumn == other.EndColumn;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + MethodToken.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        hash = hash * 31 + StartLine.GetHashCode();
        hash = hash * 31 + StartColumn.GetHashCode();
        hash = hash * 31 + EndLine.GetHashCode();
        hash = hash * 31 + EndColumn.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SequencePointProxy (");
      using (printer.IndentCookie()) {
        printer.Print("methodToken = "); MethodToken.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
        printer.Print("startLine = "); StartLine.PrintEx(printer); printer.Println();
        printer.Print("startColumn = "); StartColumn.PrintEx(printer); printer.Println();
        printer.Print("endLine = "); EndLine.PrintEx(printer); printer.Println();
        printer.Print("endColumn = "); EndColumn.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SequencePointsAndFilePair : IPrintable, IEquatable<SequencePointsAndFilePair>
  {
    //fields
    //public fields
    [NotNull] public string File {get; private set;}
    [NotNull] public List<SequencePointProxy> SequencePoints {get; private set;}
    
    //private fields
    //primary constructor
    public SequencePointsAndFilePair(
      [NotNull] string file,
      [NotNull] List<SequencePointProxy> sequencePoints
    )
    {
      if (file == null) throw new ArgumentNullException("file");
      if (sequencePoints == null) throw new ArgumentNullException("sequencePoints");
      
      File = file;
      SequencePoints = sequencePoints;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SequencePointsAndFilePair> Read = (ctx, reader) => 
    {
      var file = reader.ReadString();
      var sequencePoints = ReadSequencePointProxyList(ctx, reader);
      return new SequencePointsAndFilePair(file, sequencePoints);
    };
    public static CtxReadDelegate<List<SequencePointProxy>> ReadSequencePointProxyList = SequencePointProxy.Read.List();
    
    public static CtxWriteDelegate<SequencePointsAndFilePair> Write = (ctx, writer, value) => 
    {
      writer.Write(value.File);
      WriteSequencePointProxyList(ctx, writer, value.SequencePoints);
    };
    public static CtxWriteDelegate<List<SequencePointProxy>> WriteSequencePointProxyList = SequencePointProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SequencePointsAndFilePair) obj);
    }
    public bool Equals(SequencePointsAndFilePair other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return File == other.File && SequencePoints.SequenceEqual(other.SequencePoints);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + File.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(SequencePoints);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SequencePointsAndFilePair (");
      using (printer.IndentCookie()) {
        printer.Print("file = "); File.PrintEx(printer); printer.Println();
        printer.Print("sequencePoints = "); SequencePoints.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SessionProperties : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public IRdProperty<bool> AllowImplicitTargetInvoke { get { return _AllowImplicitTargetInvoke; }}
    [NotNull] public IRdProperty<bool> AutomaticallyRefreshWatches { get { return _AutomaticallyRefreshWatches; }}
    [NotNull] public IRdProperty<int> EvaluationTimeout { get { return _EvaluationTimeout; }}
    [NotNull] public IRdProperty<bool> BreakOnUserUnhandledExceptions { get { return _BreakOnUserUnhandledExceptions; }}
    [NotNull] public IRdProperty<bool> BreakOnUnhandledExceptions { get { return _BreakOnUnhandledExceptions; }}
    [NotNull] public IRdProperty<bool> ProcessExceptionsOutsideMyCode { get { return _ProcessExceptionsOutsideMyCode; }}
    [NotNull] public IRdProperty<bool> DisableJitOptimizationOnModuleLoad { get { return _DisableJitOptimizationOnModuleLoad; }}
    [NotNull] public IRdProperty<bool> EnableExternalSourceDebug { get { return _EnableExternalSourceDebug; }}
    [NotNull] public IRdProperty<bool> IgnoreMissingSourceFileLocations { get { return _IgnoreMissingSourceFileLocations; }}
    [NotNull] public IRdProperty<bool> ShowFullyQualifiedTypeNames { get { return _ShowFullyQualifiedTypeNames; }}
    [NotNull] public IRdProperty<bool> ObjectFlattenHierarchy { get { return _ObjectFlattenHierarchy; }}
    [NotNull] public IRdProperty<bool> EnableHeuristicPathResolve { get { return _EnableHeuristicPathResolve; }}
    [NotNull] public IRdProperty<bool> ShowHexIntegerValues { get { return _ShowHexIntegerValues; }}
    [NotNull] public IRdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind> DebugKind { get { return _DebugKind; }}
    
    //private fields
    [NotNull] private readonly RdProperty<bool> _AllowImplicitTargetInvoke;
    [NotNull] private readonly RdProperty<bool> _AutomaticallyRefreshWatches;
    [NotNull] private readonly RdProperty<int> _EvaluationTimeout;
    [NotNull] private readonly RdProperty<bool> _BreakOnUserUnhandledExceptions;
    [NotNull] private readonly RdProperty<bool> _BreakOnUnhandledExceptions;
    [NotNull] private readonly RdProperty<bool> _ProcessExceptionsOutsideMyCode;
    [NotNull] private readonly RdProperty<bool> _DisableJitOptimizationOnModuleLoad;
    [NotNull] private readonly RdProperty<bool> _EnableExternalSourceDebug;
    [NotNull] private readonly RdProperty<bool> _IgnoreMissingSourceFileLocations;
    [NotNull] private readonly RdProperty<bool> _ShowFullyQualifiedTypeNames;
    [NotNull] private readonly RdProperty<bool> _ObjectFlattenHierarchy;
    [NotNull] private readonly RdProperty<bool> _EnableHeuristicPathResolve;
    [NotNull] private readonly RdProperty<bool> _ShowHexIntegerValues;
    [NotNull] private readonly RdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind> _DebugKind;
    
    //primary constructor
    private SessionProperties(
      [NotNull] RdProperty<bool> allowImplicitTargetInvoke,
      [NotNull] RdProperty<bool> automaticallyRefreshWatches,
      [NotNull] RdProperty<int> evaluationTimeout,
      [NotNull] RdProperty<bool> breakOnUserUnhandledExceptions,
      [NotNull] RdProperty<bool> breakOnUnhandledExceptions,
      [NotNull] RdProperty<bool> processExceptionsOutsideMyCode,
      [NotNull] RdProperty<bool> disableJitOptimizationOnModuleLoad,
      [NotNull] RdProperty<bool> enableExternalSourceDebug,
      [NotNull] RdProperty<bool> ignoreMissingSourceFileLocations,
      [NotNull] RdProperty<bool> showFullyQualifiedTypeNames,
      [NotNull] RdProperty<bool> objectFlattenHierarchy,
      [NotNull] RdProperty<bool> enableHeuristicPathResolve,
      [NotNull] RdProperty<bool> showHexIntegerValues,
      [NotNull] RdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind> debugKind
    )
    {
      if (allowImplicitTargetInvoke == null) throw new ArgumentNullException("allowImplicitTargetInvoke");
      if (automaticallyRefreshWatches == null) throw new ArgumentNullException("automaticallyRefreshWatches");
      if (evaluationTimeout == null) throw new ArgumentNullException("evaluationTimeout");
      if (breakOnUserUnhandledExceptions == null) throw new ArgumentNullException("breakOnUserUnhandledExceptions");
      if (breakOnUnhandledExceptions == null) throw new ArgumentNullException("breakOnUnhandledExceptions");
      if (processExceptionsOutsideMyCode == null) throw new ArgumentNullException("processExceptionsOutsideMyCode");
      if (disableJitOptimizationOnModuleLoad == null) throw new ArgumentNullException("disableJitOptimizationOnModuleLoad");
      if (enableExternalSourceDebug == null) throw new ArgumentNullException("enableExternalSourceDebug");
      if (ignoreMissingSourceFileLocations == null) throw new ArgumentNullException("ignoreMissingSourceFileLocations");
      if (showFullyQualifiedTypeNames == null) throw new ArgumentNullException("showFullyQualifiedTypeNames");
      if (objectFlattenHierarchy == null) throw new ArgumentNullException("objectFlattenHierarchy");
      if (enableHeuristicPathResolve == null) throw new ArgumentNullException("enableHeuristicPathResolve");
      if (showHexIntegerValues == null) throw new ArgumentNullException("showHexIntegerValues");
      if (debugKind == null) throw new ArgumentNullException("debugKind");
      
      _AllowImplicitTargetInvoke = allowImplicitTargetInvoke;
      _AutomaticallyRefreshWatches = automaticallyRefreshWatches;
      _EvaluationTimeout = evaluationTimeout;
      _BreakOnUserUnhandledExceptions = breakOnUserUnhandledExceptions;
      _BreakOnUnhandledExceptions = breakOnUnhandledExceptions;
      _ProcessExceptionsOutsideMyCode = processExceptionsOutsideMyCode;
      _DisableJitOptimizationOnModuleLoad = disableJitOptimizationOnModuleLoad;
      _EnableExternalSourceDebug = enableExternalSourceDebug;
      _IgnoreMissingSourceFileLocations = ignoreMissingSourceFileLocations;
      _ShowFullyQualifiedTypeNames = showFullyQualifiedTypeNames;
      _ObjectFlattenHierarchy = objectFlattenHierarchy;
      _EnableHeuristicPathResolve = enableHeuristicPathResolve;
      _ShowHexIntegerValues = showHexIntegerValues;
      _DebugKind = debugKind;
      _AllowImplicitTargetInvoke.OptimizeNested = true;
      _AutomaticallyRefreshWatches.OptimizeNested = true;
      _EvaluationTimeout.OptimizeNested = true;
      _BreakOnUserUnhandledExceptions.OptimizeNested = true;
      _BreakOnUnhandledExceptions.OptimizeNested = true;
      _ProcessExceptionsOutsideMyCode.OptimizeNested = true;
      _DisableJitOptimizationOnModuleLoad.OptimizeNested = true;
      _EnableExternalSourceDebug.OptimizeNested = true;
      _IgnoreMissingSourceFileLocations.OptimizeNested = true;
      _ShowFullyQualifiedTypeNames.OptimizeNested = true;
      _ObjectFlattenHierarchy.OptimizeNested = true;
      _EnableHeuristicPathResolve.OptimizeNested = true;
      _ShowHexIntegerValues.OptimizeNested = true;
      _DebugKind.OptimizeNested = true;
      _AllowImplicitTargetInvoke.Async = true;
      _AutomaticallyRefreshWatches.Async = true;
      _EvaluationTimeout.Async = true;
      _BreakOnUserUnhandledExceptions.Async = true;
      _BreakOnUnhandledExceptions.Async = true;
      _ProcessExceptionsOutsideMyCode.Async = true;
      _DisableJitOptimizationOnModuleLoad.Async = true;
      _EnableExternalSourceDebug.Async = true;
      _IgnoreMissingSourceFileLocations.Async = true;
      _ShowFullyQualifiedTypeNames.Async = true;
      _ObjectFlattenHierarchy.Async = true;
      _EnableHeuristicPathResolve.Async = true;
      _ShowHexIntegerValues.Async = true;
      _DebugKind.Async = true;
      BindableChildren.Add(new KeyValuePair<string, object>("allowImplicitTargetInvoke", _AllowImplicitTargetInvoke));
      BindableChildren.Add(new KeyValuePair<string, object>("automaticallyRefreshWatches", _AutomaticallyRefreshWatches));
      BindableChildren.Add(new KeyValuePair<string, object>("evaluationTimeout", _EvaluationTimeout));
      BindableChildren.Add(new KeyValuePair<string, object>("breakOnUserUnhandledExceptions", _BreakOnUserUnhandledExceptions));
      BindableChildren.Add(new KeyValuePair<string, object>("breakOnUnhandledExceptions", _BreakOnUnhandledExceptions));
      BindableChildren.Add(new KeyValuePair<string, object>("processExceptionsOutsideMyCode", _ProcessExceptionsOutsideMyCode));
      BindableChildren.Add(new KeyValuePair<string, object>("disableJitOptimizationOnModuleLoad", _DisableJitOptimizationOnModuleLoad));
      BindableChildren.Add(new KeyValuePair<string, object>("enableExternalSourceDebug", _EnableExternalSourceDebug));
      BindableChildren.Add(new KeyValuePair<string, object>("ignoreMissingSourceFileLocations", _IgnoreMissingSourceFileLocations));
      BindableChildren.Add(new KeyValuePair<string, object>("showFullyQualifiedTypeNames", _ShowFullyQualifiedTypeNames));
      BindableChildren.Add(new KeyValuePair<string, object>("objectFlattenHierarchy", _ObjectFlattenHierarchy));
      BindableChildren.Add(new KeyValuePair<string, object>("enableHeuristicPathResolve", _EnableHeuristicPathResolve));
      BindableChildren.Add(new KeyValuePair<string, object>("showHexIntegerValues", _ShowHexIntegerValues));
      BindableChildren.Add(new KeyValuePair<string, object>("debugKind", _DebugKind));
    }
    //secondary constructor
    public SessionProperties (
    ) : this (
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<int>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind>(ReadDebugKind, WriteDebugKind)
    ) {}
    //statics
    
    public static CtxReadDelegate<SessionProperties> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var allowImplicitTargetInvoke = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var automaticallyRefreshWatches = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var evaluationTimeout = RdProperty<int>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt);
      var breakOnUserUnhandledExceptions = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var breakOnUnhandledExceptions = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var processExceptionsOutsideMyCode = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var disableJitOptimizationOnModuleLoad = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var enableExternalSourceDebug = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var ignoreMissingSourceFileLocations = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var showFullyQualifiedTypeNames = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var objectFlattenHierarchy = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var enableHeuristicPathResolve = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var showHexIntegerValues = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var debugKind = RdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind>.Read(ctx, reader, ReadDebugKind, WriteDebugKind);
      return new SessionProperties(allowImplicitTargetInvoke, automaticallyRefreshWatches, evaluationTimeout, breakOnUserUnhandledExceptions, breakOnUnhandledExceptions, processExceptionsOutsideMyCode, disableJitOptimizationOnModuleLoad, enableExternalSourceDebug, ignoreMissingSourceFileLocations, showFullyQualifiedTypeNames, objectFlattenHierarchy, enableHeuristicPathResolve, showHexIntegerValues, debugKind).WithId(_id);
    };
    public static CtxReadDelegate<JetBrains.Rider.Model.DebuggerWorker.DebugKind> ReadDebugKind = new CtxReadDelegate<JetBrains.Rider.Model.DebuggerWorker.DebugKind>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadEnum<JetBrains.Rider.Model.DebuggerWorker.DebugKind>);
    
    public static CtxWriteDelegate<SessionProperties> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      RdProperty<bool>.Write(ctx, writer, value._AllowImplicitTargetInvoke);
      RdProperty<bool>.Write(ctx, writer, value._AutomaticallyRefreshWatches);
      RdProperty<int>.Write(ctx, writer, value._EvaluationTimeout);
      RdProperty<bool>.Write(ctx, writer, value._BreakOnUserUnhandledExceptions);
      RdProperty<bool>.Write(ctx, writer, value._BreakOnUnhandledExceptions);
      RdProperty<bool>.Write(ctx, writer, value._ProcessExceptionsOutsideMyCode);
      RdProperty<bool>.Write(ctx, writer, value._DisableJitOptimizationOnModuleLoad);
      RdProperty<bool>.Write(ctx, writer, value._EnableExternalSourceDebug);
      RdProperty<bool>.Write(ctx, writer, value._IgnoreMissingSourceFileLocations);
      RdProperty<bool>.Write(ctx, writer, value._ShowFullyQualifiedTypeNames);
      RdProperty<bool>.Write(ctx, writer, value._ObjectFlattenHierarchy);
      RdProperty<bool>.Write(ctx, writer, value._EnableHeuristicPathResolve);
      RdProperty<bool>.Write(ctx, writer, value._ShowHexIntegerValues);
      RdProperty<JetBrains.Rider.Model.DebuggerWorker.DebugKind>.Write(ctx, writer, value._DebugKind);
    };
    public static CtxWriteDelegate<JetBrains.Rider.Model.DebuggerWorker.DebugKind> WriteDebugKind = new CtxWriteDelegate<JetBrains.Rider.Model.DebuggerWorker.DebugKind>(JetBrains.Platform.RdFramework.Impl.Serializers.WriteEnum<JetBrains.Rider.Model.DebuggerWorker.DebugKind>);
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("SessionProperties (");
      using (printer.IndentCookie()) {
        printer.Print("allowImplicitTargetInvoke = "); _AllowImplicitTargetInvoke.PrintEx(printer); printer.Println();
        printer.Print("automaticallyRefreshWatches = "); _AutomaticallyRefreshWatches.PrintEx(printer); printer.Println();
        printer.Print("evaluationTimeout = "); _EvaluationTimeout.PrintEx(printer); printer.Println();
        printer.Print("breakOnUserUnhandledExceptions = "); _BreakOnUserUnhandledExceptions.PrintEx(printer); printer.Println();
        printer.Print("breakOnUnhandledExceptions = "); _BreakOnUnhandledExceptions.PrintEx(printer); printer.Println();
        printer.Print("processExceptionsOutsideMyCode = "); _ProcessExceptionsOutsideMyCode.PrintEx(printer); printer.Println();
        printer.Print("disableJitOptimizationOnModuleLoad = "); _DisableJitOptimizationOnModuleLoad.PrintEx(printer); printer.Println();
        printer.Print("enableExternalSourceDebug = "); _EnableExternalSourceDebug.PrintEx(printer); printer.Println();
        printer.Print("ignoreMissingSourceFileLocations = "); _IgnoreMissingSourceFileLocations.PrintEx(printer); printer.Println();
        printer.Print("showFullyQualifiedTypeNames = "); _ShowFullyQualifiedTypeNames.PrintEx(printer); printer.Println();
        printer.Print("objectFlattenHierarchy = "); _ObjectFlattenHierarchy.PrintEx(printer); printer.Println();
        printer.Print("enableHeuristicPathResolve = "); _EnableHeuristicPathResolve.PrintEx(printer); printer.Println();
        printer.Print("showHexIntegerValues = "); _ShowHexIntegerValues.PrintEx(printer); printer.Println();
        printer.Print("debugKind = "); _DebugKind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SetValueArg : IPrintable, IEquatable<SetValueArg>
  {
    //fields
    //public fields
    public int ObjectToEditId {get; private set;}
    [NotNull] public string Value {get; private set;}
    
    //private fields
    //primary constructor
    public SetValueArg(
      int objectToEditId,
      [NotNull] string value
    )
    {
      if (value == null) throw new ArgumentNullException("value");
      
      ObjectToEditId = objectToEditId;
      Value = value;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SetValueArg> Read = (ctx, reader) => 
    {
      var objectToEditId = reader.ReadInt();
      var value = reader.ReadString();
      return new SetValueArg(objectToEditId, value);
    };
    
    public static CtxWriteDelegate<SetValueArg> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ObjectToEditId);
      writer.Write(value.Value);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SetValueArg) obj);
    }
    public bool Equals(SetValueArg other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ObjectToEditId == other.ObjectToEditId && Value == other.Value;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ObjectToEditId.GetHashCode();
        hash = hash * 31 + Value.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SetValueArg (");
      using (printer.IndentCookie()) {
        printer.Print("objectToEditId = "); ObjectToEditId.PrintEx(printer); printer.Println();
        printer.Print("value = "); Value.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class ShowNotification : IPrintable, IEquatable<ShowNotification>
  {
    //fields
    //public fields
    [NotNull] public JetBrains.Rider.Model.DebuggerWorker.OutputMessage OutputMessage {get; private set;}
    public bool AddToEventLog {get; private set;}
    
    //private fields
    //primary constructor
    public ShowNotification(
      [NotNull] JetBrains.Rider.Model.DebuggerWorker.OutputMessage outputMessage,
      bool addToEventLog
    )
    {
      if (outputMessage == null) throw new ArgumentNullException("outputMessage");
      
      OutputMessage = outputMessage;
      AddToEventLog = addToEventLog;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ShowNotification> Read = (ctx, reader) => 
    {
      var outputMessage = JetBrains.Rider.Model.DebuggerWorker.OutputMessage.Read(ctx, reader);
      var addToEventLog = reader.ReadBool();
      return new ShowNotification(outputMessage, addToEventLog);
    };
    
    public static CtxWriteDelegate<ShowNotification> Write = (ctx, writer, value) => 
    {
      JetBrains.Rider.Model.DebuggerWorker.OutputMessage.Write(ctx, writer, value.OutputMessage);
      writer.Write(value.AddToEventLog);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ShowNotification) obj);
    }
    public bool Equals(ShowNotification other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(OutputMessage, other.OutputMessage) && AddToEventLog == other.AddToEventLog;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + OutputMessage.GetHashCode();
        hash = hash * 31 + AddToEventLog.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ShowNotification (");
      using (printer.IndentCookie()) {
        printer.Print("outputMessage = "); OutputMessage.PrintEx(printer); printer.Println();
        printer.Print("addToEventLog = "); AddToEventLog.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SmartStepIntoCommandProxy : ExecutionCommandBase
  {
    //fields
    //public fields
    [NotNull] public SmartStepIntoInfoProxy Info {get; private set;}
    
    //private fields
    //primary constructor
    public SmartStepIntoCommandProxy(
      [NotNull] SmartStepIntoInfoProxy info,
      ExecutionCommandKind kind
    ) : base (
      kind
     ) 
    {
      if (info == null) throw new ArgumentNullException("info");
      
      Info = info;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<SmartStepIntoCommandProxy> Read = (ctx, reader) => 
    {
      var kind = (ExecutionCommandKind)reader.ReadInt();
      var info = SmartStepIntoInfoProxy.Read(ctx, reader);
      return new SmartStepIntoCommandProxy(info, kind);
    };
    
    public static new CtxWriteDelegate<SmartStepIntoCommandProxy> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Kind);
      SmartStepIntoInfoProxy.Write(ctx, writer, value.Info);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SmartStepIntoCommandProxy) obj);
    }
    public bool Equals(SmartStepIntoCommandProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(Info, other.Info) && Kind == other.Kind;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Info.GetHashCode();
        hash = hash * 31 + (int) Kind;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SmartStepIntoCommandProxy (");
      using (printer.IndentCookie()) {
        printer.Print("info = "); Info.PrintEx(printer); printer.Println();
        printer.Print("kind = "); Kind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SmartStepIntoElementProxy : IPrintable, IEquatable<SmartStepIntoElementProxy>
  {
    //fields
    //public fields
    [NotNull] public string Identifier {get; private set;}
    public int MetadataToken {get; private set;}
    public long Offset {get; private set;}
    [CanBeNull] public string Presentation {get; private set;}
    public int DocumentStartOffset {get; private set;}
    public int DocumentEndOffset {get; private set;}
    public bool IsHidden {get; private set;}
    
    //private fields
    //primary constructor
    public SmartStepIntoElementProxy(
      [NotNull] string identifier,
      int metadataToken,
      long offset,
      [CanBeNull] string presentation,
      int documentStartOffset,
      int documentEndOffset,
      bool isHidden
    )
    {
      if (identifier == null) throw new ArgumentNullException("identifier");
      
      Identifier = identifier;
      MetadataToken = metadataToken;
      Offset = offset;
      Presentation = presentation;
      DocumentStartOffset = documentStartOffset;
      DocumentEndOffset = documentEndOffset;
      IsHidden = isHidden;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SmartStepIntoElementProxy> Read = (ctx, reader) => 
    {
      var identifier = reader.ReadString();
      var metadataToken = reader.ReadInt();
      var offset = reader.ReadLong();
      var presentation = ReadStringNullable(ctx, reader);
      var documentStartOffset = reader.ReadInt();
      var documentEndOffset = reader.ReadInt();
      var isHidden = reader.ReadBool();
      return new SmartStepIntoElementProxy(identifier, metadataToken, offset, presentation, documentStartOffset, documentEndOffset, isHidden);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<SmartStepIntoElementProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Identifier);
      writer.Write(value.MetadataToken);
      writer.Write(value.Offset);
      WriteStringNullable(ctx, writer, value.Presentation);
      writer.Write(value.DocumentStartOffset);
      writer.Write(value.DocumentEndOffset);
      writer.Write(value.IsHidden);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SmartStepIntoElementProxy) obj);
    }
    public bool Equals(SmartStepIntoElementProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Identifier == other.Identifier && MetadataToken == other.MetadataToken && Offset == other.Offset && Equals(Presentation, other.Presentation) && DocumentStartOffset == other.DocumentStartOffset && DocumentEndOffset == other.DocumentEndOffset && IsHidden == other.IsHidden;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Identifier.GetHashCode();
        hash = hash * 31 + MetadataToken.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        hash = hash * 31 + (Presentation != null ?Presentation.GetHashCode() : 0);
        hash = hash * 31 + DocumentStartOffset.GetHashCode();
        hash = hash * 31 + DocumentEndOffset.GetHashCode();
        hash = hash * 31 + IsHidden.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SmartStepIntoElementProxy (");
      using (printer.IndentCookie()) {
        printer.Print("identifier = "); Identifier.PrintEx(printer); printer.Println();
        printer.Print("metadataToken = "); MetadataToken.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
        printer.Print("presentation = "); Presentation.PrintEx(printer); printer.Println();
        printer.Print("documentStartOffset = "); DocumentStartOffset.PrintEx(printer); printer.Println();
        printer.Print("documentEndOffset = "); DocumentEndOffset.PrintEx(printer); printer.Println();
        printer.Print("isHidden = "); IsHidden.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SmartStepIntoInfoProxy : IPrintable, IEquatable<SmartStepIntoInfoProxy>
  {
    //fields
    //public fields
    public int CurrentIndex {get; private set;}
    public int TargetIndex {get; private set;}
    [NotNull] public List<SmartStepIntoElementProxy> Elements {get; private set;}
    
    //private fields
    //primary constructor
    public SmartStepIntoInfoProxy(
      int currentIndex,
      int targetIndex,
      [NotNull] List<SmartStepIntoElementProxy> elements
    )
    {
      if (elements == null) throw new ArgumentNullException("elements");
      
      CurrentIndex = currentIndex;
      TargetIndex = targetIndex;
      Elements = elements;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SmartStepIntoInfoProxy> Read = (ctx, reader) => 
    {
      var currentIndex = reader.ReadInt();
      var targetIndex = reader.ReadInt();
      var elements = ReadSmartStepIntoElementProxyList(ctx, reader);
      return new SmartStepIntoInfoProxy(currentIndex, targetIndex, elements);
    };
    public static CtxReadDelegate<List<SmartStepIntoElementProxy>> ReadSmartStepIntoElementProxyList = SmartStepIntoElementProxy.Read.List();
    
    public static CtxWriteDelegate<SmartStepIntoInfoProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.CurrentIndex);
      writer.Write(value.TargetIndex);
      WriteSmartStepIntoElementProxyList(ctx, writer, value.Elements);
    };
    public static CtxWriteDelegate<List<SmartStepIntoElementProxy>> WriteSmartStepIntoElementProxyList = SmartStepIntoElementProxy.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SmartStepIntoInfoProxy) obj);
    }
    public bool Equals(SmartStepIntoInfoProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return CurrentIndex == other.CurrentIndex && TargetIndex == other.TargetIndex && Elements.SequenceEqual(other.Elements);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + CurrentIndex.GetHashCode();
        hash = hash * 31 + TargetIndex.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Elements);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SmartStepIntoInfoProxy (");
      using (printer.IndentCookie()) {
        printer.Print("currentIndex = "); CurrentIndex.PrintEx(printer); printer.Println();
        printer.Print("targetIndex = "); TargetIndex.PrintEx(printer); printer.Println();
        printer.Print("elements = "); Elements.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class SourcePositionProxy : IPrintable, IEquatable<SourcePositionProxy>
  {
    //fields
    //public fields
    [NotNull] public string Filename {get; private set;}
    public int Line {get; private set;}
    public int Column {get; private set;}
    public int EndLine {get; private set;}
    public int EndColumn {get; private set;}
    [NotNull] public string MethodName {get; private set;}
    
    //private fields
    //primary constructor
    public SourcePositionProxy(
      [NotNull] string filename,
      int line,
      int column,
      int endLine,
      int endColumn,
      [NotNull] string methodName
    )
    {
      if (filename == null) throw new ArgumentNullException("filename");
      if (methodName == null) throw new ArgumentNullException("methodName");
      
      Filename = filename;
      Line = line;
      Column = column;
      EndLine = endLine;
      EndColumn = endColumn;
      MethodName = methodName;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<SourcePositionProxy> Read = (ctx, reader) => 
    {
      var filename = reader.ReadString();
      var line = reader.ReadInt();
      var column = reader.ReadInt();
      var endLine = reader.ReadInt();
      var endColumn = reader.ReadInt();
      var methodName = reader.ReadString();
      return new SourcePositionProxy(filename, line, column, endLine, endColumn, methodName);
    };
    
    public static CtxWriteDelegate<SourcePositionProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Filename);
      writer.Write(value.Line);
      writer.Write(value.Column);
      writer.Write(value.EndLine);
      writer.Write(value.EndColumn);
      writer.Write(value.MethodName);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((SourcePositionProxy) obj);
    }
    public bool Equals(SourcePositionProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Filename == other.Filename && Line == other.Line && Column == other.Column && EndLine == other.EndLine && EndColumn == other.EndColumn && MethodName == other.MethodName;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Filename.GetHashCode();
        hash = hash * 31 + Line.GetHashCode();
        hash = hash * 31 + Column.GetHashCode();
        hash = hash * 31 + EndLine.GetHashCode();
        hash = hash * 31 + EndColumn.GetHashCode();
        hash = hash * 31 + MethodName.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("SourcePositionProxy (");
      using (printer.IndentCookie()) {
        printer.Print("filename = "); Filename.PrintEx(printer); printer.Println();
        printer.Print("line = "); Line.PrintEx(printer); printer.Println();
        printer.Print("column = "); Column.PrintEx(printer); printer.Println();
        printer.Print("endLine = "); EndLine.PrintEx(printer); printer.Println();
        printer.Print("endColumn = "); EndColumn.PrintEx(printer); printer.Println();
        printer.Print("methodName = "); MethodName.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class StackFrameProxy : IPrintable, IEquatable<StackFrameProxy>
  {
    //fields
    //public fields
    public int Id {get; private set;}
    [CanBeNull] public SourcePositionProxy Location {get; private set;}
    [NotNull] public string MethodName {get; private set;}
    [NotNull] public string TypeName {get; private set;}
    [NotNull] public string NamespaceName {get; private set;}
    [NotNull] public string ModuleName {get; private set;}
    public bool IsAsync {get; private set;}
    public bool IsExternalCode {get; private set;}
    public bool IsDebugInfoMatchesSources {get; private set;}
    public bool ContainsBreakpoint {get; private set;}
    public JetBrains.Rider.Model.DebuggerWorker.FrameKind FrameKind {get; private set;}
    public bool CanRefresh {get; private set;}
    
    //private fields
    //primary constructor
    public StackFrameProxy(
      int id,
      [CanBeNull] SourcePositionProxy location,
      [NotNull] string methodName,
      [NotNull] string typeName,
      [NotNull] string namespaceName,
      [NotNull] string moduleName,
      bool isAsync,
      bool isExternalCode,
      bool isDebugInfoMatchesSources,
      bool containsBreakpoint,
      JetBrains.Rider.Model.DebuggerWorker.FrameKind frameKind,
      bool canRefresh
    )
    {
      if (methodName == null) throw new ArgumentNullException("methodName");
      if (typeName == null) throw new ArgumentNullException("typeName");
      if (namespaceName == null) throw new ArgumentNullException("namespaceName");
      if (moduleName == null) throw new ArgumentNullException("moduleName");
      
      Id = id;
      Location = location;
      MethodName = methodName;
      TypeName = typeName;
      NamespaceName = namespaceName;
      ModuleName = moduleName;
      IsAsync = isAsync;
      IsExternalCode = isExternalCode;
      IsDebugInfoMatchesSources = isDebugInfoMatchesSources;
      ContainsBreakpoint = containsBreakpoint;
      FrameKind = frameKind;
      CanRefresh = canRefresh;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<StackFrameProxy> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var location = ReadSourcePositionProxyNullable(ctx, reader);
      var methodName = reader.ReadString();
      var typeName = reader.ReadString();
      var namespaceName = reader.ReadString();
      var moduleName = reader.ReadString();
      var isAsync = reader.ReadBool();
      var isExternalCode = reader.ReadBool();
      var isDebugInfoMatchesSources = reader.ReadBool();
      var containsBreakpoint = reader.ReadBool();
      var frameKind = (JetBrains.Rider.Model.DebuggerWorker.FrameKind)reader.ReadInt();
      var canRefresh = reader.ReadBool();
      return new StackFrameProxy(id, location, methodName, typeName, namespaceName, moduleName, isAsync, isExternalCode, isDebugInfoMatchesSources, containsBreakpoint, frameKind, canRefresh);
    };
    public static CtxReadDelegate<SourcePositionProxy> ReadSourcePositionProxyNullable = SourcePositionProxy.Read.NullableClass();
    
    public static CtxWriteDelegate<StackFrameProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      WriteSourcePositionProxyNullable(ctx, writer, value.Location);
      writer.Write(value.MethodName);
      writer.Write(value.TypeName);
      writer.Write(value.NamespaceName);
      writer.Write(value.ModuleName);
      writer.Write(value.IsAsync);
      writer.Write(value.IsExternalCode);
      writer.Write(value.IsDebugInfoMatchesSources);
      writer.Write(value.ContainsBreakpoint);
      writer.Write((int)value.FrameKind);
      writer.Write(value.CanRefresh);
    };
    public static CtxWriteDelegate<SourcePositionProxy> WriteSourcePositionProxyNullable = SourcePositionProxy.Write.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((StackFrameProxy) obj);
    }
    public bool Equals(StackFrameProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && Equals(Location, other.Location) && MethodName == other.MethodName && TypeName == other.TypeName && NamespaceName == other.NamespaceName && ModuleName == other.ModuleName && IsAsync == other.IsAsync && IsExternalCode == other.IsExternalCode && IsDebugInfoMatchesSources == other.IsDebugInfoMatchesSources && ContainsBreakpoint == other.ContainsBreakpoint && FrameKind == other.FrameKind && CanRefresh == other.CanRefresh;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + (Location != null ?Location.GetHashCode() : 0);
        hash = hash * 31 + MethodName.GetHashCode();
        hash = hash * 31 + TypeName.GetHashCode();
        hash = hash * 31 + NamespaceName.GetHashCode();
        hash = hash * 31 + ModuleName.GetHashCode();
        hash = hash * 31 + IsAsync.GetHashCode();
        hash = hash * 31 + IsExternalCode.GetHashCode();
        hash = hash * 31 + IsDebugInfoMatchesSources.GetHashCode();
        hash = hash * 31 + ContainsBreakpoint.GetHashCode();
        hash = hash * 31 + (int) FrameKind;
        hash = hash * 31 + CanRefresh.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("StackFrameProxy (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("location = "); Location.PrintEx(printer); printer.Println();
        printer.Print("methodName = "); MethodName.PrintEx(printer); printer.Println();
        printer.Print("typeName = "); TypeName.PrintEx(printer); printer.Println();
        printer.Print("namespaceName = "); NamespaceName.PrintEx(printer); printer.Println();
        printer.Print("moduleName = "); ModuleName.PrintEx(printer); printer.Println();
        printer.Print("isAsync = "); IsAsync.PrintEx(printer); printer.Println();
        printer.Print("isExternalCode = "); IsExternalCode.PrintEx(printer); printer.Println();
        printer.Print("isDebugInfoMatchesSources = "); IsDebugInfoMatchesSources.PrintEx(printer); printer.Println();
        printer.Print("containsBreakpoint = "); ContainsBreakpoint.PrintEx(printer); printer.Println();
        printer.Print("frameKind = "); FrameKind.PrintEx(printer); printer.Println();
        printer.Print("canRefresh = "); CanRefresh.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class StackProxy : IPrintable, IEquatable<StackProxy>
  {
    //fields
    //public fields
    public int Id {get; private set;}
    public int ThreadId {get; private set;}
    [NotNull] public string Name {get; private set;}
    public int FrameCount {get; private set;}
    [CanBeNull] public StackFrameProxy TopFrame {get; private set;}
    
    //private fields
    //primary constructor
    public StackProxy(
      int id,
      int threadId,
      [NotNull] string name,
      int frameCount,
      [CanBeNull] StackFrameProxy topFrame
    )
    {
      if (name == null) throw new ArgumentNullException("name");
      
      Id = id;
      ThreadId = threadId;
      Name = name;
      FrameCount = frameCount;
      TopFrame = topFrame;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<StackProxy> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var threadId = reader.ReadInt();
      var name = reader.ReadString();
      var frameCount = reader.ReadInt();
      var topFrame = ReadStackFrameProxyNullable(ctx, reader);
      return new StackProxy(id, threadId, name, frameCount, topFrame);
    };
    public static CtxReadDelegate<StackFrameProxy> ReadStackFrameProxyNullable = StackFrameProxy.Read.NullableClass();
    
    public static CtxWriteDelegate<StackProxy> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      writer.Write(value.ThreadId);
      writer.Write(value.Name);
      writer.Write(value.FrameCount);
      WriteStackFrameProxyNullable(ctx, writer, value.TopFrame);
    };
    public static CtxWriteDelegate<StackFrameProxy> WriteStackFrameProxyNullable = StackFrameProxy.Write.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((StackProxy) obj);
    }
    public bool Equals(StackProxy other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && ThreadId == other.ThreadId && Name == other.Name && FrameCount == other.FrameCount && Equals(TopFrame, other.TopFrame);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + ThreadId.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        hash = hash * 31 + FrameCount.GetHashCode();
        hash = hash * 31 + (TopFrame != null ?TopFrame.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("StackProxy (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("threadId = "); ThreadId.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
        printer.Print("frameCount = "); FrameCount.PrintEx(printer); printer.Println();
        printer.Print("topFrame = "); TopFrame.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class StringPair : IPrintable, IEquatable<StringPair>
  {
    //fields
    //public fields
    [NotNull] public string First {get; private set;}
    [NotNull] public string Second {get; private set;}
    
    //private fields
    //primary constructor
    public StringPair(
      [NotNull] string first,
      [NotNull] string second
    )
    {
      if (first == null) throw new ArgumentNullException("first");
      if (second == null) throw new ArgumentNullException("second");
      
      First = first;
      Second = second;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<StringPair> Read = (ctx, reader) => 
    {
      var first = reader.ReadString();
      var second = reader.ReadString();
      return new StringPair(first, second);
    };
    
    public static CtxWriteDelegate<StringPair> Write = (ctx, writer, value) => 
    {
      writer.Write(value.First);
      writer.Write(value.Second);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((StringPair) obj);
    }
    public bool Equals(StringPair other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return First == other.First && Second == other.Second;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + First.GetHashCode();
        hash = hash * 31 + Second.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("StringPair (");
      using (printer.IndentCookie()) {
        printer.Print("first = "); First.PrintEx(printer); printer.Println();
        printer.Print("second = "); Second.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class TargetExitedArgs : IPrintable, IEquatable<TargetExitedArgs>
  {
    //fields
    //public fields
    [CanBeNull] public int? ExitCode {get; private set;}
    
    //private fields
    //primary constructor
    public TargetExitedArgs(
      [CanBeNull] int? exitCode
    )
    {
      ExitCode = exitCode;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<TargetExitedArgs> Read = (ctx, reader) => 
    {
      var exitCode = ReadIntNullable(ctx, reader);
      return new TargetExitedArgs(exitCode);
    };
    public static CtxReadDelegate<int?> ReadIntNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt.NullableStruct();
    
    public static CtxWriteDelegate<TargetExitedArgs> Write = (ctx, writer, value) => 
    {
      WriteIntNullable(ctx, writer, value.ExitCode);
    };
    public static CtxWriteDelegate<int?> WriteIntNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt.NullableStruct();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((TargetExitedArgs) obj);
    }
    public bool Equals(TargetExitedArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(ExitCode, other.ExitCode);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (ExitCode != null ?ExitCode.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("TargetExitedArgs (");
      using (printer.IndentCookie()) {
        printer.Print("exitCode = "); ExitCode.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class XamarinAndroidStartInfo : XamarinStartInfoBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public XamarinAndroidStartInfo(
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      port,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinAndroidStartInfo> Read = (ctx, reader) => 
    {
      var port = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new XamarinAndroidStartInfo(port, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<XamarinAndroidStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Port);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((XamarinAndroidStartInfo) obj);
    }
    public bool Equals(XamarinAndroidStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Port == other.Port && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Port.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("XamarinAndroidStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("port = "); Port.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class XamarinIOSUsbStartInfo : XamarinStartInfoBase
  {
    //fields
    //public fields
    [NotNull] public string DeviceUdid {get; private set;}
    
    //private fields
    //primary constructor
    public XamarinIOSUsbStartInfo(
      [NotNull] string deviceUdid,
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      port,
      needToBeInitializedImmediately
     ) 
    {
      if (deviceUdid == null) throw new ArgumentNullException("deviceUdid");
      
      DeviceUdid = deviceUdid;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinIOSUsbStartInfo> Read = (ctx, reader) => 
    {
      var port = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      var deviceUdid = reader.ReadString();
      return new XamarinIOSUsbStartInfo(deviceUdid, port, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<XamarinIOSUsbStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Port);
      writer.Write(value.NeedToBeInitializedImmediately);
      writer.Write(value.DeviceUdid);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((XamarinIOSUsbStartInfo) obj);
    }
    public bool Equals(XamarinIOSUsbStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return DeviceUdid == other.DeviceUdid && Port == other.Port && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DeviceUdid.GetHashCode();
        hash = hash * 31 + Port.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("XamarinIOSUsbStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("deviceUdid = "); DeviceUdid.PrintEx(printer); printer.Println();
        printer.Print("port = "); Port.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class XamarinIOSWiFiStartInfo : XamarinStartInfoBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public XamarinIOSWiFiStartInfo(
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      port,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinIOSWiFiStartInfo> Read = (ctx, reader) => 
    {
      var port = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new XamarinIOSWiFiStartInfo(port, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<XamarinIOSWiFiStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Port);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((XamarinIOSWiFiStartInfo) obj);
    }
    public bool Equals(XamarinIOSWiFiStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Port == other.Port && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Port.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("XamarinIOSWiFiStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("port = "); Port.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class XamarinMacStartInfo : XamarinStartInfoBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public XamarinMacStartInfo(
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      port,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinMacStartInfo> Read = (ctx, reader) => 
    {
      var port = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new XamarinMacStartInfo(port, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<XamarinMacStartInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Port);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((XamarinMacStartInfo) obj);
    }
    public bool Equals(XamarinMacStartInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Port == other.Port && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Port.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("XamarinMacStartInfo (");
      using (printer.IndentCookie()) {
        printer.Print("port = "); Port.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class XamarinStartInfoBase : DebuggerStartInfoBase
  {
    //fields
    //public fields
    public int Port {get; private set;}
    
    //private fields
    //primary constructor
    protected XamarinStartInfoBase(
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      needToBeInitializedImmediately
     ) 
    {
      Port = port;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinStartInfoBase> Read = Polymorphic<XamarinStartInfoBase>.ReadAbstract(XamarinStartInfoBase_Unknown.Read);
    
    public static new CtxWriteDelegate<XamarinStartInfoBase> Write = Polymorphic<XamarinStartInfoBase>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class XamarinStartInfoBase_Unknown : XamarinStartInfoBase
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    public XamarinStartInfoBase_Unknown(
      int port,
      bool needToBeInitializedImmediately
    ) : base (
      port,
      needToBeInitializedImmediately
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<XamarinStartInfoBase_Unknown> Read = (ctx, reader) => 
    {
      var port = reader.ReadInt();
      var needToBeInitializedImmediately = reader.ReadBool();
      return new XamarinStartInfoBase_Unknown(port, needToBeInitializedImmediately);
    };
    
    public static new CtxWriteDelegate<XamarinStartInfoBase_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Port);
      writer.Write(value.NeedToBeInitializedImmediately);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((XamarinStartInfoBase_Unknown) obj);
    }
    public bool Equals(XamarinStartInfoBase_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Port == other.Port && NeedToBeInitializedImmediately == other.NeedToBeInitializedImmediately;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Port.GetHashCode();
        hash = hash * 31 + NeedToBeInitializedImmediately.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("XamarinStartInfoBase_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("port = "); Port.PrintEx(printer); printer.Println();
        printer.Print("needToBeInitializedImmediately = "); NeedToBeInitializedImmediately.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
