using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model
{
  
  
  public class DiagnosticsModel : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public RdMemoryUsageIndicator MemoryUsageIndicator {get; private set;}
    [NotNull] public IViewableMap<string, RdSpecialPath> SpecialPaths { get { return _SpecialPaths; }}
    [NotNull] public IRdProperty<bool> Recording { get { return _Recording; }}
    [NotNull] public IRdProperty<AssertMode> BackendAssertMode { get { return _BackendAssertMode; }}
    
    //private fields
    [NotNull] private readonly RdMap<string, RdSpecialPath> _SpecialPaths;
    [NotNull] private readonly RdProperty<bool> _Recording;
    [NotNull] private readonly RdProperty<AssertMode> _BackendAssertMode;
    
    //primary constructor
    private DiagnosticsModel(
      [NotNull] RdMemoryUsageIndicator memoryUsageIndicator,
      [NotNull] RdMap<string, RdSpecialPath> specialPaths,
      [NotNull] RdProperty<bool> recording,
      [NotNull] RdProperty<AssertMode> backendAssertMode
    )
    {
      if (memoryUsageIndicator == null) throw new ArgumentNullException("memoryUsageIndicator");
      if (specialPaths == null) throw new ArgumentNullException("specialPaths");
      if (recording == null) throw new ArgumentNullException("recording");
      if (backendAssertMode == null) throw new ArgumentNullException("backendAssertMode");
      
      MemoryUsageIndicator = memoryUsageIndicator;
      _SpecialPaths = specialPaths;
      _Recording = recording;
      _BackendAssertMode = backendAssertMode;
      _Recording.OptimizeNested = true;
      _BackendAssertMode.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("memoryUsageIndicator", MemoryUsageIndicator));
      BindableChildren.Add(new KeyValuePair<string, object>("specialPaths", _SpecialPaths));
      BindableChildren.Add(new KeyValuePair<string, object>("recording", _Recording));
      BindableChildren.Add(new KeyValuePair<string, object>("backendAssertMode", _BackendAssertMode));
    }
    //secondary constructor
    internal DiagnosticsModel (
    ) : this (
      new RdMemoryUsageIndicator(),
      new RdMap<string, RdSpecialPath>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RdSpecialPath.Read, RdSpecialPath.Write),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<AssertMode>(ReadAssertMode, WriteAssertMode)
    ) {}
    //statics
    
    public static CtxReadDelegate<AssertMode> ReadAssertMode = new CtxReadDelegate<AssertMode>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadEnum<AssertMode>);
    
    public static CtxWriteDelegate<AssertMode> WriteAssertMode = new CtxWriteDelegate<AssertMode>(JetBrains.Platform.RdFramework.Impl.Serializers.WriteEnum<AssertMode>);
    
    protected override long SerializationHash => -7024726161859754755L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.RegisterEnum<AssertMode>();
      serializers.Register(RdMemoryUsageIndicator.Read, RdMemoryUsageIndicator.Write);
      serializers.Register(RdSpecialPath.Read, RdSpecialPath.Write);
      serializers.Register(RdMemoryUsageState.Read, RdMemoryUsageState.Write);
      serializers.Register(RdProtocolStatistic.Read, RdProtocolStatistic.Write);
      serializers.RegisterEnum<RdSpecialPathKind>();
      
      serializers.RegisterToplevelOnce(typeof(IdeRoot), IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DiagnosticsModel (");
      using (printer.IndentCookie()) {
        printer.Print("memoryUsageIndicator = "); MemoryUsageIndicator.PrintEx(printer); printer.Println();
        printer.Print("specialPaths = "); _SpecialPaths.PrintEx(printer); printer.Println();
        printer.Print("recording = "); _Recording.PrintEx(printer); printer.Println();
        printer.Print("backendAssertMode = "); _BackendAssertMode.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  public static class SolutionDiagnosticsModelEx
   {
    public static DiagnosticsModel GetDiagnosticsModel(this Solution solution)
    {
      return solution.GetOrCreateExtension("diagnosticsModel", () => new DiagnosticsModel());
    }
  }
  
  
  public enum AssertMode {
    Relaxed,
    ThrowAssert
  }
  
  
  public class RdMemoryUsageIndicator : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public IRdProperty<bool> IsEnable { get { return _IsEnable; }}
    [NotNull] public IRdProperty<RdMemoryUsageState> State { get { return _State; }}
    [NotNull] public ISink<RdVoid> GcCollect { get { return _GcCollect; }}
    
    //private fields
    [NotNull] private readonly RdProperty<bool> _IsEnable;
    [NotNull] private readonly RdProperty<RdMemoryUsageState> _State;
    [NotNull] private readonly RdSignal<RdVoid> _GcCollect;
    
    //primary constructor
    private RdMemoryUsageIndicator(
      [NotNull] RdProperty<bool> isEnable,
      [NotNull] RdProperty<RdMemoryUsageState> state,
      [NotNull] RdSignal<RdVoid> gcCollect
    )
    {
      if (isEnable == null) throw new ArgumentNullException("isEnable");
      if (state == null) throw new ArgumentNullException("state");
      if (gcCollect == null) throw new ArgumentNullException("gcCollect");
      
      _IsEnable = isEnable;
      _State = state;
      _GcCollect = gcCollect;
      _IsEnable.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("isEnable", _IsEnable));
      BindableChildren.Add(new KeyValuePair<string, object>("state", _State));
      BindableChildren.Add(new KeyValuePair<string, object>("gcCollect", _GcCollect));
    }
    //secondary constructor
    public RdMemoryUsageIndicator (
    ) : this (
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<RdMemoryUsageState>(RdMemoryUsageState.Read, RdMemoryUsageState.Write),
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid)
    ) {}
    //statics
    
    public static CtxReadDelegate<RdMemoryUsageIndicator> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var isEnable = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var state = RdProperty<RdMemoryUsageState>.Read(ctx, reader, RdMemoryUsageState.Read, RdMemoryUsageState.Write);
      var gcCollect = RdSignal<RdVoid>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      return new RdMemoryUsageIndicator(isEnable, state, gcCollect).WithId(_id);
    };
    
    public static CtxWriteDelegate<RdMemoryUsageIndicator> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      RdProperty<bool>.Write(ctx, writer, value._IsEnable);
      RdProperty<RdMemoryUsageState>.Write(ctx, writer, value._State);
      RdSignal<RdVoid>.Write(ctx, writer, value._GcCollect);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdMemoryUsageIndicator (");
      using (printer.IndentCookie()) {
        printer.Print("isEnable = "); _IsEnable.PrintEx(printer); printer.Println();
        printer.Print("state = "); _State.PrintEx(printer); printer.Println();
        printer.Print("gcCollect = "); _GcCollect.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdMemoryUsageState : RdBindableBase
  {
    //fields
    //public fields
    public long PrivateMemorySize {get; private set;}
    public long VirtualMemorySize {get; private set;}
    public long WorkingSet {get; private set;}
    public long AllocatedManagedMemory {get; private set;}
    [NotNull] public List<int> CollectionCount {get; private set;}
    public int LohCollectionCount {get; private set;}
    [NotNull] public List<RdProtocolStatistic> ProtocolStats {get; private set;}
    
    //private fields
    //primary constructor
    public RdMemoryUsageState(
      long privateMemorySize,
      long virtualMemorySize,
      long workingSet,
      long allocatedManagedMemory,
      [NotNull] List<int> collectionCount,
      int lohCollectionCount,
      [NotNull] List<RdProtocolStatistic> protocolStats
    )
    {
      if (collectionCount == null) throw new ArgumentNullException("collectionCount");
      if (protocolStats == null) throw new ArgumentNullException("protocolStats");
      
      PrivateMemorySize = privateMemorySize;
      VirtualMemorySize = virtualMemorySize;
      WorkingSet = workingSet;
      AllocatedManagedMemory = allocatedManagedMemory;
      CollectionCount = collectionCount;
      LohCollectionCount = lohCollectionCount;
      ProtocolStats = protocolStats;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdMemoryUsageState> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var privateMemorySize = reader.ReadLong();
      var virtualMemorySize = reader.ReadLong();
      var workingSet = reader.ReadLong();
      var allocatedManagedMemory = reader.ReadLong();
      var collectionCount = ReadIntList(ctx, reader);
      var lohCollectionCount = reader.ReadInt();
      var protocolStats = ReadRdProtocolStatisticList(ctx, reader);
      return new RdMemoryUsageState(privateMemorySize, virtualMemorySize, workingSet, allocatedManagedMemory, collectionCount, lohCollectionCount, protocolStats).WithId(_id);
    };
    public static CtxReadDelegate<List<int>> ReadIntList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt.List();
    public static CtxReadDelegate<List<RdProtocolStatistic>> ReadRdProtocolStatisticList = RdProtocolStatistic.Read.List();
    
    public static CtxWriteDelegate<RdMemoryUsageState> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.PrivateMemorySize);
      writer.Write(value.VirtualMemorySize);
      writer.Write(value.WorkingSet);
      writer.Write(value.AllocatedManagedMemory);
      WriteIntList(ctx, writer, value.CollectionCount);
      writer.Write(value.LohCollectionCount);
      WriteRdProtocolStatisticList(ctx, writer, value.ProtocolStats);
    };
    public static CtxWriteDelegate<List<int>> WriteIntList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt.List();
    public static CtxWriteDelegate<List<RdProtocolStatistic>> WriteRdProtocolStatisticList = RdProtocolStatistic.Write.List();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdMemoryUsageState (");
      using (printer.IndentCookie()) {
        printer.Print("privateMemorySize = "); PrivateMemorySize.PrintEx(printer); printer.Println();
        printer.Print("virtualMemorySize = "); VirtualMemorySize.PrintEx(printer); printer.Println();
        printer.Print("workingSet = "); WorkingSet.PrintEx(printer); printer.Println();
        printer.Print("allocatedManagedMemory = "); AllocatedManagedMemory.PrintEx(printer); printer.Println();
        printer.Print("collectionCount = "); CollectionCount.PrintEx(printer); printer.Println();
        printer.Print("lohCollectionCount = "); LohCollectionCount.PrintEx(printer); printer.Println();
        printer.Print("protocolStats = "); ProtocolStats.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdProtocolStatistic : IPrintable, IEquatable<RdProtocolStatistic>
  {
    //fields
    //public fields
    [NotNull] public string ProtocolName {get; private set;}
    public long ProtocolBytesSent {get; private set;}
    public long ProtocolBytesReceived {get; private set;}
    
    //private fields
    //primary constructor
    public RdProtocolStatistic(
      [NotNull] string protocolName,
      long protocolBytesSent,
      long protocolBytesReceived
    )
    {
      if (protocolName == null) throw new ArgumentNullException("protocolName");
      
      ProtocolName = protocolName;
      ProtocolBytesSent = protocolBytesSent;
      ProtocolBytesReceived = protocolBytesReceived;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdProtocolStatistic> Read = (ctx, reader) => 
    {
      var protocolName = reader.ReadString();
      var protocolBytesSent = reader.ReadLong();
      var protocolBytesReceived = reader.ReadLong();
      return new RdProtocolStatistic(protocolName, protocolBytesSent, protocolBytesReceived);
    };
    
    public static CtxWriteDelegate<RdProtocolStatistic> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProtocolName);
      writer.Write(value.ProtocolBytesSent);
      writer.Write(value.ProtocolBytesReceived);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdProtocolStatistic) obj);
    }
    public bool Equals(RdProtocolStatistic other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProtocolName == other.ProtocolName && ProtocolBytesSent == other.ProtocolBytesSent && ProtocolBytesReceived == other.ProtocolBytesReceived;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProtocolName.GetHashCode();
        hash = hash * 31 + ProtocolBytesSent.GetHashCode();
        hash = hash * 31 + ProtocolBytesReceived.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdProtocolStatistic (");
      using (printer.IndentCookie()) {
        printer.Print("protocolName = "); ProtocolName.PrintEx(printer); printer.Println();
        printer.Print("protocolBytesSent = "); ProtocolBytesSent.PrintEx(printer); printer.Println();
        printer.Print("protocolBytesReceived = "); ProtocolBytesReceived.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdSpecialPath : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public string Id {get; private set;}
    [NotNull] public string Description {get; private set;}
    [NotNull] public string FullPath {get; private set;}
    public RdSpecialPathKind Kind {get; private set;}
    
    //private fields
    //primary constructor
    public RdSpecialPath(
      [NotNull] string id,
      [NotNull] string description,
      [NotNull] string fullPath,
      RdSpecialPathKind kind
    )
    {
      if (id == null) throw new ArgumentNullException("id");
      if (description == null) throw new ArgumentNullException("description");
      if (fullPath == null) throw new ArgumentNullException("fullPath");
      
      Id = id;
      Description = description;
      FullPath = fullPath;
      Kind = kind;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdSpecialPath> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var id = reader.ReadString();
      var description = reader.ReadString();
      var fullPath = reader.ReadString();
      var kind = (RdSpecialPathKind)reader.ReadInt();
      return new RdSpecialPath(id, description, fullPath, kind).WithId(_id);
    };
    
    public static CtxWriteDelegate<RdSpecialPath> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.Id);
      writer.Write(value.Description);
      writer.Write(value.FullPath);
      writer.Write((int)value.Kind);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdSpecialPath (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("description = "); Description.PrintEx(printer); printer.Println();
        printer.Print("fullPath = "); FullPath.PrintEx(printer); printer.Println();
        printer.Print("kind = "); Kind.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdSpecialPathKind {
    File,
    Folder
  }
}
