using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model
{
  
  
  public class FileSystemModel : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public IRdCall<string, RdOperationResult> CreateFile { get { return _CreateFile; }}
    [NotNull] public IRdCall<string, RdOperationResult> CreateDirectory { get { return _CreateDirectory; }}
    [NotNull] public IRdCall<string, RdOperationResult> Delete { get { return _Delete; }}
    [NotNull] public IRdCall<RdFromTo, RdOperationResult> Move { get { return _Move; }}
    [NotNull] public IRdCall<RdFromTo, RdOperationResult> Copy { get { return _Copy; }}
    [NotNull] public IRdCall<RdRefreshRequest, RdVoid> RefreshPaths { get { return _RefreshPaths; }}
    [NotNull] public IRdCall<FilesToSave, bool> RefreshFiles { get { return _RefreshFiles; }}
    [NotNull] public IRdProperty<RdSyncRequest> SyncRequest { get { return _SyncRequest; }}
    [NotNull] public ISink<RdDeltaBatch> Change { get { return _Change; }}
    [NotNull] public IViewableMap<string, RdWatchRootInfo> WatchRoots { get { return _WatchRoots; }}
    [NotNull] public IRdProperty<bool> FileSystemTrackerReady { get { return _FileSystemTrackerReady; }}
    [NotNull] public ISource<string> FileSystemChanged { get { return _FileSystemChanged; }}
    
    //private fields
    [NotNull] private readonly RdCall<string, RdOperationResult> _CreateFile;
    [NotNull] private readonly RdCall<string, RdOperationResult> _CreateDirectory;
    [NotNull] private readonly RdCall<string, RdOperationResult> _Delete;
    [NotNull] private readonly RdCall<RdFromTo, RdOperationResult> _Move;
    [NotNull] private readonly RdCall<RdFromTo, RdOperationResult> _Copy;
    [NotNull] private readonly RdCall<RdRefreshRequest, RdVoid> _RefreshPaths;
    [NotNull] private readonly RdCall<FilesToSave, bool> _RefreshFiles;
    [NotNull] private readonly RdProperty<RdSyncRequest> _SyncRequest;
    [NotNull] private readonly RdSignal<RdDeltaBatch> _Change;
    [NotNull] private readonly RdMap<string, RdWatchRootInfo> _WatchRoots;
    [NotNull] private readonly RdProperty<bool> _FileSystemTrackerReady;
    [NotNull] private readonly RdSignal<string> _FileSystemChanged;
    
    //primary constructor
    private FileSystemModel(
      [NotNull] RdCall<string, RdOperationResult> createFile,
      [NotNull] RdCall<string, RdOperationResult> createDirectory,
      [NotNull] RdCall<string, RdOperationResult> delete,
      [NotNull] RdCall<RdFromTo, RdOperationResult> move,
      [NotNull] RdCall<RdFromTo, RdOperationResult> copy,
      [NotNull] RdCall<RdRefreshRequest, RdVoid> refreshPaths,
      [NotNull] RdCall<FilesToSave, bool> refreshFiles,
      [NotNull] RdProperty<RdSyncRequest> syncRequest,
      [NotNull] RdSignal<RdDeltaBatch> change,
      [NotNull] RdMap<string, RdWatchRootInfo> watchRoots,
      [NotNull] RdProperty<bool> fileSystemTrackerReady,
      [NotNull] RdSignal<string> fileSystemChanged
    )
    {
      if (createFile == null) throw new ArgumentNullException("createFile");
      if (createDirectory == null) throw new ArgumentNullException("createDirectory");
      if (delete == null) throw new ArgumentNullException("delete");
      if (move == null) throw new ArgumentNullException("move");
      if (copy == null) throw new ArgumentNullException("copy");
      if (refreshPaths == null) throw new ArgumentNullException("refreshPaths");
      if (refreshFiles == null) throw new ArgumentNullException("refreshFiles");
      if (syncRequest == null) throw new ArgumentNullException("syncRequest");
      if (change == null) throw new ArgumentNullException("change");
      if (watchRoots == null) throw new ArgumentNullException("watchRoots");
      if (fileSystemTrackerReady == null) throw new ArgumentNullException("fileSystemTrackerReady");
      if (fileSystemChanged == null) throw new ArgumentNullException("fileSystemChanged");
      
      _CreateFile = createFile;
      _CreateDirectory = createDirectory;
      _Delete = delete;
      _Move = move;
      _Copy = copy;
      _RefreshPaths = refreshPaths;
      _RefreshFiles = refreshFiles;
      _SyncRequest = syncRequest;
      _Change = change;
      _WatchRoots = watchRoots;
      _FileSystemTrackerReady = fileSystemTrackerReady;
      _FileSystemChanged = fileSystemChanged;
      _WatchRoots.OptimizeNested = true;
      _FileSystemTrackerReady.OptimizeNested = true;
      _FileSystemTrackerReady.Async = true;
      _SyncRequest.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("createFile", _CreateFile));
      BindableChildren.Add(new KeyValuePair<string, object>("createDirectory", _CreateDirectory));
      BindableChildren.Add(new KeyValuePair<string, object>("delete", _Delete));
      BindableChildren.Add(new KeyValuePair<string, object>("move", _Move));
      BindableChildren.Add(new KeyValuePair<string, object>("copy", _Copy));
      BindableChildren.Add(new KeyValuePair<string, object>("refreshPaths", _RefreshPaths));
      BindableChildren.Add(new KeyValuePair<string, object>("refreshFiles", _RefreshFiles));
      BindableChildren.Add(new KeyValuePair<string, object>("syncRequest", _SyncRequest));
      BindableChildren.Add(new KeyValuePair<string, object>("change", _Change));
      BindableChildren.Add(new KeyValuePair<string, object>("watchRoots", _WatchRoots));
      BindableChildren.Add(new KeyValuePair<string, object>("fileSystemTrackerReady", _FileSystemTrackerReady));
      BindableChildren.Add(new KeyValuePair<string, object>("fileSystemChanged", _FileSystemChanged));
    }
    //secondary constructor
    internal FileSystemModel (
    ) : this (
      new RdCall<string, RdOperationResult>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RdOperationResult.Read, RdOperationResult.Write),
      new RdCall<string, RdOperationResult>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RdOperationResult.Read, RdOperationResult.Write),
      new RdCall<string, RdOperationResult>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RdOperationResult.Read, RdOperationResult.Write),
      new RdCall<RdFromTo, RdOperationResult>(RdFromTo.Read, RdFromTo.Write, RdOperationResult.Read, RdOperationResult.Write),
      new RdCall<RdFromTo, RdOperationResult>(RdFromTo.Read, RdFromTo.Write, RdOperationResult.Read, RdOperationResult.Write),
      new RdCall<RdRefreshRequest, RdVoid>(RdRefreshRequest.Read, RdRefreshRequest.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdCall<FilesToSave, bool>(FilesToSave.Read, FilesToSave.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<RdSyncRequest>(ReadRdSyncRequestNullable, WriteRdSyncRequestNullable),
      new RdSignal<RdDeltaBatch>(RdDeltaBatch.Read, RdDeltaBatch.Write),
      new RdMap<string, RdWatchRootInfo>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RdWatchRootInfo.Read, RdWatchRootInfo.Write),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString)
    ) {}
    //statics
    
    public static CtxReadDelegate<RdSyncRequest> ReadRdSyncRequestNullable = RdSyncRequest.Read.NullableClass();
    
    public static CtxWriteDelegate<RdSyncRequest> WriteRdSyncRequestNullable = RdSyncRequest.Write.NullableClass();
    
    protected override long SerializationHash => 2978447440511601031L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(RdFromTo.Read, RdFromTo.Write);
      serializers.Register(FileToSave.Read, FileToSave.Write);
      serializers.Register(FilesToSave.Read, FilesToSave.Write);
      serializers.Register(RdRefreshRequest.Read, RdRefreshRequest.Write);
      serializers.Register(RdDelta.Read, RdDelta.Write);
      serializers.Register(RdDeltaBatch.Read, RdDeltaBatch.Write);
      serializers.Register(RdOperationResult.Read, RdOperationResult.Write);
      serializers.Register(RdWatchRootInfo.Read, RdWatchRootInfo.Write);
      serializers.Register(RdSyncRequest.Read, RdSyncRequest.Write);
      serializers.RegisterEnum<RdDeltaType>();
      
      serializers.RegisterToplevelOnce(typeof(IdeRoot), IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("FileSystemModel (");
      using (printer.IndentCookie()) {
        printer.Print("createFile = "); _CreateFile.PrintEx(printer); printer.Println();
        printer.Print("createDirectory = "); _CreateDirectory.PrintEx(printer); printer.Println();
        printer.Print("delete = "); _Delete.PrintEx(printer); printer.Println();
        printer.Print("move = "); _Move.PrintEx(printer); printer.Println();
        printer.Print("copy = "); _Copy.PrintEx(printer); printer.Println();
        printer.Print("refreshPaths = "); _RefreshPaths.PrintEx(printer); printer.Println();
        printer.Print("refreshFiles = "); _RefreshFiles.PrintEx(printer); printer.Println();
        printer.Print("syncRequest = "); _SyncRequest.PrintEx(printer); printer.Println();
        printer.Print("change = "); _Change.PrintEx(printer); printer.Println();
        printer.Print("watchRoots = "); _WatchRoots.PrintEx(printer); printer.Println();
        printer.Print("fileSystemTrackerReady = "); _FileSystemTrackerReady.PrintEx(printer); printer.Println();
        printer.Print("fileSystemChanged = "); _FileSystemChanged.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  public static class SolutionFileSystemModelEx
   {
    public static FileSystemModel GetFileSystemModel(this Solution solution)
    {
      return solution.GetOrCreateExtension("fileSystemModel", () => new FileSystemModel());
    }
  }
  
  
  public class FileToSave : IPrintable, IEquatable<FileToSave>
  {
    //fields
    //public fields
    [NotNull] public string Path {get; private set;}
    [NotNull] public string FullText {get; private set;}
    
    //private fields
    //primary constructor
    public FileToSave(
      [NotNull] string path,
      [NotNull] string fullText
    )
    {
      if (path == null) throw new ArgumentNullException("path");
      if (fullText == null) throw new ArgumentNullException("fullText");
      
      Path = path;
      FullText = fullText;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<FileToSave> Read = (ctx, reader) => 
    {
      var path = reader.ReadString();
      var fullText = reader.ReadString();
      return new FileToSave(path, fullText);
    };
    
    public static CtxWriteDelegate<FileToSave> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Path);
      writer.Write(value.FullText);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((FileToSave) obj);
    }
    public bool Equals(FileToSave other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Path == other.Path && FullText == other.FullText;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Path.GetHashCode();
        hash = hash * 31 + FullText.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("FileToSave (");
      using (printer.IndentCookie()) {
        printer.Print("path = "); Path.PrintEx(printer); printer.Println();
        printer.Print("fullText = "); FullText.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class FilesToSave : IPrintable, IEquatable<FilesToSave>
  {
    //fields
    //public fields
    [NotNull] public List<FileToSave> Files {get; private set;}
    
    //private fields
    //primary constructor
    public FilesToSave(
      [NotNull] List<FileToSave> files
    )
    {
      if (files == null) throw new ArgumentNullException("files");
      
      Files = files;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<FilesToSave> Read = (ctx, reader) => 
    {
      var files = ReadFileToSaveList(ctx, reader);
      return new FilesToSave(files);
    };
    public static CtxReadDelegate<List<FileToSave>> ReadFileToSaveList = FileToSave.Read.List();
    
    public static CtxWriteDelegate<FilesToSave> Write = (ctx, writer, value) => 
    {
      WriteFileToSaveList(ctx, writer, value.Files);
    };
    public static CtxWriteDelegate<List<FileToSave>> WriteFileToSaveList = FileToSave.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((FilesToSave) obj);
    }
    public bool Equals(FilesToSave other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Files.SequenceEqual(other.Files);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Files);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("FilesToSave (");
      using (printer.IndentCookie()) {
        printer.Print("files = "); Files.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdDelta : IPrintable, IEquatable<RdDelta>
  {
    //fields
    //public fields
    [NotNull] public string Path {get; private set;}
    public RdDeltaType Type {get; private set;}
    
    //private fields
    //primary constructor
    public RdDelta(
      [NotNull] string path,
      RdDeltaType type
    )
    {
      if (path == null) throw new ArgumentNullException("path");
      
      Path = path;
      Type = type;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdDelta> Read = (ctx, reader) => 
    {
      var path = reader.ReadString();
      var type = (RdDeltaType)reader.ReadInt();
      return new RdDelta(path, type);
    };
    
    public static CtxWriteDelegate<RdDelta> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Path);
      writer.Write((int)value.Type);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdDelta) obj);
    }
    public bool Equals(RdDelta other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Path == other.Path && Type == other.Type;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Path.GetHashCode();
        hash = hash * 31 + (int) Type;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdDelta (");
      using (printer.IndentCookie()) {
        printer.Print("path = "); Path.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdDeltaBatch : IPrintable, IEquatable<RdDeltaBatch>
  {
    //fields
    //public fields
    [NotNull] public List<RdDelta> Deltas {get; private set;}
    
    //private fields
    //primary constructor
    public RdDeltaBatch(
      [NotNull] List<RdDelta> deltas
    )
    {
      if (deltas == null) throw new ArgumentNullException("deltas");
      
      Deltas = deltas;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdDeltaBatch> Read = (ctx, reader) => 
    {
      var deltas = ReadRdDeltaList(ctx, reader);
      return new RdDeltaBatch(deltas);
    };
    public static CtxReadDelegate<List<RdDelta>> ReadRdDeltaList = RdDelta.Read.List();
    
    public static CtxWriteDelegate<RdDeltaBatch> Write = (ctx, writer, value) => 
    {
      WriteRdDeltaList(ctx, writer, value.Deltas);
    };
    public static CtxWriteDelegate<List<RdDelta>> WriteRdDeltaList = RdDelta.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdDeltaBatch) obj);
    }
    public bool Equals(RdDeltaBatch other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Deltas.SequenceEqual(other.Deltas);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Deltas);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdDeltaBatch (");
      using (printer.IndentCookie()) {
        printer.Print("deltas = "); Deltas.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdDeltaType {
    Created,
    Deleted,
    Changed
  }
  
  
  public class RdFromTo : IPrintable, IEquatable<RdFromTo>
  {
    //fields
    //public fields
    [NotNull] public string From {get; private set;}
    [NotNull] public string To {get; private set;}
    public bool Overwrite {get; private set;}
    
    //private fields
    //primary constructor
    public RdFromTo(
      [NotNull] string from,
      [NotNull] string to,
      bool overwrite
    )
    {
      if (from == null) throw new ArgumentNullException("from");
      if (to == null) throw new ArgumentNullException("to");
      
      From = from;
      To = to;
      Overwrite = overwrite;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdFromTo> Read = (ctx, reader) => 
    {
      var from = reader.ReadString();
      var to = reader.ReadString();
      var overwrite = reader.ReadBool();
      return new RdFromTo(from, to, overwrite);
    };
    
    public static CtxWriteDelegate<RdFromTo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.From);
      writer.Write(value.To);
      writer.Write(value.Overwrite);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdFromTo) obj);
    }
    public bool Equals(RdFromTo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return From == other.From && To == other.To && Overwrite == other.Overwrite;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + From.GetHashCode();
        hash = hash * 31 + To.GetHashCode();
        hash = hash * 31 + Overwrite.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdFromTo (");
      using (printer.IndentCookie()) {
        printer.Print("from = "); From.PrintEx(printer); printer.Println();
        printer.Print("to = "); To.PrintEx(printer); printer.Println();
        printer.Print("overwrite = "); Overwrite.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdOperationResult : IPrintable, IEquatable<RdOperationResult>
  {
    //fields
    //public fields
    public bool Success {get; private set;}
    [CanBeNull] public string Error {get; private set;}
    
    //private fields
    //primary constructor
    public RdOperationResult(
      bool success,
      [CanBeNull] string error
    )
    {
      Success = success;
      Error = error;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdOperationResult> Read = (ctx, reader) => 
    {
      var success = reader.ReadBool();
      var error = ReadStringNullable(ctx, reader);
      return new RdOperationResult(success, error);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<RdOperationResult> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Success);
      WriteStringNullable(ctx, writer, value.Error);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdOperationResult) obj);
    }
    public bool Equals(RdOperationResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Success == other.Success && Equals(Error, other.Error);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Success.GetHashCode();
        hash = hash * 31 + (Error != null ?Error.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdOperationResult (");
      using (printer.IndentCookie()) {
        printer.Print("success = "); Success.PrintEx(printer); printer.Println();
        printer.Print("error = "); Error.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdRefreshRequest : IPrintable, IEquatable<RdRefreshRequest>
  {
    //fields
    //public fields
    [NotNull] public List<string> Paths {get; private set;}
    public bool Async {get; private set;}
    
    //private fields
    //primary constructor
    public RdRefreshRequest(
      [NotNull] List<string> paths,
      bool async
    )
    {
      if (paths == null) throw new ArgumentNullException("paths");
      
      Paths = paths;
      Async = async;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdRefreshRequest> Read = (ctx, reader) => 
    {
      var paths = ReadStringList(ctx, reader);
      var async = reader.ReadBool();
      return new RdRefreshRequest(paths, async);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<RdRefreshRequest> Write = (ctx, writer, value) => 
    {
      WriteStringList(ctx, writer, value.Paths);
      writer.Write(value.Async);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdRefreshRequest) obj);
    }
    public bool Equals(RdRefreshRequest other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Paths.SequenceEqual(other.Paths) && Async == other.Async;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Paths);
        hash = hash * 31 + Async.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdRefreshRequest (");
      using (printer.IndentCookie()) {
        printer.Print("paths = "); Paths.PrintEx(printer); printer.Println();
        printer.Print("async = "); Async.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdSyncRequest : RdBindableBase
  {
    //fields
    //public fields
    [NotNull] public ISink<RdVoid> Ready { get { return _Ready; }}
    
    //private fields
    [NotNull] private readonly RdSignal<RdVoid> _Ready;
    
    //primary constructor
    private RdSyncRequest(
      [NotNull] RdSignal<RdVoid> ready
    )
    {
      if (ready == null) throw new ArgumentNullException("ready");
      
      _Ready = ready;
      BindableChildren.Add(new KeyValuePair<string, object>("ready", _Ready));
    }
    //secondary constructor
    public RdSyncRequest (
    ) : this (
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid)
    ) {}
    //statics
    
    public static CtxReadDelegate<RdSyncRequest> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var ready = RdSignal<RdVoid>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      return new RdSyncRequest(ready).WithId(_id);
    };
    
    public static CtxWriteDelegate<RdSyncRequest> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      RdSignal<RdVoid>.Write(ctx, writer, value._Ready);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdSyncRequest (");
      using (printer.IndentCookie()) {
        printer.Print("ready = "); _Ready.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdWatchRootInfo : IPrintable, IEquatable<RdWatchRootInfo>
  {
    //fields
    //public fields
    public bool IsDirectory {get; private set;}
    public bool IsRecursive {get; private set;}
    
    //private fields
    //primary constructor
    public RdWatchRootInfo(
      bool isDirectory,
      bool isRecursive
    )
    {
      IsDirectory = isDirectory;
      IsRecursive = isRecursive;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdWatchRootInfo> Read = (ctx, reader) => 
    {
      var isDirectory = reader.ReadBool();
      var isRecursive = reader.ReadBool();
      return new RdWatchRootInfo(isDirectory, isRecursive);
    };
    
    public static CtxWriteDelegate<RdWatchRootInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.IsDirectory);
      writer.Write(value.IsRecursive);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdWatchRootInfo) obj);
    }
    public bool Equals(RdWatchRootInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return IsDirectory == other.IsDirectory && IsRecursive == other.IsRecursive;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + IsDirectory.GetHashCode();
        hash = hash * 31 + IsRecursive.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdWatchRootInfo (");
      using (printer.IndentCookie()) {
        printer.Print("isDirectory = "); IsDirectory.PrintEx(printer); printer.Println();
        printer.Print("isRecursive = "); IsRecursive.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
