using System;
using System.Linq;
using System.Collections.Generic;
using JetBrains.Application;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.debuggerHelper
{
  
  
  [ShellComponent]
  public class DebuggerHelperHostModel : RdBindableBase {
    //fields
    [NotNull] public RdEndpoint<int, bool> IsProcess64Bit { get { return _IsProcess64Bit; }}
    [NotNull] private readonly RdEndpoint<int, bool> _IsProcess64Bit;
    [NotNull] public RdEndpoint<string, bool> IsExeWillRunAs64Bit { get { return _IsExeWillRunAs64Bit; }}
    [NotNull] private readonly RdEndpoint<string, bool> _IsExeWillRunAs64Bit;
    [NotNull] public RdEndpoint<string, UnityProcessInfoList> GetUnityProcesses { get { return _GetUnityProcesses; }}
    [NotNull] private readonly RdEndpoint<string, UnityProcessInfoList> _GetUnityProcesses;
    
    //primary constructor
    public DebuggerHelperHostModel(
      [NotNull] RdEndpoint<int, bool> isProcess64Bit,
      [NotNull] RdEndpoint<string, bool> isExeWillRunAs64Bit,
      [NotNull] RdEndpoint<string, UnityProcessInfoList> getUnityProcesses
    )
    {
      _IsProcess64Bit = isProcess64Bit;
      _IsExeWillRunAs64Bit = isExeWillRunAs64Bit;
      _GetUnityProcesses = getUnityProcesses;
    }
    //secondary constructor
    //statics
    
    private void Register(ISerializers serializers)
    {
      serializers.Register(UnityProcessInfo.Read, UnityProcessInfo.Write);
      serializers.Register(UnityProcessInfoList.Read, UnityProcessInfoList.Write);
    }
    
    public DebuggerHelperHostModel(Lifetime lifetime, IProtocol protocol) : this (
      new RdEndpoint<int, bool>().Static(1006),
      new RdEndpoint<string, bool>().Static(1007),
      new RdEndpoint<string, UnityProcessInfoList>().Static(1008)
    )
    {
      Register(protocol.Serializers);
      Bind(lifetime, protocol, GetType().Name);
    }
    //init method
    protected override void Init(Lifetime lifetime) {
      _IsProcess64Bit.BindEx(lifetime, this, "isProcess64Bit");
      _IsExeWillRunAs64Bit.BindEx(lifetime, this, "isExeWillRunAs64Bit");
      _GetUnityProcesses.BindEx(lifetime, this, "getUnityProcesses");
    }
    //identify method
    public override void Identify(IIdentities ids) {
      _IsProcess64Bit.IdentifyEx(ids);
      _IsExeWillRunAs64Bit.IdentifyEx(ids);
      _GetUnityProcesses.IdentifyEx(ids);
    }
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerHelperHostModel (");
      using (printer.IndentCookie()) {
        printer.Print("isProcess64Bit = "); _IsProcess64Bit.PrintEx(printer); printer.Println();
        printer.Print("isExeWillRunAs64Bit = "); _IsExeWillRunAs64Bit.PrintEx(printer); printer.Println();
        printer.Print("getUnityProcesses = "); _GetUnityProcesses.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class UnityProcessInfo : IPrintable, IEquatable<UnityProcessInfo> {
    //fields
    public int Pid {get; private set;}
    [NotNull] public string Name {get; private set;}
    
    //primary constructor
    public UnityProcessInfo(
      int pid,
      [NotNull] string name
    )
    {
      Pid = pid;
      Name = name;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<UnityProcessInfo> Read = (ctx, reader) => 
    {
      var pid = reader.ReadInt();
      var name = reader.ReadString();
      return new UnityProcessInfo(pid, name);
    };
    
    public static CtxWriteDelegate<UnityProcessInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Pid);
      writer.Write(value.Name);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((UnityProcessInfo) obj);
    }
    public bool Equals(UnityProcessInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Pid == other.Pid && Name == other.Name;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + Pid.GetHashCode();
        __r = __r*31 + Name.GetHashCode();
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("UnityProcessInfo (");
      using (printer.IndentCookie()) {
        printer.Print("pid = "); Pid.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class UnityProcessInfoList : IPrintable, IEquatable<UnityProcessInfoList> {
    //fields
    [NotNull] public List<UnityProcessInfo> Processes {get; private set;}
    
    //primary constructor
    public UnityProcessInfoList(
      [NotNull] List<UnityProcessInfo> processes
    )
    {
      Processes = processes;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<List<UnityProcessInfo>> ReadUnityProcessInfoList = UnityProcessInfo.Read.List();
    public static CtxReadDelegate<UnityProcessInfoList> Read = (ctx, reader) => 
    {
      var processes = ReadUnityProcessInfoList(ctx, reader);
      return new UnityProcessInfoList(processes);
    };
    
    public static CtxWriteDelegate<List<UnityProcessInfo>> WriteUnityProcessInfoList = UnityProcessInfo.Write.List();
    public static CtxWriteDelegate<UnityProcessInfoList> Write = (ctx, writer, value) => 
    {
      WriteUnityProcessInfoList(ctx, writer, value.Processes);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((UnityProcessInfoList) obj);
    }
    public bool Equals(UnityProcessInfoList other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Processes.SequenceEqual(other.Processes);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + CollectionUtil.GetHashCode(Processes);
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("UnityProcessInfoList (");
      using (printer.IndentCookie()) {
        printer.Print("processes = "); Processes.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
