using System;
using System.Linq;
using System.Collections.Generic;
using JetBrains.Application;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.Loggers
{
  
  
  [ShellComponent]
  public class LoggerModel : RdBindableBase {
    //fields
    [NotNull] public ISource<LogEntry> Message { get { return _Message; }}
    [NotNull] private readonly RdSignal<LogEntry> _Message;
    [NotNull] public IRdProperty<string> TestLogDir { get { return _TestLogDir; }}
    [NotNull] private readonly RdProperty<string> _TestLogDir;
    
    //primary constructor
    public LoggerModel(
      [NotNull] RdSignal<LogEntry> message,
      [NotNull] RdProperty<string> testLogDir
    )
    {
      _Message = message;
      _TestLogDir = testLogDir;
      testLogDir.OptimizeNested = true;
    }
    //secondary constructor
    //statics
    
    private void Register(ISerializers serializers)
    {
      serializers.RegisterEnum<LogEntryType>();
      serializers.Register(LogEntry.Read, LogEntry.Write);
    }
    
    public LoggerModel(Lifetime lifetime, IProtocol protocol) : this (
      new RdSignal<LogEntry>().Static(1011),
      new RdProperty<string>().Static(1012)
    )
    {
      Register(protocol.Serializers);
      Bind(lifetime, protocol, GetType().Name);
    }
    //init method
    protected override void Init(Lifetime lifetime) {
      _Message.BindEx(lifetime, this, "message");
      _TestLogDir.BindEx(lifetime, this, "testLogDir");
    }
    //identify method
    public override void Identify(IIdentities ids) {
      _Message.IdentifyEx(ids);
      _TestLogDir.IdentifyEx(ids);
    }
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("LoggerModel (");
      using (printer.IndentCookie()) {
        printer.Print("message = "); _Message.PrintEx(printer); printer.Println();
        printer.Print("testLogDir = "); _TestLogDir.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class LogEntry : IPrintable, IEquatable<LogEntry> {
    //fields
    [NotNull] public string LogMessage {get; private set;}
    [NotNull] public string LogMessageWithSensitiveData {get; private set;}
    public LogEntryType Severity {get; private set;}
    
    //primary constructor
    public LogEntry(
      [NotNull] string logMessage,
      [NotNull] string logMessageWithSensitiveData,
      LogEntryType severity
    )
    {
      LogMessage = logMessage;
      LogMessageWithSensitiveData = logMessageWithSensitiveData;
      Severity = severity;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<LogEntry> Read = (ctx, reader) => 
    {
      var logMessage = reader.ReadString();
      var logMessageWithSensitiveData = reader.ReadString();
      var severity = (LogEntryType)reader.ReadInt();
      return new LogEntry(logMessage, logMessageWithSensitiveData, severity);
    };
    
    public static CtxWriteDelegate<LogEntry> Write = (ctx, writer, value) => 
    {
      writer.Write(value.LogMessage);
      writer.Write(value.LogMessageWithSensitiveData);
      writer.Write((int)value.Severity);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((LogEntry) obj);
    }
    public bool Equals(LogEntry other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return LogMessage == other.LogMessage && LogMessageWithSensitiveData == other.LogMessageWithSensitiveData && Severity == other.Severity;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + LogMessage.GetHashCode();
        __r = __r*31 + LogMessageWithSensitiveData.GetHashCode();
        __r = __r*31 + (int) Severity;
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("LogEntry (");
      using (printer.IndentCookie()) {
        printer.Print("logMessage = "); LogMessage.PrintEx(printer); printer.Println();
        printer.Print("logMessageWithSensitiveData = "); LogMessageWithSensitiveData.PrintEx(printer); printer.Println();
        printer.Print("severity = "); Severity.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum LogEntryType {
    ERROR,
    WARN,
    INFO,
    VERBOSE,
    TRACE
  }
}
