using System;
using System.Linq;
using System.Collections.Generic;
using JetBrains.Application;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.Notifications
{
  
  
  [ShellComponent]
  public class RdNotificationsModel : RdBindableBase {
    //fields
    [NotNull] public ISource<RdNotificationEntry> Notification { get { return _Notification; }}
    [NotNull] private readonly RdSignal<RdNotificationEntry> _Notification;
    [NotNull] public IRdCall<RdMessageBoxModel, RdMessageBoxResponse> MessageBox { get { return _MessageBox; }}
    [NotNull] private readonly RdCall<RdMessageBoxModel, RdMessageBoxResponse> _MessageBox;
    [NotNull] public IViewableMap<int, RdBackgroundTask> Tasks { get { return _Tasks; }}
    [NotNull] private readonly RdMap<int, RdBackgroundTask> _Tasks;
    
    //primary constructor
    public RdNotificationsModel(
      [NotNull] RdSignal<RdNotificationEntry> notification,
      [NotNull] RdCall<RdMessageBoxModel, RdMessageBoxResponse> messageBox,
      [NotNull] RdMap<int, RdBackgroundTask> tasks
    )
    {
      _Notification = notification;
      _MessageBox = messageBox;
      _Tasks = tasks;
    }
    //secondary constructor
    //statics
    
    private void Register(ISerializers serializers)
    {
      serializers.RegisterEnum<RdNotificationEntryType>();
      serializers.Register(RdNotificationEntry.Read, RdNotificationEntry.Write);
      serializers.RegisterEnum<RdMessageBoxResponse>();
      serializers.RegisterEnum<RdMessageBoxType>();
      serializers.Register(RdMessageBoxModel.Read, RdMessageBoxModel.Write);
      serializers.Register(RdBackgroundTask.Read, RdBackgroundTask.Write);
    }
    
    public RdNotificationsModel(Lifetime lifetime, IProtocol protocol) : this (
      new RdSignal<RdNotificationEntry>().Static(1030),
      new RdCall<RdMessageBoxModel, RdMessageBoxResponse>().Static(1031),
      new RdMap<int, RdBackgroundTask>().Static(1032)
    )
    {
      Register(protocol.Serializers);
      Bind(lifetime, protocol, GetType().Name);
    }
    //init method
    protected override void Init(Lifetime lifetime) {
      _Notification.BindEx(lifetime, this, "notification");
      _MessageBox.BindEx(lifetime, this, "messageBox");
      _Tasks.BindEx(lifetime, this, "tasks");
    }
    //identify method
    public override void Identify(IIdentities ids) {
      _Notification.IdentifyEx(ids);
      _MessageBox.IdentifyEx(ids);
      _Tasks.IdentifyEx(ids);
    }
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdNotificationsModel (");
      using (printer.IndentCookie()) {
        printer.Print("notification = "); _Notification.PrintEx(printer); printer.Println();
        printer.Print("messageBox = "); _MessageBox.PrintEx(printer); printer.Println();
        printer.Print("tasks = "); _Tasks.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdBackgroundTask : RdBindableBase {
    //fields
    [NotNull] public string Title {get; private set;}
    [NotNull] public IRdProperty<string> Header { get { return _Header; }}
    [NotNull] private readonly RdProperty<string> _Header;
    [NotNull] public IRdProperty<string> Description { get { return _Description; }}
    [NotNull] private readonly RdProperty<string> _Description;
    [NotNull] public IRdProperty<double> Progress { get { return _Progress; }}
    [NotNull] private readonly RdProperty<double> _Progress;
    public bool IsIndeterminate {get; private set;}
    public bool IsCancelable {get; private set;}
    [NotNull] public IVoidSink Cancel { get { return _Cancel; }}
    [NotNull] private readonly RdVoidSignal _Cancel;
    public int UpdateInterval {get; private set;}
    
    //primary constructor
    private RdBackgroundTask(
      [NotNull] string title,
      [NotNull] RdProperty<string> header,
      [NotNull] RdProperty<string> description,
      [NotNull] RdProperty<double> progress,
      bool isIndeterminate,
      bool isCancelable,
      [NotNull] RdVoidSignal cancel,
      int updateInterval
    )
    {
      Title = title;
      _Header = header;
      _Description = description;
      _Progress = progress;
      IsIndeterminate = isIndeterminate;
      IsCancelable = isCancelable;
      _Cancel = cancel;
      UpdateInterval = updateInterval;
      header.OptimizeNested = true;
      description.OptimizeNested = true;
      progress.OptimizeNested = true;
    }
    //secondary constructor
    public RdBackgroundTask (
      [NotNull] string Title,
      bool IsIndeterminate,
      bool IsCancelable,
      int UpdateInterval
    ) : this (
      Title,
      new RdProperty<string>(),
      new RdProperty<string>(),
      new RdProperty<double>(),
      IsIndeterminate,
      IsCancelable,
      new RdVoidSignal(),
      UpdateInterval
    ) {}
    //statics
    
    public static CtxReadDelegate<string> ReadStringNullable = Serializers.ReadString.NullableClass();
    public static CtxReadDelegate<RdBackgroundTask> Read = (ctx, reader) => 
    {
      var title = reader.ReadString();
      var header = RdProperty<string>.Read(ctx, reader);
      var description = RdProperty<string>.Read(ctx, reader);
      var progress = RdProperty<double>.Read(ctx, reader);
      var isIndeterminate = reader.ReadBool();
      var isCancelable = reader.ReadBool();
      var cancel = RdVoidSignal.Read(ctx, reader);
      var updateInterval = reader.ReadInt();
      return new RdBackgroundTask(title, header, description, progress, isIndeterminate, isCancelable, cancel, updateInterval);
    };
    
    public static CtxWriteDelegate<string> WriteStringNullable = Serializers.WriteString.NullableClass();
    public static CtxWriteDelegate<RdBackgroundTask> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Title);
      RdProperty<string>.Write(ctx, writer, value._Header);
      RdProperty<string>.Write(ctx, writer, value._Description);
      RdProperty<double>.Write(ctx, writer, value._Progress);
      writer.Write(value.IsIndeterminate);
      writer.Write(value.IsCancelable);
      RdVoidSignal.Write(ctx, writer, value._Cancel);
      writer.Write(value.UpdateInterval);
    };
    //init method
    protected override void Init(Lifetime lifetime) {
      _Header.BindEx(lifetime, this, "header");
      _Description.BindEx(lifetime, this, "description");
      _Progress.BindEx(lifetime, this, "progress");
      _Cancel.BindEx(lifetime, this, "cancel");
    }
    //identify method
    public override void Identify(IIdentities ids) {
      _Header.IdentifyEx(ids);
      _Description.IdentifyEx(ids);
      _Progress.IdentifyEx(ids);
      _Cancel.IdentifyEx(ids);
    }
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdBackgroundTask (");
      using (printer.IndentCookie()) {
        printer.Print("title = "); Title.PrintEx(printer); printer.Println();
        printer.Print("header = "); _Header.PrintEx(printer); printer.Println();
        printer.Print("description = "); _Description.PrintEx(printer); printer.Println();
        printer.Print("progress = "); _Progress.PrintEx(printer); printer.Println();
        printer.Print("isIndeterminate = "); IsIndeterminate.PrintEx(printer); printer.Println();
        printer.Print("isCancelable = "); IsCancelable.PrintEx(printer); printer.Println();
        printer.Print("cancel = "); _Cancel.PrintEx(printer); printer.Println();
        printer.Print("updateInterval = "); UpdateInterval.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdMessageBoxModel : IPrintable, IEquatable<RdMessageBoxModel> {
    //fields
    [NotNull] public string Message {get; private set;}
    [NotNull] public string Caption {get; private set;}
    public RdMessageBoxType Type {get; private set;}
    
    //primary constructor
    public RdMessageBoxModel(
      [NotNull] string message,
      [NotNull] string caption,
      RdMessageBoxType type
    )
    {
      Message = message;
      Caption = caption;
      Type = type;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdMessageBoxModel> Read = (ctx, reader) => 
    {
      var message = reader.ReadString();
      var caption = reader.ReadString();
      var type = (RdMessageBoxType)reader.ReadInt();
      return new RdMessageBoxModel(message, caption, type);
    };
    
    public static CtxWriteDelegate<RdMessageBoxModel> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Message);
      writer.Write(value.Caption);
      writer.Write((int)value.Type);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdMessageBoxModel) obj);
    }
    public bool Equals(RdMessageBoxModel other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Message == other.Message && Caption == other.Caption && Type == other.Type;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + Message.GetHashCode();
        __r = __r*31 + Caption.GetHashCode();
        __r = __r*31 + (int) Type;
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdMessageBoxModel (");
      using (printer.IndentCookie()) {
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
        printer.Print("caption = "); Caption.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdMessageBoxResponse {
    YES,
    NO,
    CANCEL,
    OK
  }
  
  
  public enum RdMessageBoxType {
    OK,
    YESNO,
    OKCANCEL
  }
  
  
  public class RdNotificationEntry : IPrintable, IEquatable<RdNotificationEntry> {
    //fields
    [NotNull] public string Title {get; private set;}
    [NotNull] public string Message {get; private set;}
    public bool ShowPopup {get; private set;}
    public RdNotificationEntryType Type {get; private set;}
    
    //primary constructor
    public RdNotificationEntry(
      [NotNull] string title,
      [NotNull] string message,
      bool showPopup,
      RdNotificationEntryType type
    )
    {
      Title = title;
      Message = message;
      ShowPopup = showPopup;
      Type = type;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdNotificationEntry> Read = (ctx, reader) => 
    {
      var title = reader.ReadString();
      var message = reader.ReadString();
      var showPopup = reader.ReadBool();
      var type = (RdNotificationEntryType)reader.ReadInt();
      return new RdNotificationEntry(title, message, showPopup, type);
    };
    
    public static CtxWriteDelegate<RdNotificationEntry> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Title);
      writer.Write(value.Message);
      writer.Write(value.ShowPopup);
      writer.Write((int)value.Type);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdNotificationEntry) obj);
    }
    public bool Equals(RdNotificationEntry other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Title == other.Title && Message == other.Message && ShowPopup == other.ShowPopup && Type == other.Type;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + Title.GetHashCode();
        __r = __r*31 + Message.GetHashCode();
        __r = __r*31 + ShowPopup.GetHashCode();
        __r = __r*31 + (int) Type;
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdNotificationEntry (");
      using (printer.IndentCookie()) {
        printer.Print("title = "); Title.PrintEx(printer); printer.Println();
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
        printer.Print("showPopup = "); ShowPopup.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdNotificationEntryType {
    ERROR,
    WARN,
    INFO
  }
}
