using System;
using System.Linq;
using System.Collections.Generic;
using JetBrains.Application;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.UiContext
{
  
  
  [ShellComponent]
  public class UiContextModel : RdBindableBase {
    //fields
    [NotNull] public IRdProperty<int> Caret { get { return _Caret; }}
    [NotNull] private readonly RdProperty<int> _Caret;
    [NotNull] public IRdProperty<int> Selection { get { return _Selection; }}
    [NotNull] private readonly RdProperty<int> _Selection;
    [NotNull] public IRdProperty<int> Focus { get { return _Focus; }}
    [NotNull] private readonly RdProperty<int> _Focus;
    [NotNull] public IRdProperty<int> TextControl { get { return _TextControl; }}
    [NotNull] private readonly RdProperty<int> _TextControl;
    [NotNull] public IRdProperty<int> Document { get { return _Document; }}
    [NotNull] private readonly RdProperty<int> _Document;
    
    //primary constructor
    public UiContextModel(
      [NotNull] RdProperty<int> caret,
      [NotNull] RdProperty<int> selection,
      [NotNull] RdProperty<int> focus,
      [NotNull] RdProperty<int> textControl,
      [NotNull] RdProperty<int> document
    )
    {
      _Caret = caret;
      _Selection = selection;
      _Focus = focus;
      _TextControl = textControl;
      _Document = document;
      caret.OptimizeNested = true;
      selection.OptimizeNested = true;
      focus.OptimizeNested = true;
      textControl.OptimizeNested = true;
      document.OptimizeNested = true;
    }
    //secondary constructor
    //statics
    
    private void Register(ISerializers serializers)
    {
      serializers.RegisterEnum<ContextKey>();
      serializers.Register(ContextPair.Read, ContextPair.Write);
    }
    
    public UiContextModel(Lifetime lifetime, IProtocol protocol) : this (
      new RdProperty<int>().Static(1052),
      new RdProperty<int>().Static(1053),
      new RdProperty<int>().Static(1054),
      new RdProperty<int>().Static(1055),
      new RdProperty<int>().Static(1056)
    )
    {
      Register(protocol.Serializers);
      Bind(lifetime, protocol, GetType().Name);
    }
    //init method
    protected override void Init(Lifetime lifetime) {
      _Caret.BindEx(lifetime, this, "caret");
      _Selection.BindEx(lifetime, this, "selection");
      _Focus.BindEx(lifetime, this, "focus");
      _TextControl.BindEx(lifetime, this, "textControl");
      _Document.BindEx(lifetime, this, "document");
    }
    //identify method
    public override void Identify(IIdentities ids) {
      _Caret.IdentifyEx(ids);
      _Selection.IdentifyEx(ids);
      _Focus.IdentifyEx(ids);
      _TextControl.IdentifyEx(ids);
      _Document.IdentifyEx(ids);
    }
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("UiContextModel (");
      using (printer.IndentCookie()) {
        printer.Print("caret = "); _Caret.PrintEx(printer); printer.Println();
        printer.Print("selection = "); _Selection.PrintEx(printer); printer.Println();
        printer.Print("focus = "); _Focus.PrintEx(printer); printer.Println();
        printer.Print("textControl = "); _TextControl.PrintEx(printer); printer.Println();
        printer.Print("document = "); _Document.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum ContextKey {
    Caret,
    Selection,
    Focus,
    TextControl,
    Document
  }
  
  
  public class ContextPair : IPrintable, IEquatable<ContextPair> {
    //fields
    public ContextKey Key {get; private set;}
    public int Timestamp {get; private set;}
    
    //primary constructor
    public ContextPair(
      ContextKey key,
      int timestamp
    )
    {
      Key = key;
      Timestamp = timestamp;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<ContextPair> Read = (ctx, reader) => 
    {
      var key = (ContextKey)reader.ReadInt();
      var timestamp = reader.ReadInt();
      return new ContextPair(key, timestamp);
    };
    
    public static CtxWriteDelegate<ContextPair> Write = (ctx, writer, value) => 
    {
      writer.Write((int)value.Key);
      writer.Write(value.Timestamp);
    };
    //init method
    //identify method
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((ContextPair) obj);
    }
    public bool Equals(ContextPair other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Key == other.Key && Timestamp == other.Timestamp;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var __r = 0;
        __r = __r*31 + (int) Key;
        __r = __r*31 + Timestamp.GetHashCode();
        return __r;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("ContextPair (");
      using (printer.IndentCookie()) {
        printer.Print("key = "); Key.PrintEx(printer); printer.Println();
        printer.Print("timestamp = "); Timestamp.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
