<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = Expand-Excluded $exclusions
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      $exclusions += $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }

  Set-MpPreference -ExclusionPath $exclusions
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIvoQYJKoZIhvcNAQcCoIIvkjCCL44CAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg9OPGiRg/9xyJzUyY
# SCqXYDVeviHTe8JjtES/dIowqHugghNYMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owA
# AAAAUdOUfzANBgkqhkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVn
# YWwtdGVybXMxOTA3BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3Ig
# YXV0aG9yaXplZCB1c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3
# MTYxMzQ1WjBpMQswCQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFC
# MEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9u
# IEF1dGhvcml0eSAtIENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAp4GP9xRFtmJD8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAi
# xqhmdU1Ug8leaBur9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToY
# yVAslyctv9oAfWN/7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tI
# urnXwYJ4hWUuf7XJwOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36
# rL9kUdFcm7T1XOdc/zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32
# KvBPpSA9aCXrYZd8D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+
# Xq14Z1tV417rx9jsTG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6
# hD5B3G5WrsYaW/RnaAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxX
# OwlyndfKt5DGzXtFkpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOj
# c2zjs2s3/+adZwGSht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6H
# RGDjqfTCGKPj/c4MhCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEn
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQG
# CCsGAQUFBwMDBggrBgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEF
# BQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAl
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAn
# MCWgI6Ahhh9odHRwOi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQW
# BBSCutY9l86fz3Hokjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHU
# bI2fkBJmqzANBgkqhkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSY
# Xlk4EwmkWZRCXlC/T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8U
# fY8dQzZks2YTXxTMpXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97c
# v+kR3RaDCNMsjX9NqBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbj
# hbhNm0VyiwfxivtJuF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24X
# iA3hxkOY14FhtoWdR+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBoMw
# ggRroAMCAQICEDWvt3udNB9q/I+ERqsxNSswDQYJKoZIhvcNAQENBQAwaTELMAkG
# A1UEBhMCVVMxFjAUBgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1
# c3QgQ29kZSBTaWduaW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBD
# U0JSMTAeFw0yMTA1MDcxOTE5NTJaFw00MDEyMjkyMzU5MDBaMGMxCzAJBgNVBAYT
# AlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNFbnRydXN0IEV4
# dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZDUzIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC+vac5yaV97F1l8fQrqYfQ560axRo7
# GM7hoVGNcvrOWB9cuCRCD0bVMZfQSk3jmzDJupeonP5FNs8ngOd7uG7BJLffp3Tc
# hfKjJFhFzJSUOwLrxI18RvVcZoLjpMHTH6xuDFMjDtQ6+tpC8YNUXzUyVvK1eJtX
# AEgeqWFhJw5zA0O21nCS+9SFmjDGs+aaXkvvHSrYZqsWRv8L2A+miUoCUVdBPEE4
# TpfHUTJtZ45moV5NnzUir9Vqt39AX2g1zGn68QXw6oWm6jKFy8HByoNpRUkG3Als
# ukllGYz5tzcnjGSeNePl7OcHoJ2ocrxvhTosphZOPZzPCOaC9UR9KMC9ia1sL9wv
# eHkR1xxwS92dCExeLvqNvafdY/Z/8FIxhG462NlchUSeYwZp0IZYeImbh7tYHKQo
# bMb+aQqcHqwRYGpeyWllLu1DLWnxeLc7LTXyqk/iH+MBb5BGqtWoDQRXoLSs4229
# nRsogCdGx9qqZ5Xx0Yd7x8gl6YQMj4k20r4z4YXAM9WgPBmLrzjy5ZOAv8bDq3uT
# xD2due5FdsDUaG8wXjy0NvnXRULgEgaA26Uh/OcFeiiNtI5ge/fItUpHrBRml6Ba
# aXIxV2tThM1hunMHFXA7ewH8pz+MLT2HjPsI1+UvF4N+gqtPCfIp4X5Vd2WUBR1Z
# 1Ardk37jFF3iuwIDAQABo4IBKzCCAScwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNV
# HQ4EFgQUzolPglGqFaKEYsoxI2HSYfv4/ngwHwYDVR0jBBgwFoAUgrrWPZfOn89x
# 6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVu
# dHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYI
# KwYBBQUHAwMwRAYDVR0gBD0wOzAwBgRVHSAAMCgwJgYIKwYBBQUHAgEWGmh0dHA6
# Ly93d3cuZW50cnVzdC5uZXQvcnBhMAcGBWeBDAEDMA0GCSqGSIb3DQEBDQUAA4IC
# AQA+AFS4KvOPZq9hFsRYk2T0QYtkVY0bNTOhm5HYq0bKq1+8vn5w5NLXB6iWB9eG
# 0VcKCNDQeE34Kt+yBPOa4dd88MEAesFAod+KjLfLtB0BYfKYmqbduIMFyqksFtyc
# LQ7+p5fkUKmXlcFLKoxR48QC4Gt8NU1TvEuUE4tTGBZepgqnuDTBbLHrLOQxfQws
# xnzhpFboLXbVshN16oMArgB3xm+pPe5jFIFQFvxbTxIZDlsPjepLzgZMiUuuIMyO
# R6Z11mXuLzDoXTSPH4JNXEKm8hRMUCCcCaJ0JFw52IkyhTyvjOVqnuYEOqUT/6od
# zUdLLgIFtGqP64VPge8K232fKY+lwj9SOFJBlTu8PltUMEIjCWPeUI2JNUX6q7gP
# j6Kte3oRk/GPKIR7aHlHauhkKU0f9B0vbR7IlIY801qZemt8qzX3KzlLT7k/FpLv
# KYyNq6wBGsxxRDnLQD4gEs4IcONH/tyA1wgA0Qtq6iG1eWYX6WqqVt3NFXixA8AU
# rT5HGHXtClNLMpPN/4CxkNYT5eRBeCLtQYLlDt+wzpUhReaLAdMZ9/QrVazZVnNj
# EOC4oG3LVJaYoJkhUQaqpgw6K2PLedpDudISzN6PVXrpCfPimtMlwXs2ktrg5VJn
# zePmVAqlndRzo9MvXQZkhQN3f2DYwrx0yZWRaLe3fmEZdTCCBuowggTSoAMCAQIC
# EDGdnUgatvXgkrzF40/3PFswDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMx
# FjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xPDA6BgNVBAMTM0VudHJ1c3QgRXh0ZW5k
# ZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcgQ0EgLSBFVkNTMjAeFw0yMjEwMTEx
# MjM2MzdaFw0yNTEwMTExMjM2MzZaMIGaMQswCQYDVQQGEwJDWjEOMAwGA1UEBxMF
# UHJhaGExEzARBgsrBgEEAYI3PAIBAxMCQ1oxGTAXBgNVBAoTEEpldEJyYWlucyBz
# LnIuby4xHTAbBgNVBA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMREwDwYDVQQFEwgy
# NjUwMjI3NTEZMBcGA1UEAxMQSmV0QnJhaW5zIHMuci5vLjCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAMmZsBOO0sV6f+qwSfowL7CAaqooz6zzYDcPWA/6
# P1BlOeO9NqKEzd4IyZEI1lccipydIwdaO1YHa2kM7/5kertoyJ6ITuMjmcyxnqXI
# AhdCHLbqO/Vq6R+4fqokQ5FvPvsTZ6bbc8sznX3roST/zjlgYA5+HDR2GRZ4sy5d
# tkt/i/MANl3f3rRv1RO0qvy+dU7GOk0CWPfXFIHrj+sszSQ7zgM52bHAP70NlKfj
# VkKdX7U4Ytz3yb1gnfLRQyxYgM/MBKGDI6BrHlUfskryxjV5gH+aWFfBaXw8+NVl
# fYr8lUdlXlzlWmHqa7J792WNCLlE31NevEYaXA+0TbxJKT8bvXWehHxBpGY9Q15z
# WjnaGk1FguXxqmXKkRQfDgBiJ4sCtGW3iVGmOtMiEKmOURSZ/hoUQSrtMf8r/itK
# re6DHoxGbjA9yjzPXZbT1dJk6eOcKZaesYANyu05Kz7S/lRX83N4UbCQEJ4xpcCI
# G+eeE4d2BrapVFMZxobUKZaFtV+SAByQFEumEjEX5hEMyGzfODoa0KnFowf9Fu5u
# lz0dM3cV6/+riIgphUIlsNYgum7swNn6dp13+iyMuaOvPEL0kBlOPufcz7Lq18a6
# o5anOBJpW9jAMf0cCfJ+hwAOj3gcTMbiKCmEo7gEoQiZvnM4DZra7tf+Nblmv730
# RtPHAgMBAAGjggFgMIIBXDAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBQKghd/k2G5
# FxhP/0MIUgmYqUN3xjAfBgNVHSMEGDAWgBTOiU+CUaoVooRiyjEjYdJh+/j+eDBn
# BggrBgEFBQcBAQRbMFkwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3Qu
# bmV0MDIGCCsGAQUFBzAChiZodHRwOi8vYWlhLmVudHJ1c3QubmV0L2V2Y3MyLWNo
# YWluLnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVudHJ1c3QubmV0
# L2V2Y3MyLmNybDAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# SwYDVR0gBEQwQjA3BgpghkgBhvpsCgECMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAHBgVngQwBAzANBgkqhkiG9w0BAQsFAAOCAgEA
# BLLwIeBU0HShBD5L1ZoZN+ZQIpMOIHkQajuxja1zSLwakCX3lA0V3MpxliDDRx3e
# y4YA3OcefN2zV6xUAVNOb6V8zB552SblNVxa4TIxzmSzKI2uifjkikt1ZogDEXsS
# Cflyak2rbBATAmNrEBFAcZBLUGDxExbK2HyZAtG+CR16jj9Qd3zDHmbSciIMlsBq
# pmgmP34/pcjr/LfjXNqa2r0Q+ISbhVgv5N4ESYdBUKh0SAMf5fcUAreP0IpTOO4v
# FkkZSoUkALhGfBQDluukYrIGUh7fjmNVAwTQ1UjLRb8VDQwsIhFMBJkzevbxkqZA
# 3O6JLdisMuRL6/CFkdnftPAeiBJbs2jRQzADDGylIdIMv8V6A/yymtg9kZH9eCNw
# JibhEhTPHsEJX5Unwk8vE7POUqCKoB7+ULkGURTrUtEBBYmShPcbjQH0l6pcb66J
# oCX78Cbzd/Zr9Fm6mLjjlNbZcyjBSGDyuZq6SpKLL90YYCXHNEFKJdFCtnxtnbM4
# ipy3TZi9Xhx5OSfLd3T7/WgFQYuLHacteeGVonYBGXZbQiFKbUophNfYFRF0N5ZE
# cc7Vy3Vm+JSAvX496v6GRR3/C2zLI9ffZBqYLFhMGo0qIIUNLuIBbhX6pzw3mrlj
# VIdeZGNRj3vWqhAp9A7IzS6x3t2RrmSrkV1an7k6Um0xghuhMIIbnQIBATB3MGMx
# CzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNF
# bnRydXN0IEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZD
# UzICEDGdnUgatvXgkrzF40/3PFswCwYJYIZIAWUDBAIBoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCAcdR8jR9bL9V3fqV5tBa/+1UUg1PuQ4DNM/0aeA9a25jBC
# BgorBgEEAYI3AgEMMTQwMqAwgC4AZABlAGYAZQBuAGQAZQByAC0AZQB4AGMAbAB1
# AHMAaQBvAG4AcwAuAHAAcwAxMA0GCSqGSIb3DQEBAQUABIICAFocIHHprDje8cVC
# mFj74BgwsoTqkNRouDVx3ix/abJdz8ysrV70+kXJrIsRqJ0HwTBdDYsRXV+dCBiI
# 2Q7bFBgo9lioCk3huX0JURLqa0r7Af5yNX1N3eEY1vNxg3rB5nK8VQjbYpHE30yU
# sePGc0FgqmuwReOR6Ycsdi8WRHOAQnHUEPjHvJ1bAEJn26Cbi/GJxa7SRSy4GTtS
# SbmIsIDcOyw3g0euA6CnFKV5IqI3DZBI3V8bwzN0rP/WV2dcbZYqL6oM9B84uLZh
# JgfTPYH/16fqlnm3e4gQx0Cz2RSqVSKn/Iol0044KxTkce+jj8Gm/xAdpkLOp2yt
# JatMn4Ld3sFiIsdTLRxdZW5zLUSqX6wSLrx8p78+UPi1rWXV6Jns7bkNzgNfJhmA
# GPPPFYgwtbumLWuvrcOJB3DYdiNMFSNUokc+5UmPEbOlynNv9weIQM9LpMlGWtvR
# Mz0ViyCUJKFzjm+JT8Aq0L0mdxtL4CmXQRkEJrGiM2AIDEOvpCBxDPGcQUBmSQCP
# EyxMUrRcU4t1jmi9BJYrINT9MJwTb14J0UYxFYT9A+lSME0gtISz6LnVQUphIWQ+
# dK43eYaN3d0cYwa4gx8hpA1YUc2ZNldEuk/RWow9UHu+Ungn8oFIC17s+d+CE5z9
# GvJM1iZFjq3s963H2BDcRf8GpR9VoYIYTDCCGEgGCisGAQQBgjcDAwExghg4MIIY
# NAYJKoZIhvcNAQcCoIIYJTCCGCECAQMxDTALBglghkgBZQMEAgMwgc8GCyqGSIb3
# DQEJEAEEoIG/BIG8MIG5AgEBBgpghkgBhvpsCgMFMC8wCwYJYIZIAWUDBAIBBCC7
# NYKg7QqOLPritokNYfaJlwf3j2+SbV2h661uu5V0ZgIJAMAan8BeQtGKGA8yMDI1
# MDYyNDA0MjMxMFowAwIBAaBWpFQwUjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUVu
# dHJ1c3QsIEluYy4xKzApBgNVBAMTIkVudHJ1c3QgVGltZXN0YW1wIEF1dGhvcml0
# eSAtIFRTQTKgghMZMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owAAAAAUdOUfzANBgkq
# hkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUVudHJ1c3QsIElu
# Yy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVnYWwtdGVybXMxOTA3
# BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3IgYXV0aG9yaXplZCB1
# c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRpZmljYXRpb24gQXV0
# aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3MTYxMzQ1WjBpMQsw
# CQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFCMEAGA1UEAww5RW50
# cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eSAt
# IENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAp4GP9xRFtmJD
# 8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAixqhmdU1Ug8leaBur
# 9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToYyVAslyctv9oAfWN/
# 7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tIurnXwYJ4hWUuf7XJ
# wOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36rL9kUdFcm7T1XOdc
# /zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32KvBPpSA9aCXrYZd8
# D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+Xq14Z1tV417rx9js
# TG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6hD5B3G5WrsYaW/Rn
# aAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxXOwlyndfKt5DGzXtF
# kpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOjc2zjs2s3/+adZwGS
# ht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6HRGDjqfTCGKPj/c4M
# hCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEnMA4GA1UdDwEB/wQE
# AwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQGCCsGAQUFBwMDBggr
# BgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEFBQcCARYaaHR0cDov
# L3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzAB
# hhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAnMCWgI6Ahhh9odHRw
# Oi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQWBBSCutY9l86fz3Ho
# kjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHUbI2fkBJmqzANBgkq
# hkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSYXlk4EwmkWZRCXlC/
# T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8UfY8dQzZks2YTXxTM
# pXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97cv+kR3RaDCNMsjX9N
# qBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbjhbhNm0VyiwfxivtJ
# uF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24XiA3hxkOY14FhtoWd
# R+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBm8wggRXoAMCAQICECW8
# K/MpyhB/Hqm6iIXUnTswDQYJKoZIhvcNAQENBQAwaTELMAkGA1UEBhMCVVMxFjAU
# BgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1c3QgQ29kZSBTaWdu
# aW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBDU0JSMTAeFw0yMTA1
# MDcxOTIyMTRaFw00MDEyMjkyMzU5MDBaME4xCzAJBgNVBAYTAlVTMRYwFAYDVQQK
# Ew1FbnRydXN0LCBJbmMuMScwJQYDVQQDEx5FbnRydXN0IFRpbWUgU3RhbXBpbmcg
# Q0EgLSBUUzIwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC1AyoGtoRP
# NMyeMb7qjsZ7biAkDwPXvYE2M+Zv0j67xJ6qoMxmXUJgNFHiLWGDujyeaLhLw2aO
# pd4rupstQaXe0MtXBS2I2cBGiG08NQ0ZkKy4DBnwTMXbRVvcO8K8jUQA4Dj//13I
# zwiaPdSy63uVw8SlAOBiAWRZX4zje4up+UW3xrCiCjdDuEaBq4Z+fy/e8F/rzSDM
# pS0x46gumZvgeN30212CY30wOYh+JAbmfGCEeMhcKeWVy/V7T89Y3JDPp6J7FFTE
# 4DeYMMGbtq6cKfZrJUPnEmo+GYu+wOeB10owCH58jd8880iTId6Bg2qdAD7XYLrR
# s2IIlum2SQA49Fx2Ddp3aj2gld4eocxZel6fz+l2XUDytRW1YGgs81rJI4PY9Rpr
# aSikttSuYgbeJkW93ulWd6rcZLBBzcwT8V1xdLKUCEtPMm5+cLh36dUyN8J63kIS
# 6HEc4thiv6prQYYGW+ZpviYJ9JfC/kz0gHKEbtvexQepjhWibeEb4AkP9aAHoLvE
# d3MJPAeTjQG1EmctTRm1uMXJEKtwz0L/pScd1hLW5BhEYPs5XYS7ZrVTEp0DFIJl
# KbTsSXL9s0PlwwIpJLof+Li+XaO3Lqn8z2LZ+pfEE3jjVblaeoTr/7vPaYjAtvmL
# YIVBEFDHBRDSXnadPjXs9k+K+RJ7P68LNwIDAQABo4IBLDCCASgwEgYDVR0TAQH/
# BAgwBgEB/wIBADAdBgNVHQ4EFgQUJg/wxEgIG83dkfVUVLazs/yZ8QgwHwYDVR0j
# BBgwFoAUgrrWPZfOn89x6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsG
# AQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAi
# hiBodHRwOi8vY3JsLmVudHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMC
# AYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwRQYDVR0gBD4wPDAwBgRVHSAAMCgwJgYI
# KwYBBQUHAgEWGmh0dHA6Ly93d3cuZW50cnVzdC5uZXQvcnBhMAgGBmeBDAEEAjAN
# BgkqhkiG9w0BAQ0FAAOCAgEAdj1GaIVfCcDOyfjHuNd+p1w7C0ZzziJTizj2Ebp3
# xMKHIY8n2QyV6+hL5VzXkBVvqCosimrgIhE0efq9lnnIdhbNsUTqcVEPm1XJGHzV
# gnmc86a3k6kFOHICBpehqLJ5fl4I4m5seZqoh5TOf49VNkAPnz9R1Wa+e6uG5m6H
# uk5jXbHYjh/LZ8MNcNp665OyFITSPn2TPxYMNqBceQCfC27lhCrYiMFtBLc385Ka
# cOA7A/3NuyeCzi/8jeSyyr74JYXG7XTIPTVfOAk9eU/rG+BBXqV0gT9RFcD4SYiP
# ursF1K1FgjN5wSWNX1Q9keS4nxeYAF2tKOVPXxv7+FS1pcQk/PB2O/gNXsxHsMqq
# u25R31O1SRrxYIe3+f1pBnVfc9YRkPKAWI7lww8DmIwEU7Mph98/97DpTFeBJER5
# aP4bNgfWZT3sb9bCtaphfGYG7NLlaYD4cZIuXOIRRhhFS9b6BWTvu94GykMlvd+N
# yQF0YYjb8MemPeMMcbx/S+fI4G7g2oD5AJ7AayXVo7pcK/7EYCAUSgcjMeUay5FE
# spp7Q/FbmLUhS7gxOyJU7nlh95qUG2YnKsbf4WVd73E55lAl/Yc0ua5dfCc752WT
# +CiEsW+GkyyTk7Zwr6HuyKRhqYQ7+wq3+LhtJu5HTvVeBfqcDxF918uRrkMg9xVZ
# Y7wwgga/MIIEp6ADAgECAhEAxRT2yVF1rv52Adu5FRlGujANBgkqhkiG9w0BAQ0F
# ADBOMQswCQYDVQQGEwJVUzEWMBQGA1UEChMNRW50cnVzdCwgSW5jLjEnMCUGA1UE
# AxMeRW50cnVzdCBUaW1lIFN0YW1waW5nIENBIC0gVFMyMB4XDTI1MDEyMjE5MTUy
# MloXDTMwMDExODE5MTUyMVowUjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUVudHJ1
# c3QsIEluYy4xKzApBgNVBAMTIkVudHJ1c3QgVGltZXN0YW1wIEF1dGhvcml0eSAt
# IFRTQTIwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC5Q2M++sgGUBnv
# iuHZdqlmaEz+1XSsCZYPQhgv1zz6YgGcSjkcpb5FRd3ZAyUkhAr3zhCuemUFYaUT
# 9u9rAxYKF7Frb0eNTIrVqjeyl/pTakkYYO5OYI9FAwhCVHi4y+Ypo3lkNw+stmny
# NRla5E8AbVsIVl25LBEtAIa+TKwvZDWIxpRBR169Omf9ejgt9tNSm8Ro8Glb6U3i
# NVYAqRFnQCbzK8Dkv9EHYeBoTOvMUVIN/Mz4X9HOiY092EPc36LMVDlyonz9TTzl
# aEEu7Kx4XvshY07NrmcWUEuoDju4uVdbEG5siU6SvpnHi3pKzywvsIJ9JtZuuiqx
# 0kDgOkUG3SKJM+5OyTqLRKxbxKd6bbI58DKGyeux2eks9nz8c4yD09ZeTGnuOWAw
# UniRcgO4D8GJ8/ahXF7Ii5iST3e6vdleGTjhYSBr/vdM8QrkPBcxeYygzrCNcoI2
# M0T4+lEBMKmVomptYx+r+Y6TS++qopg6IAToECvrrpXbbbyKckNeyC1s7vhWyQcJ
# yyYGAYUgtih46W1SNr00nCDcG30zxmeo2URDl3QEKYTP7d4haBsBb4mSu2Hr4B3P
# 84jvUwxOkqHOGdOvMBEqdnjpZsTVk5wNbrc0hImiMj45xhGVhp/y5Ge4w5Gz65VK
# F0t+8BgAEBJ4CBoigIAUCkNczLyyvQIDAQABo4IBkjCCAY4wDAYDVR0TAQH/BAIw
# ADAdBgNVHQ4EFgQUqQbsAk8ngw0Uw666FYVQHFJ4ET0wHwYDVR0jBBgwFoAUJg/w
# xEgIG83dkfVUVLazs/yZ8QgwDgYDVR0PAQH/BAQDAgeAMBYGA1UdJQEB/wQMMAoG
# CCsGAQUFBwMIMGgGCCsGAQUFBwEBBFwwWjAjBggrBgEFBQcwAYYXaHR0cDovL29j
# c3AuZW50cnVzdC5uZXQwMwYIKwYBBQUHMAKGJ2h0dHA6Ly9haWEuZW50cnVzdC5u
# ZXQvdHMyLWNoYWluMjU2LnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3Js
# LmVudHJ1c3QubmV0L3RzMmNhLmNybDBMBgNVHSAERTBDMAgGBmeBDAEEAjA3Bgpg
# hkgBhvpsCgEHMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8vd3d3LmVudHJ1c3QubmV0
# L3JwYTArBgNVHRAEJDAigA8yMDI1MDEyMjE5MTUyMlqBDzIwMjYwNDIxMTkxNTIx
# WjANBgkqhkiG9w0BAQ0FAAOCAgEANLaZNVZXVd6w4gq4N2iAEO2XFk/pJniTtbKr
# z7AIIbezNEEabwZ9duakOVk7HWj4DCEs8d5FbNr+PC7q9C57s7f//STIcK3JWTsB
# iZLiqI8Wops7UiBY7Ox8pz7GvAfG+f5UVRtJiZV8Bf3ZWMuXeQdYfmDwujZmSvo4
# i8zim1LLxGFgdtoD0pPZkuXi6pGtaTjm76kXFkZLwMzE15bWs14vgAf78z3IA9U+
# 7o3Eqv5jRT1oVk4ujEwH9AiWRp557brW5C+WGFekPaAhgaQEsmTYQgbHlQzVN/eM
# OmzIZ0R6/teuqNTucFRWz6C9P2BNGfMxYBI4FOjiL2anh1DzyV4nbG+mxh+a8OIG
# JtDheyyHVjczJhD5eMP/6f48586OFtxL2a5aFH2mdb2ga38hsJlCYG7K4EgklZ5u
# xRm3OvNE2FrnM04iu86U+zv9a9OPtC0nQJ9y/u48hbpLEYDKEu38C3oEGM3qyH4e
# l/c3K1k4B/igfWHpfZ610chbG8IhTxUyvcNirG96vh0RXTocGCSXwQxkdNjHo/MJ
# ea42tzyfq0fomVgwADr3rwGsPEiB0Z0XaJQX9Igwc2pcafVOzTtYugPpc4h+ck9r
# 7Lq3Zwnl0JxucQ0O1rk6JWwcv6llKp9DOtcw4udfoon/aSZF/ckTDaly2D/fXnPd
# 4gkMQwoxggQcMIIEGAIBATBjME4xCzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRy
# dXN0LCBJbmMuMScwJQYDVQQDEx5FbnRydXN0IFRpbWUgU3RhbXBpbmcgQ0EgLSBU
# UzICEQDFFPbJUXWu/nYB27kVGUa6MAsGCWCGSAFlAwQCA6CCAYwwGgYJKoZIhvcN
# AQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yNTA2MjQwNDIzMTBa
# MCsGCSqGSIb3DQEJNDEeMBwwCwYJYIZIAWUDBAIDoQ0GCSqGSIb3DQEBDQUAME8G
# CSqGSIb3DQEJBDFCBEAK4d0JAvRwAq+xab480xqHSzv82RFOBohgKKUg1cmo/Q56
# Bigu3Wax9khurtlg3Hp1r0DlDYq+nsKm2hQH5YBcMIHRBgsqhkiG9w0BCRACLzGB
# wTCBvjCBuzCBuDALBglghkgBZQMEAgMEQKs2IQVFGhSX6q6PiopGzsTFVZZ3K2fl
# NRLaIXpuORqJMqvOm7cLqff9zdYvkM27yDMFxZWsiiVPlm5n0UD0ArAwZzBSpFAw
# TjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xJzAlBgNVBAMT
# HkVudHJ1c3QgVGltZSBTdGFtcGluZyBDQSAtIFRTMgIRAMUU9slRda7+dgHbuRUZ
# RrowDQYJKoZIhvcNAQENBQAEggIAEDCuGWKHOh+2ml4h28NEbkzlOAEcdQsyz77d
# FLZHq8+DvXm59HNEAGZYdhlF0376vEOr9BIowjWh1nu4E+kd5YHnM6q30LqBK9Gl
# DlNUZ++ClpcJWrdhs2GrMqbxmrbCkpM3NaBY1OVrQe5QA++0vHWlAAbaIcLHsFSL
# A80cjzNUMbe8OCMeo7YphNaZKhQPNBYoK95xEF4EWvWYB35slPDWOFhWPcVuW2Q8
# aC65O/9uZ7Ug6MP1FCgJF3O8OCPbSFp305y1IRVL8gGSbCTCKvV6y46JUwePWX+O
# UNwlPeHZGUBqnI/+HtHsZfgBZ/ZioHIXLFfFM6ic9a5GjCP/VZ5bxrLiSn3+oGmw
# GvkU4+FRT9PwP9uOGssqdBGN0umLe7kLMSGjFk9jNzqAlXTddyfYokshVH6DD9c5
# dopeFvToVYkbwLGgVDE1neV8hax41cT0AV80YfZ0Kn8I7WaF7LqzDWAlsTOrJJ76
# hWMJskUCNphp3xh9BssWhX2tu2MBxAtqrH4bCFCnGEEcZp/kEeTLZ6Vr9pACNZW5
# TI2pg5qDaKW0jKF9Ty4+aapp3Z9NXC7RWBYnYsndnIWtFTmLw7BqMWB4hGG8iQwq
# GLzD0aOhRqb1JmpbVUi42BzF42EHL7gyz3XJqOgfhKyUQxJkzmVeFEhEGV3x1WHW
# Frlo6Dw=
# SIG # End signature block
