<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.

   See https://learn.microsoft.com/en-us/defender-endpoint/configure-extension-file-exclusions-microsoft-defender-antivirus
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = @(Expand-Excluded $exclusions)
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      Add-MpPreference -ExclusionPath $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIuywYJKoZIhvcNAQcCoIIuvDCCLrgCAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQgeTbSdh+QTc1PlIl/
# aTZyoemLeBftCRJOcd5Eh1jG7PygghNYMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owA
# AAAAUdOUfzANBgkqhkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVn
# YWwtdGVybXMxOTA3BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3Ig
# YXV0aG9yaXplZCB1c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3
# MTYxMzQ1WjBpMQswCQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFC
# MEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9u
# IEF1dGhvcml0eSAtIENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAp4GP9xRFtmJD8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAi
# xqhmdU1Ug8leaBur9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToY
# yVAslyctv9oAfWN/7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tI
# urnXwYJ4hWUuf7XJwOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36
# rL9kUdFcm7T1XOdc/zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32
# KvBPpSA9aCXrYZd8D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+
# Xq14Z1tV417rx9jsTG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6
# hD5B3G5WrsYaW/RnaAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxX
# OwlyndfKt5DGzXtFkpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOj
# c2zjs2s3/+adZwGSht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6H
# RGDjqfTCGKPj/c4MhCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEn
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQG
# CCsGAQUFBwMDBggrBgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEF
# BQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAl
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAn
# MCWgI6Ahhh9odHRwOi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQW
# BBSCutY9l86fz3Hokjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHU
# bI2fkBJmqzANBgkqhkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSY
# Xlk4EwmkWZRCXlC/T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8U
# fY8dQzZks2YTXxTMpXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97c
# v+kR3RaDCNMsjX9NqBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbj
# hbhNm0VyiwfxivtJuF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24X
# iA3hxkOY14FhtoWdR+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBoMw
# ggRroAMCAQICEDWvt3udNB9q/I+ERqsxNSswDQYJKoZIhvcNAQENBQAwaTELMAkG
# A1UEBhMCVVMxFjAUBgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1
# c3QgQ29kZSBTaWduaW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBD
# U0JSMTAeFw0yMTA1MDcxOTE5NTJaFw00MDEyMjkyMzU5MDBaMGMxCzAJBgNVBAYT
# AlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNFbnRydXN0IEV4
# dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZDUzIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC+vac5yaV97F1l8fQrqYfQ560axRo7
# GM7hoVGNcvrOWB9cuCRCD0bVMZfQSk3jmzDJupeonP5FNs8ngOd7uG7BJLffp3Tc
# hfKjJFhFzJSUOwLrxI18RvVcZoLjpMHTH6xuDFMjDtQ6+tpC8YNUXzUyVvK1eJtX
# AEgeqWFhJw5zA0O21nCS+9SFmjDGs+aaXkvvHSrYZqsWRv8L2A+miUoCUVdBPEE4
# TpfHUTJtZ45moV5NnzUir9Vqt39AX2g1zGn68QXw6oWm6jKFy8HByoNpRUkG3Als
# ukllGYz5tzcnjGSeNePl7OcHoJ2ocrxvhTosphZOPZzPCOaC9UR9KMC9ia1sL9wv
# eHkR1xxwS92dCExeLvqNvafdY/Z/8FIxhG462NlchUSeYwZp0IZYeImbh7tYHKQo
# bMb+aQqcHqwRYGpeyWllLu1DLWnxeLc7LTXyqk/iH+MBb5BGqtWoDQRXoLSs4229
# nRsogCdGx9qqZ5Xx0Yd7x8gl6YQMj4k20r4z4YXAM9WgPBmLrzjy5ZOAv8bDq3uT
# xD2due5FdsDUaG8wXjy0NvnXRULgEgaA26Uh/OcFeiiNtI5ge/fItUpHrBRml6Ba
# aXIxV2tThM1hunMHFXA7ewH8pz+MLT2HjPsI1+UvF4N+gqtPCfIp4X5Vd2WUBR1Z
# 1Ardk37jFF3iuwIDAQABo4IBKzCCAScwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNV
# HQ4EFgQUzolPglGqFaKEYsoxI2HSYfv4/ngwHwYDVR0jBBgwFoAUgrrWPZfOn89x
# 6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVu
# dHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYI
# KwYBBQUHAwMwRAYDVR0gBD0wOzAwBgRVHSAAMCgwJgYIKwYBBQUHAgEWGmh0dHA6
# Ly93d3cuZW50cnVzdC5uZXQvcnBhMAcGBWeBDAEDMA0GCSqGSIb3DQEBDQUAA4IC
# AQA+AFS4KvOPZq9hFsRYk2T0QYtkVY0bNTOhm5HYq0bKq1+8vn5w5NLXB6iWB9eG
# 0VcKCNDQeE34Kt+yBPOa4dd88MEAesFAod+KjLfLtB0BYfKYmqbduIMFyqksFtyc
# LQ7+p5fkUKmXlcFLKoxR48QC4Gt8NU1TvEuUE4tTGBZepgqnuDTBbLHrLOQxfQws
# xnzhpFboLXbVshN16oMArgB3xm+pPe5jFIFQFvxbTxIZDlsPjepLzgZMiUuuIMyO
# R6Z11mXuLzDoXTSPH4JNXEKm8hRMUCCcCaJ0JFw52IkyhTyvjOVqnuYEOqUT/6od
# zUdLLgIFtGqP64VPge8K232fKY+lwj9SOFJBlTu8PltUMEIjCWPeUI2JNUX6q7gP
# j6Kte3oRk/GPKIR7aHlHauhkKU0f9B0vbR7IlIY801qZemt8qzX3KzlLT7k/FpLv
# KYyNq6wBGsxxRDnLQD4gEs4IcONH/tyA1wgA0Qtq6iG1eWYX6WqqVt3NFXixA8AU
# rT5HGHXtClNLMpPN/4CxkNYT5eRBeCLtQYLlDt+wzpUhReaLAdMZ9/QrVazZVnNj
# EOC4oG3LVJaYoJkhUQaqpgw6K2PLedpDudISzN6PVXrpCfPimtMlwXs2ktrg5VJn
# zePmVAqlndRzo9MvXQZkhQN3f2DYwrx0yZWRaLe3fmEZdTCCBuowggTSoAMCAQIC
# EDGdnUgatvXgkrzF40/3PFswDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMx
# FjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xPDA6BgNVBAMTM0VudHJ1c3QgRXh0ZW5k
# ZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcgQ0EgLSBFVkNTMjAeFw0yMjEwMTEx
# MjM2MzdaFw0yNTEwMTExMjM2MzZaMIGaMQswCQYDVQQGEwJDWjEOMAwGA1UEBxMF
# UHJhaGExEzARBgsrBgEEAYI3PAIBAxMCQ1oxGTAXBgNVBAoTEEpldEJyYWlucyBz
# LnIuby4xHTAbBgNVBA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMREwDwYDVQQFEwgy
# NjUwMjI3NTEZMBcGA1UEAxMQSmV0QnJhaW5zIHMuci5vLjCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAMmZsBOO0sV6f+qwSfowL7CAaqooz6zzYDcPWA/6
# P1BlOeO9NqKEzd4IyZEI1lccipydIwdaO1YHa2kM7/5kertoyJ6ITuMjmcyxnqXI
# AhdCHLbqO/Vq6R+4fqokQ5FvPvsTZ6bbc8sznX3roST/zjlgYA5+HDR2GRZ4sy5d
# tkt/i/MANl3f3rRv1RO0qvy+dU7GOk0CWPfXFIHrj+sszSQ7zgM52bHAP70NlKfj
# VkKdX7U4Ytz3yb1gnfLRQyxYgM/MBKGDI6BrHlUfskryxjV5gH+aWFfBaXw8+NVl
# fYr8lUdlXlzlWmHqa7J792WNCLlE31NevEYaXA+0TbxJKT8bvXWehHxBpGY9Q15z
# WjnaGk1FguXxqmXKkRQfDgBiJ4sCtGW3iVGmOtMiEKmOURSZ/hoUQSrtMf8r/itK
# re6DHoxGbjA9yjzPXZbT1dJk6eOcKZaesYANyu05Kz7S/lRX83N4UbCQEJ4xpcCI
# G+eeE4d2BrapVFMZxobUKZaFtV+SAByQFEumEjEX5hEMyGzfODoa0KnFowf9Fu5u
# lz0dM3cV6/+riIgphUIlsNYgum7swNn6dp13+iyMuaOvPEL0kBlOPufcz7Lq18a6
# o5anOBJpW9jAMf0cCfJ+hwAOj3gcTMbiKCmEo7gEoQiZvnM4DZra7tf+Nblmv730
# RtPHAgMBAAGjggFgMIIBXDAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBQKghd/k2G5
# FxhP/0MIUgmYqUN3xjAfBgNVHSMEGDAWgBTOiU+CUaoVooRiyjEjYdJh+/j+eDBn
# BggrBgEFBQcBAQRbMFkwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3Qu
# bmV0MDIGCCsGAQUFBzAChiZodHRwOi8vYWlhLmVudHJ1c3QubmV0L2V2Y3MyLWNo
# YWluLnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVudHJ1c3QubmV0
# L2V2Y3MyLmNybDAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# SwYDVR0gBEQwQjA3BgpghkgBhvpsCgECMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAHBgVngQwBAzANBgkqhkiG9w0BAQsFAAOCAgEA
# BLLwIeBU0HShBD5L1ZoZN+ZQIpMOIHkQajuxja1zSLwakCX3lA0V3MpxliDDRx3e
# y4YA3OcefN2zV6xUAVNOb6V8zB552SblNVxa4TIxzmSzKI2uifjkikt1ZogDEXsS
# Cflyak2rbBATAmNrEBFAcZBLUGDxExbK2HyZAtG+CR16jj9Qd3zDHmbSciIMlsBq
# pmgmP34/pcjr/LfjXNqa2r0Q+ISbhVgv5N4ESYdBUKh0SAMf5fcUAreP0IpTOO4v
# FkkZSoUkALhGfBQDluukYrIGUh7fjmNVAwTQ1UjLRb8VDQwsIhFMBJkzevbxkqZA
# 3O6JLdisMuRL6/CFkdnftPAeiBJbs2jRQzADDGylIdIMv8V6A/yymtg9kZH9eCNw
# JibhEhTPHsEJX5Unwk8vE7POUqCKoB7+ULkGURTrUtEBBYmShPcbjQH0l6pcb66J
# oCX78Cbzd/Zr9Fm6mLjjlNbZcyjBSGDyuZq6SpKLL90YYCXHNEFKJdFCtnxtnbM4
# ipy3TZi9Xhx5OSfLd3T7/WgFQYuLHacteeGVonYBGXZbQiFKbUophNfYFRF0N5ZE
# cc7Vy3Vm+JSAvX496v6GRR3/C2zLI9ffZBqYLFhMGo0qIIUNLuIBbhX6pzw3mrlj
# VIdeZGNRj3vWqhAp9A7IzS6x3t2RrmSrkV1an7k6Um0xghrLMIIaxwIBATB3MGMx
# CzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNF
# bnRydXN0IEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZD
# UzICEDGdnUgatvXgkrzF40/3PFswCwYJYIZIAWUDBAIBoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCCsAEoIoDvG+9FEYqddEncOUXgFwLzokG6oGzSNQMnzhTBC
# BgorBgEEAYI3AgEMMTQwMqAwgC4AZABlAGYAZQBuAGQAZQByAC0AZQB4AGMAbAB1
# AHMAaQBvAG4AcwAuAHAAcwAxMA0GCSqGSIb3DQEBAQUABIICACPKOV3L6geSMuj0
# PjErTOnuASfIhIP3Dtq1KEJ/torGUhRvYZV29niqmV8OJ9NjHnJjmFrkzctGydEQ
# Zh/1jEq+ghGntWOHLzWXNaXA/aeaYNruOyRnCFHbndwEYCFh95LfBBvmJVG3UuDh
# ERi3hXdZMOdn9+ds7CjEptHHJosovduj6ODf3NrbchhSPBTYBurW6uul9ml4Euba
# fyJa0oA617DT4lfPtB7/4Z+YaCEuO4PNHjGFzsjmCDF/yR6H+k9u7QjTZjABIfqp
# mN062CzYXohu/MoUzpdsJQO4GQY7bkHbCAVnR4M8kqfBIe5rMCZTgsV+PxLvorys
# QELeoIG8Vw1BDXjFA7nPmYl4as1lKNPIJzmui26htUMUGG1QVUQ2yxm0JDDNtNas
# jdOyrzsB/kPdx1Fd4+P0ke/BE0hkJ+WlOkI5b5Qgk79BsWdwpfCd1jeg84B6GGmM
# Req1W/ukEnj5OrDxCC9gLAvtuuxWrcj3iW5DAdIHtjb6B1Kkmqy1jRxHLWQKDD58
# iT633nuX70lYFqQjsm9L/6iyjGI+HtvKN/OSRcOz/yDdIt8Nqcc6BZWtyPClqdwR
# 9AoSX5oQ7Bljj2SKk5UGOIowkrq5nhqCqQVzq0gDuHowW9doAf90agapQnmr4Cnp
# RlR/mrj97/G7jcaxtiIzza4PTMYioYIXdjCCF3IGCisGAQQBgjcDAwExghdiMIIX
# XgYJKoZIhvcNAQcCoIIXTzCCF0sCAQMxDzANBglghkgBZQMEAgEFADB3BgsqhkiG
# 9w0BCRABBKBoBGYwZAIBAQYJYIZIAYb9bAcBMDEwDQYJYIZIAWUDBAIBBQAEIAB7
# HkW6sP19ZHuaKF2xB2KbcOwB7zN1AFRES8NVtR14AhA+na18iYWrO5QVxPH9Jck1
# GA8yMDI1MDkxOTEyNDgwN1qgghM6MIIG7TCCBNWgAwIBAgIQCoDvGEuN8QWC0cR2
# p5V0aDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0
# YW1waW5nIFJTQTQwOTYgU0hBMjU2IDIwMjUgQ0ExMB4XDTI1MDYwNDAwMDAwMFoX
# DTM2MDkwMzIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0
# LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBTSEEyNTYgUlNBNDA5NiBUaW1lc3Rh
# bXAgUmVzcG9uZGVyIDIwMjUgMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBANBGrC0Sxp7Q6q5gVrMrV7pvUf+GcAoB38o3zBlCMGMyqJnfFNZx+wvA69HF
# TBdwbHwBSOeLpvPnZ8ZN+vo8dE2/pPvOx/Vj8TchTySA2R4QKpVD7dvNZh6wW2R6
# kSu9RJt/4QhguSssp3qome7MrxVyfQO9sMx6ZAWjFDYOzDi8SOhPUWlLnh00Cll8
# pjrUcCV3K3E0zz09ldQ//nBZZREr4h/GI6Dxb2UoyrN0ijtUDVHRXdmncOOMA3Co
# B/iUSROUINDT98oksouTMYFOnHoRh6+86Ltc5zjPKHW5KqCvpSduSwhwUmotuQhc
# g9tw2YD3w6ySSSu+3qU8DD+nigNJFmt6LAHvH3KSuNLoZLc1Hf2JNMVL4Q1Opbyb
# pMe46YceNA0LfNsnqcnpJeItK/DhKbPxTTuGoX7wJNdoRORVbPR1VVnDuSeHVZlc
# 4seAO+6d2sC26/PQPdP51ho1zBp+xUIZkpSFA8vWdoUoHLWnqWU3dCCyFG1roSrg
# HjSHlq8xymLnjCbSLZ49kPmk8iyyizNDIXj//cOgrY7rlRyTlaCCfw7aSUROwnu7
# zER6EaJ+AliL7ojTdS5PWPsWeupWs7NpChUk555K096V1hE0yZIXe+giAwW00aHz
# rDchIc2bQhpp0IoKRR7YufAkprxMiXAJQ1XCmnCfgPf8+3mnAgMBAAGjggGVMIIB
# kTAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBTkO/zyMe39/dfzkXFjGVBDz2GM6DAf
# BgNVHSMEGDAWgBTvb1NK6eQGfHrK4pBW9i/USezLTjAOBgNVHQ8BAf8EBAMCB4Aw
# FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwgZUGCCsGAQUFBwEBBIGIMIGFMCQGCCsG
# AQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXQYIKwYBBQUHMAKGUWh0
# dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFRpbWVT
# dGFtcGluZ1JTQTQwOTZTSEEyNTYyMDI1Q0ExLmNydDBfBgNVHR8EWDBWMFSgUqBQ
# hk5odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1l
# U3RhbXBpbmdSU0E0MDk2U0hBMjU2MjAyNUNBMS5jcmwwIAYDVR0gBBkwFzAIBgZn
# gQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQBlKq3xHCcEua5g
# QezRCESeY0ByIfjk9iJP2zWLpQq1b4URGnwWBdEZD9gBq9fNaNmFj6Eh8/YmRDfx
# T7C0k8FUFqNh+tshgb4O6Lgjg8K8elC4+oWCqnU/ML9lFfim8/9yJmZSe2F8AQ/U
# dKFOtj7YMTmqPO9mzskgiC3QYIUP2S3HQvHG1FDu+WUqW4daIqToXFE/JQ/EABgf
# ZXLWU0ziTN6R3ygQBHMUBaB5bdrPbF6MRYs03h4obEMnxYOX8VBRKe1uNnzQVTeL
# ni2nHkX/QqvXnNb+YkDFkxUGtMTaiLR9wjxUxu2hECZpqyU1d0IbX6Wq8/gVutDo
# jBIFeRlqAcuEVT0cKsb+zJNEsuEB7O7/cuvTQasnM9AWcIQfVjnzrvwiCZ85EE8L
# UkqRhoS3Y50OHgaY7T/lwd6UArb+BOVAkg2oOvol/DJgddJ35XTxfUlQ+8Hggt8l
# 2Yv7roancJIFcbojBcxlRcGG0LIhp6GvReQGgMgYxQbV1S3CrWqZzBt1R9xJgKf4
# 7CdxVRd/ndUlQ05oxYy2zRWVFjF7mcr4C34Mj3ocCVccAvlKV9jEnstrniLvUxxV
# ZE/rptb7IRE2lskKPIJgbaP5t2nGj/ULLi49xTcBZU8atufk+EMF/cWuiC7POGT7
# 5qaL6vdCvHlshtjdNXOCIUjsarfNZzCCBrQwggScoAMCAQICEA3HrFcF/yGZLkBD
# Igw6SYYwDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERp
# Z2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMY
# RGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTI1MDUwNzAwMDAwMFoXDTM4MDEx
# NDIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBSU0E0
# MDk2IFNIQTI1NiAyMDI1IENBMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBALR4MdMKmEFyvjxGwBysddujRmh0tFEXnU2tjQ2UtZmWgyxU7UNqEY81FzJs
# Qqr5G7A6c+Gh/qm8Xi4aPCOo2N8S9SLrC6Kbltqn7SWCWgzbNfiR+2fkHUiljNOq
# nIVD/gG3SYDEAd4dg2dDGpeZGKe+42DFUF0mR/vtLa4+gKPsYfwEu7EEbkC9+0F2
# w4QJLVSTEG8yAR2CQWIM1iI5PHg62IVwxKSpO0XaF9DPfNBKS7Zazch8NF5vp7ea
# Z2CVNxpqumzTCNSOxm+SAWSuIr21Qomb+zzQWKhxKTVVgtmUPAW35xUUFREmDrMx
# SNlr/NsJyUXzdtFUUt4aS4CEeIY8y9IaaGBpPNXKFifinT7zL2gdFpBP9qh8SdLn
# Eut/GcalNeJQ55IuwnKCgs+nrpuQNfVmUB5KlCX3ZA4x5HHKS+rqBvKWxdCyQEEG
# cbLe1b8Aw4wJkhU1JrPsFfxW1gaou30yZ46t4Y9F20HHfIY4/6vHespYMQmUiote
# 8ladjS/nJ0+k6MvqzfpzPDOy5y6gqztiT96Fv/9bH7mQyogxG9QEPHrPV6/7umw0
# 52AkyiLA6tQbZl1KhBtTasySkuJDpsZGKdlsjg4u70EwgWbVRSX1Wd4+zoFpp4Ra
# +MlKM2baoD6x0VR4RjSpWM8o5a6D8bpfm4CLKczsG7ZrIGNTAgMBAAGjggFdMIIB
# WTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBTvb1NK6eQGfHrK4pBW9i/U
# SezLTjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8BAf8E
# BAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBpMCQGCCsG
# AQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKGNWh0
# dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQu
# Y3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9E
# aWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsG
# CWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAF877FoAc/gc9EXZxML2+C8i1
# NKZ/zdCHxYgaMH9Pw5tcBnPw6O6FTGNpoV2V4wzSUGvI9NAzaoQk97frPBtIj+ZL
# zdp+yXdhOP4hCFATuNT+ReOPK0mCefSG+tXqGpYZ3essBS3q8nL2UwM+NMvEuBd/
# 2vmdYxDCvwzJv2sRUoKEfJ+nN57mQfQXwcAEGCvRR2qKtntujB71WPYAgwPyWLKu
# 6RnaID/B0ba2H3LUiwDRAXx1Neq9ydOal95CHfmTnM4I+ZI2rVQfjXQA1WSjjf4J
# 2a7jLzWGNqNX+DF0SQzHU0pTi4dBwp9nEC8EAqoxW6q17r0z0noDjs6+BFo+z7bK
# SBwZXTRNivYuve3L2oiKNqetRHdqfMTCW/NmKLJ9M+MtucVGyOxiDf06VXxyKkOi
# rv6o02OoXN4bFzK0vlNMsvhlqgF2puE6FndlENSmE+9JGYxOGLS/D284NHNboDGc
# mWXfwXRy4kbu4QFhOm0xJuF2EZAOk5eCkhSxZON3rGlHqhpB/8MluDezooIs8CVn
# rpHMiD2wL40mm53+/j7tFaxYKIqL0Q4ssd8xHZnIn/7GELH3IdvG2XlM9q7WP/Uw
# gOkw/HQtyRN62JK4S1C8uw3PdBunvAZapsiI5YKdvlarEvf8EA+8hcpSM9LHJmyr
# xaFtoza2zNaQ9k+5t1wwggWNMIIEdaADAgECAhAOmxiO+dAt5+/bUOIIQBhaMA0G
# CSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJ
# bmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNVBAMTG0RpZ2lDZXJ0
# IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAwMDBaFw0zMTExMDkyMzU5
# NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNV
# BAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQg
# Um9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL/mkHNo3rvk
# XUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsbhA3EMB/zG6Q4FutWxpdt
# HauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iTcMKyunWZanMylNEQRBAu
# 34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGbNOsFxl7sWxq868nPzaw0
# QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclPXuU15zHL2pNe3I6PgNq2
# kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCrVYJBMtfbBHMqbpEBfCFM
# 1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFPObURWBf3JFxGj2T3wWmI
# dph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTvkpI6nj3cAORFJYm2mkQZ
# K37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWMcCxBYKqxYxhElRp2Yn72
# gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls5Q5SUUd0viastkF13nqs
# X40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBRa2+xq4aLT8LWRV+dIPyh
# HsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6MIIBNjAPBgNVHRMBAf8E
# BTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qYrhwPTzAfBgNVHSMEGDAW
# gBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8EBAMCAYYweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDARBgNVHSAE
# CjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCgv0NcVec4X6CjdBs9thbX
# 979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQTSnovLbc47/T/gLn4offy
# ct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh65ZyoUi0mcudT6cGAxN3
# J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSwuKFWjuyk1T3osdz9HNj0
# d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAOQGPFmCLBsln1VWvPJ6ts
# ds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjDTZ9ztwGpn1eqXijiuZQx
# ggN8MIIDeAIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcg
# UlNBNDA5NiBTSEEyNTYgMjAyNSBDQTECEAqA7xhLjfEFgtHEdqeVdGgwDQYJYIZI
# AWUDBAIBBQCggdEwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3
# DQEJBTEPFw0yNTA5MTkxMjQ4MDdaMCsGCyqGSIb3DQEJEAIMMRwwGjAYMBYEFN1i
# MKyGCi0wa9o4sWh5UjAH+0F+MC8GCSqGSIb3DQEJBDEiBCDS/YlYZA9Z8FSMIs2n
# X/nMHNa208+DsSkg7n9oGOi3IjA3BgsqhkiG9w0BCRACLzEoMCYwJDAiBCBKoD+i
# LNdchMVck4+CjmdrnK7Ksz/jbSaaozTxRhEKMzANBgkqhkiG9w0BAQEFAASCAgC3
# rGAkBkP5GDBT1fPuPX14oNZ0QFKxOdl0CLuTVWB0nzQpMwqXKa3wfmbtk4eoxcfn
# uTzoC81nq7usmVRPrIy1TkGfbwXta2788acfB5at04FAmBBqgwm+ol2rvuhGrAXr
# QDNd4SGNZKDPcFv5oAJZbqVGu0fmQZZ/IcU2yQrMnwq7r7zf9Cv3eUvfx9/m8157
# dWL6a1lJ5W1OydBQx+XNEqVk13bCT+XYvmEt3qM3Ht/vISwzwYnn98eilFIzb8pU
# KPp6JUn5kT+feMVXnZ/qw5/2aM8bYFa529w7thP7/cH+ATypTLhWSY6qi11cyWpX
# eZI4bdHloOqbYE19HBrrKZRT5nyluEIDaJNsW2+YG+6e5H9VCQ041CtfxEMdKevt
# xWEgHIqVI/7lnOfBCX1P6KWpM3vFWJO0BbZ/XttizJB38mtJOqfdEg3VaV/ftL4S
# ycUmZys1WgHPmrBv2Y2LqK/t6GwJ2aKW3UEWLPDtozOPvFQotfoYUIbosFX16SGk
# b15acErKOSN2AywTN0IcKjuBJPk3aKBtaldMx/2NAATJMpqIOWyInBcKtdqSLvsN
# /JYz/ZWXNNin5eKuGf51Bnx1+Zl7YoKx5fBtuyTy8Klj52H8253ZaEWz72EZg/WR
# b6y7M0gUPFDtgXtq/ytqhoe0fwqbO1yY4PgGDgeOww==
# SIG # End signature block
