/*
 * Decompiled with CFR 0.152.
 */
package com.jediterm.terminal.model.hyperlinks;

import com.jediterm.terminal.HyperlinkStyle;
import com.jediterm.terminal.TextStyle;
import com.jediterm.terminal.model.CharBuffer;
import com.jediterm.terminal.model.LinesStorage;
import com.jediterm.terminal.model.TerminalHyperlinkListener;
import com.jediterm.terminal.model.TerminalLine;
import com.jediterm.terminal.model.TerminalLineUtil;
import com.jediterm.terminal.model.TerminalTextBuffer;
import com.jediterm.terminal.model.TextBufferChangesListener;
import com.jediterm.terminal.model.hyperlinks.AsyncHyperlinkFilter;
import com.jediterm.terminal.model.hyperlinks.HyperlinkFilter;
import com.jediterm.terminal.model.hyperlinks.LinkResult;
import com.jediterm.terminal.model.hyperlinks.LinkResultItem;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TextProcessing {
    private static final Logger LOG = LoggerFactory.getLogger(TextProcessing.class);
    private static final int MAX_RESCHEDULING_ATTEMPTS = 5;
    private final List<AsyncHyperlinkFilter> myHyperlinkFilters = new CopyOnWriteArrayList<AsyncHyperlinkFilter>();
    private final TextStyle myHyperlinkColor;
    private final HyperlinkStyle.HighlightMode myHighlightMode;
    private TerminalTextBuffer myTerminalTextBuffer;
    private final List<TerminalHyperlinkListener> myHyperlinkListeners = new CopyOnWriteArrayList<TerminalHyperlinkListener>();

    public TextProcessing(@NotNull TextStyle hyperlinkColor, @NotNull HyperlinkStyle.HighlightMode highlightMode) {
        this.myHyperlinkColor = hyperlinkColor;
        this.myHighlightMode = highlightMode;
    }

    public void setTerminalTextBuffer(@NotNull TerminalTextBuffer terminalTextBuffer) {
        this.myTerminalTextBuffer = terminalTextBuffer;
        terminalTextBuffer.addChangesListener(new TextBufferChangesListener(){

            @Override
            public void linesDiscardedFromHistory(@NotNull @NotNull List<@NotNull TerminalLine> lines2) {
                for (TerminalLine line : lines2) {
                    TerminalLineUtil.INSTANCE.incModificationCount$core(line);
                }
            }

            @Override
            public void historyCleared() {
            }

            @Override
            public void widthResized() {
            }

            @Override
            public void linesChanged(int fromIndex) {
            }
        });
    }

    public void processHyperlinks(@NotNull LinesStorage linesStorage, @NotNull TerminalLine updatedLine) {
        if (!this.myHyperlinkFilters.isEmpty()) {
            this.doProcessHyperlinks(linesStorage, updatedLine);
        }
    }

    private void doProcessHyperlinks(@NotNull LinesStorage linesStorage, @NotNull TerminalLine updatedLine) {
        LineInfoImpl lineInfo = this.buildLineInfo(linesStorage, updatedLine);
        if (lineInfo != null) {
            this.doProcessHyperlinks(linesStorage, lineInfo, 1);
        }
    }

    private void doProcessHyperlinks(@NotNull LinesStorage linesStorage, @NotNull LineInfoImpl lineInfo, int attemptNumber) {
        for (AsyncHyperlinkFilter filter2 : this.myHyperlinkFilters) {
            CompletableFuture<LinkResult> resultFuture = filter2.apply(lineInfo);
            resultFuture.whenComplete((result2, error) -> {
                if (result2 != null) {
                    this.applyLinkResultsOrReschedule(linesStorage, lineInfo, result2.getItems(), attemptNumber);
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void applyLinkResultsOrReschedule(@NotNull LinesStorage linesStorage, @NotNull LineInfoImpl lineInfo, @NotNull List<LinkResultItem> resultItems, int attemptNumber) {
        if (resultItems.isEmpty()) {
            return;
        }
        this.myTerminalTextBuffer.lock();
        try {
            String lineStr = lineInfo.getLine();
            if (lineStr == null) {
                return;
            }
            int terminalWidth = this.myTerminalTextBuffer.getWidth();
            if (lineInfo.myTerminalWidth == terminalWidth) {
                this.applyLinkResults(resultItems, lineInfo, lineStr);
            } else if (attemptNumber < 5) {
                List<List<TerminalLine>> matchedWrappedLines = new TerminalLineFinder(this.myTerminalTextBuffer, lineStr).findMatchedLines(200);
                for (List<TerminalLine> wrappedLine : matchedWrappedLines) {
                    LineInfoImpl newLineInfo = new LineInfoImpl(wrappedLine, terminalWidth);
                    this.doProcessHyperlinks(linesStorage, newLineInfo, attemptNumber + 1);
                }
            }
        }
        finally {
            this.myTerminalTextBuffer.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    private LineInfoImpl buildLineInfo(@NotNull LinesStorage linesStorage, @NotNull TerminalLine updatedLine) {
        this.myTerminalTextBuffer.lock();
        try {
            int updatedLineInd = linesStorage.indexOf(updatedLine);
            if (updatedLineInd == -1) {
                LinesStorage historyLinesStorage = this.myTerminalTextBuffer.getHistoryLinesStorage();
                updatedLineInd = this.findHistoryLineInd(historyLinesStorage, updatedLine);
                if (updatedLineInd == -1) {
                    LOG.debug("Cannot find line for links processing");
                    LineInfoImpl lineInfoImpl = null;
                    return lineInfoImpl;
                }
                linesStorage = historyLinesStorage;
            }
            int startLineInd = this.findStartLineInd(linesStorage, updatedLineInd);
            List<TerminalLine> linesToProcess = this.collectLines(linesStorage, startLineInd, updatedLineInd);
            LineInfoImpl lineInfoImpl = new LineInfoImpl(linesToProcess, this.myTerminalTextBuffer.getWidth());
            return lineInfoImpl;
        }
        finally {
            this.myTerminalTextBuffer.unlock();
        }
    }

    @NotNull
    private List<TerminalLine> collectLines(@NotNull LinesStorage linesStorage, int startLineInd, int updatedLineInd) {
        if (startLineInd == updatedLineInd) {
            return List.of(linesStorage.get(startLineInd));
        }
        ArrayList<TerminalLine> result2 = new ArrayList<TerminalLine>(updatedLineInd - startLineInd + 1);
        for (int i2 = startLineInd; i2 <= updatedLineInd; ++i2) {
            result2.add(linesStorage.get(i2));
        }
        return result2;
    }

    private int findStartLineInd(@NotNull LinesStorage linesStorage, int lineInd) {
        int startLineInd;
        for (startLineInd = lineInd; startLineInd > 0 && linesStorage.get(startLineInd - 1).isWrapped(); --startLineInd) {
        }
        return startLineInd;
    }

    private void applyLinkResults(@NotNull List<LinkResultItem> linkResultItems, @NotNull LineInfoImpl lineInfo, @NotNull String lineStr) {
        boolean linkAdded = false;
        int terminalWidth = lineInfo.myTerminalWidth;
        String actualLineStr = TextProcessing.joinLines(lineInfo.myLinesToProcess, terminalWidth);
        if (!actualLineStr.equals(lineStr)) {
            LOG.warn("Outdated lines when applying hyperlinks");
            return;
        }
        for (LinkResultItem item : linkResultItems) {
            if (item.getStartOffset() < 0 || item.getEndOffset() > lineStr.length()) continue;
            HyperlinkStyle style2 = new HyperlinkStyle(this.myHyperlinkColor.getForeground(), this.myHyperlinkColor.getBackground(), item.getLinkInfo(), this.myHighlightMode, null);
            int prevLinesLength = 0;
            for (TerminalLine line : lineInfo.myLinesToProcess) {
                int endLineOffset;
                int startLineOffset = Math.max(prevLinesLength, item.getStartOffset());
                if (startLineOffset < (endLineOffset = Math.min(prevLinesLength + lineInfo.myTerminalWidth, item.getEndOffset()))) {
                    line.writeString(startLineOffset - prevLinesLength, new CharBuffer(lineStr.substring(startLineOffset, endLineOffset)), style2);
                    linkAdded = true;
                }
                prevLinesLength += terminalWidth;
            }
        }
        if (linkAdded) {
            this.fireHyperlinksChanged();
        }
    }

    private void fireHyperlinksChanged() {
        for (TerminalHyperlinkListener myHyperlinkListener : this.myHyperlinkListeners) {
            myHyperlinkListener.hyperlinksChanged();
        }
    }

    public void addHyperlinkListener(@NotNull TerminalHyperlinkListener listener) {
        this.myHyperlinkListeners.add(listener);
    }

    private int findHistoryLineInd(@NotNull LinesStorage historyLinesStorage, @NotNull TerminalLine line) {
        int lastLineInd = Math.max(0, historyLinesStorage.getSize() - 200);
        for (int i2 = historyLinesStorage.getSize() - 1; i2 >= lastLineInd; --i2) {
            if (historyLinesStorage.get(i2) != line) continue;
            return i2;
        }
        return -1;
    }

    public void addHyperlinkFilter(final @NotNull HyperlinkFilter filter2) {
        this.addAsyncHyperlinkFilter(new AsyncHyperlinkFilter(){

            @Override
            public @NotNull CompletableFuture<@Nullable LinkResult> apply(@NotNull AsyncHyperlinkFilter.LineInfo lineInfo) {
                String lineStr = lineInfo.getLine();
                if (lineStr == null) {
                    return CompletableFuture.completedFuture(null);
                }
                LinkResult result2 = filter2.apply(lineStr);
                return CompletableFuture.completedFuture(result2);
            }
        });
    }

    public void addAsyncHyperlinkFilter(@NotNull AsyncHyperlinkFilter filter2) {
        this.myHyperlinkFilters.add(filter2);
    }

    @NotNull
    public List<LinkResultItem> applyFilter(final @NotNull String line) {
        return this.myHyperlinkFilters.stream().map(filter2 -> {
            CompletableFuture<@Nullable LinkResult> resultFuture = filter2.apply(new AsyncHyperlinkFilter.LineInfo(){

                @Override
                @NotNull
                public String getLine() {
                    return line;
                }
            });
            try {
                return resultFuture.get(2L, TimeUnit.SECONDS);
            }
            catch (Exception e2) {
                LOG.info("Failed to find links in {}", (Object)line, (Object)e2);
                return null;
            }
        }).filter(Objects::nonNull).flatMap(result2 -> result2.getItems().stream()).collect(Collectors.toList());
    }

    @NotNull
    private static String joinLines(@NotNull List<TerminalLine> lines2, int terminalWidth) {
        StringBuilder result2 = new StringBuilder();
        int size = lines2.size();
        for (int i2 = 0; i2 < size; ++i2) {
            String text = lines2.get(i2).getText();
            result2.append(text);
            if (i2 >= size - 1 || text.length() >= terminalWidth) continue;
            result2.append(new CharBuffer(' ', terminalWidth - text.length()));
        }
        return result2.toString();
    }

    private static class TerminalLineFinder {
        private final List<TerminalLine> myCurrentLine = new ArrayList<TerminalLine>();
        private final List<List<TerminalLine>> myMatchedLines = new ArrayList<List<TerminalLine>>();
        private final TerminalTextBuffer myTextBuffer;
        private final String myLineToFind;

        public TerminalLineFinder(@NotNull TerminalTextBuffer textBuffer, @NotNull String lineToFind) {
            this.myTextBuffer = textBuffer;
            this.myLineToFind = lineToFind;
        }

        @NotNull
        List<List<TerminalLine>> findMatchedLines(int topHistoryCount) {
            for (int i2 = -Math.min(topHistoryCount, this.myTextBuffer.getHistoryLinesCount()); i2 < this.myTextBuffer.getScreenLinesCount(); ++i2) {
                this.add(this.myTextBuffer.getLine(i2));
            }
            return this.myMatchedLines;
        }

        private void add(@NotNull TerminalLine line) {
            this.myCurrentLine.add(line);
            if (!line.isWrapped()) {
                String lineStr = TextProcessing.joinLines(this.myCurrentLine, this.myTextBuffer.getWidth());
                if (lineStr.equals(this.myLineToFind)) {
                    this.myMatchedLines.add(new ArrayList<TerminalLine>(this.myCurrentLine));
                }
                this.myCurrentLine.clear();
            }
        }
    }

    private class LineInfoImpl
    implements AsyncHyperlinkFilter.LineInfo {
        private final List<TerminalLine> myLinesToProcess;
        private final int[] initialModificationCounts;
        private final int myTerminalWidth;
        private String myCachedLineStr;
        private volatile boolean isUpToDate = true;

        LineInfoImpl(List<TerminalLine> linesToProcess, int terminalWidth) {
            this.myLinesToProcess = linesToProcess;
            this.initialModificationCounts = new int[linesToProcess.size()];
            int i2 = 0;
            for (TerminalLine line : linesToProcess) {
                TerminalLineUtil.INSTANCE.incModificationCount$core(line);
                this.initialModificationCounts[i2++] = TerminalLineUtil.INSTANCE.getModificationCount$core(line);
            }
            this.myTerminalWidth = terminalWidth;
        }

        private boolean isUpToDate() {
            boolean isUpToDate = this.isUpToDate;
            if (isUpToDate) {
                this.isUpToDate = isUpToDate = this.areLinesUpToDate();
            }
            return isUpToDate;
        }

        private boolean areLinesUpToDate() {
            for (int i2 = 0; i2 < this.myLinesToProcess.size(); ++i2) {
                TerminalLine line = this.myLinesToProcess.get(i2);
                if (TerminalLineUtil.INSTANCE.getModificationCount$core(line) == this.initialModificationCounts[i2]) continue;
                return false;
            }
            return true;
        }

        @Override
        @Nullable
        public String getLine() {
            if (!this.isUpToDate()) {
                return null;
            }
            if (this.myCachedLineStr == null) {
                TextProcessing.this.myTerminalTextBuffer.lock();
                try {
                    this.myCachedLineStr = TextProcessing.joinLines(this.myLinesToProcess, this.myTerminalWidth);
                }
                finally {
                    TextProcessing.this.myTerminalTextBuffer.unlock();
                }
            }
            return this.myCachedLineStr;
        }
    }
}

