/*
 * Decompiled with CFR 0.152.
 */
package net.i2p.crypto.eddsa;

import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import net.i2p.crypto.eddsa.EdDSAKey;
import net.i2p.crypto.eddsa.EdDSAPrivateKey;
import net.i2p.crypto.eddsa.EdDSAPublicKey;
import net.i2p.crypto.eddsa.math.Curve;
import net.i2p.crypto.eddsa.math.GroupElement;
import net.i2p.crypto.eddsa.math.ScalarOps;
import sun.security.x509.X509Key;

public final class EdDSAEngine
extends Signature {
    public static final String SIGNATURE_ALGORITHM = "NONEwithEdDSA";
    private MessageDigest digest;
    private ByteArrayOutputStream baos;
    private EdDSAKey key;
    private boolean oneShotMode;
    private byte[] oneShotBytes;
    private int oneShotOffset;
    private int oneShotLength;
    public static final AlgorithmParameterSpec ONE_SHOT_MODE = new OneShotSpec();

    public EdDSAEngine() {
        super(SIGNATURE_ALGORITHM);
    }

    public EdDSAEngine(MessageDigest digest) {
        this();
        this.digest = digest;
    }

    private void reset() {
        if (this.digest != null) {
            this.digest.reset();
        }
        if (this.baos != null) {
            this.baos.reset();
        }
        this.oneShotMode = false;
        this.oneShotBytes = null;
    }

    @Override
    protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException {
        EdDSAPrivateKey privKey;
        this.reset();
        if (privateKey instanceof EdDSAPrivateKey) {
            privKey = (EdDSAPrivateKey)privateKey;
            this.key = privKey;
            if (this.digest == null) {
                try {
                    this.digest = MessageDigest.getInstance(this.key.getParams().getHashAlgorithm());
                }
                catch (NoSuchAlgorithmException e2) {
                    throw new InvalidKeyException("cannot get required digest " + this.key.getParams().getHashAlgorithm() + " for private key.");
                }
            } else if (!this.key.getParams().getHashAlgorithm().equals(this.digest.getAlgorithm())) {
                throw new InvalidKeyException("Key hash algorithm does not match chosen digest");
            }
        } else {
            throw new InvalidKeyException("cannot identify EdDSA private key: " + privateKey.getClass());
        }
        this.digestInitSign(privKey);
    }

    private void digestInitSign(EdDSAPrivateKey privKey) {
        int b2 = privKey.getParams().getCurve().getField().getb();
        this.digest.update(privKey.getH(), b2 / 8, b2 / 4 - b2 / 8);
    }

    @Override
    protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException {
        this.reset();
        if (publicKey instanceof EdDSAPublicKey) {
            this.key = (EdDSAPublicKey)publicKey;
            if (this.digest == null) {
                try {
                    this.digest = MessageDigest.getInstance(this.key.getParams().getHashAlgorithm());
                }
                catch (NoSuchAlgorithmException e2) {
                    throw new InvalidKeyException("cannot get required digest " + this.key.getParams().getHashAlgorithm() + " for private key.");
                }
            } else if (!this.key.getParams().getHashAlgorithm().equals(this.digest.getAlgorithm())) {
                throw new InvalidKeyException("Key hash algorithm does not match chosen digest");
            }
        } else if (publicKey instanceof X509Key) {
            EdDSAPublicKey parsedPublicKey;
            try {
                parsedPublicKey = new EdDSAPublicKey(new X509EncodedKeySpec(publicKey.getEncoded()));
            }
            catch (InvalidKeySpecException ex2) {
                throw new InvalidKeyException("cannot handle X.509 EdDSA public key: " + publicKey.getAlgorithm());
            }
            this.engineInitVerify(parsedPublicKey);
        } else {
            throw new InvalidKeyException("cannot identify EdDSA public key: " + publicKey.getClass());
        }
    }

    @Override
    protected void engineUpdate(byte b2) throws SignatureException {
        if (this.oneShotMode) {
            throw new SignatureException("unsupported in one-shot mode");
        }
        if (this.baos == null) {
            this.baos = new ByteArrayOutputStream(256);
        }
        this.baos.write(b2);
    }

    @Override
    protected void engineUpdate(byte[] b2, int off, int len) throws SignatureException {
        if (this.oneShotMode) {
            if (this.oneShotBytes != null) {
                throw new SignatureException("update() already called");
            }
            this.oneShotBytes = b2;
            this.oneShotOffset = off;
            this.oneShotLength = len;
        } else {
            if (this.baos == null) {
                this.baos = new ByteArrayOutputStream(256);
            }
            this.baos.write(b2, off, len);
        }
    }

    @Override
    protected byte[] engineSign() throws SignatureException {
        try {
            byte[] byArray = this.x_engineSign();
            return byArray;
        }
        finally {
            this.reset();
            EdDSAPrivateKey privKey = (EdDSAPrivateKey)this.key;
            this.digestInitSign(privKey);
        }
    }

    private byte[] x_engineSign() throws SignatureException {
        int length;
        int offset;
        byte[] message;
        Curve curve = this.key.getParams().getCurve();
        ScalarOps sc = this.key.getParams().getScalarOps();
        byte[] a2 = ((EdDSAPrivateKey)this.key).geta();
        if (this.oneShotMode) {
            if (this.oneShotBytes == null) {
                throw new SignatureException("update() not called first");
            }
            message = this.oneShotBytes;
            offset = this.oneShotOffset;
            length = this.oneShotLength;
        } else {
            message = this.baos == null ? new byte[]{} : this.baos.toByteArray();
            offset = 0;
            length = message.length;
        }
        this.digest.update(message, offset, length);
        byte[] r2 = this.digest.digest();
        r2 = sc.reduce(r2);
        GroupElement R2 = this.key.getParams().getB().scalarMultiply(r2);
        byte[] Rbyte = R2.toByteArray();
        this.digest.update(Rbyte);
        this.digest.update(((EdDSAPrivateKey)this.key).getAbyte());
        this.digest.update(message, offset, length);
        byte[] h7 = this.digest.digest();
        h7 = sc.reduce(h7);
        byte[] S2 = sc.multiplyAndAdd(h7, a2, r2);
        int b2 = curve.getField().getb();
        ByteBuffer out = ByteBuffer.allocate(b2 / 4);
        out.put(Rbyte).put(S2);
        return out.array();
    }

    @Override
    protected boolean engineVerify(byte[] sigBytes) throws SignatureException {
        try {
            boolean bl2 = this.x_engineVerify(sigBytes);
            return bl2;
        }
        finally {
            this.reset();
        }
    }

    private boolean x_engineVerify(byte[] sigBytes) throws SignatureException {
        int length;
        int offset;
        byte[] message;
        Curve curve = this.key.getParams().getCurve();
        int b2 = curve.getField().getb();
        if (sigBytes.length != b2 / 4) {
            throw new SignatureException("signature length is wrong");
        }
        this.digest.update(sigBytes, 0, b2 / 8);
        this.digest.update(((EdDSAPublicKey)this.key).getAbyte());
        if (this.oneShotMode) {
            if (this.oneShotBytes == null) {
                throw new SignatureException("update() not called first");
            }
            message = this.oneShotBytes;
            offset = this.oneShotOffset;
            length = this.oneShotLength;
        } else {
            message = this.baos == null ? new byte[]{} : this.baos.toByteArray();
            offset = 0;
            length = message.length;
        }
        this.digest.update(message, offset, length);
        byte[] h7 = this.digest.digest();
        h7 = this.key.getParams().getScalarOps().reduce(h7);
        byte[] Sbyte = Arrays.copyOfRange(sigBytes, b2 / 8, b2 / 4);
        GroupElement R2 = this.key.getParams().getB().doubleScalarMultiplyVariableTime(((EdDSAPublicKey)this.key).getNegativeA(), h7, Sbyte);
        byte[] Rcalc = R2.toByteArray();
        for (int i2 = 0; i2 < Rcalc.length; ++i2) {
            if (Rcalc[i2] == sigBytes[i2]) continue;
            return false;
        }
        return true;
    }

    public byte[] signOneShot(byte[] data) throws SignatureException {
        return this.signOneShot(data, 0, data.length);
    }

    public byte[] signOneShot(byte[] data, int off, int len) throws SignatureException {
        this.oneShotMode = true;
        this.update(data, off, len);
        return this.sign();
    }

    public boolean verifyOneShot(byte[] data, byte[] signature) throws SignatureException {
        return this.verifyOneShot(data, 0, data.length, signature, 0, signature.length);
    }

    public boolean verifyOneShot(byte[] data, int off, int len, byte[] signature) throws SignatureException {
        return this.verifyOneShot(data, off, len, signature, 0, signature.length);
    }

    public boolean verifyOneShot(byte[] data, byte[] signature, int sigoff, int siglen) throws SignatureException {
        return this.verifyOneShot(data, 0, data.length, signature, sigoff, siglen);
    }

    public boolean verifyOneShot(byte[] data, int off, int len, byte[] signature, int sigoff, int siglen) throws SignatureException {
        this.oneShotMode = true;
        this.update(data, off, len);
        return this.verify(signature, sigoff, siglen);
    }

    @Override
    protected void engineSetParameter(AlgorithmParameterSpec spec) throws InvalidAlgorithmParameterException {
        if (spec.equals(ONE_SHOT_MODE)) {
            if (this.oneShotBytes != null || this.baos != null && this.baos.size() > 0) {
                throw new InvalidAlgorithmParameterException("update() already called");
            }
            this.oneShotMode = true;
        } else {
            super.engineSetParameter(spec);
        }
    }

    @Override
    protected void engineSetParameter(String param2, Object value) {
        throw new UnsupportedOperationException("engineSetParameter unsupported");
    }

    @Override
    protected Object engineGetParameter(String param2) {
        throw new UnsupportedOperationException("engineSetParameter unsupported");
    }

    private static class OneShotSpec
    implements AlgorithmParameterSpec {
        private OneShotSpec() {
        }
    }
}

