package com.intellij.database.datagrid;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

public interface AutoValueDescriptor extends JdbcColumnDescriptor {
  boolean isAutogenerated();
  boolean isComputed();
  boolean hasDefault();

  record TypeName(@Nullable String value) {}

  class DelegateDescriptor<T extends JdbcColumnDescriptor> implements AutoValueDescriptor {
    private final T myDelegate;
    private final boolean myAutogenerated;
    private final boolean myHasDefault;
    private final boolean myComputed;
    private final @Nullable TypeName myTypeName;

    public DelegateDescriptor(@NotNull T delegate, boolean autogenerated, boolean hasDefault, boolean computed) {
      myDelegate = delegate;
      myAutogenerated = autogenerated;
      myHasDefault = hasDefault;
      myComputed = computed;
      myTypeName = null;
    }

    public DelegateDescriptor(@NotNull T delegate, boolean autogenerated, boolean hasDefault, boolean computed, @Nullable String typeName) {
      myDelegate = delegate;
      myAutogenerated = autogenerated;
      myHasDefault = hasDefault;
      myComputed = computed;
      myTypeName = new TypeName(typeName);
    }

    @Override
    public int getScale() {
      return myDelegate.getScale();
    }

    @Override
    public int getSize() {
      return myDelegate.getSize();
    }

    @Override
    public int getType() {
      return myDelegate.getType();
    }

    @Override
    public String getName() {
      return myDelegate.getName();
    }

    @Override
    public String getTypeName() {
      if (myTypeName != null) {
        return myTypeName.value();
      }
      return myDelegate.getTypeName();
    }

    @Override
    public @Nullable String getJavaClassName() {
      return myDelegate.getJavaClassName();
    }

    @Override
    public @NotNull Set<Attribute> getAttributes() {
      return myDelegate.getAttributes();
    }

    @Override
    public boolean isAutogenerated() {
      return myAutogenerated;
    }

    @Override
    public boolean isComputed() {
      return myComputed;
    }

    @Override
    public boolean hasDefault() {
      return myHasDefault;
    }

    public @NotNull T getDelegate() {
      return myDelegate;
    }

    @Override
    public String toString() {
      return myDelegate.toString();
    }
  }
}