import {__log, __null_aware_to_string, Variables, Environment, InPlaceVariables} from 'js-graalvm-stdlib.js'
import {jwt} from "js-graalvm-jwt.js"

function ContentType(mimeType, charset) {
  this.mimeType = mimeType
  this.charset = charset
}

function __null_aware_lowercase(value) {
  return value != null ? ("" + value).toLowerCase() : null
}

Object.defineProperty(globalThis, "request", {
  value: new Request(__args.requestElementsProvider)
});


function HttpClient() {
  this.tests = {}
  const self = this
  this.global = new Variables(__args.global)
  this.variables = {
    global: self.global,
    environment: new Environment(__args.elementsProvider),
    file: new InPlaceVariables(__args.elementsProvider),
    request: request.variables,
  }

  this.test = function (testName, func) {
    let exception
    try {
      throw new Error("error")
    }
    catch (e) {
      exception = e
    }
    this.tests[testName] = {
      "func": func || null,
      "creationException": exception,
    }
  }

  this.assert = function (condition, message) {
    if (!condition) {
      __args.assert(message, this)
    }
  }

  this.log = __log

  this.exit = function () {
    __args.cancelable.accept(this)
  }
}

function ResponseHeaders(headers) {
  this.headers = headers

  this.valueOf = function (headerName) {
    const len = this.headers.length
    for (let i = 0; i < len; i++) {
      if (__null_aware_lowercase(headerName) === __null_aware_lowercase(this.headers[i].name)) {
        return __null_aware_to_string(this.headers[i].value())
      }
    }
    return null
  }

  this.valuesOf = function (headerName) {
    let values = []
    let len = this.headers.length
    for (let i = 0; i < len; i++) {
      if (__null_aware_lowercase(headerName) === __null_aware_lowercase(this.headers[i].name)) {
        values.push(__null_aware_to_string(headers[i].value()))
      }
    }
    return values
  }

  this.toJSON = function () {
    const obj = {};
    const len = this.headers.length

    for (let i = 0; i < len; i++) {
      let key = "" + headers[i].name;
      let headerValue = obj[key];
      if (headerValue === undefined) {
        obj[key] = headers[i].value()
      }
      else if (headerValue instanceof Array) {
        headerValue.push(headers[i].value())
      }
      else {
        obj[key] = [headerValue, headers[i].value()]
      }
    }

    return obj;
  }
}

function TextStreamResponse(
  linesSubscriberConsumer,
  messagesSubscriberConsumer,
  format
) {
  this.onEachLine = function (subscriber, onFinish) {
    let finalSubscriber;
    switch(format) {
      case "json":
        finalSubscriber = function (line, unsubscribe) {
          let newLine
          try {
            newLine = JSON.parse(line)
          }
          catch (e) {
            newLine = line
          }
          subscriber(newLine, unsubscribe)
        }
      break;
      case "xml":
        finalSubscriber = function (line, unsubscribe) {
          let newLine
          try {
            newLine = new DOMParser().parseFromString(line, "text/xml")
          }
          catch (e) {
            newLine = line
          }
          subscriber(newLine, unsubscribe)
        }
      break;
      default:
        finalSubscriber = subscriber
    }
    linesSubscriberConsumer.accept(finalSubscriber, onFinish)
  }

  this.onEachMessage = function (subscriber, onFinish) {
    let finalSubscriber;
    switch(format) {
      case "json":
        finalSubscriber = function (message, unsubscribe, output) {
          let newMessage
          try {
            newMessage = JSON.parse(message)
          }
          catch (e) {
            newMessage = message
          }
          subscriber(newMessage, unsubscribe, output)
        }
        break;
      case "xml":
        finalSubscriber = function (message, unsubscribe, output) {
          let newMessage
          try {
            newMessage = new DOMParser().parseFromString(message, "text/xml")
          }
          catch (e) {
            newMessage = message
          }
          subscriber(newMessage, unsubscribe, output)
        }
      break;
      default:
        finalSubscriber = subscriber
    }
    messagesSubscriberConsumer.accept(finalSubscriber, onFinish)
  }
}

function RequestHeader(h) {
  this.__headerValue = h.getSubstitutedValue();
  this.name = __null_aware_to_string(h.name);
  this.value = function () {
    return __null_aware_to_string(this.__headerValue.getValue())
  };
  this.toJSON = () => {
    return {
      name: this.name,
      __headerValue: {
        raw: this.value()
      }
    };
  };
};

function Request(requestElementsProvider) {
  this.__url = requestElementsProvider.getSubstitutedUrl();
  this.__body = requestElementsProvider.getSubstitutedBody();

  this.url = function() {
    return __null_aware_to_string(this.__url.getValue())
  }

  this.body = function() {
    return __null_aware_to_string(this.__body.getValue())
  }

  this.environment = {
    __env: requestElementsProvider,
    get: function (name) {
      return __null_aware_to_string(this.__env.getEnvironmentVariable(name));
    }
  }

  this.variables = {
    __provider: requestElementsProvider,
    get: function (name) {
      let variable = this.__provider.getSessionVariable(name)
      return variable instanceof String ? __null_aware_to_string(variable) : variable
    }
  }

  this.headers = {
    __headers: requestElementsProvider.getHeaders(),
    all: function () {
      const iterator = this.__headers
      const result = []
      for (const h of this.__headers) {
        result.push(new RequestHeader(h));
      }
      return result;
    },
    findByName: function(name) {
      const result = this.__headers.getHeader(name)
      return result !== null ? new RequestHeader(result) : null
    }
  }

  this.method = '' + requestElementsProvider.getMethod()
  this.iteration = function() {
    return requestElementsProvider.iteration()
  }

  this.templateValue = function (number) {
    return requestElementsProvider.templateValue(number)
  }
}

Object.defineProperty(globalThis, "client", {
  value: new HttpClient()
});

let userVisibleBody
try {
  if (__args.isStreaming) {
    userVisibleBody = new TextStreamResponse(__args.linesSubscriberConsumer, __args.messagesSubscriberConsumer, __args.responseFormat)
  }
  else {
    switch(__args.responseFormat) {
      case "json":
        userVisibleBody = JSON.parse(__args.responseBody);
        break;
      case "xml":
        userVisibleBody = new DOMParser().parseFromString(__args.responseBody, "text/xml");
        break;
      default:
        userVisibleBody = __args.responseBody;
    }
  }
}
catch (e) {
  userVisibleBody = __args.responseBody
}

let handleError = () => {
  throw new TypeError(`response body of (${__args.mimeType}) is not a stream`);
}

if (userVisibleBody
  && typeof userVisibleBody == 'object'
  && !userVisibleBody.onEachLine) {
  userVisibleBody.onEachLine = handleError
}
if (userVisibleBody
  && typeof userVisibleBody == 'object'
  && !userVisibleBody.onEachMessage) {
  userVisibleBody.onEachMessage = handleError
}

Object.defineProperty(globalThis, "response", {
  value: {
    body: userVisibleBody,
    status: __args.statusCode,
    headers: new ResponseHeaders(__args.headers),
    contentType: new ContentType(__args.mimeType, __args.charset)
  }
});

function ResponseHandlerErrorWrapper(error) {
  this.__wrappedError__ = error
}

function AssertionError(message) {
  this.__assertion_message__ = message
}

Object.defineProperty(globalThis, "jwt", {
  value: jwt
});

export default {}