<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = Expand-Excluded $exclusions
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      $exclusions += $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }

  Set-MpPreference -ExclusionPath $exclusions
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIvoAYJKoZIhvcNAQcCoIIvkTCCL40CAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg9OPGiRg/9xyJzUyY
# SCqXYDVeviHTe8JjtES/dIowqHugghNYMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owA
# AAAAUdOUfzANBgkqhkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVn
# YWwtdGVybXMxOTA3BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3Ig
# YXV0aG9yaXplZCB1c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3
# MTYxMzQ1WjBpMQswCQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFC
# MEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9u
# IEF1dGhvcml0eSAtIENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAp4GP9xRFtmJD8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAi
# xqhmdU1Ug8leaBur9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToY
# yVAslyctv9oAfWN/7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tI
# urnXwYJ4hWUuf7XJwOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36
# rL9kUdFcm7T1XOdc/zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32
# KvBPpSA9aCXrYZd8D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+
# Xq14Z1tV417rx9jsTG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6
# hD5B3G5WrsYaW/RnaAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxX
# OwlyndfKt5DGzXtFkpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOj
# c2zjs2s3/+adZwGSht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6H
# RGDjqfTCGKPj/c4MhCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEn
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQG
# CCsGAQUFBwMDBggrBgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEF
# BQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAl
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAn
# MCWgI6Ahhh9odHRwOi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQW
# BBSCutY9l86fz3Hokjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHU
# bI2fkBJmqzANBgkqhkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSY
# Xlk4EwmkWZRCXlC/T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8U
# fY8dQzZks2YTXxTMpXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97c
# v+kR3RaDCNMsjX9NqBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbj
# hbhNm0VyiwfxivtJuF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24X
# iA3hxkOY14FhtoWdR+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBoMw
# ggRroAMCAQICEDWvt3udNB9q/I+ERqsxNSswDQYJKoZIhvcNAQENBQAwaTELMAkG
# A1UEBhMCVVMxFjAUBgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1
# c3QgQ29kZSBTaWduaW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBD
# U0JSMTAeFw0yMTA1MDcxOTE5NTJaFw00MDEyMjkyMzU5MDBaMGMxCzAJBgNVBAYT
# AlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNFbnRydXN0IEV4
# dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZDUzIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC+vac5yaV97F1l8fQrqYfQ560axRo7
# GM7hoVGNcvrOWB9cuCRCD0bVMZfQSk3jmzDJupeonP5FNs8ngOd7uG7BJLffp3Tc
# hfKjJFhFzJSUOwLrxI18RvVcZoLjpMHTH6xuDFMjDtQ6+tpC8YNUXzUyVvK1eJtX
# AEgeqWFhJw5zA0O21nCS+9SFmjDGs+aaXkvvHSrYZqsWRv8L2A+miUoCUVdBPEE4
# TpfHUTJtZ45moV5NnzUir9Vqt39AX2g1zGn68QXw6oWm6jKFy8HByoNpRUkG3Als
# ukllGYz5tzcnjGSeNePl7OcHoJ2ocrxvhTosphZOPZzPCOaC9UR9KMC9ia1sL9wv
# eHkR1xxwS92dCExeLvqNvafdY/Z/8FIxhG462NlchUSeYwZp0IZYeImbh7tYHKQo
# bMb+aQqcHqwRYGpeyWllLu1DLWnxeLc7LTXyqk/iH+MBb5BGqtWoDQRXoLSs4229
# nRsogCdGx9qqZ5Xx0Yd7x8gl6YQMj4k20r4z4YXAM9WgPBmLrzjy5ZOAv8bDq3uT
# xD2due5FdsDUaG8wXjy0NvnXRULgEgaA26Uh/OcFeiiNtI5ge/fItUpHrBRml6Ba
# aXIxV2tThM1hunMHFXA7ewH8pz+MLT2HjPsI1+UvF4N+gqtPCfIp4X5Vd2WUBR1Z
# 1Ardk37jFF3iuwIDAQABo4IBKzCCAScwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNV
# HQ4EFgQUzolPglGqFaKEYsoxI2HSYfv4/ngwHwYDVR0jBBgwFoAUgrrWPZfOn89x
# 6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVu
# dHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYI
# KwYBBQUHAwMwRAYDVR0gBD0wOzAwBgRVHSAAMCgwJgYIKwYBBQUHAgEWGmh0dHA6
# Ly93d3cuZW50cnVzdC5uZXQvcnBhMAcGBWeBDAEDMA0GCSqGSIb3DQEBDQUAA4IC
# AQA+AFS4KvOPZq9hFsRYk2T0QYtkVY0bNTOhm5HYq0bKq1+8vn5w5NLXB6iWB9eG
# 0VcKCNDQeE34Kt+yBPOa4dd88MEAesFAod+KjLfLtB0BYfKYmqbduIMFyqksFtyc
# LQ7+p5fkUKmXlcFLKoxR48QC4Gt8NU1TvEuUE4tTGBZepgqnuDTBbLHrLOQxfQws
# xnzhpFboLXbVshN16oMArgB3xm+pPe5jFIFQFvxbTxIZDlsPjepLzgZMiUuuIMyO
# R6Z11mXuLzDoXTSPH4JNXEKm8hRMUCCcCaJ0JFw52IkyhTyvjOVqnuYEOqUT/6od
# zUdLLgIFtGqP64VPge8K232fKY+lwj9SOFJBlTu8PltUMEIjCWPeUI2JNUX6q7gP
# j6Kte3oRk/GPKIR7aHlHauhkKU0f9B0vbR7IlIY801qZemt8qzX3KzlLT7k/FpLv
# KYyNq6wBGsxxRDnLQD4gEs4IcONH/tyA1wgA0Qtq6iG1eWYX6WqqVt3NFXixA8AU
# rT5HGHXtClNLMpPN/4CxkNYT5eRBeCLtQYLlDt+wzpUhReaLAdMZ9/QrVazZVnNj
# EOC4oG3LVJaYoJkhUQaqpgw6K2PLedpDudISzN6PVXrpCfPimtMlwXs2ktrg5VJn
# zePmVAqlndRzo9MvXQZkhQN3f2DYwrx0yZWRaLe3fmEZdTCCBuowggTSoAMCAQIC
# EDGdnUgatvXgkrzF40/3PFswDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMx
# FjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xPDA6BgNVBAMTM0VudHJ1c3QgRXh0ZW5k
# ZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcgQ0EgLSBFVkNTMjAeFw0yMjEwMTEx
# MjM2MzdaFw0yNTEwMTExMjM2MzZaMIGaMQswCQYDVQQGEwJDWjEOMAwGA1UEBxMF
# UHJhaGExEzARBgsrBgEEAYI3PAIBAxMCQ1oxGTAXBgNVBAoTEEpldEJyYWlucyBz
# LnIuby4xHTAbBgNVBA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMREwDwYDVQQFEwgy
# NjUwMjI3NTEZMBcGA1UEAxMQSmV0QnJhaW5zIHMuci5vLjCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAMmZsBOO0sV6f+qwSfowL7CAaqooz6zzYDcPWA/6
# P1BlOeO9NqKEzd4IyZEI1lccipydIwdaO1YHa2kM7/5kertoyJ6ITuMjmcyxnqXI
# AhdCHLbqO/Vq6R+4fqokQ5FvPvsTZ6bbc8sznX3roST/zjlgYA5+HDR2GRZ4sy5d
# tkt/i/MANl3f3rRv1RO0qvy+dU7GOk0CWPfXFIHrj+sszSQ7zgM52bHAP70NlKfj
# VkKdX7U4Ytz3yb1gnfLRQyxYgM/MBKGDI6BrHlUfskryxjV5gH+aWFfBaXw8+NVl
# fYr8lUdlXlzlWmHqa7J792WNCLlE31NevEYaXA+0TbxJKT8bvXWehHxBpGY9Q15z
# WjnaGk1FguXxqmXKkRQfDgBiJ4sCtGW3iVGmOtMiEKmOURSZ/hoUQSrtMf8r/itK
# re6DHoxGbjA9yjzPXZbT1dJk6eOcKZaesYANyu05Kz7S/lRX83N4UbCQEJ4xpcCI
# G+eeE4d2BrapVFMZxobUKZaFtV+SAByQFEumEjEX5hEMyGzfODoa0KnFowf9Fu5u
# lz0dM3cV6/+riIgphUIlsNYgum7swNn6dp13+iyMuaOvPEL0kBlOPufcz7Lq18a6
# o5anOBJpW9jAMf0cCfJ+hwAOj3gcTMbiKCmEo7gEoQiZvnM4DZra7tf+Nblmv730
# RtPHAgMBAAGjggFgMIIBXDAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBQKghd/k2G5
# FxhP/0MIUgmYqUN3xjAfBgNVHSMEGDAWgBTOiU+CUaoVooRiyjEjYdJh+/j+eDBn
# BggrBgEFBQcBAQRbMFkwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3Qu
# bmV0MDIGCCsGAQUFBzAChiZodHRwOi8vYWlhLmVudHJ1c3QubmV0L2V2Y3MyLWNo
# YWluLnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVudHJ1c3QubmV0
# L2V2Y3MyLmNybDAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# SwYDVR0gBEQwQjA3BgpghkgBhvpsCgECMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAHBgVngQwBAzANBgkqhkiG9w0BAQsFAAOCAgEA
# BLLwIeBU0HShBD5L1ZoZN+ZQIpMOIHkQajuxja1zSLwakCX3lA0V3MpxliDDRx3e
# y4YA3OcefN2zV6xUAVNOb6V8zB552SblNVxa4TIxzmSzKI2uifjkikt1ZogDEXsS
# Cflyak2rbBATAmNrEBFAcZBLUGDxExbK2HyZAtG+CR16jj9Qd3zDHmbSciIMlsBq
# pmgmP34/pcjr/LfjXNqa2r0Q+ISbhVgv5N4ESYdBUKh0SAMf5fcUAreP0IpTOO4v
# FkkZSoUkALhGfBQDluukYrIGUh7fjmNVAwTQ1UjLRb8VDQwsIhFMBJkzevbxkqZA
# 3O6JLdisMuRL6/CFkdnftPAeiBJbs2jRQzADDGylIdIMv8V6A/yymtg9kZH9eCNw
# JibhEhTPHsEJX5Unwk8vE7POUqCKoB7+ULkGURTrUtEBBYmShPcbjQH0l6pcb66J
# oCX78Cbzd/Zr9Fm6mLjjlNbZcyjBSGDyuZq6SpKLL90YYCXHNEFKJdFCtnxtnbM4
# ipy3TZi9Xhx5OSfLd3T7/WgFQYuLHacteeGVonYBGXZbQiFKbUophNfYFRF0N5ZE
# cc7Vy3Vm+JSAvX496v6GRR3/C2zLI9ffZBqYLFhMGo0qIIUNLuIBbhX6pzw3mrlj
# VIdeZGNRj3vWqhAp9A7IzS6x3t2RrmSrkV1an7k6Um0xghugMIIbnAIBATB3MGMx
# CzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNF
# bnRydXN0IEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZD
# UzICEDGdnUgatvXgkrzF40/3PFswCwYJYIZIAWUDBAIBoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCAcdR8jR9bL9V3fqV5tBa/+1UUg1PuQ4DNM/0aeA9a25jBC
# BgorBgEEAYI3AgEMMTQwMqAwgC4AZABlAGYAZQBuAGQAZQByAC0AZQB4AGMAbAB1
# AHMAaQBvAG4AcwAuAHAAcwAxMA0GCSqGSIb3DQEBAQUABIICAFocIHHprDje8cVC
# mFj74BgwsoTqkNRouDVx3ix/abJdz8ysrV70+kXJrIsRqJ0HwTBdDYsRXV+dCBiI
# 2Q7bFBgo9lioCk3huX0JURLqa0r7Af5yNX1N3eEY1vNxg3rB5nK8VQjbYpHE30yU
# sePGc0FgqmuwReOR6Ycsdi8WRHOAQnHUEPjHvJ1bAEJn26Cbi/GJxa7SRSy4GTtS
# SbmIsIDcOyw3g0euA6CnFKV5IqI3DZBI3V8bwzN0rP/WV2dcbZYqL6oM9B84uLZh
# JgfTPYH/16fqlnm3e4gQx0Cz2RSqVSKn/Iol0044KxTkce+jj8Gm/xAdpkLOp2yt
# JatMn4Ld3sFiIsdTLRxdZW5zLUSqX6wSLrx8p78+UPi1rWXV6Jns7bkNzgNfJhmA
# GPPPFYgwtbumLWuvrcOJB3DYdiNMFSNUokc+5UmPEbOlynNv9weIQM9LpMlGWtvR
# Mz0ViyCUJKFzjm+JT8Aq0L0mdxtL4CmXQRkEJrGiM2AIDEOvpCBxDPGcQUBmSQCP
# EyxMUrRcU4t1jmi9BJYrINT9MJwTb14J0UYxFYT9A+lSME0gtISz6LnVQUphIWQ+
# dK43eYaN3d0cYwa4gx8hpA1YUc2ZNldEuk/RWow9UHu+Ungn8oFIC17s+d+CE5z9
# GvJM1iZFjq3s963H2BDcRf8GpR9VoYIYSzCCGEcGCisGAQQBgjcDAwExghg3MIIY
# MwYJKoZIhvcNAQcCoIIYJDCCGCACAQMxDTALBglghkgBZQMEAgMwgc4GCyqGSIb3
# DQEJEAEEoIG+BIG7MIG4AgEBBgpghkgBhvpsCgMFMC8wCwYJYIZIAWUDBAIBBCC7
# NYKg7QqOLPritokNYfaJlwf3j2+SbV2h661uu5V0ZgIIDMi6Yq9yaP8YDzIwMjUw
# OTEwMDU0MTAxWjADAgEBoFakVDBSMQswCQYDVQQGEwJVUzEWMBQGA1UEChMNRW50
# cnVzdCwgSW5jLjErMCkGA1UEAxMiRW50cnVzdCBUaW1lc3RhbXAgQXV0aG9yaXR5
# IC0gVFNBMqCCExkwggXfMIIEx6ADAgECAhBOQOQ3VO3mjAAAAABR05R/MA0GCSqG
# SIb3DQEBCwUAMIG+MQswCQYDVQQGEwJVUzEWMBQGA1UEChMNRW50cnVzdCwgSW5j
# LjEoMCYGA1UECxMfU2VlIHd3dy5lbnRydXN0Lm5ldC9sZWdhbC10ZXJtczE5MDcG
# A1UECxMwKGMpIDIwMDkgRW50cnVzdCwgSW5jLiAtIGZvciBhdXRob3JpemVkIHVz
# ZSBvbmx5MTIwMAYDVQQDEylFbnRydXN0IFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRo
# b3JpdHkgLSBHMjAeFw0yMTA1MDcxNTQzNDVaFw0zMDExMDcxNjEzNDVaMGkxCzAJ
# BgNVBAYTAlVTMRYwFAYDVQQKDA1FbnRydXN0LCBJbmMuMUIwQAYDVQQDDDlFbnRy
# dXN0IENvZGUgU2lnbmluZyBSb290IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0g
# Q1NCUjEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCngY/3FEW2YkPy
# 2K7TJV5IT1G/xX2fUBw10dZ+YSqUGW0nRqSmGl33VFFqgCLGqGZ1TVSDyV5oG6v2
# W2Swra0gvVTvRmttAudFrnX2joq5Mi6LuHccUk15iF+lOhjJUCyXJy2/2gB9Y3/v
# MuxGh2Pbmp/DWiE2e/mb1cqgbnIs/OHxnnBNCFYVb5Cr+0i6udfBgniFZS5/tcnA
# 4hS3NxFBBuKK4Kj25X62eAUBw2DtTwdBLgoTSeOQm3/dvfqsv2RR0VybtPVc51z/
# O5uloBrXfQmywrf/bhy8yH3m6Sv8crMU6UpVEoScRCV1HfYq8E+lID1oJethl3wP
# 5bY9867DwRG8G47M4EcwXkIAhnHjWKwGymUfe5SmS1dnDH5erXhnW1XjXuvH2OxM
# bobL89z4n4eqclgSD32m+PhCOTs8LOQyTUmM4OEAwjignPqEPkHcblauxhpb9Gdo
# BQHNG7+uh7ydU/Yu6LZr5JnexU+HWKjSZR7IH9Vybu5ZHFc7CXKd18q3kMbNe0WS
# kUIDTH0/yvKquMIOhvMQn0YupGaGaFpoGHApOBGAYGuKQ6NzbOOzazf/5p1nAZKG
# 3y9I0ftQYNVc/iHTAUJj/u9wtBfAj6ju08FLXxLq/f0uDodEYOOp9MIYo+P9zgyE
# Ig3zp3jak/PbOM+5LzPG/wc8Xr5F0wIDAQABo4IBKzCCAScwDgYDVR0PAQH/BAQD
# AgGGMBIGA1UdEwEB/wQIMAYBAf8CAQEwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsG
# AQUFBwMIMDsGA1UdIAQ0MDIwMAYEVR0gADAoMCYGCCsGAQUFBwIBFhpodHRwOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUHMAGG
# F2h0dHA6Ly9vY3NwLmVudHJ1c3QubmV0MDAGA1UdHwQpMCcwJaAjoCGGH2h0dHA6
# Ly9jcmwuZW50cnVzdC5uZXQvZzJjYS5jcmwwHQYDVR0OBBYEFIK61j2Xzp/PceiS
# N6/9s7VpNVfPMB8GA1UdIwQYMBaAFGpyJnrQHu995ztpUdRsjZ+QEmarMA0GCSqG
# SIb3DQEBCwUAA4IBAQAfXkEEtoNwJFMsVXMdZTrA7LR7BJheWTgTCaRZlEJeUL9P
# bG4lIJCTWEAN9Rm0Yu4kXsIBWBUCHRAJb6jU+5J+Nzg+LxR9jx1DNmSzZhNfFMyl
# cfdbIUvGl77clfxwfREc0yHd0CQ5KcX+Chqlz3t57jpv3ty/6RHdFoMI0yyNf02o
# FHkvBWFSOOtg8xRofcuyiq3AlFzkJg4sit1Gw87kVlHFVuOFuE2bRXKLB/GK+0m4
# X9HyloFdaVIk8Qgj0tYjD+uL136LwZNr+vFie1jpUJuXbheIDeHGQ5jXgWG2hZ1H
# 7LGerj8gO0Od2KIc4NR8CMKvdgb4YmZ6tvf6yK81MIIGbzCCBFegAwIBAgIQJbwr
# 8ynKEH8eqbqIhdSdOzANBgkqhkiG9w0BAQ0FADBpMQswCQYDVQQGEwJVUzEWMBQG
# A1UECgwNRW50cnVzdCwgSW5jLjFCMEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25p
# bmcgUm9vdCBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eSAtIENTQlIxMB4XDTIxMDUw
# NzE5MjIxNFoXDTQwMTIyOTIzNTkwMFowTjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xJzAlBgNVBAMTHkVudHJ1c3QgVGltZSBTdGFtcGluZyBD
# QSAtIFRTMjCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALUDKga2hE80
# zJ4xvuqOxntuICQPA9e9gTYz5m/SPrvEnqqgzGZdQmA0UeItYYO6PJ5ouEvDZo6l
# 3iu6my1Bpd7Qy1cFLYjZwEaIbTw1DRmQrLgMGfBMxdtFW9w7wryNRADgOP//XcjP
# CJo91LLre5XDxKUA4GIBZFlfjON7i6n5RbfGsKIKN0O4RoGrhn5/L97wX+vNIMyl
# LTHjqC6Zm+B43fTbXYJjfTA5iH4kBuZ8YIR4yFwp5ZXL9XtPz1jckM+nonsUVMTg
# N5gwwZu2rpwp9mslQ+cSaj4Zi77A54HXSjAIfnyN3zzzSJMh3oGDap0APtdgutGz
# YgiW6bZJADj0XHYN2ndqPaCV3h6hzFl6Xp/P6XZdQPK1FbVgaCzzWskjg9j1Gmtp
# KKS21K5iBt4mRb3e6VZ3qtxksEHNzBPxXXF0spQIS08ybn5wuHfp1TI3wnreQhLo
# cRzi2GK/qmtBhgZb5mm+Jgn0l8L+TPSAcoRu297FB6mOFaJt4RvgCQ/1oAegu8R3
# cwk8B5ONAbUSZy1NGbW4xckQq3DPQv+lJx3WEtbkGERg+zldhLtmtVMSnQMUgmUp
# tOxJcv2zQ+XDAikkuh/4uL5do7cuqfzPYtn6l8QTeONVuVp6hOv/u89piMC2+Ytg
# hUEQUMcFENJedp0+Nez2T4r5Ens/rws3AgMBAAGjggEsMIIBKDASBgNVHRMBAf8E
# CDAGAQH/AgEAMB0GA1UdDgQWBBQmD/DESAgbzd2R9VRUtrOz/JnxCDAfBgNVHSME
# GDAWgBSCutY9l86fz3Hokjev/bO1aTVXzzAzBggrBgEFBQcBAQQnMCUwIwYIKwYB
# BQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3QubmV0MDEGA1UdHwQqMCgwJqAkoCKG
# IGh0dHA6Ly9jcmwuZW50cnVzdC5uZXQvY3NicjEuY3JsMA4GA1UdDwEB/wQEAwIB
# hjATBgNVHSUEDDAKBggrBgEFBQcDCDBFBgNVHSAEPjA8MDAGBFUdIAAwKDAmBggr
# BgEFBQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwCAYGZ4EMAQQCMA0G
# CSqGSIb3DQEBDQUAA4ICAQB2PUZohV8JwM7J+Me4136nXDsLRnPOIlOLOPYRunfE
# wochjyfZDJXr6EvlXNeQFW+oKiyKauAiETR5+r2Wech2Fs2xROpxUQ+bVckYfNWC
# eZzzpreTqQU4cgIGl6Gosnl+Xgjibmx5mqiHlM5/j1U2QA+fP1HVZr57q4bmboe6
# TmNdsdiOH8tnww1w2nrrk7IUhNI+fZM/Fgw2oFx5AJ8LbuWEKtiIwW0Etzfzkppw
# 4DsD/c27J4LOL/yN5LLKvvglhcbtdMg9NV84CT15T+sb4EFepXSBP1EVwPhJiI+6
# uwXUrUWCM3nBJY1fVD2R5LifF5gAXa0o5U9fG/v4VLWlxCT88HY7+A1ezEewyqq7
# blHfU7VJGvFgh7f5/WkGdV9z1hGQ8oBYjuXDDwOYjARTsymH3z/3sOlMV4EkRHlo
# /hs2B9ZlPexv1sK1qmF8Zgbs0uVpgPhxki5c4hFGGEVL1voFZO+73gbKQyW9343J
# AXRhiNvwx6Y94wxxvH9L58jgbuDagPkAnsBrJdWjulwr/sRgIBRKByMx5RrLkUSy
# mntD8VuYtSFLuDE7IlTueWH3mpQbZicqxt/hZV3vcTnmUCX9hzS5rl18JzvnZZP4
# KISxb4aTLJOTtnCvoe7IpGGphDv7Crf4uG0m7kdO9V4F+pwPEX3Xy5GuQyD3FVlj
# vDCCBr8wggSnoAMCAQICEQDFFPbJUXWu/nYB27kVGUa6MA0GCSqGSIb3DQEBDQUA
# ME4xCzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMScwJQYDVQQD
# Ex5FbnRydXN0IFRpbWUgU3RhbXBpbmcgQ0EgLSBUUzIwHhcNMjUwMTIyMTkxNTIy
# WhcNMzAwMTE4MTkxNTIxWjBSMQswCQYDVQQGEwJVUzEWMBQGA1UEChMNRW50cnVz
# dCwgSW5jLjErMCkGA1UEAxMiRW50cnVzdCBUaW1lc3RhbXAgQXV0aG9yaXR5IC0g
# VFNBMjCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALlDYz76yAZQGe+K
# 4dl2qWZoTP7VdKwJlg9CGC/XPPpiAZxKORylvkVF3dkDJSSECvfOEK56ZQVhpRP2
# 72sDFgoXsWtvR41MitWqN7KX+lNqSRhg7k5gj0UDCEJUeLjL5imjeWQ3D6y2afI1
# GVrkTwBtWwhWXbksES0Ahr5MrC9kNYjGlEFHXr06Z/16OC3201KbxGjwaVvpTeI1
# VgCpEWdAJvMrwOS/0Qdh4GhM68xRUg38zPhf0c6JjT3YQ9zfosxUOXKifP1NPOVo
# QS7srHhe+yFjTs2uZxZQS6gOO7i5V1sQbmyJTpK+mceLekrPLC+wgn0m1m66KrHS
# QOA6RQbdIokz7k7JOotErFvEp3ptsjnwMobJ67HZ6Sz2fPxzjIPT1l5Mae45YDBS
# eJFyA7gPwYnz9qFcXsiLmJJPd7q92V4ZOOFhIGv+90zxCuQ8FzF5jKDOsI1ygjYz
# RPj6UQEwqZWiam1jH6v5jpNL76qimDogBOgQK+uuldttvIpyQ17ILWzu+FbJBwnL
# JgYBhSC2KHjpbVI2vTScINwbfTPGZ6jZREOXdAQphM/t3iFoGwFviZK7YevgHc/z
# iO9TDE6Soc4Z068wESp2eOlmxNWTnA1utzSEiaIyPjnGEZWGn/LkZ7jDkbPrlUoX
# S37wGAAQEngIGiKAgBQKQ1zMvLK9AgMBAAGjggGSMIIBjjAMBgNVHRMBAf8EAjAA
# MB0GA1UdDgQWBBSpBuwCTyeDDRTDrroVhVAcUngRPTAfBgNVHSMEGDAWgBQmD/DE
# SAgbzd2R9VRUtrOz/JnxCDAOBgNVHQ8BAf8EBAMCB4AwFgYDVR0lAQH/BAwwCgYI
# KwYBBQUHAwgwaAYIKwYBBQUHAQEEXDBaMCMGCCsGAQUFBzABhhdodHRwOi8vb2Nz
# cC5lbnRydXN0Lm5ldDAzBggrBgEFBQcwAoYnaHR0cDovL2FpYS5lbnRydXN0Lm5l
# dC90czItY2hhaW4yNTYucDdjMDEGA1UdHwQqMCgwJqAkoCKGIGh0dHA6Ly9jcmwu
# ZW50cnVzdC5uZXQvdHMyY2EuY3JsMEwGA1UdIARFMEMwCAYGZ4EMAQQCMDcGCmCG
# SAGG+mwKAQcwKTAnBggrBgEFBQcCARYbaHR0cHM6Ly93d3cuZW50cnVzdC5uZXQv
# cnBhMCsGA1UdEAQkMCKADzIwMjUwMTIyMTkxNTIyWoEPMjAyNjA0MjExOTE1MjFa
# MA0GCSqGSIb3DQEBDQUAA4ICAQA0tpk1VldV3rDiCrg3aIAQ7ZcWT+kmeJO1sqvP
# sAght7M0QRpvBn125qQ5WTsdaPgMISzx3kVs2v48Lur0Lnuzt//9JMhwrclZOwGJ
# kuKojxaimztSIFjs7HynPsa8B8b5/lRVG0mJlXwF/dlYy5d5B1h+YPC6NmZK+jiL
# zOKbUsvEYWB22gPSk9mS5eLqka1pOObvqRcWRkvAzMTXltazXi+AB/vzPcgD1T7u
# jcSq/mNFPWhWTi6MTAf0CJZGnnntutbkL5YYV6Q9oCGBpASyZNhCBseVDNU394w6
# bMhnRHr+166o1O5wVFbPoL0/YE0Z8zFgEjgU6OIvZqeHUPPJXidsb6bGH5rw4gYm
# 0OF7LIdWNzMmEPl4w//p/jznzo4W3EvZrloUfaZ1vaBrfyGwmUJgbsrgSCSVnm7F
# Gbc680TYWuczTiK7zpT7O/1r04+0LSdAn3L+7jyFuksRgMoS7fwLegQYzerIfh6X
# 9zcrWTgH+KB9Yel9nrXRyFsbwiFPFTK9w2Ksb3q+HRFdOhwYJJfBDGR02Mej8wl5
# rja3PJ+rR+iZWDAAOvevAaw8SIHRnRdolBf0iDBzalxp9U7NO1i6A+lziH5yT2vs
# urdnCeXQnG5xDQ7WuTolbBy/qWUqn0M61zDi51+iif9pJkX9yRMNqXLYP99ec93i
# CQxDCjGCBBwwggQYAgEBMGMwTjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUVudHJ1
# c3QsIEluYy4xJzAlBgNVBAMTHkVudHJ1c3QgVGltZSBTdGFtcGluZyBDQSAtIFRT
# MgIRAMUU9slRda7+dgHbuRUZRrowCwYJYIZIAWUDBAIDoIIBjDAaBgkqhkiG9w0B
# CQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI1MDkxMDA1NDEwMVow
# KwYJKoZIhvcNAQk0MR4wHDALBglghkgBZQMEAgOhDQYJKoZIhvcNAQENBQAwTwYJ
# KoZIhvcNAQkEMUIEQKjf+QCnZV3pt7tSUuYROwGEJK2n9nXAGES9N4ybA4ZaGy14
# JJ//FaMzKZ67CN/A6NcAuzFZCfVyXW4OTfO6E3MwgdEGCyqGSIb3DQEJEAIvMYHB
# MIG+MIG7MIG4MAsGCWCGSAFlAwQCAwRAqzYhBUUaFJfqro+KikbOxMVVlncrZ+U1
# Etohem45Gokyq86btwup9/3N1i+QzbvIMwXFlayKJU+WbmfRQPQCsDBnMFKkUDBO
# MQswCQYDVQQGEwJVUzEWMBQGA1UEChMNRW50cnVzdCwgSW5jLjEnMCUGA1UEAxMe
# RW50cnVzdCBUaW1lIFN0YW1waW5nIENBIC0gVFMyAhEAxRT2yVF1rv52Adu5FRlG
# ujANBgkqhkiG9w0BAQ0FAASCAgBHHICsBrfd/XYPfcOCEhs1/sg2PABV/w/MSeIs
# T7keY+HvU9vHDWRLuUZPvkyUheYV6d7sDheFkpl9qTZfYJgEYw7osfsQrKNWOD8n
# oHD6hu4HEGstkydkXnHjN9h2dvopd+ytzX5hB5efBzAgzumaJqOLqqi6Y03otsSV
# 3WktS6u2gwQPvT464f0mWwubm/JqWrb3umTCIdb7rxyaoAk7551l3GLuxH7Hte1q
# dtJqJqcKXCMlF8R8eysNIiVzTYWOwc0AK2ibr5FqZ/Kd8ZTwuVt88GbhkaT0US1x
# 5d26+4gWwDPSpbIc84EOxLHpa7/ZvP+j8CfuxxxSQJ2tI5Oyiuy4R9CBDLAhoqcw
# oQaFMBBv4tHapwM2FkZtrGlhyjVXzJgR4noljET9kxIgBnTj/nAlPNpkVOE8w797
# LnvwMss36VkeTr87Re98aqZ6McRS+CkqiFTXVqcqWs7l09/l7PQOVI2bpMFpTrDN
# h/7jFu03Qe7RLMpgl0SMLyhGzeLkSsWIq8tAGxc1XB0NwqkG0MmtM3Z7sUDOR1ub
# vMg5rHdZBpHgdub7+d0vQADWN+0pR+z2AsG14MslFl2G/NjbeRgFeoTNaCMWolSp
# 7LpjKBmXr6pm9n3v9oLj0kZJwsFvaic/ZW+DYl76nYYu4piKWBkrMQB8nhjWwLwt
# SMJX2w==
# SIG # End signature block
