<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.

   See https://learn.microsoft.com/en-us/defender-endpoint/configure-extension-file-exclusions-microsoft-defender-antivirus
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = @(Expand-Excluded $exclusions)
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      Add-MpPreference -ExclusionPath $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIpiwYJKoZIhvcNAQcCoIIpfDCCKXgCAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQgeTbSdh+QTc1PlIl/
# aTZyoemLeBftCRJOcd5Eh1jG7Pyggg4JMIIGtzCCBJ+gAwIBAgIQB2Asv251eUAD
# glbwrYde1jANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMM
# RGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQD
# ExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjMwMjE0MDAwMDAwWhcNMzMw
# MjEzMjM1OTU5WjBwMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIElu
# Yy4xSDBGBgNVBAMTP0RpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIEV1
# cm9wZSBSU0E0MDk2IFNIQTM4NCAyMDIzIENBMTCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAK/M63p6cQzFK8iIb4EA1jkqmULJtwPvV8p5XVaj1AnSeUBg
# Mc3z+9lc+6KHxZnts0e1iJmE5aZRlPRl5/Ku7NlvBVtZbvmxdfm0ku6RCtxOB+AG
# mexDPu8CJ9Q9O8dqISPY4kavx1xXR9JoxTq0SU51fiEu+63lt0F2VRb/YHL1B3/5
# Fx/0qIYcjudZeQEByQrRE4AOpZO/aZJei6jT3PBK++nqWnCWYP0sfGSW9gAwuO7M
# 6FdHLU5yLWl89p6OWmM3jSGgjNz9OYcYD+giEUofoYvHD6Utu+SdbWAW7L33dhQ6
# 6k74MU2bDnD0AXlPFUy+qOf6PhLYGnmKym12dwjRkx2g1NTC9skPb36jtXG+jjaV
# pVsS0jlL1UiB0iRSBcttj56RBHYOh2TkiTjXCc2PEW16ewhIaEGSkG6XMNiqb7Gy
# 8mMZAJo5YXmrw3Ed3mGo/to2DAhXf2bSaKs+1+3nY7FZ0ra1kerY4pJwp83n6ax8
# AtNm55BJYxUZ482yvP4dIj6HjCabb6S+PXStO8OiYusP56K70ixaHL5KDVt8HfZI
# z3sl89Hc2TW8Ap4th5IcMQLJkHnqHOCZOaBBqmGLXogjD9qBZqrqEZ7FyIhfJaKQ
# VCFoG5eIjM/bjslNB1nGDgmceYnC7/6PCMuRnfpuEuxCNCkO7p57NchlppHjAgMB
# AAGjggFZMIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBT69DbjOUhY
# 4boDpXUcRxhtX9ZhFjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAO
# BgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYIKwYBBQUHAQEE
# azBpMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYB
# BQUHMAKGNWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
# ZWRSb290RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2lj
# ZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwGA1UdIAQVMBMwBwYF
# Z4EMAQMwCAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQCimOO0oKisz8g5qVWH
# T7Djbx6HCAUqvZOTFJc1H81yOOlzt4a6LtPwclHMCrNih7+TuzzWjQVvNCSVT8Sx
# sDYgP0ISnFx61N2j1ubujz8isItnWTvL+4WsFj7Mb9IDI8XyeOz7LCwExRRSRIry
# GwNyJgxRpTJxy37FrxjlTyp99gJmlJjG4cJggtw3Umo/i6wYRLLczhXuxyDuo+m9
# MmSmb4gNO24v+Qi0qM4vsLCRtvEWZsjO6k94vdAXdOKpOKx7qhr7H+HQaqBieanC
# EGi5KwjcGijrWe62BegSOt/QwZkDoUX64YoATYYpYQkfhMrVNLj+zu7RcMVQZ1Cs
# CquYU10dvslcnYK+xuUexUJ97CR1E7tV2IqgroBccejuiWKwzhKuA3T6FEjGdOSV
# nv4rq7cxlfTA5Zpx0YhVDcwfdvRiMG9LSzadMYuPMa4v1LVL3Cfi/tPtpqkrpo1F
# hARTVkFz0TTP8I5q3GE+8zlQmt1MWDQ4IfdAxxWWbsfk7G6OuMQfyiJK+WOB1Orm
# +hh+MjwcBdzlGazZ454WW+TyTPMc6PV7g0KsGpGhrkhc1rXxoL5BVmXl0ZP6c3Qv
# Co/R2o2gt1+UEiMhQFAc9gqAYnQegKoPFInnCPV6kpoZ1uqTi7x53UWiHEGV4k0Y
# jfnyuQWP3YrxSuA7QdwcGPUTOjCCB0owggUyoAMCAQICEAg+oTiEvf/Ojl2dXK0u
# +94wDQYJKoZIhvcNAQELBQAwcDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUgwRgYDVQQDEz9EaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2ln
# bmluZyBFdXJvcGUgUlNBNDA5NiBTSEEzODQgMjAyMyBDQTEwHhcNMjUwODI2MDAw
# MDAwWhcNMjgwODI1MjM1OTU5WjCBmzETMBEGCysGAQQBgjc8AgEDEwJDWjEdMBsG
# A1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xETAPBgNVBAUTCDI2NTAyMjc1MQsw
# CQYDVQQGEwJDWjEPMA0GA1UEBxMGUHJhZ3VlMRkwFwYDVQQKExBKZXRCcmFpbnMg
# cy5yLm8uMRkwFwYDVQQDExBKZXRCcmFpbnMgcy5yLm8uMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAqZKaFkS+uECz+dpb19DnKoU+ZsNEyg/xJmOkI0h9
# 36AkpzIIA8pjBSBFAIlsZuRb87PSd1FON2EX/veQ4NBzp6jQWq8En8eTSYy0QR10
# Nr6n+FFdNqX3KD1qmQMUV3vGli8af/by+f1/m9k2w2Nfh6qxy6aw1Ez9Hbr5oH+o
# 6PCszPkdSgWh3eOypq8zabpPuZZwmH8CNSPMAxLb78vopabs8Nhr03fmO5ksxzgC
# 6i/y9prBoXk/y05BvuBJhkldWkJHTigVhMcIUaE3tmVWb2u8BSW0lRh9wq1u1fHu
# Vj/AwyBCAx06SE6drLiEMTWpPzd37N8PIPMabce1NpWLA5t958IwWY4EdBQfBa4R
# bBVnocoF1G70kT7CqIhnkaQwqjjePtVTUYl0S+yuTk6bHfjz3EWN+16Vtg4kR11x
# sg2lrphTgMdMevMaFxqr/lgw7Owmcb5tSoQMOVAzOEMTE0juDCrRADjdEB4xxOfg
# +v3DZcOavru1qRpQenVmvUHCDgeG/Ivjib73lLLtIyxyvRPoJOX72mihSH2pA6h/
# D1DsqisvfGSRgX6MkggQPHHwCS6X07TApwsYKnBw/T2ttxV8kJwOi+UdzQABQMHy
# SucLC1rOrbkEtiZF6Dst9S/SDm6xrYNmzjUf7475El3vJj13jhHD5kE1Xx86htcH
# uXcCAwEAAaOCAbIwggGuMB8GA1UdIwQYMBaAFPr0NuM5SFjhugOldRxHGG1f1mEW
# MB0GA1UdDgQWBBQ8pDHZkGU6ehH6QCnluo8K/DvlczA9BgNVHSAENjA0MDIGBWeB
# DAEDMCkwJwYIKwYBBQUHAgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAO
# BgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwYgYDVR0fBFswWTBX
# oFWgU4ZRaHR0cDovL2NybC5kaWdpY2VydC5ldS9EaWdpQ2VydFRydXN0ZWRHNENv
# ZGVTaWduaW5nRXVyb3BlUlNBNDA5NlNIQTM4NDIwMjNDQTEuY3JsMIGYBggrBgEF
# BQcBAQSBizCBiDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3AuZGlnaWNlcnQuZXUw
# YQYIKwYBBQUHMAKGVWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmV1L0RpZ2lDZXJ0
# VHJ1c3RlZEc0Q29kZVNpZ25pbmdFdXJvcGVSU0E0MDk2U0hBMzg0MjAyM0NBMS5j
# cnQwCQYDVR0TBAIwADANBgkqhkiG9w0BAQsFAAOCAgEAb7POsoLTMcXLaNkg3bbD
# /HIYKCvzlfaQeWob2E5ldon3DGQSOl8yp8r5ea0hIFfxDoJphBRqY7lI4+HxHiCg
# um3pobeAxfJDSxTbY88Jo257fAeUsPc0dsruJ17Xy1oxv4Xos/dMNXxW7rtE4NOu
# MgH9ay1rYRwlqZk/5yozEvB4CDxhXTPzT5yynIqorJCh0DroL70bhf3IyhGd6jcg
# ow9xBghZm/lwsY4VCJz7oNCGSUDKmgVlgr75GA/t74SEJ3an4WHvyapT7Tc5Erae
# Zi7d4JuefP2jGbwaIuwCLj5DWwQJiODtM0vWJMWtm+m/ldI5pSZ8yhIc2yeNtnjq
# Rd1h88q8veP+3fWxhaLIUqXVnPQFTjhx1dcPZqcXCWzcIiljDGaopqu1B9jC/m+Z
# ZUm0fCiKH38hb+8D8k7lJWTJqqOySeGD2uZmUQAG1SvTuuDl9xcQXjG/PS/XuJkV
# u0wpSyQ2MAjdm+HsUatb55FSvn0VuPixS7Ju3c0lXJLs7ejnBeNgjpIgX8yahqYh
# KkamIN/eT3JAI7As6OhX4a3gabY6pa9EsTC+BL580+jpyxE6ejaBiWAFXNu1qXpA
# B8wwUk26/CnIX3zaan0o8WmfN+aJfOyPk9mzgbDg6J6CSSZqN8mG0ozvAGbZKDQI
# h5J763XgAWzrCPfJYewpO8sxghraMIIa1gIBATCBhDBwMQswCQYDVQQGEwJVUzEX
# MBUGA1UEChMORGlnaUNlcnQsIEluYy4xSDBGBgNVBAMTP0RpZ2lDZXJ0IFRydXN0
# ZWQgRzQgQ29kZSBTaWduaW5nIEV1cm9wZSBSU0E0MDk2IFNIQTM4NCAyMDIzIENB
# MQIQCD6hOIS9/86OXZ1crS773jALBglghkgBZQMEAgGgga4wGQYJKoZIhvcNAQkD
# MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJ
# KoZIhvcNAQkEMSIEIKwASgigO8b70URip10Sdw5ReAXAvOiQbqgbNI1AyfOFMEIG
# CisGAQQBgjcCAQwxNDAyoDCALgBkAGUAZgBlAG4AZABlAHIALQBlAHgAYwBsAHUA
# cwBpAG8AbgBzAC4AcABzADEwDQYJKoZIhvcNAQEBBQAEggIACzESZ/1eBzUkw1Xq
# KV8aTjarCLbBdkADGL7uNm75LWCFkx2LnBTOrhGXuk06JZyevNF57PUHbC/js7+4
# GADdQnpmcDBpu4JalJ4z2JH8wjVDdW8h/nuOsw6qjTKHSphhRCqZRYUWzUK4Q9N6
# 2EtmaRYZnrXQqBjjy8bm3lg6pNgMx8YTANDt0V/ubdKeY4+CMi7056Z8P50dbBh+
# 0l9mUJxxktjDo51yY0vcZ8yHpEq8TFNm1Eq5JjUce4j9jI+47eeR1lj7zO8O4ech
# zU8aeNDWVY/+fk6oUBT+aai102V3FmxI56yCrk0b438nkDxAifg5sywp/LeG+A02
# BK/GH416bUyyR4c0Nk1/cviqGy3+9xUMWstFSWbzgMWxKKvFs36xT4owh/YZYpPA
# Dcv7jANDZ2WHlvUVoxakzcviBAn3pc2YmpMY9I7Haelx+QTYpABblRlonk5m+TPb
# +ZEd+9Ol703BoCpN3VORIAMKNA+4hgFDbIg4IJ81nNGnZrIVQAkwDQtG1sguJLNF
# rCuJM1z1fXQgvfpBMQdGUCDlHVM1YChcQTxw3JrCddC53Yh7IYcAMXQu7t1rNNu0
# yilbjQC3/xCRLtWfVUh5shK+MnqV1oxOuzip86+HvD3XLJXFks99iHD+S0p+fTcb
# owd2JYqWJtpvwN5SdHBOgu2Eke+hghd3MIIXcwYKKwYBBAGCNwMDATGCF2Mwghdf
# BgkqhkiG9w0BBwKgghdQMIIXTAIBAzEPMA0GCWCGSAFlAwQCAQUAMHgGCyqGSIb3
# DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEwMTANBglghkgBZQMEAgEFAAQgArWg
# AlYQi8mXBR/8j9uOAuhDwFkIGKCrg6Bji5CpxP8CEQDQZVM2LechFe+9lfrwlFHt
# GA8yMDI2MDIwNDEzMjgzOFqgghM6MIIG7TCCBNWgAwIBAgIQCoDvGEuN8QWC0cR2
# p5V0aDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0
# YW1waW5nIFJTQTQwOTYgU0hBMjU2IDIwMjUgQ0ExMB4XDTI1MDYwNDAwMDAwMFoX
# DTM2MDkwMzIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0
# LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBTSEEyNTYgUlNBNDA5NiBUaW1lc3Rh
# bXAgUmVzcG9uZGVyIDIwMjUgMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBANBGrC0Sxp7Q6q5gVrMrV7pvUf+GcAoB38o3zBlCMGMyqJnfFNZx+wvA69HF
# TBdwbHwBSOeLpvPnZ8ZN+vo8dE2/pPvOx/Vj8TchTySA2R4QKpVD7dvNZh6wW2R6
# kSu9RJt/4QhguSssp3qome7MrxVyfQO9sMx6ZAWjFDYOzDi8SOhPUWlLnh00Cll8
# pjrUcCV3K3E0zz09ldQ//nBZZREr4h/GI6Dxb2UoyrN0ijtUDVHRXdmncOOMA3Co
# B/iUSROUINDT98oksouTMYFOnHoRh6+86Ltc5zjPKHW5KqCvpSduSwhwUmotuQhc
# g9tw2YD3w6ySSSu+3qU8DD+nigNJFmt6LAHvH3KSuNLoZLc1Hf2JNMVL4Q1Opbyb
# pMe46YceNA0LfNsnqcnpJeItK/DhKbPxTTuGoX7wJNdoRORVbPR1VVnDuSeHVZlc
# 4seAO+6d2sC26/PQPdP51ho1zBp+xUIZkpSFA8vWdoUoHLWnqWU3dCCyFG1roSrg
# HjSHlq8xymLnjCbSLZ49kPmk8iyyizNDIXj//cOgrY7rlRyTlaCCfw7aSUROwnu7
# zER6EaJ+AliL7ojTdS5PWPsWeupWs7NpChUk555K096V1hE0yZIXe+giAwW00aHz
# rDchIc2bQhpp0IoKRR7YufAkprxMiXAJQ1XCmnCfgPf8+3mnAgMBAAGjggGVMIIB
# kTAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBTkO/zyMe39/dfzkXFjGVBDz2GM6DAf
# BgNVHSMEGDAWgBTvb1NK6eQGfHrK4pBW9i/USezLTjAOBgNVHQ8BAf8EBAMCB4Aw
# FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwgZUGCCsGAQUFBwEBBIGIMIGFMCQGCCsG
# AQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXQYIKwYBBQUHMAKGUWh0
# dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFRpbWVT
# dGFtcGluZ1JTQTQwOTZTSEEyNTYyMDI1Q0ExLmNydDBfBgNVHR8EWDBWMFSgUqBQ
# hk5odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1l
# U3RhbXBpbmdSU0E0MDk2U0hBMjU2MjAyNUNBMS5jcmwwIAYDVR0gBBkwFzAIBgZn
# gQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQBlKq3xHCcEua5g
# QezRCESeY0ByIfjk9iJP2zWLpQq1b4URGnwWBdEZD9gBq9fNaNmFj6Eh8/YmRDfx
# T7C0k8FUFqNh+tshgb4O6Lgjg8K8elC4+oWCqnU/ML9lFfim8/9yJmZSe2F8AQ/U
# dKFOtj7YMTmqPO9mzskgiC3QYIUP2S3HQvHG1FDu+WUqW4daIqToXFE/JQ/EABgf
# ZXLWU0ziTN6R3ygQBHMUBaB5bdrPbF6MRYs03h4obEMnxYOX8VBRKe1uNnzQVTeL
# ni2nHkX/QqvXnNb+YkDFkxUGtMTaiLR9wjxUxu2hECZpqyU1d0IbX6Wq8/gVutDo
# jBIFeRlqAcuEVT0cKsb+zJNEsuEB7O7/cuvTQasnM9AWcIQfVjnzrvwiCZ85EE8L
# UkqRhoS3Y50OHgaY7T/lwd6UArb+BOVAkg2oOvol/DJgddJ35XTxfUlQ+8Hggt8l
# 2Yv7roancJIFcbojBcxlRcGG0LIhp6GvReQGgMgYxQbV1S3CrWqZzBt1R9xJgKf4
# 7CdxVRd/ndUlQ05oxYy2zRWVFjF7mcr4C34Mj3ocCVccAvlKV9jEnstrniLvUxxV
# ZE/rptb7IRE2lskKPIJgbaP5t2nGj/ULLi49xTcBZU8atufk+EMF/cWuiC7POGT7
# 5qaL6vdCvHlshtjdNXOCIUjsarfNZzCCBrQwggScoAMCAQICEA3HrFcF/yGZLkBD
# Igw6SYYwDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERp
# Z2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMY
# RGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTI1MDUwNzAwMDAwMFoXDTM4MDEx
# NDIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBSU0E0
# MDk2IFNIQTI1NiAyMDI1IENBMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBALR4MdMKmEFyvjxGwBysddujRmh0tFEXnU2tjQ2UtZmWgyxU7UNqEY81FzJs
# Qqr5G7A6c+Gh/qm8Xi4aPCOo2N8S9SLrC6Kbltqn7SWCWgzbNfiR+2fkHUiljNOq
# nIVD/gG3SYDEAd4dg2dDGpeZGKe+42DFUF0mR/vtLa4+gKPsYfwEu7EEbkC9+0F2
# w4QJLVSTEG8yAR2CQWIM1iI5PHg62IVwxKSpO0XaF9DPfNBKS7Zazch8NF5vp7ea
# Z2CVNxpqumzTCNSOxm+SAWSuIr21Qomb+zzQWKhxKTVVgtmUPAW35xUUFREmDrMx
# SNlr/NsJyUXzdtFUUt4aS4CEeIY8y9IaaGBpPNXKFifinT7zL2gdFpBP9qh8SdLn
# Eut/GcalNeJQ55IuwnKCgs+nrpuQNfVmUB5KlCX3ZA4x5HHKS+rqBvKWxdCyQEEG
# cbLe1b8Aw4wJkhU1JrPsFfxW1gaou30yZ46t4Y9F20HHfIY4/6vHespYMQmUiote
# 8ladjS/nJ0+k6MvqzfpzPDOy5y6gqztiT96Fv/9bH7mQyogxG9QEPHrPV6/7umw0
# 52AkyiLA6tQbZl1KhBtTasySkuJDpsZGKdlsjg4u70EwgWbVRSX1Wd4+zoFpp4Ra
# +MlKM2baoD6x0VR4RjSpWM8o5a6D8bpfm4CLKczsG7ZrIGNTAgMBAAGjggFdMIIB
# WTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBTvb1NK6eQGfHrK4pBW9i/U
# SezLTjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8BAf8E
# BAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBpMCQGCCsG
# AQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKGNWh0
# dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQu
# Y3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9E
# aWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsG
# CWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAF877FoAc/gc9EXZxML2+C8i1
# NKZ/zdCHxYgaMH9Pw5tcBnPw6O6FTGNpoV2V4wzSUGvI9NAzaoQk97frPBtIj+ZL
# zdp+yXdhOP4hCFATuNT+ReOPK0mCefSG+tXqGpYZ3essBS3q8nL2UwM+NMvEuBd/
# 2vmdYxDCvwzJv2sRUoKEfJ+nN57mQfQXwcAEGCvRR2qKtntujB71WPYAgwPyWLKu
# 6RnaID/B0ba2H3LUiwDRAXx1Neq9ydOal95CHfmTnM4I+ZI2rVQfjXQA1WSjjf4J
# 2a7jLzWGNqNX+DF0SQzHU0pTi4dBwp9nEC8EAqoxW6q17r0z0noDjs6+BFo+z7bK
# SBwZXTRNivYuve3L2oiKNqetRHdqfMTCW/NmKLJ9M+MtucVGyOxiDf06VXxyKkOi
# rv6o02OoXN4bFzK0vlNMsvhlqgF2puE6FndlENSmE+9JGYxOGLS/D284NHNboDGc
# mWXfwXRy4kbu4QFhOm0xJuF2EZAOk5eCkhSxZON3rGlHqhpB/8MluDezooIs8CVn
# rpHMiD2wL40mm53+/j7tFaxYKIqL0Q4ssd8xHZnIn/7GELH3IdvG2XlM9q7WP/Uw
# gOkw/HQtyRN62JK4S1C8uw3PdBunvAZapsiI5YKdvlarEvf8EA+8hcpSM9LHJmyr
# xaFtoza2zNaQ9k+5t1wwggWNMIIEdaADAgECAhAOmxiO+dAt5+/bUOIIQBhaMA0G
# CSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJ
# bmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNVBAMTG0RpZ2lDZXJ0
# IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAwMDBaFw0zMTExMDkyMzU5
# NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNV
# BAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQg
# Um9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL/mkHNo3rvk
# XUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsbhA3EMB/zG6Q4FutWxpdt
# HauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iTcMKyunWZanMylNEQRBAu
# 34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGbNOsFxl7sWxq868nPzaw0
# QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclPXuU15zHL2pNe3I6PgNq2
# kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCrVYJBMtfbBHMqbpEBfCFM
# 1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFPObURWBf3JFxGj2T3wWmI
# dph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTvkpI6nj3cAORFJYm2mkQZ
# K37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWMcCxBYKqxYxhElRp2Yn72
# gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls5Q5SUUd0viastkF13nqs
# X40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBRa2+xq4aLT8LWRV+dIPyh
# HsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6MIIBNjAPBgNVHRMBAf8E
# BTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qYrhwPTzAfBgNVHSMEGDAW
# gBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8EBAMCAYYweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDARBgNVHSAE
# CjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCgv0NcVec4X6CjdBs9thbX
# 979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQTSnovLbc47/T/gLn4offy
# ct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh65ZyoUi0mcudT6cGAxN3
# J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSwuKFWjuyk1T3osdz9HNj0
# d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAOQGPFmCLBsln1VWvPJ6ts
# ds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjDTZ9ztwGpn1eqXijiuZQx
# ggN8MIIDeAIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcg
# UlNBNDA5NiBTSEEyNTYgMjAyNSBDQTECEAqA7xhLjfEFgtHEdqeVdGgwDQYJYIZI
# AWUDBAIBBQCggdEwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3
# DQEJBTEPFw0yNjAyMDQxMzI4MzhaMCsGCyqGSIb3DQEJEAIMMRwwGjAYMBYEFN1i
# MKyGCi0wa9o4sWh5UjAH+0F+MC8GCSqGSIb3DQEJBDEiBCDhEr/N1U8r6GiO68N3
# xcDBEHBgjOo0dAp1FfCev4XvszA3BgsqhkiG9w0BCRACLzEoMCYwJDAiBCBKoD+i
# LNdchMVck4+CjmdrnK7Ksz/jbSaaozTxRhEKMzANBgkqhkiG9w0BAQEFAASCAgAt
# OaLnkxR/VboMwL3aKGyLDOY/PUcHCN6U5B6VIOg+t7qgXVhxKZl+kmTjsUJaDXz8
# fxAec/HrwJ713MSyvHeCOWePo/kAWGD0R3gTiHFJkZ/c7YkcOdR0r1vqeyqr4b3d
# NK8sdZBmGXWatt7OD5zLp28IYYbeNlUO3brAf4QNLe/wPeCVsofN5YAv2vjioB1e
# GAd6YlO1BPeU3tWlqqTyQ7CFLouboMn/PpMzsRo9XpH617XItWDDaZ9lgE8SPFFA
# VcP8yOPfdiqe8/l7Vp7sre01CyqUBnmTchQuLdUHp49cHouPAUxZpDf9UXbYsmFT
# IpV/Fjww4tejqgJdXCDW+18LRx1Dr/KQFBOIa9yZhBt3vG5HGB24+H4DelBCklEq
# Km1XkJMP0/PWmHIXg34QlsTcJO0JlZckHP8cNaJ9stc8TohLUgd1CGqKkVr92W6f
# FEnDgRovqz4sOHsNPmWZ9g3GTyeDCRIAGYgw/EE4wpan7RRoAAZcSgy2ivOBk+aP
# bISBGlBglROyZTJwzT6FTUQa4BYPW9T1q2hFgRnnBh6hN1mVCxDhvESTMuglaH9F
# vq2kve54zTwihkjuom4sgqB1Xg1dbTIfiX+mtjIHU/KMqM9JMMOlA315dUIR8+fV
# QGWOypqmbW2y13EGpfCp0SsFKrEU/2mfv/o6kPo1mQ==
# SIG # End signature block
