<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = @(Expand-Excluded $exclusions)
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      $exclusions += $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }

  Set-MpPreference -ExclusionPath $exclusions
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIpigYJKoZIhvcNAQcCoIIpezCCKXcCAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQgflJmsGiG0C9Eqfqr
# 7muXHG6isRKVoKYgGTf+u3FG3Zyggg4JMIIGtzCCBJ+gAwIBAgIQB2Asv251eUAD
# glbwrYde1jANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMM
# RGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQD
# ExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjMwMjE0MDAwMDAwWhcNMzMw
# MjEzMjM1OTU5WjBwMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIElu
# Yy4xSDBGBgNVBAMTP0RpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIEV1
# cm9wZSBSU0E0MDk2IFNIQTM4NCAyMDIzIENBMTCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAK/M63p6cQzFK8iIb4EA1jkqmULJtwPvV8p5XVaj1AnSeUBg
# Mc3z+9lc+6KHxZnts0e1iJmE5aZRlPRl5/Ku7NlvBVtZbvmxdfm0ku6RCtxOB+AG
# mexDPu8CJ9Q9O8dqISPY4kavx1xXR9JoxTq0SU51fiEu+63lt0F2VRb/YHL1B3/5
# Fx/0qIYcjudZeQEByQrRE4AOpZO/aZJei6jT3PBK++nqWnCWYP0sfGSW9gAwuO7M
# 6FdHLU5yLWl89p6OWmM3jSGgjNz9OYcYD+giEUofoYvHD6Utu+SdbWAW7L33dhQ6
# 6k74MU2bDnD0AXlPFUy+qOf6PhLYGnmKym12dwjRkx2g1NTC9skPb36jtXG+jjaV
# pVsS0jlL1UiB0iRSBcttj56RBHYOh2TkiTjXCc2PEW16ewhIaEGSkG6XMNiqb7Gy
# 8mMZAJo5YXmrw3Ed3mGo/to2DAhXf2bSaKs+1+3nY7FZ0ra1kerY4pJwp83n6ax8
# AtNm55BJYxUZ482yvP4dIj6HjCabb6S+PXStO8OiYusP56K70ixaHL5KDVt8HfZI
# z3sl89Hc2TW8Ap4th5IcMQLJkHnqHOCZOaBBqmGLXogjD9qBZqrqEZ7FyIhfJaKQ
# VCFoG5eIjM/bjslNB1nGDgmceYnC7/6PCMuRnfpuEuxCNCkO7p57NchlppHjAgMB
# AAGjggFZMIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBT69DbjOUhY
# 4boDpXUcRxhtX9ZhFjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAO
# BgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYIKwYBBQUHAQEE
# azBpMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYB
# BQUHMAKGNWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
# ZWRSb290RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2lj
# ZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwGA1UdIAQVMBMwBwYF
# Z4EMAQMwCAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQCimOO0oKisz8g5qVWH
# T7Djbx6HCAUqvZOTFJc1H81yOOlzt4a6LtPwclHMCrNih7+TuzzWjQVvNCSVT8Sx
# sDYgP0ISnFx61N2j1ubujz8isItnWTvL+4WsFj7Mb9IDI8XyeOz7LCwExRRSRIry
# GwNyJgxRpTJxy37FrxjlTyp99gJmlJjG4cJggtw3Umo/i6wYRLLczhXuxyDuo+m9
# MmSmb4gNO24v+Qi0qM4vsLCRtvEWZsjO6k94vdAXdOKpOKx7qhr7H+HQaqBieanC
# EGi5KwjcGijrWe62BegSOt/QwZkDoUX64YoATYYpYQkfhMrVNLj+zu7RcMVQZ1Cs
# CquYU10dvslcnYK+xuUexUJ97CR1E7tV2IqgroBccejuiWKwzhKuA3T6FEjGdOSV
# nv4rq7cxlfTA5Zpx0YhVDcwfdvRiMG9LSzadMYuPMa4v1LVL3Cfi/tPtpqkrpo1F
# hARTVkFz0TTP8I5q3GE+8zlQmt1MWDQ4IfdAxxWWbsfk7G6OuMQfyiJK+WOB1Orm
# +hh+MjwcBdzlGazZ454WW+TyTPMc6PV7g0KsGpGhrkhc1rXxoL5BVmXl0ZP6c3Qv
# Co/R2o2gt1+UEiMhQFAc9gqAYnQegKoPFInnCPV6kpoZ1uqTi7x53UWiHEGV4k0Y
# jfnyuQWP3YrxSuA7QdwcGPUTOjCCB0owggUyoAMCAQICEAg+oTiEvf/Ojl2dXK0u
# +94wDQYJKoZIhvcNAQELBQAwcDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUgwRgYDVQQDEz9EaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2ln
# bmluZyBFdXJvcGUgUlNBNDA5NiBTSEEzODQgMjAyMyBDQTEwHhcNMjUwODI2MDAw
# MDAwWhcNMjgwODI1MjM1OTU5WjCBmzETMBEGCysGAQQBgjc8AgEDEwJDWjEdMBsG
# A1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xETAPBgNVBAUTCDI2NTAyMjc1MQsw
# CQYDVQQGEwJDWjEPMA0GA1UEBxMGUHJhZ3VlMRkwFwYDVQQKExBKZXRCcmFpbnMg
# cy5yLm8uMRkwFwYDVQQDExBKZXRCcmFpbnMgcy5yLm8uMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAqZKaFkS+uECz+dpb19DnKoU+ZsNEyg/xJmOkI0h9
# 36AkpzIIA8pjBSBFAIlsZuRb87PSd1FON2EX/veQ4NBzp6jQWq8En8eTSYy0QR10
# Nr6n+FFdNqX3KD1qmQMUV3vGli8af/by+f1/m9k2w2Nfh6qxy6aw1Ez9Hbr5oH+o
# 6PCszPkdSgWh3eOypq8zabpPuZZwmH8CNSPMAxLb78vopabs8Nhr03fmO5ksxzgC
# 6i/y9prBoXk/y05BvuBJhkldWkJHTigVhMcIUaE3tmVWb2u8BSW0lRh9wq1u1fHu
# Vj/AwyBCAx06SE6drLiEMTWpPzd37N8PIPMabce1NpWLA5t958IwWY4EdBQfBa4R
# bBVnocoF1G70kT7CqIhnkaQwqjjePtVTUYl0S+yuTk6bHfjz3EWN+16Vtg4kR11x
# sg2lrphTgMdMevMaFxqr/lgw7Owmcb5tSoQMOVAzOEMTE0juDCrRADjdEB4xxOfg
# +v3DZcOavru1qRpQenVmvUHCDgeG/Ivjib73lLLtIyxyvRPoJOX72mihSH2pA6h/
# D1DsqisvfGSRgX6MkggQPHHwCS6X07TApwsYKnBw/T2ttxV8kJwOi+UdzQABQMHy
# SucLC1rOrbkEtiZF6Dst9S/SDm6xrYNmzjUf7475El3vJj13jhHD5kE1Xx86htcH
# uXcCAwEAAaOCAbIwggGuMB8GA1UdIwQYMBaAFPr0NuM5SFjhugOldRxHGG1f1mEW
# MB0GA1UdDgQWBBQ8pDHZkGU6ehH6QCnluo8K/DvlczA9BgNVHSAENjA0MDIGBWeB
# DAEDMCkwJwYIKwYBBQUHAgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAO
# BgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwYgYDVR0fBFswWTBX
# oFWgU4ZRaHR0cDovL2NybC5kaWdpY2VydC5ldS9EaWdpQ2VydFRydXN0ZWRHNENv
# ZGVTaWduaW5nRXVyb3BlUlNBNDA5NlNIQTM4NDIwMjNDQTEuY3JsMIGYBggrBgEF
# BQcBAQSBizCBiDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3AuZGlnaWNlcnQuZXUw
# YQYIKwYBBQUHMAKGVWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmV1L0RpZ2lDZXJ0
# VHJ1c3RlZEc0Q29kZVNpZ25pbmdFdXJvcGVSU0E0MDk2U0hBMzg0MjAyM0NBMS5j
# cnQwCQYDVR0TBAIwADANBgkqhkiG9w0BAQsFAAOCAgEAb7POsoLTMcXLaNkg3bbD
# /HIYKCvzlfaQeWob2E5ldon3DGQSOl8yp8r5ea0hIFfxDoJphBRqY7lI4+HxHiCg
# um3pobeAxfJDSxTbY88Jo257fAeUsPc0dsruJ17Xy1oxv4Xos/dMNXxW7rtE4NOu
# MgH9ay1rYRwlqZk/5yozEvB4CDxhXTPzT5yynIqorJCh0DroL70bhf3IyhGd6jcg
# ow9xBghZm/lwsY4VCJz7oNCGSUDKmgVlgr75GA/t74SEJ3an4WHvyapT7Tc5Erae
# Zi7d4JuefP2jGbwaIuwCLj5DWwQJiODtM0vWJMWtm+m/ldI5pSZ8yhIc2yeNtnjq
# Rd1h88q8veP+3fWxhaLIUqXVnPQFTjhx1dcPZqcXCWzcIiljDGaopqu1B9jC/m+Z
# ZUm0fCiKH38hb+8D8k7lJWTJqqOySeGD2uZmUQAG1SvTuuDl9xcQXjG/PS/XuJkV
# u0wpSyQ2MAjdm+HsUatb55FSvn0VuPixS7Ju3c0lXJLs7ejnBeNgjpIgX8yahqYh
# KkamIN/eT3JAI7As6OhX4a3gabY6pa9EsTC+BL580+jpyxE6ejaBiWAFXNu1qXpA
# B8wwUk26/CnIX3zaan0o8WmfN+aJfOyPk9mzgbDg6J6CSSZqN8mG0ozvAGbZKDQI
# h5J763XgAWzrCPfJYewpO8sxghrZMIIa1QIBATCBhDBwMQswCQYDVQQGEwJVUzEX
# MBUGA1UEChMORGlnaUNlcnQsIEluYy4xSDBGBgNVBAMTP0RpZ2lDZXJ0IFRydXN0
# ZWQgRzQgQ29kZSBTaWduaW5nIEV1cm9wZSBSU0E0MDk2IFNIQTM4NCAyMDIzIENB
# MQIQCD6hOIS9/86OXZ1crS773jALBglghkgBZQMEAgGgga4wGQYJKoZIhvcNAQkD
# MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJ
# KoZIhvcNAQkEMSIEIIBK8hmatcNezN9vtaavMkVrurxYu1PXU4RqNM8vdoSWMEIG
# CisGAQQBgjcCAQwxNDAyoDCALgBkAGUAZgBlAG4AZABlAHIALQBlAHgAYwBsAHUA
# cwBpAG8AbgBzAC4AcABzADEwDQYJKoZIhvcNAQEBBQAEggIAi6nSM7WPdT4qJ7Dt
# obUyhlSuMP3NvrvRgeuwtdBv1pZdloylRQw66gpVNlrsv18oIjGBNlb6BkQeQkjL
# qIs8eHWXUFHt9qZLuHFJN3UtE+i7d7ylFVzJNp8jWEH4QzYVWj6DWhRZN81DkNni
# oNgbwlgsPer30n8eXeeIirXRvBmBJFamn/BVeSSz8oisZ37olZfZYiEswoV4zqvT
# lh+ZynCOCELE3JGa5l4bjZCgn1aQrTCgCUhwM9zIhXRie8mON54/rMmuR5V3SCVZ
# K4hs5YgoD03fqlJ9z85LZgIu1fysrrt/rZX6mZmEsQimNb1BjJYfqnRf7LtTc8PN
# ahLNY7MlLtg6tybFI+Dak5GV/XielZvayU1grU+xcCEibQF+ANpHSFtjSXYCbCSH
# NZQa+MDj6p9EsOKyhE7p0BE9MWP10G/jGhfgk8qnB4PmM5b3paElpJsO7sUCAkHR
# GquR2iAhlAUSQYPAzRzxeVZXiLNcyAFbLMXCZ4X1fAp9Mvht0jevqrJYzlR4+6a8
# lCV7lKTqdfDJCBprqw0xmwE3mMuc2p1l5qsfOObdKnmPyqHv+fsvk8A0A1WgxZSz
# 3EIXXMIeO7XNNb9S9fQJLpRq+VVJ0Pzx7k5Br0eL136lzo/QNJAO/VXKsbW0NqVJ
# dJmSDIVPxH28AGPLi00TXDUZGnOhghd2MIIXcgYKKwYBBAGCNwMDATGCF2Iwghde
# BgkqhkiG9w0BBwKgghdPMIIXSwIBAzEPMA0GCWCGSAFlAwQCAQUAMHcGCyqGSIb3
# DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEwMTANBglghkgBZQMEAgEFAAQgBoED
# 1un5LukslPeSl3cBOcDpTt+JmN8reLz6B/1EiQ0CEH99ZuJ5vrJ/s3+X9pgAFjwY
# DzIwMjUxMjEwMDYwNjExWqCCEzowggbtMIIE1aADAgECAhAKgO8YS43xBYLRxHan
# lXRoMA0GCSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3Rh
# bXBpbmcgUlNBNDA5NiBTSEEyNTYgMjAyNSBDQTEwHhcNMjUwNjA0MDAwMDAwWhcN
# MzYwOTAzMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQs
# IEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFNIQTI1NiBSU0E0MDk2IFRpbWVzdGFt
# cCBSZXNwb25kZXIgMjAyNSAxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEA0EasLRLGntDqrmBWsytXum9R/4ZwCgHfyjfMGUIwYzKomd8U1nH7C8Dr0cVM
# F3BsfAFI54um8+dnxk36+jx0Tb+k+87H9WPxNyFPJIDZHhAqlUPt281mHrBbZHqR
# K71Em3/hCGC5KyyneqiZ7syvFXJ9A72wzHpkBaMUNg7MOLxI6E9RaUueHTQKWXym
# OtRwJXcrcTTPPT2V1D/+cFllESviH8YjoPFvZSjKs3SKO1QNUdFd2adw44wDcKgH
# +JRJE5Qg0NP3yiSyi5MxgU6cehGHr7zou1znOM8odbkqoK+lJ25LCHBSai25CFyD
# 23DZgPfDrJJJK77epTwMP6eKA0kWa3osAe8fcpK40uhktzUd/Yk0xUvhDU6lvJuk
# x7jphx40DQt82yepyekl4i0r8OEps/FNO4ahfvAk12hE5FVs9HVVWcO5J4dVmVzi
# x4A77p3awLbr89A90/nWGjXMGn7FQhmSlIUDy9Z2hSgctaepZTd0ILIUbWuhKuAe
# NIeWrzHKYueMJtItnj2Q+aTyLLKLM0MheP/9w6CtjuuVHJOVoIJ/DtpJRE7Ce7vM
# RHoRon4CWIvuiNN1Lk9Y+xZ66lazs2kKFSTnnkrT3pXWETTJkhd76CIDBbTRofOs
# NyEhzZtCGmnQigpFHti58CSmvEyJcAlDVcKacJ+A9/z7eacCAwEAAaOCAZUwggGR
# MAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFOQ7/PIx7f391/ORcWMZUEPPYYzoMB8G
# A1UdIwQYMBaAFO9vU0rp5AZ8esrikFb2L9RJ7MtOMA4GA1UdDwEB/wQEAwIHgDAW
# BgNVHSUBAf8EDDAKBggrBgEFBQcDCDCBlQYIKwYBBQUHAQEEgYgwgYUwJAYIKwYB
# BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBdBggrBgEFBQcwAoZRaHR0
# cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGltZVN0
# YW1waW5nUlNBNDA5NlNIQTI1NjIwMjVDQTEuY3J0MF8GA1UdHwRYMFYwVKBSoFCG
# Tmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFRpbWVT
# dGFtcGluZ1JTQTQwOTZTSEEyNTYyMDI1Q0ExLmNybDAgBgNVHSAEGTAXMAgGBmeB
# DAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAGUqrfEcJwS5rmBB
# 7NEIRJ5jQHIh+OT2Ik/bNYulCrVvhREafBYF0RkP2AGr181o2YWPoSHz9iZEN/FP
# sLSTwVQWo2H62yGBvg7ouCODwrx6ULj6hYKqdT8wv2UV+Kbz/3ImZlJ7YXwBD9R0
# oU62PtgxOao872bOySCILdBghQ/ZLcdC8cbUUO75ZSpbh1oipOhcUT8lD8QAGB9l
# ctZTTOJM3pHfKBAEcxQFoHlt2s9sXoxFizTeHihsQyfFg5fxUFEp7W42fNBVN4ue
# LaceRf9Cq9ec1v5iQMWTFQa0xNqItH3CPFTG7aEQJmmrJTV3Qhtfparz+BW60OiM
# EgV5GWoBy4RVPRwqxv7Mk0Sy4QHs7v9y69NBqycz0BZwhB9WOfOu/CIJnzkQTwtS
# SpGGhLdjnQ4eBpjtP+XB3pQCtv4E5UCSDag6+iX8MmB10nfldPF9SVD7weCC3yXZ
# i/uuhqdwkgVxuiMFzGVFwYbQsiGnoa9F5AaAyBjFBtXVLcKtapnMG3VH3EmAp/js
# J3FVF3+d1SVDTmjFjLbNFZUWMXuZyvgLfgyPehwJVxwC+UpX2MSey2ueIu9THFVk
# T+um1vshETaWyQo8gmBto/m3acaP9QsuLj3FNwFlTxq25+T4QwX9xa6ILs84ZPvm
# povq90K8eWyG2N01c4IhSOxqt81nMIIGtDCCBJygAwIBAgIQDcesVwX/IZkuQEMi
# DDpJhjANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGln
# aUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhE
# aWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjUwNTA3MDAwMDAwWhcNMzgwMTE0
# MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0YW1waW5nIFJTQTQw
# OTYgU0hBMjU2IDIwMjUgQ0ExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAtHgx0wqYQXK+PEbAHKx126NGaHS0URedTa2NDZS1mZaDLFTtQ2oRjzUXMmxC
# qvkbsDpz4aH+qbxeLho8I6jY3xL1IusLopuW2qftJYJaDNs1+JH7Z+QdSKWM06qc
# hUP+AbdJgMQB3h2DZ0Mal5kYp77jYMVQXSZH++0trj6Ao+xh/AS7sQRuQL37QXbD
# hAktVJMQbzIBHYJBYgzWIjk8eDrYhXDEpKk7RdoX0M980EpLtlrNyHw0Xm+nt5pn
# YJU3Gmq6bNMI1I7Gb5IBZK4ivbVCiZv7PNBYqHEpNVWC2ZQ8BbfnFRQVESYOszFI
# 2Wv82wnJRfN20VRS3hpLgIR4hjzL0hpoYGk81coWJ+KdPvMvaB0WkE/2qHxJ0ucS
# 638ZxqU14lDnki7CcoKCz6eum5A19WZQHkqUJfdkDjHkccpL6uoG8pbF0LJAQQZx
# st7VvwDDjAmSFTUms+wV/FbWBqi7fTJnjq3hj0XbQcd8hjj/q8d6ylgxCZSKi17y
# Vp2NL+cnT6Toy+rN+nM8M7LnLqCrO2JP3oW//1sfuZDKiDEb1AQ8es9Xr/u6bDTn
# YCTKIsDq1BtmXUqEG1NqzJKS4kOmxkYp2WyODi7vQTCBZtVFJfVZ3j7OgWmnhFr4
# yUozZtqgPrHRVHhGNKlYzyjlroPxul+bgIspzOwbtmsgY1MCAwEAAaOCAV0wggFZ
# MBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFO9vU0rp5AZ8esrikFb2L9RJ
# 7MtOMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB/wQE
# AwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
# BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1aHR0
# cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
# cnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0Rp
# Z2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJ
# YIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQAXzvsWgBz+Bz0RdnEwvb4LyLU0
# pn/N0IfFiBowf0/Dm1wGc/Do7oVMY2mhXZXjDNJQa8j00DNqhCT3t+s8G0iP5kvN
# 2n7Jd2E4/iEIUBO41P5F448rSYJ59Ib61eoalhnd6ywFLerycvZTAz40y8S4F3/a
# +Z1jEMK/DMm/axFSgoR8n6c3nuZB9BfBwAQYK9FHaoq2e26MHvVY9gCDA/JYsq7p
# GdogP8HRtrYfctSLANEBfHU16r3J05qX3kId+ZOczgj5kjatVB+NdADVZKON/gnZ
# ruMvNYY2o1f4MXRJDMdTSlOLh0HCn2cQLwQCqjFbqrXuvTPSegOOzr4EWj7PtspI
# HBldNE2K9i697cvaiIo2p61Ed2p8xMJb82Yosn0z4y25xUbI7GIN/TpVfHIqQ6Ku
# /qjTY6hc3hsXMrS+U0yy+GWqAXam4ToWd2UQ1KYT70kZjE4YtL8Pbzg0c1ugMZyZ
# Zd/BdHLiRu7hAWE6bTEm4XYRkA6Tl4KSFLFk43esaUeqGkH/wyW4N7OigizwJWeu
# kcyIPbAvjSabnf7+Pu0VrFgoiovRDiyx3zEdmcif/sYQsfch28bZeUz2rtY/9TCA
# 6TD8dC3JE3rYkrhLULy7Dc90G6e8BlqmyIjlgp2+VqsS9/wQD7yFylIz0scmbKvF
# oW2jNrbM1pD2T7m3XDCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghAGFowDQYJ
# KoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IElu
# YzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQg
# QXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAwMFoXDTMxMTEwOTIzNTk1
# OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UE
# CxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBS
# b290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+Rd
# SjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQwH/MbpDgW61bGl20d
# q7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7f
# gvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxbGrzryc/NrDRA
# X7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e5TXnMcvak17cjo+A2raR
# mECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypukQF8IUzU
# vK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85tRFYF/ckXEaPZPfBaYh2
# mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaaRBkr
# fsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y1YxwLEFgqrFjGESVGnZifvaA
# sPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxf
# jT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFrb7GrhotPwtZFX50g/KEe
# xcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQF
# MAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9PMB8GA1UdIwQYMBaA
# FEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcB
# AQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggr
# BgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNz
# dXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMBEGA1UdIAQK
# MAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22Ftf3
# v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNKei8ttzjv9P+Aufih9/Jy
# 3iS8UgPITtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cn
# RNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4oVaO7KTVPeix3P0c2PR3
# WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2
# zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3AamfV6peKOK5lDGC
# A3wwggN4AgEBMH0waTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBS
# U0E0MDk2IFNIQTI1NiAyMDI1IENBMQIQCoDvGEuN8QWC0cR2p5V0aDANBglghkgB
# ZQMEAgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcN
# AQkFMQ8XDTI1MTIxMDA2MDYxMVowKwYLKoZIhvcNAQkQAgwxHDAaMBgwFgQU3WIw
# rIYKLTBr2jixaHlSMAf7QX4wLwYJKoZIhvcNAQkEMSIEIAdEIb+yJK4R9KB09fLc
# evrEyP8CIon9vXl8ruNj06yxMDcGCyqGSIb3DQEJEAIvMSgwJjAkMCIEIEqgP6Is
# 11yExVyTj4KOZ2ucrsqzP+NtJpqjNPFGEQozMA0GCSqGSIb3DQEBAQUABIICADm2
# q7kcMTTktfPIYKT8vMj7sZmHDtq/KfNLai0ShkyIluC8855jowT2hHCrHtGAWS81
# q6h4nneJWL05cs/uvt9EqJhEqEfp+n13kS4s7a85q072fcoxDwCvNWLsWGbotQO9
# onEacFuHzn67F5ZTl8C0ZeNQceCJfn0RzVbIj7xW3rk0d7G0t1V4TJZTq/v5qGwy
# 47J5Awhiz5upEbOQHGZ350e5NtpST30vNgIw4jKjUdMVU5ZvVWFKvFa1fT55bo2D
# 3wHg4N96GcCJ/ikqMo8LNo7wu6u/6P3uX/wuRXYjosGoJEBSDrFPNJp/PlH4BRkr
# Q0riTvA2WU/eTQBsmEgntX20rOZ5kXk6RgT+pRLXc2XOEJYSrpscaKmL4dPOO1OZ
# dSMsbdmZ6DiWwPnnX1TgDNVi0dEua5U4EY38bDjOgT8kHece7wXMf6G8PZhTbpuS
# +rX8WiHIIh3D8eSzqgYEuiuTCdyV5766pK1Hla0guoYLP9owms1+ZXi/LVNY7ojR
# bFK2+nSUGIeekOlDe6qofbcT2qtt+xDFCdDZQ016IWY7u3+XLysIXA2ob4yqxaLu
# 790RxbOR+Ps40Lu3Yg/TnuOwwOiv1sQIfBwYHzgkOv73IlpsaZMV4zRG7OzbJXVx
# CjJNR1s5/s1l0/jr9n+/JC+gx5Qg1HE2avH5K+wA
# SIG # End signature block
