import type * as ts from "./tsserverlibrary.shim"
import type {
  AreTypesMutuallyAssignableArguments,
  AreTypesMutuallyAssignableResponse,
  GetCompletionSymbolsArguments,
  GetCompletionSymbolsResponse,
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetResolvedSignatureArguments,
  GetResolvedSignatureResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments,
  GetTypePropertyArguments,
  GetTypeTextArguments,
  GetTypeTextResponse,
  SymbolResponse,
} from "./protocol"
import {
  areTypesMutuallyAssignable,
  getCompletionSymbols,
  getElementTypeByOffsets,
  getResolvedSignature,
  getSymbolType,
  getTypeProperties,
  getTypeProperty,
  getTypeText,
} from "./ide-get-element-type"
import {throwIdeError} from "./utils"

const decoratedLanguageServices = new WeakSet();

export function decorateLanguageService(languageService: ts.LanguageService) {
  if (decoratedLanguageServices.has(languageService)) {
    return
  }
  decoratedLanguageServices.add(languageService);

  languageService.webStormGetCompletionSymbols = (options): GetCompletionSymbolsResponse => {
    const {ts, ls, fileName, position, cancellationToken, reverseMapper} = options;
    const program = languageService.getProgram();
    if (!program) {
      return undefined;
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined;
    }

    return getCompletionSymbols({
      ls,
      ts,
      program,
      sourceFileName: sourceFile.fileName,
      position,
      cancellationToken,
      reverseMapper
    });
  }

  languageService.webStormGetElementType = (options) => {
    const {fileName} = options;
    // see getQuickInfoAtPosition
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    return getElementTypeByOffsets({
      ...options,
      ideProjectId: languageService.ideProjectId,
      program,
      sourceFile,
    })
  }

  languageService.webStormGetSymbolType = (options) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getSymbolType({...options, program})
  }

  languageService.webStormGetTypeText = (options) => {
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeText({...options, program});
  }

  languageService.webStormGetTypeProperties = (options) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperties({...options, program})
  }

  languageService.webStormGetTypeProperty = (options) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperty({...options, program})
  }

  languageService.webStormAreTypesMutuallyAssignable = (options) => {
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }

    return areTypesMutuallyAssignable({...options, program})
  }

  languageService.webStormGetResolvedSignature = (options) => {
    const {ts, fileName, startOffset, endOffset, cancellationToken, reverseMapper} = options;
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    // Convert offsets to Range
    const start = ts.getLineAndCharacterOfPosition(sourceFile, startOffset);
    const end = ts.getLineAndCharacterOfPosition(sourceFile, endOffset);
    const range = {start, end};

    return getResolvedSignature({
      ts,
      ideProjectId: languageService.ideProjectId,
      program,
      sourceFile,
      range,
      cancellationToken,
      reverseMapper,
    })
  }
}

export function getCompletionSymbolsTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetCompletionSymbolsArguments
): GetCompletionSymbolsResponse {
  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file,
    requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined;

  let { line, character } = requestArguments.position
  let position = ts.getPositionOfLineAndCharacter(sourceFile, line, character)
  let languageService = project.getLanguageService()
  return languageService.webStormGetCompletionSymbols({
    ts,
    ls: languageService,
    fileName,
    position,
    cancellationToken: projectService.cancellationToken
  })
}

export function getElementTypeTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetElementTypeArguments,
): GetElementTypeResponse {
  const fileName = ts.server.toNormalizedPath(requestArguments.file)
  const {project, sourceFile} = projectService.ideProjectService
    .getProjectAndSourceFile(requestArguments.file, requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined

  const range = requestArguments.range
  const startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  const endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)

  return project.getLanguageService().webStormGetElementType({
    ts,
    fileName,
    startOffset,
    endOffset,
    typeRequestKind: requestArguments.typeRequestKind,
    forceReturnType: requestArguments.forceReturnType,
    cancellationToken: projectService.cancellationToken,
  })
}

export function getSymbolTypeTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetSymbolTypeArguments,
): GetSymbolTypeResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetSymbolType({
    ts,
    symbolId: requestArguments.symbolId,
    cancellationToken: projectService.cancellationToken,
  })
}

export function getTypeTextTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetTypeTextArguments
): GetTypeTextResponse {
  const languageService = findLanguageService(
    projectService,
    requestArguments.ideProjectId,
    requestArguments.ideTypeCheckerId
  );

  if (!languageService) return undefined;

  return languageService.webStormGetTypeText({
    ts,
    symbolId: requestArguments.symbolId,
    cancellationToken: projectService.cancellationToken,
    flags: requestArguments.flags,
  });
}

export function getTypePropertiesTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetTypePropertiesArguments,
): GetElementTypeResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperties({
    ts,
    typeId: requestArguments.typeId,
    cancellationToken: projectService.cancellationToken,
  })
}

export function getTypePropertyTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetTypePropertyArguments,
): SymbolResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperty({
    ts,
    typeId: requestArguments.typeId,
    propertyName: requestArguments.propertyName,
    cancellationToken: projectService.cancellationToken,
  })
}

export function areTypesMutuallyAssignableTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: AreTypesMutuallyAssignableArguments,
): AreTypesMutuallyAssignableResponse {
  const languageService = findLanguageService(
    projectService,
    requestArguments.ideProjectId,
    requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }

  return languageService.webStormAreTypesMutuallyAssignable({
    ts,
    type1Id: requestArguments.type1Id,
    type2Id: requestArguments.type2Id,
    cancellationToken: projectService.cancellationToken,
  })
}

export function getResolvedSignatureTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetResolvedSignatureArguments,
): GetResolvedSignatureResponse {
  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService
    .getProjectAndSourceFile(requestArguments.file, requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined

  let range = requestArguments.range
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return project.getLanguageService().webStormGetResolvedSignature({
    ts, fileName, startOffset, endOffset,
    cancellationToken: projectService.cancellationToken,
  })
}

function findLanguageService(
  projectService: ts.server.ProjectService,
  ideProjectId: number,
  ideTypeCheckerId: number,
): ts.LanguageService | undefined {
  for (let [, project] of projectService.configuredProjects as Map<string, ts.server.ConfiguredProject>) {
    if (project.ideProjectId === ideProjectId) {
      return getLanguageService(project)
    }
  }

  for (let inferredProject of projectService.inferredProjects) {
    if (inferredProject.ideProjectId === ideProjectId) {
      return getLanguageService(inferredProject)
    }
  }

  for (let externalProject of projectService.externalProjects) {
    if (externalProject.ideProjectId === ideProjectId) {
      return getLanguageService(externalProject)
    }
  }

  function getLanguageService(project: ts.server.Project) {
    let program = project.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return project.getLanguageService()
    }
    throwIdeError("OutdatedTypeCheckerIdException")
  }

  console.error(`findLanguageService - failed to find language service for ideProjectId ${ideProjectId}`)
}
