/**
 * Object for accessing HTTP Client Crypto API
 */
// @ts-ignore
declare const crypto: CryptoSupport;

declare const jwt: JwtDigestInterface;

/**
 * Some useful function for cryptography
 */
interface CryptoSupport {
    /**
     * SHA-1 digest builder
     */
    sha1(): DigestBuilder;

    /**
     * SHA-3 digest builder
     * @param bits SHA-3 digest size in bits. Supported values: 224, 256, 384, 512.
     */
    sha3(bits: string): DigestBuilder;

    /**
     * SHA-256 digest builder
     */
    sha256(): DigestBuilder;

    /**
     * SHA-384 digest builder
     */
    sha384(): DigestBuilder;

    /**
     * SHA-512 digest builder
     */
    sha512(): DigestBuilder;

    /**
     * MD-5 digest builder
     */
    md5(): DigestBuilder;

    /**
     * UUID generation
     */
    randomUUID(): string;

    /**
     * API for hmac
     */
    hmac: HmacSupport;
    /**
     *
     */
    subtle: SubtleCrypto;
}

/**
 * Builder for digests.
 * Sequential calls of `update*` methods append bytes to result message.
 */
interface DigestBuilder {
    /**
     * Append data presented as text
     * @param textInput data for appending to message
     * @param encoding encoding for decoding text to bytes. By default, UTF-8
     */
    updateWithText(textInput: string, encoding?: string): DigestBuilder;

    /**
     * Append data presented as 16-radix HEX text
     * @param hexInput data for appending to message
     */
    updateWithHex(hexInput: string): DigestBuilder;

    /**
     * Append data presented as Base64 encoded text
     * @param base64Input data for appending to message
     * @param urlSafe is `base64Input` encoded as urlSafe Base64 variant. By default, false
     */
    updateWithBase64(base64Input: string, urlSafe?: boolean): DigestBuilder;

    /**
     * Constructs digest from containing message.
     */
    digest(): Digest;
}

/**
 * Object containing bytes of digest
 */
interface Digest {
    /**
     * Returns bytes encoded as 16-radix HEX string
     */
    toHex(): string;

    /**
     * Returns bytes encoded as Base64 string
     * @param urlSafe if true, will be used url-safe variant of Base64. By default, false
     */
    toBase64(urlSafe?: boolean): string;
}

/**
 * API for HMAC
 */
interface HmacSupport {
    /**
     * SHA-1 HMAC builder
     */
    sha1(): HmacInitializer;

    /**
     * SHA-3 HMAC builder
     * @param bits SHA-3 digest size in bits. Supported values: 224, 256, 384, 512.
     */
    sha3(bits: string): HmacInitializer;
    /**
     * SHA-256 HMAC builder
     */
    sha256(): HmacInitializer;

    /**
     * SHA-384 HMAC builder
     */
    sha384(): HmacInitializer;

    /**
     * SHA-512 HMAC builder
     */
    sha512(): HmacInitializer;

    /**
     * MD-5 HMAC builder
     */
    md5(): HmacInitializer;
}

interface SubtleCrypto {
    generateKey(algorithm: KeyAlgorithm, extractable: boolean, keyUsages: string[]): CryptoKeyPair;
    importKey(format: string, keyData: ArrayBuffer, algorithm: KeyAlgorithm, extractable: boolean, keyUsages: string[]): CryptoKey;
    exportKey(format: string, key: CryptoKey): ArrayBuffer;
    sign(algorithm: string, key: CryptoKey, data: ArrayBuffer): ArrayBuffer;
    verify(algorithm: string, key: CryptoKey, signature: ArrayBuffer, data: ArrayBuffer): boolean;
    encrypt(algorithm: string, key: CryptoKey, data: ArrayBuffer): ArrayBuffer;
    decrypt(algorithm: string, key: CryptoKey, data: ArrayBuffer): ArrayBuffer;
}

interface CryptoKeyPair {
    privateKey: CryptoKey;
    publicKey: CryptoKey;
}

interface CryptoKey {
    readonly algorithm: KeyAlgorithm;
    readonly extractable: boolean;
    readonly type: KeyType;
    readonly usages: KeyUsage[];
}



interface KeyAlgorithm {
    name: string;
    hash: string;
    modulusLength: number;
    publicExponent: BigInteger;
}

/**
 * Object for initializing HMAC with private key (secret).
 */
interface HmacInitializer {
    /**
     * Initializes HMAC with secret presented as text. Converts to bytes using encoding
     * @param textSecret HMAC secret
     * @param encoding encoding for decoding text. By default, UTF-8
     */
    withTextSecret(textSecret: string, encoding?: string): DigestBuilder;

    /**
     * Initializes HMAC with secret presented as 16-radix HEX string.
     * @param hexSecret HMAC secret
     */
    withHexSecret(hexSecret: string): DigestBuilder;

    /**
     * Initializes HMAC with secret presented as Base64 string.
     * @param base64Secret HMAC secret
     * @param urlSafe is `base64Secret` encoded using urlSafe Base64-variant. By default, false
     */
    withBase64Secret(base64Secret: string, urlSafe?: string): DigestBuilder;
}

/**
 * Object for signing and verifying JWTs.
 */
interface JwtDigestInterface {
    /**
     * Signs JWT with provided secret or key.
     * @param claim JWT claim
     * @param secretOrKey secret or public key for signing
     * @param options signing options
     */
    sign(claim: JwtClaim, secretOrKey: string|CryptoKey, options: JwtSigningOptions): string;

    /**
     * Verifies JWT signature.
     * @param jwtSignature JWT signature
     * @param secretOrKey secret or private key for verifying
     * @param options verifying options
     */
    verify(jwtSignature: string, secretOrKey: string|CryptoKey, options: JwtVerifyOptions): boolean;

    /**
     * Decodes JWT signature.
     * @param jwtSignature
     */
    decode(jwtSignature: string): JwtDecodedToken;
}

/**
 * Object for JWT claim.
 * @see https://tools.ietf.org/html/rfc7519#section-4.1
 * @interface
 */
interface JwtClaim {
    /**
     * The "iat" (issued at) claim identifies the time at which the JWT was
     * issued in seconds.
     * @type {number}
     */
    iat?: number,
    /**
     * The "exp" (expiration time) claim identifies the expiration time on
     * or after which the JWT MUST NOT be accepted for processing.
     * in seconds. It can be modified in JwtSigningOptions::expiresIn.
     * @type {number}
     */
    exp?: number,
    /**
     * The "nbf" (not before) claim identifies the time before which the JWT
     * MUST NOT be accepted for processing. It can be modified in JwtSigningOptions::notBefore.
     * @type {number}
     */
    nbf?: number,
    /**
     * The "jti" (JWT ID) claim provides a unique identifier for the JWT.
     * It can be modified in JwtSigningOptions::jwtid.
     * @type {string}
     */
    jti?: string,
    /**
     * The "iss" (issuer) claim identifies the principal that issued the JWT.
     * It can be modified in JwtSigningOptions::issuer.
     * @type {string}
     */
    iss?: string,
    /**
     * The "sub" (subject) claim identifies the principal that is the subject
     * of the JWT. It can be modified in JwtSigningOptions::subject.
     * @type {string}
     */
    sub?: string,
    /**
     * The "aud" (audience) claim identifies the recipients that the JWT is
     * intended for. it can be modified in JwtSigningOptions::audience
     * @type {string|string[]}
     */
    aud?: string|string[],
    /**
     * The "nonce" (nonce) claim is used to prevent replay attacks.
     * @type {string}
     */
    nonce?: string,
    /**
     * The "header" (header) claim contains additional information about the JWT.
     * It can be modified in JwtSigningOptions::header
     * @type {object}
     */
    header?: object,
}
interface JwtSigningOptions {
    /**
     * The "header" (header) claim contains additional information about the JWT.
     * it modifies JwtClaim::header
     * @type {object}
     */
    header?: object,
    /**
     * The "audience" (aud) claim identifies the recipients that the JWT is
     * intended for. It modifies JwtClaim::aud
     * @type {string}
     */
    audience?: string,
    /**
     * The "issuer" claim identifies the principal that issued the JWT.
     * It modifies JwtClaim::iss
     * @type {string}
     */
    issuer?: string,
    /**
     * The "expiresIn" (expiresIn) claim identifies the expiration time on
     * or after which the JWT MUST NOT be accepted for processing.
     * It modifies JwtClaim::exp
     * @type {number}
     */
    expiresIn?: number,
    /**
     * The "notBefore" (notBefore) claim identifies the time before which the JWT MUST NOT be accepted for processing.
     * It modifies JwtClaim::nbf
     * @type {number}
     */
    notBefore?: number,
    /**
     * The "jti" (jti) claim provides a unique identifier for the JWT.
     * It modifies JwtClaim::jti
     * @type {string}
     */
    jwtid?: string,
    /**
     * The "subject" (sub) claim identifies the principal that is the subject
     * of the JWT. It modifies JwtClaim::sub
     * @type {string}
     */
    subject?: string,
    /**
     * The "noTimestamp" (noTimestamp) claim prevert JWTClaim::iat generation.
     * It modifies JwtClaim::noTimestamp
     * @type {boolean}
     */
    noTimestamp?: boolean,
    keyid?: string,
    /**
     * The "algorithm" (algorithm) claim identifies the algorithm used to sign the JWT.
     * possible values: "HS256", "HS384", "HS512", "RS256", "RS384", "RS512", "ES256", "ES384", "ES512", "PS256", "PS384", "PS512", "none"
     * @type {string}
     */
    algorithm: string,
}
interface JwtVerifyOptions {
    /**
     * verify
     * @type {string|string[]}
     */
    audience?: string|string[],
    /**
     * The "issuer" (iss) claim identifies the principal that issued the JWT.
     * @type {string}
     */
    issuer?: string,
    /**
     * The "clockTolerance" (clockTolerance) claim allows a some small leeway to account for clock skew.
     * @type {number}
     */
    clockTolerance?: number,
    /**
     * The "ignoreExpiration" (ignoreExpiration) claim allows to ignore the expiration time.
     * @type {boolean}
     */
    ignoreExpiration?: boolean,
    /**
     * The "ignoreNotBefore" (ignoreNotBefore) claim allows to ignore the not before time.
     * @type {boolean}
     */
    ignoreNotBefore?: boolean,
    /**
     * The "ignoreAudience" (ignoreAudience) claim allows to ignore the audience.
     * @type {boolean}
     */
    ignoreAudience?: boolean,
    /**
     * The "ignoreIssuer" (ignoreIssuer) claim allows to ignore the issuer.
     * @type {boolean}
     */
    ignoreIssuer?: boolean,
    /**
     * The "subject" (sub) claim identifies the principal that is the subject
     * of the JWT. It can be modified in JwtVerifyOptions::subject
     * @type {string}
     */
    subject?: string,
    /**
     * The "jti" (jti) claim provides a unique identifier for the JWT.
     * It can be modified in JwtVerifyOptions::jwtid
     * @type {string}
     */
    jwtid?: string,
}

interface JwtDecodedToken {
    header: object,
    payload: object,
    signature: string,
}
