// Copyright 2000-2023 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php;

import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.Ref;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import com.jetbrains.php.lang.documentation.phpdoc.psi.PhpDocProperty;
import com.jetbrains.php.lang.psi.elements.*;
import com.jetbrains.php.lang.psi.resolve.types.PhpType;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Predicate;

import static com.intellij.util.containers.ContainerUtil.addIfNotNull;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toCollection;

public final class PhpClassHierarchyUtils {
  private static final NextElementsAppender<PhpClass> SUPER_CLASS_APPENDER_NOT_AMBIGUITY =
    (element, phpClasses) -> addIfNotNull(phpClasses, element.getSuperClass());
  private static final NextElementsAppender<PhpClass> SUPER_CLASS_APPENDER_AMBIGUITY =
    (element, phpClasses) -> phpClasses.addAll(element.getSuperClasses());
  private static final NextElementsAppender<PhpClass> SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String interfaceName : element.getInterfaceNames()) {
      final Collection<PhpClass> interfacesByFQN = phpIndex.getInterfacesByFQN(interfaceName);
      if (interfacesByFQN.size() == 1) {
        phpClasses.add(interfacesByFQN.iterator().next());
      }
    }
    phpClasses.addAll(element.getTypeAwareImplicitEnumInterfaces());
  };
  private static final NextElementsAppender<PhpClass> SUPER_INTERFACE_APPENDER_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String interfaceName : element.getInterfaceNames()) {
      phpClasses.addAll(phpIndex.getInterfacesByFQN(interfaceName));
    }
    phpClasses.addAll(element.getTypeAwareImplicitEnumInterfaces());
  };

  private static final NextElementsAppender<PhpClass> MIXINS_APPENDER =
    (element, phpClasses) -> ContainerUtil.addAll(phpClasses, element.getMixins());
  private static final NextElementsAppender<PhpClass> SUPER_TRAIT_APPENDER_NOT_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String name : element.getTraitNames()) {
      final Collection<PhpClass> classes = phpIndex.getTraitsByFQN(name);
      if (classes.size() == 1) {
        phpClasses.add(classes.iterator().next());
      }
    }
  };
  private static final NextElementsAppender<PhpClass> SUPER_TRAIT_APPENDER_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    final String[] names = element.getTraitNames();
    for (String name : names) {
      phpClasses.addAll(phpIndex.getTraitsByFQN(name));
    }
  };


  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_NOT_AMBIGUITY =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_NOT_AMBIGUITY, SUPER_CLASS_APPENDER_NOT_AMBIGUITY, SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY);

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_AMBIGUITY =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_AMBIGUITY, SUPER_CLASS_APPENDER_AMBIGUITY, SUPER_INTERFACE_APPENDER_AMBIGUITY);

  public static boolean subclassesOfBaseContainsFQN(Project project,
                                                    @NotNull String fqn,
                                                    @NotNull String baseFQN, Key<CachedValue<Collection<String>>> key) {
      return fqn.equalsIgnoreCase(baseFQN) || CachedValuesManager.getManager(project).getCachedValue(project, key, () -> {
        PhpIndex instance = PhpIndex.getInstance(project);
        Collection<String> res = new HashSet<>();
        instance.processAllSubclasses(baseFQN, c -> {
          res.add(c.getFQN());
          return true;
        });
        return CachedValueProvider.Result.createSingleDependency(res, PsiModificationTracker.MODIFICATION_COUNT);
      }, false).contains(fqn);
    }

  private static final class CompositeNextElementsAppender<T> implements NextElementsAppender<T> {

    private final NextElementsAppender<T>[] myAppenders;

    private CompositeNextElementsAppender(NextElementsAppender<T>... appenders) {
      myAppenders = appenders;
    }

    @Override
    public void appendNextElements(@NotNull T element, @NotNull Collection<? super T> collection) {
      for (NextElementsAppender<T> appender : myAppenders) {
        appender.appendNextElements(element, collection);
      }
    }
  }

  private PhpClassHierarchyUtils() {
  }

  @FunctionalInterface @ApiStatus.Internal
  public interface NextElementsAppender<T> {

    void appendNextElements(@NotNull T element, @NotNull Collection<? super T> collection);

  }

  private static <T extends PhpClass> void process(@NotNull final T initialElement,
                                                   boolean processSelf,
                                                   @NotNull final Processor<? super T> processor,
                                                   @NotNull final NextElementsAppender<T> appender) {
    process(initialElement, processSelf, processor, appender, new HashSet<>());
  }

  private static <T extends PhpClass> void process(@NotNull final T initialElement,
                                                   boolean processSelf,
                                                   @NotNull final Processor<? super T> processor,
                                                   @NotNull final NextElementsAppender<T> appender,
                                                   Collection<T> processed) {
    Set<String> processedFQNs = new HashSet<>();
    final Deque<T> processorPool = new ArrayDeque<>();
    if (processSelf) {
      processorPool.add(initialElement);
    }
    else {
      appender.appendNextElements(initialElement, processorPool);
    }
    while (processorPool.size() > 0) {
      ProgressManager.checkCanceled();
      T first = processorPool.pollFirst();
      processedFQNs.add(first.getFQN());
      if (processed.add(first)) {
        if (processor.process(first)) {
          appender.appendNextElements(first, processorPool);
        }
        else {
          for (T remainingClassWithSameFQN : processorPool) {
            ProgressManager.checkCanceled();
            if (processedFQNs.contains(remainingClassWithSameFQN.getFQN())) {
              processor.process(remainingClassWithSameFQN);
            }
          }
          return;
        }
      }
    }
  }

  public static void processSuperClasses(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_CLASS_APPENDER_AMBIGUITY : SUPER_CLASS_APPENDER_NOT_AMBIGUITY);
  }

  public static void processSuperTraits(@NotNull final PhpClass clazz, @NotNull Processor<? super PhpClass> processor) {
    process(clazz, false, processor, SUPER_TRAIT_APPENDER_AMBIGUITY);
  }

  public static void processSuperInterfaces(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_INTERFACE_APPENDER_AMBIGUITY : SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY);
  }

  /**
   * Process {@link PhpClass#getMixins()} as well
   * @see #processSuperWithoutMixins
   */
  public static void processSupers(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_APPENDER_AMBIGUITY_WITH_MIXINS : SUPER_APPENDER_NOT_AMBIGUITY_WITH_MIXINS);
  }


  // external usage
  @SuppressWarnings("unused")
  public static void processSuperWithoutMixins(@NotNull final PhpClass clazz,
                                               boolean processSelf,
                                               final boolean allowAmbiguity,
                                               @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_APPENDER_AMBIGUITY : SUPER_APPENDER_NOT_AMBIGUITY, new HashSet<>());
  }

  @ApiStatus.Internal
  public static void processSuperWithoutMixins(@NotNull final PhpClass clazz,
                                               boolean processSelf,
                                               final boolean allowAmbiguity,
                                               @NotNull final Processor<? super PhpClass> processor, Collection<PhpClass> processed) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_APPENDER_AMBIGUITY : SUPER_APPENDER_NOT_AMBIGUITY, processed);
  }

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_NOT_AMBIGUITY_WITH_MIXINS =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_NOT_AMBIGUITY, SUPER_CLASS_APPENDER_NOT_AMBIGUITY, SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY, MIXINS_APPENDER);

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_AMBIGUITY_WITHOUT_TRAITS =
    new CompositeNextElementsAppender<>(SUPER_CLASS_APPENDER_AMBIGUITY, SUPER_INTERFACE_APPENDER_AMBIGUITY, MIXINS_APPENDER);

  @ApiStatus.Internal
  public static final NextElementsAppender<PhpClass> SUPER_APPENDER_AMBIGUITY_WITH_MIXINS =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_AMBIGUITY, SUPER_APPENDER_AMBIGUITY_WITHOUT_TRAITS);

  @ApiStatus.Internal
  public static final HierarchyClassMemberProcessor FALSE_PROCESSOR = (superMember, subClass, baseClass) -> false;

  public static boolean isSuperClass(@NotNull final PhpClass superClass, @NotNull final PhpClass subClass, boolean allowAmbiguity) {
    return isSuperClass(superClass, subClass, allowAmbiguity, true);
  }

  public static boolean isSuperClass(@Nullable final PhpClass superClass,
                                     @Nullable final PhpClass subClass,
                                     boolean allowAmbiguity,
                                     boolean strict) {
    if (superClass == null || subClass == null) return false;
    final Ref<Boolean> isSuperClassRef = new Ref<>(false);
    processSuperClasses(subClass, !strict, allowAmbiguity, curClass -> {
      if (classesEqual(curClass, superClass)) {
        isSuperClassRef.set(true);
      }
      return !isSuperClassRef.get();
    });
    return isSuperClassRef.get();
  }

  public static void processMethods(PhpClass phpClass,
                                    PhpClass initialClass,
                                    @NotNull HierarchyMethodProcessor methodProcessor,
                                    boolean processOwnMembersOnly, boolean processMixins) {
    processMembersInternal(phpClass, new HashSet<>(), null, initialClass, methodProcessor, Method.INSTANCEOF,
                           processOwnMembersOnly, processMixins, new HashSet<>());
  }

  public static void processMethodsWithGenericMixins(PhpClass phpClass,
                                                     PhpClass initialClass,
                                                     @NotNull HierarchyMethodProcessor methodProcessor,
                                                     boolean processOwnMembersOnly, boolean processMixins,
                                                     String genericInstantiationType) {
    processMembersInternalWithGenericMixins(phpClass, new HashSet<>(), null, initialClass, methodProcessor, Method.INSTANCEOF,
                                            processOwnMembersOnly, processMixins, new HashSet<>(), genericInstantiationType);
  }

  public static void processFields(PhpClass phpClass,
                                   PhpClass initialClass,
                                   @NotNull HierarchyFieldProcessor fieldProcessor,
                                   boolean processOwnMembersOnly, boolean processMixins) {
    processMembersInternal(phpClass, new HashSet<>(), null, initialClass, fieldProcessor, Field.INSTANCEOF, processOwnMembersOnly, processMixins, new HashSet<>());
  }

  public static void processFieldsWithGenericMixins(PhpClass phpClass,
                                                    PhpClass initialClass,
                                                    @NotNull HierarchyFieldProcessor fieldProcessor,
                                                    boolean processOwnMembersOnly, boolean processMixins,
                                                    String genericInstantiationType) {
    processMembersInternalWithGenericMixins(phpClass, new HashSet<>(), null, initialClass, fieldProcessor, Field.INSTANCEOF, processOwnMembersOnly, processMixins, new HashSet<>(), genericInstantiationType);
  }

  private static <T extends PhpClassMember> boolean processMembersInternal(@Nullable PhpClass phpClass,
                                                                           @NotNull Set<? super PhpClass> visited,
                                                                           @Nullable Map<String, PhpTraitUseRule> conflictResolution,
                                                                           PhpClass initialClass,
                                                                           @NotNull TypedHierarchyMemberProcessor<T> processor,
                                                                           Condition<PsiElement> condition,
                                                                           boolean processOwnMembersOnly, boolean processMixins, Set<String> nonDocVisitedMembersNames) {
    return processMembersInternalWithGenericMixins(phpClass, visited, conflictResolution, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames, null);
  }

  private static <T extends PhpClassMember> boolean processMembersInternalWithGenericMixins(@Nullable PhpClass phpClass,
                                                                                            @NotNull Set<? super PhpClass> visited,
                                                                                            @Nullable Map<String, PhpTraitUseRule> conflictResolution,
                                                                                            PhpClass initialClass,
                                                                                            @NotNull TypedHierarchyMemberProcessor<T> processor,
                                                                                            Condition<PsiElement> condition,
                                                                                            boolean processOwnMembersOnly, boolean processMixins,
                                                                                            Set<String> nonDocVisitedMembersNames,
                                                                                            String genericInstantiationType) {
    if (phpClass == null || !visited.add(phpClass)) {
      return true;
    }
    ProgressManager.checkCanceled();
    //System.out.println(phpClass.getFQN() + " conflictResolution = " + conflictResolution);
    boolean processMethods = condition == Method.INSTANCEOF;
    Set<String> currentNonDocVisitedMembersNames = new HashSet<>();
    PhpClassMember[] members = processMethods ? phpClass.getOwnMethods() : phpClass.getOwnFields(true);
    for (PhpClassMember member : members) {
      if (!processMethods) {
        if (member instanceof PhpDocProperty) {
          if (nonDocVisitedMembersNames.contains(member.getName())) {
            continue;
          }
        } else {
          currentNonDocVisitedMembersNames.add(member.getName());
        }
      }
      PhpTraitUseRule rule = conflictResolution != null ? conflictResolution.get(member.getFQN()) : null;
      //System.out.println("member = " + member.getFQN());
      //System.out.println("rule = " + (rule!=null?rule.getText():"null") );
      if (rule != null && rule.getAlias() == null && processMethods) {
        //System.out.println("*** OVERRIDE");
      }
      else {
        if (processMethods ?
            !((HierarchyMethodProcessor)processor).process((Method)member, phpClass, initialClass) :
            !((HierarchyFieldProcessor)processor).process((Field)member, phpClass, initialClass)
          ) return false;
      }
    }

    if (!processMethods) {
      nonDocVisitedMembersNames = ContainerUtil.union(currentNonDocVisitedMembersNames, nonDocVisitedMembersNames);
    }

    if(!processOwnMembersOnly && phpClass.hasTraitUses()) {
      List<PhpTraitUseRule> rules = phpClass.traitUseRules().toList();
      Map<String, PhpTraitUseRule> newConflictResolution = getTraitUseRulesConflictResolutions(rules);
      if (processMethods) {
        for (PhpTraitUseRule rule : rules) {
          if (!rule.isInsteadOf()) {
            Method member = rule.getMethod();
            if (member != null && member.isValid() && !((HierarchyMethodProcessor)processor).process(member, phpClass, initialClass)) return false;
          }
        }
      }
      // @link http://www.php.net/manual/en/language.oop5.traits.php
      // An inherited member from a base class is overridden by a member inserted by a Trait.
      // The precedence order is that members from the current class override Trait methods, which in return override inherited methods.
      for (PhpClass trait : phpClass.getTraits()) {
        if (!processMembersInternal(trait, visited, newConflictResolution, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames)) return false;
      }
    }

    if (!processOwnMembersOnly){
      for (PhpClass superClass : phpClass.getSuperClasses()) {
        if (!processMembersInternal(superClass, visited, null, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames)) return false;
      }
      for (PhpClass phpInterface : phpClass.getDirectImplementedInterfaces()) {
        if (!processMembersInternal(phpInterface, visited, null, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames)) return false;
      }
      if (phpClass.isEnum()) {
        for (PhpClass phpInterface : phpClass.getTypeAwareImplicitEnumInterfaces()) {
          if (!processMembersInternal(phpInterface, visited, null, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames)) return false;
        }
      }
      if (processMixins) {
        if (genericInstantiationType != null && phpClass.hasGenericMixins()) {
          PhpClass[] mixins = phpClass.getMixinsIncludingGeneric(genericInstantiationType);
          for (PhpClass mixin : mixins) {
            if (!processMembersInternal(mixin, visited, null, initialClass, processor, condition, processOwnMembersOnly, processMixins,
                                        nonDocVisitedMembersNames)) {
              return false;
            }
          }
          return true;
        }

        for (PhpClass mixin : phpClass.getMixins()) {
          if (!processMembersInternal(mixin, visited, null, initialClass, processor, condition, processOwnMembersOnly, processMixins, nonDocVisitedMembersNames)) return false;
        }
      }
    }

    return true;
  }

  @NotNull
  public static Map<String, PhpTraitUseRule> getTraitUseRulesConflictResolutions(Collection<PhpTraitUseRule> rules) {
    Map<String, PhpTraitUseRule> newConflictResolution = new HashMap<>();
    for (PhpTraitUseRule rule : rules) {
      for (String overridesFqn : rule.getOverriddenMethodFqns()) {
        newConflictResolution.put(overridesFqn, rule);
      }
    }
    return newConflictResolution;
  }

  public static boolean processOverridingFields(@NotNull Field field, TypedHierarchyMemberProcessor<? super Field> memberProcessor) {
    PhpIndex phpIndex = PhpIndex.getInstance(field.getProject());
    final PhpClass me = field.getContainingClass();
    if (me != null) {
      final String fieldName = field.getName();
      Ref<Boolean> res = new Ref<>(true);
      boolean isConstant = field.isConstant();
      phpIndex.processAllSubclasses(me.getFQN(), myChild -> {
        final Field overriddenField = myChild.findOwnFieldByName(fieldName, isConstant);
        if (overriddenField != null && !memberProcessor.process(overriddenField, myChild, me)) {
          res.set(false);
          return false;
        }
        return true;
      });
      return res.get();
    }
    return false;
  }

  public static boolean processOverridingMethods(@NotNull Method method, TypedHierarchyMemberProcessor<? super Method> memberProcessor) {
    return processOverridingMethods(method, method.getContainingClass(), memberProcessor);
  }

  public static boolean processOverridingMethods(@NotNull Method method, @Nullable PhpClass containingClass, TypedHierarchyMemberProcessor<? super Method> memberProcessor) {
    PhpIndex phpIndex = PhpIndex.getInstance(method.getProject());
    if (!methodCanHaveOverride(method)) return true;
    final String methodName = method.getName();
    if (containingClass != null && containingClass.isValid()) {
      Collection<PhpClass> allSubclasses = new ArrayList<>();
      if (processSubclasses(memberProcessor, phpIndex, methodName, containingClass, allSubclasses) &&
          processTraitUsages(memberProcessor, phpIndex, methodName, containingClass, allSubclasses)) {
        if (!containingClass.isTrait() &&
            collectUsedTraits(allSubclasses, trait -> trait == containingClass ||
                                                      processOverridingMethod(memberProcessor, methodName, containingClass, trait))) {
          return true;
        }
      }
      return false;
    }
    return true;
  }

  private static boolean processTraitUsages(TypedHierarchyMemberProcessor<? super Method> memberProcessor,
                                   PhpIndex phpIndex,
                                   String methodName,
                                   PhpClass me,
                                   Collection<PhpClass> allSubclasses) {
    for (PhpClass c : phpIndex.getTraitUsages(me)) {
      allSubclasses.add(c);
      if (!processOverridingMethod(memberProcessor, methodName, me, c)) return false;
    }
    return true;
  }

  public static void processMethodInSubclasses(TypedHierarchyMemberProcessor<? super Method> memberProcessor,
                                               PhpIndex phpIndex,
                                               String methodName,
                                               PhpClass me) {
    processSubclasses(memberProcessor, phpIndex, methodName, me, null);
  }

  private static boolean processSubclasses(TypedHierarchyMemberProcessor<? super Method> memberProcessor,
                                   PhpIndex phpIndex,
                                   String methodName,
                                   PhpClass me,
                                   @Nullable Collection<PhpClass> allSubclasses) {
    Ref<Boolean> res = new Ref<>(true);
    phpIndex.processAllSubclasses(me.getFQN(), c -> {
      if (allSubclasses != null) {
        allSubclasses.add(c);
      }
      if (!processOverridingMethod(memberProcessor, methodName, me, c)) {
        res.set(false);
        return false;
      }
      return true;
    });
    return res.get();
  }

  private static boolean processOverridingMethod(TypedHierarchyMemberProcessor<? super Method> memberProcessor,
                                   String methodName,
                                   PhpClass me,
                                   PhpClass myChild) {
    Method overriddenMethod = myChild.findOwnMethodByName(methodName);
    return overriddenMethod == null || memberProcessor.process(overriddenMethod, myChild, me);
  }

  public static boolean processOverridingMembers(PhpClassMember member, HierarchyClassMemberProcessor memberProcessor){
    if (member instanceof Method){
      return processOverridingMethods((Method)member, (method, subClass, baseClass) -> memberProcessor.process(method,subClass,baseClass));
    } else {
      return processOverridingFields((Field)member, (field, subClass, baseClass) -> memberProcessor.process(field,subClass,baseClass));
    }
  }

  private static boolean collectUsedTraits(@NotNull Collection<PhpClass> classes, Processor<PhpClass> traitProcessor) {
    Set<PhpClass> result = new HashSet<>();
    Deque<PhpClass> traitsToProcess = classes.stream().flatMap(e -> stream(e.getTraits())).collect(toCollection(ArrayDeque::new));
    while (!traitsToProcess.isEmpty()) {
      PhpClass trait = traitsToProcess.pollFirst();
      if (trait == null || !result.add(trait)) continue;
      if (!traitProcessor.process(trait)) return false;
      for (PhpClass innerTrait : trait.getTraits()) {
        traitsToProcess.addLast(innerTrait);
      }
    }
    return true;
  }


  public static boolean methodCanHaveOverride(@NotNull Method element) {
    PhpClass containingClass = element.getContainingClass();
    return containingClass != null && !containingClass.isFinal() && !element.getAccess().isPrivate() && !element.isFinal();
  }

  public static boolean hasSuperMethod(@NotNull Method method) {
    Ref<Boolean> hasSuperMethod = new Ref<>(false);
    processSuperMethods(method, (member, subclass, superClass) -> {
      hasSuperMethod.set(true);
      return false;
    });
    return hasSuperMethod.get();
  }

  interface MemberFounder<T extends PhpClassMember> {
    Collection<T> findMember(PhpClass superClass, T member, HashSet<PhpClass> visitedClasses);
  }

  private static <T extends PhpClassMember> boolean processSuperMembersInternal(@NotNull T member, @NotNull MemberFounder<T> memberFounder, @NotNull TypedHierarchyMemberProcessor<T> memberProcessor) {
    HashSet<PhpClass> processed = new HashSet<>();
    Set<PhpClassMember> processedMember = new HashSet<>();
    final Queue<PhpClassMember> members = new ArrayDeque<>();
    members.add(member);
    while (!members.isEmpty()) {
      final PhpClass me = members.poll().getContainingClass();
      if (me == null) return false;
      List<PhpClass> parents = getImmediateParents(me);
      if (me.isTrait()) {
        for (PhpClass usage : collectClassesWithTraitUsage(me)) {
          parents.addAll(usage.getSuperClasses());
          Collections.addAll(parents, usage.getImplementedInterfaces());
        }
      }
      Collection<String> overriddenClassesFqns = new HashSet<>();
      if (member instanceof Method) {
        String methodName = member.getName();
        me.traitUseRules()
          .filter(PhpTraitUseRule::isInsteadOf)
          .flatMap(e -> e.getOverriddenMethodFqns())
          .filter(fqn -> fqn.substring(fqn.lastIndexOf('.') + 1).equalsIgnoreCase(methodName))
          .map(fqn -> fqn.substring(0, fqn.lastIndexOf('.')))
          .forEach(overriddenClassesFqns::add);
      }
      boolean res = true;
      for (PhpClass mySuper : parents) {
        if ( mySuper.isTrait() && overriddenClassesFqns.contains(mySuper.getFQN())) continue;
        for (T foundMember : memberFounder.findMember(mySuper, member, processed)) {
          if (foundMember != null && processedMember.add(foundMember)) {
            PhpModifier modifier = foundMember.getModifier();
            if (!modifier.isPrivate() ||
                foundMember instanceof Method &&
                (isAbstractTraitClass((Method)foundMember, modifier) ||
                 ((Method)foundMember).getMethodType(true) == Method.MethodType.CONSTRUCTOR)) {
              members.add(foundMember);
              res = memberProcessor.process(foundMember, me, mySuper) && res;
            }
          }
        }
      }
      if (!res) {
        return false;
      }
    }
    return true;

  }

  private static boolean isAbstractTraitClass(Method foundMember, PhpModifier modifier) {
    PhpClass aClass = foundMember.getContainingClass();
    return modifier.isAbstract() && aClass != null && aClass.isTrait();
  }

  public static boolean processSuperMethods(@NotNull Method member, @NotNull HierarchyMethodProcessor memberProcessor) {
    return processSuperMembersInternal(member, (superClass, method, visitedClasses) -> {
      return superClass.findMethodsByName(member.getName(), visitedClasses);
    }, memberProcessor);
  }

  public static boolean processSuperFields(@NotNull Field member, @NotNull final HierarchyFieldProcessor memberProcessor) {
    return processSuperMembersInternal(member, (superClass, field, visitedClasses) -> Collections.singleton(superClass.findFieldByName(member.getName(), member.isConstant())),
                                       memberProcessor);
  }

  public static boolean processSuperMembers(@NotNull PhpClassMember member, @NotNull final HierarchyClassMemberProcessor memberProcessor) {
    if (member instanceof Method) {
      return processSuperMethods(((Method)member), (member1, subClass, baseClass) -> memberProcessor.process(member1, subClass, baseClass));
    } else {
      return processSuperFields(((Field)member), (member1, subClass, baseClass) -> memberProcessor.process(member1, subClass, baseClass));
    }
  }

  @NotNull
  private static Collection<PhpClass> collectClassesWithTraitUsage(@NotNull PhpClass trait) {
    PhpIndex index = PhpIndex.getInstance(trait.getProject());
    Deque<PhpClass> usages = new ArrayDeque<>(index.getTraitUsages(trait));
    Set<PhpClass> result = new HashSet<>();
    while (!usages.isEmpty()) {
      PhpClass classWithTraitUsage = usages.pollFirst();
      if (classWithTraitUsage == null || !result.add(classWithTraitUsage)) continue;
      if (classWithTraitUsage.isTrait()) usages.addAll(index.getTraitUsages(classWithTraitUsage));
    }
    return result;
  }

  @NotNull
  public static Collection<PhpClass> getSuperClasses(@NotNull PhpClass aClass) {
    String superName = aClass.getSuperFQN();
    final PhpIndex phpIndex = PhpIndex.getInstance(aClass.getProject());
    Collection<PhpClass> classes = aClass.isInterface() ?
                                   phpIndex.getInterfacesByFQN(superName) :
                                   phpIndex.getClassesByFQN(superName);
    return ContainerUtil.filter(classes, c -> c != aClass);
  }

  public static List<PhpClass> getImmediateParents(PhpClass me) {
    List<PhpClass> parents = new ArrayList<>(getSuperClasses(me));
    ContainerUtil.addAll(parents, me.getImplementedInterfaces());
    ContainerUtil.addAll(parents, me.getTraits());
    return parents;
  }

  @FunctionalInterface
  public interface HierarchyClassMemberProcessor extends HierarchyMemberProcessor {
    boolean process(PhpClassMember classMember, PhpClass subClass, PhpClass baseClass);
  }

  @FunctionalInterface
  public interface HierarchyMethodProcessor extends TypedHierarchyMemberProcessor<Method> {

  }

  @FunctionalInterface
  public interface HierarchyFieldProcessor extends TypedHierarchyMemberProcessor<Field> {

  }

  public interface HierarchyMemberProcessor {
  }

  public interface TypedHierarchyMemberProcessor<T extends PhpClassMember> {
    boolean process(T member, PhpClass subClass, PhpClass baseClass);
  }

  public static Collection<PhpClass> getDirectSubclasses(@NotNull PhpClass psiClass) {
    if (psiClass.isFinal()) {
      return Collections.emptyList();
    }

    final PhpIndex phpIndex = PhpIndex.getInstance(psiClass.getProject());
    return phpIndex.getDirectSubclasses(psiClass.getFQN());
  }

  /**
   * @deprecated Will traverse whole class hierarchy which can be heavy, consider to user {@link PhpIndex#processAllSubclasses(String, Processor)} for lazy processing and early return
   */
  @Deprecated
  public static Collection<PhpClass> getAllSubclasses(@NotNull PhpClass psiClass) {
    if (psiClass.isFinal()) {
      return Collections.emptyList();
    }

    final PhpIndex phpIndex = PhpIndex.getInstance(psiClass.getProject());
    return phpIndex.getAllSubclasses(psiClass.getFQN());
  }

  public static boolean anySubclassExists(@NotNull PhpClass psiClass, Predicate<PhpClass> predicate) {
    return !psiClass.isFinal() && anySubclassExists(psiClass.getProject(), psiClass.getFQN(), predicate);
  }

  public static boolean subclassExists(@NotNull PhpClass psiClass) {
    return anySubclassExists(psiClass, aClass -> true);
  }

  public static boolean anySubclassExists(@NotNull Project project, @NotNull String fqn, Predicate<PhpClass> predicate) {
    final PhpIndex phpIndex = PhpIndex.getInstance(project);
    Ref<Boolean> res = new Ref<>(false);
    phpIndex.processAllSubclasses(fqn, r -> {
      if (predicate.test(r)) {
        res.set(true);
        return false;
      }
      return true;
    });
    return res.get();
  }

  @Nullable
  public static PhpClass getObject(@NotNull final Project project) {
    Iterator iterator = PhpIndex.getInstance(project).getClassesByFQN(PhpType._OBJECT_FQN).iterator();
    return iterator.hasNext() ? (PhpClass)iterator.next() : null;
  }

  public static boolean isMyTrait(@NotNull final PhpClass me, @NotNull final PhpClass trait, @Nullable Collection<? super PhpClass> visited) {
    if (!trait.isTrait()) {
      return false;
    }
    if (visited == null) {
      visited = new HashSet<>();
    }
    for (final PhpClass candidate : me.getTraits()) {
      if (!visited.add(candidate)) {
        continue;
      }
      if (classesEqual(trait, candidate) || isMyTrait(candidate, trait, visited)) {
        return true;
      }
    }
    return false;
  }

  public static boolean classesEqual(@Nullable PhpClass one, @Nullable PhpClass another) {
    if (one != null && another != null) {
      if (one == another) {
        return true;
      }
      else if (one instanceof PhpClassAlias || another instanceof PhpClassAlias) {
        if (one instanceof PhpClassAlias) one = ((PhpClassAlias)one).getOriginal();
        if (another instanceof PhpClassAlias) another = ((PhpClassAlias)another).getOriginal();
        return classesEqual(one, another);
      }
      else if (StringUtil.equalsIgnoreCase(one.getFQN(), another.getFQN())) {
        return true;
      }
    }
    return false;
  }

}
