// Copyright 2000-2019 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.classAlias;

import com.intellij.lang.ASTNode;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.jetbrains.php.lang.psi.elements.FunctionReference;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;

/**
 * The extension point allows to mark an arbitrary {@code FunctionReference} as a class alias holder. A wrapper is created for such holders.
 * The wrapper implements the {@code PhpClassAlias} interface and serves as a delegating {@code PhpClass} implementation.
 * It's emitted by {@code PhpIndex} thus the core functionality such as completion and inspections can benefit from it.
 *
 * @see FunctionReference
 * @see PhpClassAliasIndex
 * @see com.jetbrains.php.lang.psi.elements.PhpClassAlias
 */
public interface PhpClassAliasProvider {
  ExtensionPointName<PhpClassAliasProvider> EP_NAME = ExtensionPointName.create("com.jetbrains.php.classAliasProvider");

  /**
   * Only files which contain the search string will be a subject for indexing.
   *
   * @return a string to be searched in a file
   */
  CharSequence getSearchString();

  /**
   * The method should return result as fast as possible as it's called pretty often.
   *
   * @param reference a function reference to be checked
   * @return {@code true} if a function reference is a class alias declaration
   */
  boolean isClassAliasDeclaration(@NotNull FunctionReference reference);

  /**
   * Only function references which return both an alias FQN and original class FQN are indexed.
   *
   * @param reference a function reference for which {@code #isClassAliasDeclaration()} returns {@code true}
   * @return an alisa FQN
   */
  @Nullable
  String getAliasFqn(@NotNull FunctionReference reference);

  /**
   * Only function references which return both an alias FQN and original class FQN are indexed.
   *
   * @param reference a function reference for which {@code #isClassAliasDeclaration()} returns {@code true}
   * @return an original class FQN
   */
  @Nullable
  String getOriginalFqn(@NotNull FunctionReference reference);

  /**
   * The node is used as a name node of {@code PhpClassAlias} e.g. for renaming.
   *
   * @param reference a function reference for which {@code #isClassAliasDeclaration()} returns {@code true}
   * @return a name node
   */
  @Nullable
  ASTNode getNameNode(@NotNull FunctionReference reference);

  /**
   * The method is intended to perform renaming of a class alias.
   *
   * @param nameNode a name node provided by {@code #getNameNode()}.
   * @param newName a new name
   * @param project a project
   */
  void setName(@NotNull ASTNode nameNode, @NotNull String newName, @NotNull Project project);

  /**
   * The version is used by indices. It must be updated each time the implementation amended.
   *
   * @return a version
   */
  int getVersion();
}
