// Copyright 2000-2023 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.resolve.types;

import com.intellij.codeInsight.intention.FileModifier;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.Ref;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.util.ObjectUtils;
import com.intellij.util.Range;
import com.intellij.util.containers.CollectionFactory;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.HashingStrategy;
import com.jetbrains.php.PhpClassHierarchyUtils;
import com.jetbrains.php.PhpIndex;
import com.jetbrains.php.lang.psi.elements.PhpClass;
import com.jetbrains.php.lang.psi.elements.PhpTypedElement;
import com.jetbrains.php.lang.psi.elements.Variable;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.intellij.openapi.util.text.StringUtil.startsWithChar;

@FileModifier.SafeTypeForPreview
public class PhpType {
  private static final Logger LOG = Logger.getInstance(PhpType.class);

  public static final @NlsSafe String PHPSTORM_HELPERS = "___PHPSTORM_HELPERS";
  public static final @NlsSafe String _PHPSTORM_HELPERS_FQN = "\\___PHPSTORM_HELPERS";

  public static final @NlsSafe String _OBJECT_FQN = "\\" + PHPSTORM_HELPERS + "\\object"; //TODO - USAGES?
  //public static final @NlsSafe String _PHPSTORM_HELPERS_STATIC = "\\" + PHPSTORM_HELPERS + "\\static";
  //public static final @NlsSafe String _PHPSTORM_HELPERS_$THIS = "\\" + PHPSTORM_HELPERS + "\\this";

  public static final @NlsSafe String _OBJECT = "\\object";
  public static final @NlsSafe String _MIXED = "\\mixed";
  public static final @NlsSafe String _VOID = "\\void";
  public static final @NlsSafe String _NEVER = "\\never";
  public static final @NlsSafe String _NULL = "\\null";
  public static final @NlsSafe String _ARRAY = "\\array";
  public static final @NlsSafe String _ITERABLE = "\\iterable";
  public static final @NlsSafe String _INT = "\\int";
  public static final @NlsSafe String _INTEGER = "\\integer";
  public static final @NlsSafe String _BOOL = "\\bool";
  public static final @NlsSafe String _BOOLEAN = "\\boolean";
  public static final @NlsSafe String _TRUE = "\\true";
  public static final @NlsSafe String _FALSE = "\\false";
  public static final @NlsSafe String _STRING = "\\string";
  public static final @NlsSafe String _FLOAT = "\\float";
  public static final @NlsSafe String _DOUBLE = "\\double";
  public static final @NlsSafe String _CLOSURE = "\\Closure";
  /**
   * use CALLABLE in our code, but doc literal should be OK
   */
  public static final @NlsSafe String _CALLBACK = "\\callback";
  public static final @NlsSafe String _CALLABLE = "\\callable";
  public static final @NlsSafe String _NUMBER = "\\number";
  public static final @NlsSafe String _RESOURCE = "\\resource";
  public static final @NlsSafe String _EXCEPTION = "\\Exception";
  public static final @NlsSafe String _THROWABLE = "\\Throwable";
  public static final @NlsSafe String _STRINGABLE = "\\Stringable";
  public static final @NlsSafe String _CLASS_STRING = "\\class-string";

  //primitive types
  public static final PhpType EMPTY = builder().build();
  /**
   * This type can be used to indicate uncertainty about inferred type.
   * Most of the type-based inspections won't (and should not) highlight any error on the element with this type.
   */
  public static final PhpType MIXED = builder().add(_MIXED).build();
  public static final PhpType NULL = builder().add(_NULL).build();
  public static final PhpType STRING = builder().add(_STRING).build();
  /**
   * @see #isBoolean()
   */
  public static final PhpType BOOLEAN = builder().add(_BOOL).build();
  public static final PhpType FALSE = builder().add(_FALSE).build();
  public static final PhpType TRUE = builder().add(_TRUE).build();
  public static final PhpType INT = builder().add(_INT).build();
  public static final PhpType FLOAT = builder().add(_FLOAT).build();
  public static final PhpType OBJECT = builder().add(_OBJECT).build();
  public static final PhpType CLOSURE = builder().add(_CLOSURE).build();
  public static final PhpType CALLABLE = builder().add(_CALLABLE).build();
  public static final PhpType RESOURCE = builder().add(_RESOURCE).build();
  /**
   * Represents an array of elements with unknown type, {@link #elementType()} on this type will produce {@link #MIXED}.
   * Array of elements with specific types can be represented using {@link #pluralise()}.
   * Consider to use {@link #isArray(PhpType)} to check both conditions: whether the type is exactly this instance or plural one.
   */
  public static final PhpType ARRAY = builder().add(_ARRAY).build();
  public static final PhpType ITERABLE = builder().add(_ITERABLE).build();

  //aliases
  public static final PhpType NUMBER = builder().add(_NUMBER).build();

  public static final PhpType VOID = builder().add(_VOID).build();
  public static final PhpType NEVER = builder().add(_NEVER).build();

  public static final PhpType CLASS_STRING = builder().add(_CLASS_STRING).build();

  //complex types
  public static final PhpType NUMERIC = builder().add(STRING).add(INT).build();
  public static final PhpType SCALAR = builder().add(INT).add(FLOAT).add(STRING).add(BOOLEAN).add(FALSE).add(TRUE).build();
  public static final PhpType FLOAT_INT = builder().add(FLOAT).add(INT).build();

  public static final PhpType UNSET = builder().add("unset").build();
  public static final PhpType STATIC = builder().add(PhpClass.STATIC).build();
  public static final PhpType EXCEPTION = builder().add(_EXCEPTION).build();
  public static final PhpType THROWABLE = builder().add(_THROWABLE).build();
  public static final PhpType $THIS = builder().add(Variable.$THIS).build();

  public static final @NlsSafe String _TRAVERSABLE = "\\Traversable";
  public static final PhpType TRAVERSABLE = new PhpType().add(_TRAVERSABLE);
  public static final PhpType ARRAY_TRAVERSABLE_TYPE = new PhpType().add(_ARRAY).add(_TRAVERSABLE);
  public static final String EXCLUDED_INCOMPLETE_TYPE_SEPARATOR = "∆";
  public static final String INTERSECTION_TYPE_DELIMITER = "&";
  private static final char PARAMETRISED_TYPE_START = '<';
  private static final char PARAMETRISED_TYPE_END = '>';
  private boolean myTypeHasParametrizedParts = false;

  public PhpType createImmutableType() {
    return this instanceof ImmutablePhpType ? this : new ImmutablePhpType().addInternal(this);
  }

  @NotNull
  public PhpType map(@NotNull Function<@NotNull String, @Nullable String> typeMapper) {
    if (isEmpty()) {
      return this;
    }
    PhpType res = new PhpType();
    for (String type : getTypesWithParametrisedParts()) {
      res.add(typeMapper.apply(type));
    }
    return res;
  }

  public PhpType filterOutIntermediateTypes() {
    return filterOut(PhpIntermediateTypeProvider::providerExists);
  }

  public boolean hasIntersectionType() {
    return ContainerUtil.exists(getTypes(), PhpType::isIntersectionType);
  }

  public @NotNull PhpType removeParametrisedParts() {
    return map(PhpType::removeParametrisedType);
  }

  public boolean containsAll(@NotNull PhpType type) {
    return getTypes().containsAll(type.getTypes());
  }
  public static boolean isIntersectionType(String s) {
    return s.contains(INTERSECTION_TYPE_DELIMITER);
  }

  public static @NlsSafe String unpluralize(@NotNull  @NlsSafe String type, int dimension) {
    return type.substring(0, type.length() - dimension * 2);
  }

  public static @NlsSafe String pluralise(@NotNull @NlsSafe String type, int dimension) {
    return type + StringUtil.repeat("[]", dimension);
  }

  public static boolean isPrimitiveClassAccess(@NlsSafe String subject) {
    String originalSubject = subject;
    while (isSignedType(subject)) {
      char key = subject.charAt(1);
      if (PhpTypeSignatureKey.FIELD.is(key) || PhpTypeSignatureKey.METHOD.is(key)) {
        int i = subject.lastIndexOf('.');
        if (i < 0) {
          LOG.debug(String.format("Invalid subject: %s\nProcessed subject: %s", originalSubject, subject));
          return false;
        }
        subject = subject.substring(2, i);
      }
      else {
        break;
      }
    }
    return isSignedType(subject) && PhpTypeSignatureKey.CLASS.is(subject.charAt(1)) && isPrimitiveType(subject.substring(2));
  }

  public static @NotNull String unpluralize(@NotNull String type) {
    return unpluralize(type, getPluralDimension(type));
  }

  @ApiStatus.Internal
  public static boolean isResourceOrNumberType(@Nullable String typeName) {
    if (typeName == null) return false;
    String fqn = !typeName.startsWith("\\") ? "\\" + typeName : typeName;
    return _RESOURCE.equalsIgnoreCase(fqn) || _NUMBER.equalsIgnoreCase(fqn);
  }

  public static @NotNull String applyIntersectionTypesAware(@NotNull String fqn, Function<@NotNull String, @NotNull String> function) {
    if (fqn.contains(INTERSECTION_TYPE_DELIMITER)) {
      return StringUtil.split(fqn, INTERSECTION_TYPE_DELIMITER).stream()
        .map(function)
        .collect(Collectors.joining(INTERSECTION_TYPE_DELIMITER));
    } else {
      return function.apply(fqn);
    }
  }

  public static @NotNull PhpType createParametrized(@NotNull String baseName, String... parameters) {
    return from(createParametrizedType(baseName, parameters));
  }

  @NotNull
  public static String createParametrizedType(@NotNull String baseName, Collection<String> parameters) {
    if (parameters.isEmpty()) {
      return baseName;
    }
    if (!baseName.startsWith("\\") && !baseName.startsWith("#")) {
      baseName = "\\" + baseName;
    }
    return baseName + PARAMETRISED_TYPE_START + StringUtil.join(parameters, ",") + PARAMETRISED_TYPE_END;
  }


  @NotNull
  public static String createParametrizedType(@NotNull String baseName, String... parameters) {
    return createParametrizedType(baseName, Arrays.asList(parameters));
  }

  public static @Nullable Range<Integer> getIntRangeBounds(String type) {
    if (type != null && type.startsWith(_INT + PARAMETRISED_TYPE_START)) {
      List<String> parts = getParametrizedParts(type);
      if (parts.size() == 2) {
        Integer min = parseBound(parts.get(0));
        Integer max = parseBound(parts.get(1));
        return min != null && max != null ? new Range<>(min, max) : null;
      }
    }
    return null;
  }

  public static @NotNull List<String> getParametrizedParts(@NotNull String type) {
    if (!hasParameterizedPart(type)) {
      return Collections.emptyList();
    }
    int l = type.indexOf(PARAMETRISED_TYPE_START);
    int r = type.lastIndexOf(PARAMETRISED_TYPE_END);
    if (l >= 0 && r >= 0) {
      String parametrizedSubstring = type.substring(l + 1, r);
      parametrizedSubstring = escapeAllCommasInParametrizedParts(parametrizedSubstring);
      return ContainerUtil.map(StringUtil.split(parametrizedSubstring, ",", true, false), t -> t.replace(';', ','));
    }
    return Collections.emptyList();
  }

  private static String escapeAllCommasInParametrizedParts(String type) {
    Collection<Pair<Integer, Integer>> parametrizedParts = collectParametrizedParts(type);
    int i = -1;
    while ((i = type.indexOf(',', i)) >= 0) {
      int pos = i;
      if (ContainerUtil.exists(parametrizedParts, p -> pos > p.getFirst() && pos < p.getSecond())) {
        type = type.substring(0, i) + ';' + type.substring(i + 1);
      }
      i++;
    }
    return type;
  }

  private static Collection<Pair<Integer, Integer>> collectParametrizedParts(String type) {
    Collection<Pair<Integer, Integer>> res = new ArrayList<>();
    ArrayDeque<Integer> stack = new ArrayDeque<>();
    for (int i = 0; i < type.length(); i++) {
      if (type.charAt(i) == '<') {
        stack.addLast(i);
      }
      else if (type.charAt(i) == '>' && !stack.isEmpty()) {
        res.add(Pair.create(stack.pollFirst(), i));
      }
    }
    return res;
  }

  private static @Nullable Integer parseBound(String s) {
    if (s.equals("max")) {
      return Integer.MAX_VALUE;
    }
    else if (s.equals("min")) {
      return Integer.MIN_VALUE;
    }
    try {
      return Integer.parseInt(s);
    }
    catch (NumberFormatException e) {
      return null;
    }
  }

  public static boolean globalTypeEquals(PhpTypedElement f, PhpTypedElement s) {
    PhpType fType = f.getType();
    PhpType sType = s.getType();
    var fWithoutUnknown = replaceTrueFalseWithBoolean(fType.filterUnknown().getTypes());
    var sWithoutUnknown = replaceTrueFalseWithBoolean(sType.filterUnknown().getTypes());
    if (!fWithoutUnknown.isEmpty() &&
        !sWithoutUnknown.isEmpty() &&
        ContainerUtil.exists(fWithoutUnknown, t -> !sWithoutUnknown.contains(t)) ||
        ContainerUtil.exists(sWithoutUnknown, t -> !fWithoutUnknown.contains(t))) {
      return false;
    }

    Project project = f.getProject();
    return fType.hasUnresolved() ? globalTypeEquals(project, fType, sType) : globalTypeEquals(project, sType, fType);
  }

  public static boolean globalTypeEquals(PhpTypedElement element, PhpType globalType) {
    return globalTypeEquals(element.getProject(), element.getType(), globalType);
  }

  public static boolean globalTypeEquals(Project project, PhpType localType, PhpType globalType) {
    if (!globalType.hasUnresolved()) {
      PhpType typeWithoutUnknown = localType.filterUnknown();
      if (!replaceTrueFalseWithBoolean(globalType.getTypes()).containsAll(replaceTrueFalseWithBoolean(typeWithoutUnknown.getTypes()))) {
        return false;
      }
    }
    return localType.global(project).equals(globalType);
  }

  public static boolean isBooleanGlobal(Project project, PhpType type) {
    return !ContainerUtil.exists(type.filterUnknown().getTypes(), t -> !isBoolean(t)) && type.global(project).isBoolean();
  }

  public interface PhpTypeExclusion {
    boolean isNotApplicableType(@Nullable Project project, @NotNull String type);

    @NlsSafe String getCode();

    default boolean filterOnlyUnresolved() {
      return true;
    }

    default String filterIncompleteType(String type) {
      int i = 0;
      String suffixToAdd = EXCLUDED_INCOMPLETE_TYPE_SEPARATOR + getCode();
      while (type.startsWith("#-", i) ) {
        if (StringUtil.endsWith(type, 0, type.length() - i, suffixToAdd)) {
          return type;
        }
        i += 2;
      }
      return String.format("#-%s%s", type, suffixToAdd);
    }
  }

  public enum ExcludeCode implements PhpTypeExclusion {
    NOT_NULL("n") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isNull(type);
      }
    },
    NOT_PRIMITIVE("p") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return !isMixedType(type) && (isArray(type) || isNotExtendablePrimitiveType(type));
      }

      @Override
      public String filterIncompleteType(String type) {
        // These signature keys can't provide primitive type on resolving in any case
        // On the other hand there are some user code that makes decision based on #isSigned from these keys
        // so prepending exclude prefix will break this logic
        if (PhpTypeSignatureKey.SELF_CLASS.isSigned(type) ||
            PhpTypeSignatureKey.SELF_CLASS_IN_TRAIT.isSigned(type) ||
            PhpTypeSignatureKey.POLYMORPHIC_CLASS.isSigned(type)) {
          return type;
        }
        return super.filterIncompleteType(type);
      }
    }

    ,NOT_FALSE("f") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return _FALSE.equals(type);
      }
    }

    ,NOT_MIXED("m") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isMixedType(type);
      }
    }

    ,NOT_OBJECT("o") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isObject(type);
      }
    }

    , ONLY_PLURAL("s") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return !isPluralType(type);
      }
    }

    ,ONLY_NUMERIC("x") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return !NUMERIC.getTypes().contains(type);
      }
    }

    ;

    @NotNull
    private final String myCode;

    ExcludeCode(@NotNull String code) {
      myCode = code;
    }
    @Override
    public abstract boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type);

    @Override
    public @NotNull String getCode() {
      return myCode;
    }
  }

  @Nullable
  public static PhpType.PhpTypeExclusion fromCode(@NotNull String code) {
    return ContainerUtil.find(ExcludeCode.values(), v -> v.getCode().equals(code));
  }

  @Nullable
  private Set<String> types;
  private boolean isComplete = true;
  private boolean dirty = false;
  private @NlsSafe String myStringResolved;
  private @NlsSafe String myString;

  public PhpType() {
  }

  public boolean isBoolean() {
    return !isEmpty() && filterOut(PhpType::isBoolean).isEmpty();
  }

  private static boolean isBoolean(String s) {
    return _BOOL.equals(s) || _TRUE.equals(s) || _FALSE.equals(s);
  }

  public static boolean isArray(@Nullable PhpType type) {
    if (type == null) return false;
    if (type.isEmpty()) return false;
    return type.getTypes().stream().allMatch(t -> isArray(t) || isPluralType(t));
  }

  public static boolean hasArray(@Nullable PhpType type) {
    if (type == null) return false;
    return type.getTypes().stream().anyMatch(t -> isArray(t) || isPluralType(t));
  }

  public static boolean hasTypedArray(@Nullable PhpType type) {
    if (type == null) return false;
    return type.getTypes().stream().anyMatch(t -> isPluralType(t));
  }

  @NotNull
  public static PhpTypeBuilder builder() {
    return new PhpTypeBuilder();
  }

  @NotNull
  public PhpType add(@Nullable @NlsSafe String aClass) {
    if (aClass == null || aClass.length() <= 0) {
      return this;
    }

    if (aClass.length() > 1 && aClass.charAt(0) == '#') {
      isComplete = false;
    }

    String trimmed = aClass;
    while (StringUtil.endsWith(trimmed, "[]")) {
      trimmed = StringUtil.trimEnd(trimmed, "[]");
    }
    if ((isPrimitiveType(trimmed) || isResourceOrNumberType(trimmed)) && !aClass.startsWith("\\")) {
      aClass = "\\" + aClass; //intern will help here
    }
    if (aClass.equalsIgnoreCase(_INTEGER)) {
      aClass = _INT;
    }
    else if (aClass.equals(_STRING)) {
      aClass = _STRING;
    }
    else if (aClass.equalsIgnoreCase(_ARRAY)) {
      aClass = _ARRAY;
    }
    else if (aClass.equalsIgnoreCase(_BOOL)) {
      aClass = _BOOL;
    }
    else if (aClass.equalsIgnoreCase(_MIXED)) {
      aClass = _MIXED;
    }
    else if (aClass.equalsIgnoreCase(_BOOLEAN)) {
      aClass = _BOOL;
    }
    else if (aClass.equalsIgnoreCase(_CALLBACK)) {
      aClass = _CALLABLE;
    }
    else if (aClass.equalsIgnoreCase(_ITERABLE)) {
      aClass = _ITERABLE;
    }
    else if (aClass.equalsIgnoreCase(_DOUBLE)) {
      aClass = _FLOAT;
    }
    if (types == null) {
      types = createEmptyTypesSet();
    }
    if (types.size() > 50 && getTypes().size() > 50) {
      if (ApplicationManager.getApplication().isInternal()) {
        LOG.trace("too much type variants: " + getTypes());
      }
    }
    else {
      if (!myTypeHasParametrizedParts && hasParameterizedPart(aClass)) {
        myTypeHasParametrizedParts = true;
      }
      types.add(aClass);
    }
    dirty = true;
    return this;
  }

  @NotNull
  public PhpType add(@Nullable PsiElement other) {
    if(other instanceof PhpTypedElement) {
      final PhpType type = ((PhpTypedElement)other).getType();
      add(type);
    }
    return this;
  }

  public @NotNull PhpType add(PhpType type) {
    if (type == null || type.isEmpty()) {
      return this;
    }

    Set<@NlsSafe String> parts = type.getTypesWithParametrisedParts();
    if (!myTypeHasParametrizedParts && ContainerUtil.exists(parts, PhpType::hasParameterizedPart)) {
      myTypeHasParametrizedParts = true;
    }
    try {
      isComplete &= type.isComplete;
      if (types == null) {
        types = createTypesSet(parts);
      }
      else {
        types.addAll(parts);
        if (types.size() > 50 && getTypes().size() > 50 && ApplicationManager.getApplication().isInternal()) {
          LOG.trace("too much type variants: " + getTypesWithParametrisedParts());
        }
      }
      dirty = true;
    }
    catch (NoSuchElementException e) {
      throw new RuntimeException("NSEE @" + parts, e);
    }
    return this;
  }

  public int size() {
    return getTypes().size();
  }

  public boolean isAmbiguous() {
    return isEmpty() || hasUnknown() || intersects(this, MIXED);
  }

  /**
   * @return UNORDERED set of contained type literals
   */
  @NotNull
  public Set<@NlsSafe String> getTypes() {
    return sanitizeTypes();
  }

  @NotNull
  private Set<@NlsSafe String> sanitizeTypes() {
    if (myTypeHasParametrizedParts) {
      return sanitizeTypes(types);
    } else {
      return removeFalseAndTrueIfNeeded(types);
    }
  }

  @NotNull
  public Set<@NlsSafe String> getTypesWithParametrisedParts() {
    return removeFalseAndTrueIfNeeded(types);
  }

  @ApiStatus.Internal @NotNull
  public Stream<@NlsSafe String> types() {
    return sanitizeTypes().stream();
  }

  @NotNull
  public Set<Collection<@NlsSafe String>> getTypesIntersections() {
    return getTypesWithParametrisedParts().stream()
      .map(t -> new HashSet<>(StringUtil.split(t, INTERSECTION_TYPE_DELIMITER)))
      .collect(Collectors.toSet());
  }

  @NotNull
  private static Set<@NlsSafe String> sanitizeTypes(Set<@NlsSafe String> types) {
    if (types == null || types.isEmpty()) {
      return Collections.emptySet();
    }
    boolean sorted = types instanceof SortedSet;
    types = types.stream()
      .map(PhpType::convertTypedArrayToPluralOrSanitize)
      .map(PhpType::convertClassStringToString)
      .collect(Collectors.toCollection(() -> sorted ? createSortedSet() : createEmptyTypesSet()));
    return removeFalseAndTrueIfNeeded(types);
  }

  public static boolean hasParameterizedPart(@NotNull String type) {
    if (isPluralType(type)) {
      return hasParameterizedPart(unpluralize(type));
    }
    int startIndex = type.indexOf(PARAMETRISED_TYPE_START);
    return startIndex >= 0 && StringUtil.endsWithChar(type, PARAMETRISED_TYPE_END);
  }

  @NotNull
  public static String removeParametrisedType(String t) {
    int l = t.indexOf(PARAMETRISED_TYPE_START);
    int r = t.lastIndexOf(PARAMETRISED_TYPE_END);
    return l >= 0 && r >= 0 ? t.substring(0, l) + t.substring(r + 1) : t;
  }

  @NotNull
  private static Set<String> removeFalseAndTrueIfNeeded(Set<@NlsSafe String> types) {
    if(types != null) {
      if ((types.contains(_FALSE) || types.contains(_TRUE)) && types.contains(_BOOL) ||
          types.contains(_FALSE) && types.contains(_TRUE)) {
        return replaceTrueFalseWithBoolean(types);
      }
      return types;
    } else return Collections.emptySet();
  }

  @NotNull
  private static Set<String> replaceTrueFalseWithBoolean(Set<@NlsSafe String> types) {
    Set<String> typesCopy = createTypesSet(types);
    typesCopy.remove(_FALSE);
    typesCopy.remove(_TRUE);
    typesCopy.add(_BOOL);
    return typesCopy;
  }

  /**
   * @return more costly, ORDERED set of contained type literals (cached), should be avoided
   * TODO - factor usages out?
   */
  public @NotNull Set<@NlsSafe String> getTypesSorted() {
    if (types == null) {
      return Collections.emptySet();
    }
    else {
      sortIfNeeded();
      assert types != null;
      return sanitizeTypes();
    }
  }

  private @NotNull Set<@NlsSafe String> getTypesWithParametrisedPartsSorted() {
    if (types == null) {
      return Collections.emptySet();
    }
    else {
      sortIfNeeded();
      assert types != null;
      return removeFalseAndTrueIfNeeded(types);
    }
  }

  public String toString() {
    if(types==null) return "";
    if(!dirty && myString!=null) return myString;
    String typesString = sortedResolvedTypes().collect(Collectors.joining("|"));
    return myString = isComplete ? typesString : typesString + "|?";
  }

  /**
   * @param type string to trim-root-ns-if-required
   * @return trimmed string
   */
  public static String toString(@NlsSafe String type) {
    type = convertTypedArrayToPluralOrSanitize(type);
    String unpluralized = StringUtil.trimEnd(type, "[]");
    if ((isPrimitiveType(unpluralized) || isResourceOrNumberType(unpluralized)) &&
        type.startsWith("\\")) {
      return type.substring(1);
    }
    return type;
  }

  @NotNull
  private static String convertTypedArrayToPluralOrSanitize(String type) {
    String s = removeParametrisedType(type);
    if ("array".equals(StringUtil.trimStart(s, "\\"))) {
      List<String> parts = getParametrizedParts(type);
      if (!parts.isEmpty()) {
        String elementType = removeParametrisedType(ContainerUtil.getLastItem(parts));
        if (StringUtil.trimStart(elementType, "\\").equals("mixed")) {
          return _ARRAY;
        }
        return elementType + "[]";
      }
    }
    return s;
  }

  @NotNull
  private static String convertClassStringToString(String type) {
    String s = removeParametrisedType(type);
    if ("\\class-string".equals(s)) {
      return _STRING;
    }
    return s;
  }

  private void sortIfNeeded() {
    if (types != null && !(types instanceof SortedSet)) {
      Set<String> sorted = createSortedSet();
      sorted.addAll(getTypesWithParametrisedParts());
      types = sorted;
    }
  }

  @NotNull
  private static TreeSet<String> createSortedSet() {
    Comparator<String> comparator = Comparator.comparing(PhpType::removeParametrisedType, String.CASE_INSENSITIVE_ORDER)
      .thenComparing(String.CASE_INSENSITIVE_ORDER);
    return new TreeSet<>(comparator);
  }

  /**
   * NOTE: apply to .global() type only!
   * @return presentable name
   */
  public @NlsSafe String toStringResolved() {
    //PhpContractUtil.assertCompleteType(this);
    if (!dirty && myStringResolved != null) {
      return myStringResolved;
    }
    myStringResolved = toStringRelativized(null);
    return myStringResolved;
  }

  /**
   * NOTE: apply to .global() type only!
   * @return relativized presentable name
   */
  public @NlsSafe String toStringRelativized(@Nullable @NlsSafe final String currentNamespaceName) {
    //PhpContractUtil.assertCompleteType(this);
    if (types == null) {
      return "";
    }
    final StringBuilder builder = new StringBuilder();
    for (String type : getSortedTypesStrings(currentNamespaceName)) {
      builder.append(type).append('|');
    }
    if (builder.length() > 1) {
      builder.setLength(builder.length() - 1);
    }
    if (!isComplete) {
      builder.append("|?");
    }
    return builder.toString();
  }

  @NotNull
  public Collection<@NlsSafe String> getSortedTypesStrings(@Nullable @NlsSafe String currentNamespaceName) {
    return sortedResolvedTypes()
      .map(t -> applyIntersectionTypesAware(t, t1 -> trimNamespace(currentNamespaceName, t1)))
      .collect(Collectors.toList());
  }

  private static String trimNamespace(@Nullable @NlsSafe String currentNamespaceName, @NlsSafe String t) {
    return currentNamespaceName != null && t.startsWith(currentNamespaceName) ? t.substring(currentNamespaceName.length()) : t;
  }

  @NotNull
  private Stream<@NlsSafe String> sortedResolvedTypes() {
    return getTypesWithParametrisedPartsSorted().stream()
      .filter(t -> !startsWithChar(t, '?'))
      .map(PhpType::toString)
      .distinct();
  }

  public boolean isConvertibleFromGlobal(Project project, @NotNull PhpType type) {
    if (isConvertibleLocal(project, type)) return true;
    PhpType globalType = global(project);
    return globalType.isAmbiguous() || globalType.isConvertibleFrom(type.global(project), PhpIndex.getInstance(project));
  }

  public boolean isConvertibleLocal(Project project, @NotNull PhpType type) {
    if (hasUnknown() || hasUnresolved() || type.hasUnknown() || type.hasUnresolved()) {
      PhpType t1 = filterUnknown();
      PhpType t2 = type.filterUnknown();
      if (!t1.isEmpty() && !t2.isEmpty() && t1.isConvertibleFrom(t2, PhpIndex.getInstance(project))) {
        return true;
      }
    }
    return false;
  }

  @ApiStatus.Internal
  public boolean isConvertibleFrom(Project project, @NotNull PhpType otherType) {
    return isConvertibleFrom(otherType, PhpIndex.getInstance(project));
  }

  public boolean isConvertibleFrom(@NotNull PhpType otherType, @NotNull PhpIndex index) {
    if(isAmbiguous() || otherType.isAmbiguous()) return true;
    if(this.equals(NULL)) return true;
    Set<String> otherTypes = sanitizeTypes(otherType.types);
    Set<String> myTypes = sanitizeTypes();
    if (ContainerUtil.intersects(otherTypes, myTypes)) {
      return true;
    }
    if (otherTypes.contains(_ITERABLE) && myTypes.contains(_ARRAY) && myTypes.contains(_TRAVERSABLE)) {
      return true;
    }
    for (String other : otherTypes) {
      for (final String my : myTypes) {
        if (isConvertibleFrom(my, other, index)) return true;
      }
    }
    //System.out.println("\tisConvertibleFrom == false");
    return false;
  }

  private static @Nullable Boolean isConvertibleIntersectionAware(String mySuper,
                                                                  String otherChild,
                                                                  BiFunction<String, String, Boolean> convertibilityFunction) {
    if (mySuper.contains(INTERSECTION_TYPE_DELIMITER)) {
      return ContainerUtil.all(StringUtil.split(mySuper, INTERSECTION_TYPE_DELIMITER),
                               intersectedSuper -> convertibilityFunction.apply(intersectedSuper, otherChild));
    }
    else if (otherChild.contains(INTERSECTION_TYPE_DELIMITER)) {
      return ContainerUtil.exists(StringUtil.split(otherChild, INTERSECTION_TYPE_DELIMITER),
                                  intersectedChild -> convertibilityFunction.apply(mySuper, intersectedChild));
    }
    else {
      return null;
    }
  }

  private static boolean isConvertibleFrom(String mySuper, String otherChild, @NotNull PhpIndex index) {
    BiFunction<String, String, Boolean> convertibilityFunction = (superType, childType) -> isConvertibleFrom(superType, childType, index);
    Boolean convertibleIntersectionAware = isConvertibleIntersectionAware(mySuper, otherChild, convertibilityFunction);
    if (convertibleIntersectionAware != null) {
      return convertibleIntersectionAware;
    }
    if (isPluralType(mySuper) && isPluralType(otherChild)) {
      boolean elementTypesConvertible =
        new PhpType().add(mySuper).unpluralize().isConvertibleFrom(new PhpType().add(otherChild).unpluralize(), index);
      if (elementTypesConvertible) return true;
    }
    if (mySuper.equalsIgnoreCase(otherChild)
        || otherChild.equalsIgnoreCase(_NEVER)
        || isPluralType(mySuper) && otherChild.equalsIgnoreCase(_ARRAY)
        || isPluralType(otherChild) && (mySuper.equalsIgnoreCase(_ARRAY) || mySuper.equalsIgnoreCase(_ITERABLE))
        || ((mySuper.equalsIgnoreCase(_STRING) || mySuper.equalsIgnoreCase(_STRINGABLE) && !isNotExtendablePrimitiveType(otherChild)) &&
            !otherChild.equalsIgnoreCase(_ARRAY) && !isPluralType(otherChild) && !nonPrimitiveWithoutToString(otherChild, index))
        || (otherChild.equalsIgnoreCase(_STRING) && mySuper.equalsIgnoreCase(_CALLABLE))
        || (otherChild.equalsIgnoreCase(_ARRAY) && mySuper.equalsIgnoreCase(_CALLABLE))
        || (isPluralType(otherChild) && mySuper.equalsIgnoreCase(_CALLABLE))
        || (otherChild.equalsIgnoreCase(_STRING) && mySuper.equalsIgnoreCase(_INT))
        || (otherChild.equalsIgnoreCase(_STRING) && mySuper.equalsIgnoreCase(_FLOAT))
        || (otherChild.equalsIgnoreCase(_STRING) && mySuper.equalsIgnoreCase(_BOOL))
        || (otherChild.equalsIgnoreCase(_STRING) && mySuper.equalsIgnoreCase(_FALSE))
        || mySuper.equalsIgnoreCase(_ITERABLE) && otherChild.equalsIgnoreCase(_ARRAY)
        || mySuper.equalsIgnoreCase(_ITERABLE) && isPluralType(otherChild)
        || isBidi(mySuper, otherChild, _TRUE, _BOOL)
        || isBidi(mySuper, otherChild, _FALSE, _BOOL)
        || isBidi(mySuper, otherChild, _INT, _FLOAT)
        || isBidi(mySuper, otherChild, _BOOL, _INT)
        || isBidi(mySuper, otherChild, _FALSE, _INT)
        || isBidi(mySuper, otherChild, _TRUE, _INT)
        || isBidi(mySuper, otherChild, _BOOL, _FLOAT)
        || isBidi(mySuper, otherChild, _FALSE, _FLOAT)
        || isBidi(mySuper, otherChild, _TRUE, _FLOAT)
        || isBidi(mySuper, otherChild, _NUMBER, _FLOAT)
        || isBidi(mySuper, otherChild, _NUMBER, _INT)
        || isBidi(mySuper, otherChild, _OBJECT, "\\stdClass")
        || mySuper.equalsIgnoreCase(_STRING) && otherChild.equalsIgnoreCase(_CLASS_STRING)
        || mySuper.equalsIgnoreCase(_CLASS_STRING) && otherChild.equalsIgnoreCase(_STRING)
      ) return true;
    if (!mySuper.equalsIgnoreCase(_CALLABLE) && !otherChild.equalsIgnoreCase(_CALLABLE) && isPrimitiveType(mySuper) && isPrimitiveType(otherChild)) {
      return false;
    }
    if (findSuper(mySuper, otherChild, index) || isSuperWithOnlyPolymorphicTarget(mySuper, otherChild, index)) return true;
    if (!isPluralType(otherChild) && !isPrimitiveType(otherChild) && _OBJECT.equalsIgnoreCase(mySuper)) return true;
    if (mySuper.equalsIgnoreCase(_CALLABLE) && checkInvoke(otherChild, index)
        || otherChild.equalsIgnoreCase(_CALLABLE) && checkInvoke(mySuper, index)) {
      return true;
    }
    return false;
  }

  private static boolean isSuperWithOnlyPolymorphicTarget(String my, String other, @NotNull PhpIndex index) {
    Collection<PhpClass> classes = index.getAnyByFQN(other);
    if (!classes.isEmpty() && ContainerUtil.and(classes, PhpClass::isAbstract)) {
      Collection<PhpClass> subClasses = new HashSet<>();
      index.processAllSubclasses(other, new HashSet<>(), c -> {
        if (!c.isAbstract()) {
          subClasses.add(c);
        }
        return subClasses.size() <= 1;
      });
      PhpClass onlySubClass = ContainerUtil.getOnlyItem(subClasses);
      return onlySubClass != null && findSuper(my, onlySubClass.getFQN(), index);
    }
    return false;
  }

  @ApiStatus.Internal
  public static boolean nonPrimitiveWithoutToString(String fqn, @NotNull PhpIndex index) {
    if (isPluralType(fqn) || isNotExtendablePrimitiveType(fqn)) {
      return false;
    }
    Collection<PhpClass> classes = index.getAnyByFQN(fqn);
    return !classes.isEmpty() && !hasToString(classes) && !hasToStringInSubclasses(fqn, index, new HashSet<>(classes));
  }

  private static boolean hasToStringInSubclasses(String fqn, PhpIndex phpIndex, HashSet<PhpClass> alreadyProcessedSubClasses) {
    Ref<Boolean> hasToStringInSubclasses = new Ref<>(false);
    phpIndex.processAllSubclasses(fqn, phpClass -> {
      if (phpClass.findMethodByName(PhpClass.TO_STRING, alreadyProcessedSubClasses) != null) {
        hasToStringInSubclasses.set(true);
        return false;
      }
      alreadyProcessedSubClasses.add(phpClass);
      return true;
    });
    return hasToStringInSubclasses.get();
  }

  private static boolean hasToString(Collection<PhpClass> classes) {
    return ContainerUtil.exists(classes, c -> c.findMethodByName(PhpClass.TO_STRING) != null);
  }

  private static boolean isBidi(@NlsSafe String my, @NlsSafe String other, @NlsSafe String type1, @NlsSafe String type2) {
    return (my.equalsIgnoreCase(type1) && other.equalsIgnoreCase(type2)) ||
           (other.equalsIgnoreCase(type1) && my.equalsIgnoreCase(type2));
  }

  @NotNull
  public PhpType filterUnknown() {
    return filterOut(s -> s.startsWith("#") || s.startsWith("?"));
  }

  @NotNull
  public PhpType filterPrimitives() {
    return filterOut(PhpType::isPrimitiveType);
  }

  public boolean hasUnknown() {
    for (String type : getTypes()) {
      if (StringUtil.startsWith(type, "?")) {
        return true;
      }
    }
    return false;
  }

  private static boolean checkInvoke(@NotNull @NlsSafe String some, @NotNull PhpIndex index) {
    final Collection<PhpClass> candidates = index.getAnyByFQN(some);
    for (PhpClass candidate : candidates) {
      if(candidate.findMethodByName(PhpClass.INVOKE)!=null) return true;
    }
    return false;
  }

  public static boolean findSuper(@NotNull @NlsSafe String mySuper, @Nullable @NlsSafe String otherChild, @NotNull PhpIndex index) {
    return findSuper(mySuper, otherChild, index, new HashSet<>());
  }

  @ApiStatus.Internal
  public static boolean findSuper(@NotNull @NlsSafe String mySuper,
                                  @Nullable @NlsSafe String otherChild,
                                  @NotNull PhpIndex index,
                                  Collection<PhpClass> visited) {
    if (otherChild == null) return false;
    BiFunction<String, String, Boolean> convertibilityFunction = (superType, childType) -> findSuper(superType, childType, index);
    Boolean convertibleIntersectionAware = isConvertibleIntersectionAware(mySuper, otherChild, convertibilityFunction);
    if (convertibleIntersectionAware != null) {
      return convertibleIntersectionAware;
    }
    if (mySuper.endsWith("[]") != otherChild.endsWith("[]")) return false;
    mySuper = StringUtil.trimEnd(mySuper, "[]");
    otherChild = StringUtil.trimEnd(otherChild, "[]");
    if (!mySuper.startsWith("\\")) mySuper = "\\" + mySuper;
    if (!otherChild.startsWith("\\")) otherChild = "\\" + otherChild;
    if (mySuper.equalsIgnoreCase(otherChild)) {
      return true;
    }
    Collection<PhpClass> mes = index.getAnyByFQN(mySuper);
    if (mes.isEmpty()) {
      return false;
    }
    Collection<PhpClass> childClasses = index.getAnyByFQN(otherChild);
    if (ContainerUtil.all(mes, PhpClass::isFinal)) {
      return ContainerUtil.intersects(mes, childClasses);
    }
    Ref<Boolean> result = new Ref<>(false);
    return ContainerUtil.exists(childClasses, phpClass -> {
      PhpClassHierarchyUtils.processSuperWithoutMixins(phpClass, true, true, aSuper -> {
        for (final PhpClass me : mes) {
          if (PhpClassHierarchyUtils.classesEqual(me, aSuper)) {
            result.set(true);
            break;
          }
        }
        return !result.get();
      }, visited);
      return result.get();
    });
  }

  /**
   * @return type info is complete from all possible sources/is incomplete - non local stuff omitted. Defaults to true.
   */
  public boolean isComplete() {
    return isComplete;
  }

  /**
   * @param context - important for resolution of SELF, THIS, STATIC
   * @deprecated just use .global
   * @return resolved type
   */
  @Deprecated(forRemoval = true)
  public PhpType globalLocationAware(@NotNull PsiElement context) {
    return global(context.getProject());
  }

  public PhpType global(@NotNull Project p) {
    try {
      return PhpIndex.getInstance(p).completeType(p, this, null);
    }
    catch (StackOverflowError e) {
      if (ApplicationManager.getApplication().isUnitTestMode()) {
        throw e;
      } else {
        LOG.warn("SOE in PhpType.global @ " + this);
      }
      return EMPTY;
    }
  }

  public boolean isEmpty() {
    return types == null || types.isEmpty();
  }

  @NotNull
  public PhpType elementType() {
    return elementType(PhpTypeSignatureKey.ARRAY_ELEMENT);
  }

  @ApiStatus.Internal
  @NotNull
  public PhpType elementType(PhpTypeKey typeKey) {
    final PhpType elementType = new PhpType();
    PhpType keysType = new PhpType();
    for (String type : getTypesWithParametrisedParts()) {
      if (PhpKeyTypeProvider.isArrayKeySignature(type)) {
        String unsignedKeyType = type.substring(2);
        if (PhpKeyTypeProvider.isArrayKeySignature(unsignedKeyType)) {
          keysType.add(unsignedKeyType);
        }
        continue;
      }
      String base = removeParametrisedType(type);
      List<String> parts = getParametrizedParts(type);
      if (base.equals(_ARRAY) && parts.size() == 2) {
        elementType.add(ContainerUtil.getLastItem(parts));
        continue;
      }
      if (isPrimitiveClassAccess(type)) {
        elementType.add(_MIXED);
        continue;
      }
      if (type.equalsIgnoreCase(_ARRAY)) {
        elementType.add(MIXED);
      }
      else if (isPluralType(type) && !type.contains("#π")) {
        elementType.add(type.substring(0, type.length() - 2));
      }
      else if (isPrimitiveType(type)) {
        if (type.equals(_STRING)) {
          elementType.add(STRING);
        }
        else {
          elementType.add(MIXED);
        }
      }
      else {
        elementType.add(typeKey.sign(PhpTypeSignatureKey.CLASS.signIfUnsigned(type)));
      }
    }
    if (elementType.isEmpty()) elementType.add(_MIXED);
    return elementType.add(keysType);
  }

  @NotNull
  public PhpType pluralise() {
    return pluralise(1);
  }

  public PhpType pluralise(int c) {
    if (c == 0) {
      return this;
    }
    PhpType elementType = new PhpType();
    for (String type : getTypesWithParametrisedParts()) {
      String pluralise = pluraliseMixedAware(type, c);
      elementType.add(pluralise);
    }
    if (elementType.isEmpty()) {
      elementType.add(ARRAY);
    }
    return elementType;
  }

  @NotNull
  public static String pluraliseMixedAware(String type, int c) {
    return _MIXED.equals(type) ? pluralise(_ARRAY, c - 1) : pluralise(type, c);
  }

  @NotNull
  public PhpType unpluralize() {
    if (getTypes().isEmpty()) return getEmpty();
    final PhpType unpluralized = new PhpType();
    for (final String type : getTypesWithParametrisedParts()) {
      unpluralized.add(unpluralizeType(type));
    }
    return unpluralized;
  }

  @NotNull
  private static String unpluralizeType(String type) {
    String s = removeParametrisedType(type);
    List<String> parts = getParametrizedParts(type);
    if (_ARRAY.equals(s) && !parts.isEmpty()) {
      return ContainerUtil.getLastItem(parts);
    }
    return _ARRAY.equalsIgnoreCase(type) ? _MIXED : StringUtil.trimEnd(type, "[]");
  }

  public static boolean isPrimitiveType(@Nullable String type) {
    if(type == null) return true;
    if(type.length() < 3 || type.length() > 11) return false;
    if(type.charAt(0) == '#') return false;
    if(!type.startsWith("\\")) type = "\\" + type;
    return isNotExtendablePrimitiveType(type) ||
           isArray(type) ||
           _OBJECT.equalsIgnoreCase(type) ||
           _CALLABLE.equalsIgnoreCase(type) ||
           _ITERABLE.equalsIgnoreCase(type);
  }

  /**
   * Some primitive types are not exclusive, e.g. variable can be an {@code array} and {@code ArrayAccess} class at the same time
   * or {@code callable} and any other class type with implemented {@code __invoke} method. <br/><br/>
   *
   * This method is indented to check if type consists of exclusive primitive types,
   * i.e. types that can't be extended by class types
   * @return {@code true} if type consists of exclusive primitive types, {@code false} otherwise
   */
  public boolean isNotExtendablePrimitiveType() {
    if (isEmpty()) return false;
    return ContainerUtil.and(getTypes(), PhpType::isNotExtendablePrimitiveType);
  }

  public static boolean isNotExtendablePrimitiveType(@Nullable @NlsSafe String type) {
    if(type == null) return true;
    if(type.length() < 3 || type.length() > 12) return false;
    if(type.charAt(0) == '#') return false;
    if(!type.startsWith("\\")) type = "\\" + type;
    return
      _MIXED.equalsIgnoreCase(type) ||
      _STRING.equalsIgnoreCase(type) ||
      _CLASS_STRING.equalsIgnoreCase(type) ||
      _INT.equalsIgnoreCase(type) ||
      _INTEGER.equalsIgnoreCase(type) ||
      _BOOL.equalsIgnoreCase(type) ||
      _BOOLEAN.equalsIgnoreCase(type) ||
      _TRUE.equalsIgnoreCase(type) ||
      _FALSE.equalsIgnoreCase(type) ||
      _FLOAT.equalsIgnoreCase(type) ||
      _NULL.equalsIgnoreCase(type) ||
      _VOID.equalsIgnoreCase(type) ||
      _NEVER.equalsIgnoreCase(type) ||
      _DOUBLE.equalsIgnoreCase(type)
      ;
  }

  public static boolean isArray(@NotNull final @NlsSafe String type){
    return _ARRAY.equals(removeParametrisedType(type));
  }

  public static boolean isString(@NotNull final @NlsSafe String type){
    return _STRING.equals(type);
  }

  public static boolean isObject(@NotNull @NlsSafe String type){
    return _OBJECT.equals(type);
  }

  public static boolean isMixedType(@NotNull @NlsSafe String type){
    return _MIXED.equals(type);
  }

  public static boolean isCallableType(@NotNull @NlsSafe String type){
    return _CALLABLE.equals(type);
  }

  public static boolean isPluralType(@NotNull @NlsSafe String type){
    return type.endsWith("[]");
  }

  public static int getPluralDimension(@NotNull @NlsSafe String type) {
    int res = 0;
    while (StringUtil.endsWith(type, 0, type.length() - res * 2, "[]")) {
      res++;
    }
    return res;
  }

  public static boolean isPluralPrimitiveType(@NotNull @NlsSafe String type){
    return type.endsWith("[]") && isPrimitiveType(type.substring(0, type.length()-2));
  }

  public static boolean isAnonymousClass(@Nullable @NlsSafe String type) {
    return type != null && StringUtil.startsWith(type, PhpClass.ANONYMOUS);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof PhpType phpType)) return false;

    return isComplete == phpType.isComplete && Objects.equals(getTypes(), phpType.getTypes());
  }

  @Override
  public int hashCode() {
    int result = computeHashCode(getTypes());
    result = 31 * result + (isComplete ? 1 : 0);
    return result;
  }

  private static int computeHashCode(@NotNull Set<@NlsSafe String> types) {
    int result = 0;
    for (final String type : types) {
      result += HashingStrategy.caseInsensitiveCharSequence().hashCode(type); // the same as THashSet does
    }
    return result;
  }

  public static boolean isUnresolved(@NotNull @NlsSafe String type){
    return type.indexOf('#')!=-1;
  }

  public static boolean isNull(@NotNull @NlsSafe String type){
    return _NULL.equalsIgnoreCase(type);
  }

  @Contract(pure = true)
  public boolean isNullable() {
    return getTypes().stream().anyMatch(PhpType::isNull);
  }

  private static boolean isScalar(@NotNull @NlsSafe String type){
    return _BOOL.equals(type)     ||
           _BOOLEAN.equals(type)  ||
           _FLOAT.equals(type)    ||
           _STRING.equals(type)   ||
           _INT.equals(type)      ||
           _INTEGER.equals(type);
  }

  public static boolean isScalar(@NotNull final PhpType type, @NotNull final Project project){
    final PhpType completedType = type.global(project);
    for(String curType : completedType.getTypes()){
      if(!isScalar(curType)){
        return false;
      }
    }
    return true;
  }

  public boolean hasUnresolved() {
    for (String type : getTypes()) {
      if (isSignedType(type)) {
        return true;
      }
    }
    return false;
  }

  public static boolean isSignedType(@Nullable String type) {
    return startsWithChar(type, '#');
  }

  public static boolean intersectsGlobal(Project project, @NotNull PhpType f, @NotNull PhpType s) {
    return intersectsLocal(f, s) || intersects(f.global(project), s.global(project));
  }

  private static boolean intersectsLocal(@NotNull PhpType f, @NotNull PhpType s) {
    if (f.hasUnknown() || f.hasUnresolved() || s.hasUnknown() || s.hasUnresolved()) {
      PhpType f1 = f.filterUnknown();
      PhpType s1 = s.filterUnknown();
      if (!f1.isEmpty() && !s1.isEmpty() && intersects(f1, s1)) {
        return true;
      }
    }
    return false;
  }

  public static boolean intersects(@NotNull PhpType phpType1,@NotNull PhpType phpType2){
    //PhpContractUtil.assertCompleteType(phpType1, phpType2);
    final Set<String> phpTypeSet1 = phpType1.getTypes();
    final Set<String> phpTypeSet2 = phpType2.getTypes();
    for(String type1 : phpTypeSet1){
      if (phpTypeSet2.contains(type1) ||
          (_FALSE.equals(type1) || _TRUE.equals(type1)) && phpTypeSet2.contains(_BOOL) ||
          _BOOL.equals(type1) && (phpTypeSet2.contains(_FALSE) || phpTypeSet2.contains(_TRUE))) {
        return true;
      }
    }
    return false;
  }

  public static boolean isSubType(@NotNull PhpType phpType1, @NotNull PhpType phpType2){
    //PhpContractUtil.assertCompleteType(phpType1, phpType2);
    final Set<String> typeSet1 = phpType1.getTypes();
    if(typeSet1.size() == 0){
      return false;
    }
    final Set<String> typeSet2 = phpType2.getTypes();
    if(typeSet2.size() == 0){
      return false;
    }
    for(String type1 : typeSet1){
      if ((_FALSE.equals(type1) || _TRUE.equals(type1)) && typeSet2.contains(_BOOL)) {
        continue;
      }
      if(!typeSet2.contains(type1)){
        return false;
      }
    }
    return true;
  }

  @NotNull
  public static PhpType and(@NotNull PhpType phpType1, @NotNull PhpType phpType2) {
    final Set<String> phpTypeSet1 = phpType1.getTypesWithParametrisedParts();
    final Set<String> phpTypeSet2 = phpType2.getTypesWithParametrisedParts();
    final PhpType phpType = new PhpType();
    for (String type1 : phpTypeSet1) {
      if (phpTypeSet2.contains(type1)) {
        phpType.add(type1);
      }
    }
    return phpType;
  }

  @NotNull
  public static PhpType or(@Nullable PhpType phpType1, @Nullable PhpType phpType2) {
    if (phpType1 == null || phpType1.isEmpty()) {
      return ObjectUtils.notNull(phpType2, EMPTY);
    }
    if (phpType2 == null || phpType2.isEmpty()) {
      return ObjectUtils.notNull(phpType1, EMPTY);
    }
    return new PhpType().add(phpType1).add(phpType2);
  }

  @NotNull
  public PhpType filterNull() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_NULL);
  }

  public PhpType filterScalarPrimitives() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_PRIMITIVE);
  }

  public PhpType filterFalse() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_FALSE);
  }

  @ApiStatus.Internal
  public PhpType filterOutIncompleteTypesAware(PhpTypeExclusion excludeCodeForIncompleteType) {
    if (isEmpty()) {
      return this instanceof ImmutablePhpType ? getEmpty() : new PhpType();
    }
    final PhpType phpType = new PhpType();
    for (String type : getTypesWithParametrisedParts()) {
      boolean isUnresolved = isSignedType(type);
      if (!isUnresolved && excludeCodeForIncompleteType.isNotApplicableType(null, removeParametrisedType(type))) {
        continue;
      }
      if ((!excludeCodeForIncompleteType.filterOnlyUnresolved() || isUnresolved) && !StringUtil.containsChar(type, '|')) {
        phpType.add(excludeCodeForIncompleteType.filterIncompleteType(type));
      } else {
        phpType.add(type);
      }
    }
    return phpType;
  }

  protected PhpType getEmpty() {
    return new PhpType();
  }

  @NotNull
  public PhpType filterMixed() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_MIXED);
  }

  @ApiStatus.Internal
  public PhpType filterObject() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_OBJECT);
  }

  @NotNull
  public PhpType filterPlurals() {
    return filterOut(PhpType::isPluralType);
  }

  @NotNull
  public PhpType filterOut(@NotNull Predicate<String> typeExcludePredicate) {
    return filterOut(typeExcludePredicate, false);
  }

  @NotNull
  public PhpType filterOut(@NotNull Predicate<String> typeExcludePredicate, boolean checkParametrisedParts) {
    if(isEmpty()) return getEmpty();
    final PhpType phpType = new PhpType();
    for (String type : getTypesWithParametrisedParts()) {
      if(!typeExcludePredicate.test(removeParametrisedType(type)) || checkParametrisedParts && !typeExcludePredicate.test(type)) {
        phpType.add(type);
      }
    }
    return phpType;
  }

  @NotNull
  public PhpType filter(@NotNull final PhpType sieve) {
    if (isEmpty()) return getEmpty();
    final Set<String> sieveTypes = sieve.getTypes();
    if (ContainerUtil.isEmpty(sieveTypes)) return new PhpType().add(this);
    final PhpType phpType = new PhpType();
    if (sieveTypes.size() == 1) {
      final String sieveType = ContainerUtil.getFirstItem(sieveTypes);
      assert sieveType != null;
      getTypesWithParametrisedParts().stream().filter(type -> !sieveType.equals(removeParametrisedType(type))).forEach(phpType::add);
    }
    else {
      getTypesWithParametrisedParts().stream().filter(type -> !sieveTypes.contains(removeParametrisedType(type))).forEach(phpType::add);
    }
    return phpType;
  }

  private static class ImmutablePhpType extends PhpType {

    @NotNull
    @Override
    public PhpType add(@Nullable @NlsSafe String aClass) {
      throw getException();
    }

    @NotNull
    @Override
    public PhpType add(@Nullable PsiElement other) {
      throw getException();
    }

    @NotNull
    @Override
    public PhpType add(@Nullable PhpType type) {
      throw getException();
    }

    @Override
    protected PhpType getEmpty() {
      return EMPTY;
    }

    private PhpType addInternal(@Nullable PhpType type) {
      return super.add(type);
    }

    @NotNull
    private static RuntimeException getException() {
      return new UnsupportedOperationException("This PHP type is immutable");
    }
  }

  public static class PhpTypeBuilder {

    private final PhpType temp = new PhpType();

    @NotNull
    public PhpTypeBuilder add(@Nullable @NlsSafe String aClass) {
      temp.add(aClass);
      return this;
    }

    @NotNull
    public PhpTypeBuilder add(@Nullable @NlsSafe PsiElement other) {
      temp.add(other);
      return this;
    }

    @NotNull
    public PhpTypeBuilder add(@Nullable final PhpType type) {
      temp.add(type);
      return this;
    }

    @NotNull
    public PhpTypeBuilder merge(@NotNull final PhpTypeBuilder builder) {
      temp.add(builder.temp);
      return this;
    }

    @NotNull
    public PhpType build() {
      final PhpType type = new ImmutablePhpType();
      Set<@NlsSafe String> types = temp.getTypesWithParametrisedParts();
      switch (types.size()) {
        case 0 -> type.types = Collections.emptySet();
        case 1 -> type.types = Collections.unmodifiableSet(createTypesSet(types));
        default -> {
          type.types = types;
          type.dirty = temp.dirty;
        }
      }
      type.isComplete = temp.isComplete;
      type.myTypeHasParametrizedParts = ContainerUtil.exists(types, PhpType::hasParameterizedPart);
      return type;
    }
  }
  
  private static @NotNull Set<String> createEmptyTypesSet() {
    return CollectionFactory.createCaseInsensitiveStringSet();
  }

  private static @NotNull Set<String> createTypesSet(@NotNull Collection<String> types) {
    return CollectionFactory.createCaseInsensitiveStringSet(types);
  }

  @ApiStatus.Internal
  public static @NotNull PhpType from(String... types) {
    if (types == null) {
      return EMPTY;
    }
    return fromStrings(Arrays.asList(types));
  }

  @ApiStatus.Internal
  public static @NotNull PhpType fromStrings(@NotNull Collection<String> types) {
    if (ContainerUtil.and(types, e -> e == null)) return EMPTY;
    PhpType type = new PhpType();
    for (String s : types) {
      type.add(s);
    }
    return type;
  }


  @ApiStatus.Internal
  public static @NotNull PhpType global(Project project, String... types) {
    return from(types).global(project);
  }

  @ApiStatus.Internal
  public static @NotNull PhpType from(PsiElement... elements) {
    return from(Arrays.asList(elements));
  }

  @ApiStatus.Internal
  public static @NotNull PhpType from(Collection<? extends PsiElement> elements) {
    if (elements == null || ContainerUtil.and(elements, e -> e == null)) return EMPTY;
    PhpType type = new PhpType();
    for (PsiElement s : elements) {
      type.add(s);
    }
    return type;
  }

  @ApiStatus.Internal
  public static @NotNull PhpType global(PsiElement... elements) {
    if (elements == null) {
      return EMPTY;
    }
    PsiElement notNullElement = ContainerUtil.find(elements, e -> e != null);
    return notNullElement != null ? from(elements).global(notNullElement.getProject()) : EMPTY;
  }

  @ApiStatus.Internal
  public static @NotNull PhpType from(PhpType... types) {
    if (types == null || ContainerUtil.and(types, e -> e == null)) return EMPTY;
    PhpType type = new PhpType();
    for (PhpType s : types) {
      type.add(s);
    }
    return type;
  }

  @ApiStatus.Internal
  public static @NotNull PhpType global(Project project, PhpType... types) {
    return from(types).global(project);
  }
}
