// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.classAlias;

import com.intellij.openapi.project.Project;
import com.intellij.psi.search.GlobalSearchScope;
import com.jetbrains.php.lang.psi.elements.PhpClassAlias;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.function.Predicate;

/**
 * This index is different from any other because it returns fake PSI elements which serve as holders to class alias declarations.
 * As such declarations don't provide any information whether the alias is a usual class, interface or trait
 * the filtering has to be done at runtime. It leads to the fact that each method of the index performs several calls
 * to underlying indices what may potentially be slow. Hopefully, there are class aliases are not heavily used in real PHP projects
 * so the performance impact remains negligible.
 * <p/>
 * In general, there's no need to use the index directly. All the aliases are provided by the appropriate methods of {@code PhpIndex}.
 *
 * @see com.jetbrains.php.PhpIndex
 * @see PhpClassAliasProvider
 * @see PhpClassAlias
 */
public interface PhpClassAliasIndex {

  static @NotNull PhpClassAliasIndex getInstance(final @NotNull Project project) {
    return project.getService(PhpClassAliasIndex.class);
  }

  @NotNull
  Collection<String> getClassNames();

  @NotNull
  Collection<String> getInterfaceNames();

  @NotNull
  Collection<String> getTraitNames();

  @NotNull
  Collection<PhpClassAlias> getClassesByName(@NotNull String name);

  @NotNull
  Collection<PhpClassAlias> getInterfacesByName(@NotNull String name);

  @NotNull
  Collection<PhpClassAlias> getTraitsByName(@NotNull String name);

  @NotNull
  Collection<PhpClassAlias> getAllAliasesByName(@NotNull String name);

  @NotNull
  Collection<PhpClassAlias> getAliasesByName(@NotNull String name, @Nullable Predicate<PhpClassAlias> filter, @NotNull GlobalSearchScope scope);

  @NotNull
  Collection<String> getClassFqnsByAlias(@NotNull String fqn);

  @NotNull
  Collection<String> getAliasFqnsByClass(@NotNull String fqn);
}
