// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.classAlias;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.jetbrains.php.lang.psi.elements.FunctionReference;
import com.jetbrains.php.lang.psi.elements.MethodReference;
import com.jetbrains.php.lang.psi.visitors.PhpRecursiveElementVisitor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.BiConsumer;

public final class PhpClassAliasUtil {

  /**
   * The method is intended to return a list of providers relevant to the text.
   *
   * @param text to be searched for class alias declarations
   * @return list of providers
   * @see PhpClassAliasProvider#getSearchString()
   */
  public static List<PhpClassAliasProvider> getRelevantProviders(final @NotNull CharSequence text) {
    final List<PhpClassAliasProvider> providers = new ArrayList<>();
    for (final PhpClassAliasProvider provider : getClassAliasProviders()) {
      if (StringUtil.contains(text, provider.getSearchString())) {
        providers.add(provider);
      }
    }
    return providers;
  }

  public static List<PhpClassAliasProvider> getClassAliasProviders() {
    return PhpClassAliasProvider.EP_NAME.getExtensionList();
  }

  public static int getVersion() {
    return getClassAliasProviders().stream().mapToInt(PhpClassAliasProvider::getVersion).sum();
  }

  /**
   * The method is intended to help processing function/method references which are class alias declarations.
   *
   * @param scope    to find class alias declarations in
   * @param consumer to be provided with a reference and an extract result, both are never {@code null}
   * @see PhpClassAliasUtil#processClassAliasDeclarations(PsiElement, Collection, BiConsumer)
   */
  public static void processClassAliasDeclarations(final @NotNull PsiElement scope,
                                                   final @NotNull BiConsumer<? super FunctionReference, ? super ExtractResult> consumer) {
    processClassAliasDeclarations(scope, getClassAliasProviders(), consumer);
  }

  /**
   * The method is intended to help processing function/method references which are class alias declarations.
   * Working with a file it typically makes sense to filter providers first using {@link PhpClassAliasProvider#getSearchString()}
   * and only then checking against applicable providers or skipping the check at all.
   *
   * @param scope     to find class alias declarations in
   * @param providers to be checked against
   * @param consumer  to be provided with a reference and an extract result, both are never {@code null}
   * @see PhpClassAliasUtil#processClassAliasDeclarations(PsiElement, BiConsumer)
   */
  public static void processClassAliasDeclarations(final @NotNull PsiElement scope,
                                                   final @NotNull Collection<? extends PhpClassAliasProvider> providers,
                                                   final @NotNull BiConsumer<? super FunctionReference, ? super ExtractResult> consumer) {
    if (!providers.isEmpty()) {
      scope.accept(new PhpRecursiveElementVisitor() {
        @Override
        public void visitPhpFunctionCall(final FunctionReference reference) {
          process(reference);
        }

        @Override
        public void visitPhpMethodReference(final MethodReference reference) {
          process(reference);
        }

        private void process(final FunctionReference reference) {
          final ExtractResult result = extractClassAlias(reference, providers);
          if (result != null) {
            consumer.accept(reference, result);
          }
        }
      });
    }
  }

  /**
   * The method is intended to extract a class alias declaration from a function/method reference.
   *
   * @param reference which probably contains a class alias declaration
   * @return a pair of FQNs or {@code null} if the reference doesn't contain a class alias
   * @see PhpClassAliasUtil#extractClassAlias(FunctionReference, Collection)
   */
  public static @Nullable ExtractResult extractClassAlias(final @NotNull FunctionReference reference) {
    return extractClassAlias(reference, getClassAliasProviders());
  }

  /**
   * The method is intended to extract a class alias declaration from a function/method reference.
   * Working with a file it typically makes sense to filter providers first using {@link PhpClassAliasProvider#getSearchString()}
   * and only then checking against applicable providers or skipping the check at all.
   *
   * @param reference which probably contains a class alias declaration
   * @param providers to be checked against
   * @return a pair of FQNs or {@code null} if the reference doesn't contain a class alias
   * @see PhpClassAliasUtil#extractClassAlias(FunctionReference)
   */
  public static @Nullable ExtractResult extractClassAlias(final @NotNull FunctionReference reference,
                                                final @NotNull Collection<? extends PhpClassAliasProvider> providers) {
    for (final PhpClassAliasProvider provider : providers) {
      if (provider.isClassAliasDeclaration(reference)) {
        final String aliasFqn = provider.getAliasFqn(reference);
        final String originalFqn = provider.getOriginalFqn(reference);
        if (aliasFqn != null && originalFqn != null) {
          return new ExtractResult(aliasFqn, originalFqn);
        }
      }
    }
    return null;
  }

  public static final class ExtractResult {

    private final String myAliasFqn;
    private final String myOriginalFqn;

    private ExtractResult(final @NotNull String aliasFqn, final @NotNull String originalFqn) {
      myAliasFqn = aliasFqn;
      myOriginalFqn = originalFqn;
    }

    public @NotNull String getAliasFqn() {
      return myAliasFqn;
    }

    public @NotNull String getOriginalFqn() {
      return myOriginalFqn;
    }
  }
}
