// Copyright 2000-2022 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.resolve.types;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiElement;
import com.jetbrains.php.lang.psi.elements.PhpNamedElement;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

/**
 * Provides ability to add and resolve additional types for PHP PSI expressions.
 * <p>
 * #getType will be invoked during indexing for each expression, so only local information is accessible there.
 * #complete will be invoked in global mode and result of #getType will be passed as an argument to that method.
 * This argument then can be safely decoded using indices lookups
 * <p>
 * <a href="https://plugins.jetbrains.com/docs/intellij/php-open-api-php-type-providers.html">Example usage</a>
 */
public interface PhpTypeProvider4 {

  ExtensionPointName<PhpTypeProvider4> EP_NAME = ExtensionPointName.create("com.jetbrains.php.typeProvider4");

  /**
   * @return Custom signature key that should be used inside #getType to encode data.
   * Should be unique across all registered providers (special runtime assertion will check this)
   */
  char getKey();

  /**
   * @param element to deduce type for - using only local info only.
   * @return type for element, null if no insight. You can return a custom signature here to be later decoded #complete method.
   */
  @Nullable
  PhpType getType(PsiElement element);


  /**
   * @param expression All types that are starting signed by key equals to #getKey
   * @return type for element, null if no insight. You can return a custom signature here to be later decoded by getBySignature.
   */
  @Nullable
  PhpType complete(String expression, Project project);


  /**
   * Here you can extend the signature lookups
   * @param expression Signature expression to decode. You can use PhpIndex.getBySignature() to look up expression internals.
   * @param visited Recursion guard: please pass this on into any phpIndex calls having same parameter
   * @param depth Recursion guard: please pass this on into any phpIndex calls having same parameter
   * @return null if no match
   */
  Collection<? extends PhpNamedElement> getBySignature(String expression, Set<String> visited, int depth, Project project);

  default boolean emptyResultIsComplete() {
    return false;
  }

  @ApiStatus.Internal
  default boolean interceptsNativeSignature() {
    return false;
  }
}
