// Copyright 2000-2025 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.config.library;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.extensions.PluginAware;
import com.intellij.openapi.extensions.PluginDescriptor;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.vfs.VfsUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.xmlb.annotations.Attribute;
import com.intellij.util.xmlb.annotations.Transient;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.net.URL;
import java.util.stream.Stream;

/**
 * Allows to add a custom library root or to override bundled PHP Runtime.
 *
 * @see PhpLibraryRootProvider
 */
@ApiStatus.Internal
public final class PhpLibraryRoot implements PluginAware {
  private static final Logger LOG = Logger.getInstance(PhpLibraryRoot.class);

  public static final ExtensionPointName<PhpLibraryRoot> EP_NAME = new ExtensionPointName<>("com.jetbrains.php.libraryRoot");

  /**
   * This attribute specifies a path to a library root in resources.
   */
  @Attribute("path")
  public String path;

  /**
   * This attribute states that a library root is intended to override bundled PHP Runtime.
   */
  @Attribute("runtime")
  public boolean runtime;

  /**
   * This attribute specifies a qualified name of a custom implementation of the {@link PhpLibraryRootProvider} interface.
   * If it is specified, other attributes are ignored.
   */
  @Attribute("implementation")
  public String implementationClass;
  private PluginDescriptor pluginDescriptor;

  @Override
  @Transient
  public void setPluginDescriptor(@NotNull PluginDescriptor pluginDescriptor) {
    this.pluginDescriptor = pluginDescriptor;
  }

  private final NotNullLazyValue<PhpLibraryRootProvider> myInstanceHolder = NotNullLazyValue.atomicLazy(() -> {
    if (implementationClass == null) {
      return new MyPhpLibraryRootProvider();
    }
    return ApplicationManager.getApplication().instantiateClass(implementationClass, pluginDescriptor);
  });

  public @NotNull PhpLibraryRootProvider getProvider() {
    return myInstanceHolder.getValue();
  }

  @ApiStatus.Internal
  @NotNull
  public Stream<VirtualFile> getPathBasedLibraryRoots() {
    String patchedPath = path;
    patchedPath = patchedPath.substring(patchedPath.startsWith("/") ? 1 : 0, patchedPath.length() - (patchedPath.endsWith("/") ? 1 : 0));
    VirtualFile root = findRoot(patchedPath);
    if (root != null) {
      if (root.isDirectory()) {
        return Stream.of(root);
      }
      LOG.error("Library root must be a directory", patchedPath);
      return Stream.empty();
    }
    LOG.error("Please provide either a valid path or an appropriate implementation", patchedPath);
    return Stream.empty();
  }

  private @Nullable VirtualFile findRoot(String path) {
    URL url = findURL(path);
    if (url != null) {
      VirtualFile root = VfsUtil.findFileByURL(url);
      if (root != null) {
        return root;
      }
    }
    if (ApplicationManager.getApplication().isUnitTestMode()) {
      /*
        This is a temporary hack for the plugins that are running via intellij-gradle tests facility
        The thing is that after 221 it's not possible to fetch directory "stubs" directly from php.jar by plugin classloader
        (see https://jetbrains.slack.com/archives/C0CJS122E/p1636906941040800)
        <p>
        Proper fix on intellij-gradle side is required, as tmp workaround we fetch some file in this directory and obtains "stubs" directory
        as it's parent
       */
      url = findURL(path + "/.gitignore"); // stubs/.gitignore
      if (url != null) {
        VirtualFile root = VfsUtil.findFileByURL(url);
        if (root != null) {
          return root.getParent(); // stubs
        }
      }
    }
    return null;
  }

  private @Nullable URL findURL(String path) {
    URL url = pluginDescriptor.getClassLoader().getResource(path);
    if (url != null) {
      return url;
    }
    return PhpLibraryRoot.class.getClassLoader().getResource(path);
  }

  private final class MyPhpLibraryRootProvider implements PhpLibraryRootProvider {
    /**
     * {@inheritDoc}
     */
    @Override
    public @NotNull Stream<VirtualFile> getLibraryRoots(@NotNull Project project) {
      return PhpLibraryRoot.this.getPathBasedLibraryRoots();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isRuntime() {
      return runtime;
    }
  }
}
