// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.elements;

import com.intellij.lang.ASTNode;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiPolyVariantReference;
import com.intellij.psi.ResolveResult;
import com.intellij.util.ObjectUtils;
import com.jetbrains.php.lang.documentation.phpdoc.psi.PhpReferenceBase;
import com.jetbrains.php.lang.psi.resolve.types.PhpType;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;

public interface PhpReference extends PhpExpression, PhpPsiElement, PhpTypedElement, PsiPolyVariantReference, PhpReferenceBase {
  Condition<PsiElement> INSTANCEOF = e -> e instanceof PhpReference;

  @Override
  @Nullable
  String getName();

  @Nullable
  CharSequence getNameCS();

  @Override
  @Nullable
  ASTNode getNameNode();

  /**
   * Find targets in local file
   *
   * @return targets
   */
  @NotNull
  Collection<? extends PhpNamedElement> resolveLocal();

  @NotNull
  PhpType resolveLocalType();

  /**
   * Find targets in all files
   *
   * @param incompleteCode called from code completion
   * @return targets
   */
  @NotNull
  Collection<? extends PhpNamedElement> resolveGlobal(boolean incompleteCode);

  default @NotNull String getSignature() {
    return StringUtil.join(getSignatureParts(), "|");
  }

  @ApiStatus.Internal
  default @NotNull Collection<String> getSignatureParts() {
    return StringUtil.split(getSignature(), "|");
  }

  /**
   * @return computed namespace
   */
  @NotNull
  String getNamespaceName();

  @ApiStatus.Internal
  default <T extends PhpNamedElement> Collection<T> multiResolveStrict(Class<T> clazz) {
    ResolveResult[] results = multiResolve(false);
    if (results.length == 0) {
      return Collections.emptyList();
    }
    List<T> elementsWithClasses = Arrays.stream(results)
      .map(ResolveResult::getElement)
      .map(e -> ObjectUtils.tryCast(e, clazz)).filter(Objects::nonNull)
      .collect(Collectors.toList());
    return elementsWithClasses.size() == results.length ? elementsWithClasses : Collections.emptyList();
  }

  /**
   * @return namespace explicitly written in current reference
   */
  @Override
  @NotNull
  String getImmediateNamespaceName();

  /**
   * @return true if reference locally defined as absolute - starts with \ or is always absolute by context
   */
  boolean isAbsolute();

  @Override
  @Nullable
  String getFQN();
}
