// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package com.intellij.psi.impl.source.tree.injected;

import com.intellij.extapi.psi.PsiFileBase;
import com.intellij.injected.editor.DocumentWindow;
import com.intellij.injected.editor.EditorWindow;
import com.intellij.injected.editor.VirtualFileWindow;
import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.lang.injection.MultiHostRegistrar;
import com.intellij.openapi.editor.Caret;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.editor.colors.TextAttributesKey;
import com.intellij.openapi.editor.ex.DocumentEx;
import com.intellij.openapi.fileEditor.FileEditorManager;
import com.intellij.openapi.fileEditor.OpenFileDescriptor;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.*;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.*;
import com.intellij.psi.impl.BooleanRunnable;
import com.intellij.psi.impl.PsiDocumentManagerBase;
import com.intellij.psi.util.PsiUtilCore;
import com.intellij.util.ObjectUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.function.BiConsumer;

public final class InjectedLanguageUtil extends InjectedLanguageUtilBase {
  static final class ImplServiceImpl implements InjectedLanguageEditorUtil.ImplService {
    @Override
    public @Nullable Editor getEditorForInjectedLanguageNoCommit(@Nullable Editor editor, @Nullable PsiFile file) {
      return InjectedLanguageUtil.getEditorForInjectedLanguageNoCommit(editor, file);
    }
  }

  /**
   * A key used internally for forcing the association of an injected editor.
   * It is used for artificially created editors (to emulate behaviors for actions)
   */
  @ApiStatus.Internal
  public static final Key<Editor> FORCE_INJECTED_EDITOR_KEY = Key.create("FORCE_INJECTED_EDITOR_KEY");

  /**
   * A key used internally to identify and handle temporarily injected modified language copies.
   * This key serves as a marker in the context of language injections, enabling operations
   * related to managing and retrieving injected PSI elements.
   * It is used mostly for modified copies, because in this case it is impossible to match elements directly.
   */
  @ApiStatus.Internal
  public static final Key<PsiElement> FORCE_INJECTED_COPY_ELEMENT_KEY = Key.create("FORCE_INJECTED_COPY_ELEMENT_KEY");

  /**
   * {@link InjectedLanguageManager#FRANKENSTEIN_INJECTION}
   * @deprecated Use {@link InjectedLanguageManager#isFrankensteinInjection(PsiElement)} or
   *             {@link MultiHostRegistrar#frankensteinInjection(boolean)} instead
   */
  @Deprecated(forRemoval = true)
  public static final Key<Boolean> FRANKENSTEIN_INJECTION = InjectedLanguageManager.FRANKENSTEIN_INJECTION;

  private static final Comparator<PsiFile> LONGEST_INJECTION_HOST_RANGE_COMPARATOR = Comparator.comparing(
    psiFile -> InjectedLanguageManager.getInstance(psiFile.getProject()).getInjectionHost(psiFile),
    Comparator.nullsLast(Comparator.comparingInt(PsiElement::getTextLength))
  );

  public static void enumerate(@NotNull DocumentWindow documentWindow,
                               @NotNull PsiFile hostPsiFile,
                               @NotNull PsiLanguageInjectionHost.InjectedPsiVisitor visitor) {
    Segment[] ranges = documentWindow.getHostRanges();
    Segment rangeMarker = ranges.length > 0 ? ranges[0] : null;
    PsiElement element = rangeMarker == null ? null : hostPsiFile.findElementAt(rangeMarker.getStartOffset());
    if (element != null) {
      enumerate(element, hostPsiFile, true, visitor);
    }
  }

  /**
   * Invocation of this method on uncommitted {@code file} can lead to unexpected results, including throwing an exception!
   */
  @Contract("null,_->null;!null,_->!null")
  public static Editor getEditorForInjectedLanguageNoCommit(@Nullable Editor editor, @Nullable PsiFile file) {
    if (editor == null || file == null || editor instanceof EditorWindow) return editor;

    int offset = editor.getCaretModel().getOffset();
    return getEditorForInjectedLanguageNoCommit(editor, file, offset);
  }

  /**
   * Invocation of this method on uncommitted {@code file} can lead to unexpected results, including throwing an exception!
   */
  public static Editor getEditorForInjectedLanguageNoCommit(@Nullable Editor editor, @Nullable Caret caret, @Nullable PsiFile file) {
    if (editor == null || file == null || editor instanceof EditorWindow || caret == null) return editor;

    PsiFile injectedFile = findInjectedPsiNoCommit(file, caret.getOffset());
    return getInjectedEditorForInjectedFile(editor, caret, injectedFile);
  }

  /**
   * Invocation of this method on uncommitted {@code file} can lead to unexpected results, including throwing an exception!
   */
  public static Caret getCaretForInjectedLanguageNoCommit(@Nullable Caret caret, @Nullable PsiFile file) {
    if (caret == null || file == null || caret instanceof InjectedCaret) return caret;

    PsiFile injectedFile = findInjectedPsiNoCommit(file, caret.getOffset());
    Editor injectedEditor = getInjectedEditorForInjectedFile(caret.getEditor(), injectedFile);
    if (!(injectedEditor instanceof EditorWindow)) {
      return caret;
    }
    for (Caret injectedCaret : injectedEditor.getCaretModel().getAllCarets()) {
      if (((InjectedCaret)injectedCaret).getDelegate() == caret) {
        return injectedCaret;
      }
    }
    return null;
  }

  /**
   * Finds injected language in expression
   *
   * @param expression  where to find
   * @param classToFind class that represents language we look for
   * @param <T>         class that represents language we look for
   * @return instance of class that represents language we look for or null of not found
   */
  // We check types dynamically (using isAssignableFrom)
  @SuppressWarnings("unchecked")
  public static @Nullable <T extends PsiFileBase> T findInjectedFile(final @NotNull PsiElement expression,
                                                                     final @NotNull Class<T> classToFind) {
    final List<Pair<PsiElement, TextRange>> files =
      InjectedLanguageManager.getInstance(expression.getProject()).getInjectedPsiFiles(expression);
    if (files == null) {
      return null;
    }
    for (final Pair<PsiElement, TextRange> fileInfo : files) {
      final PsiElement injectedFile = fileInfo.first;
      if (classToFind.isAssignableFrom(injectedFile.getClass())) {
        return (T)injectedFile;
      }
    }
    return null;
  }

  /**
   * Invocation of this method on uncommitted {@code file} can lead to unexpected results, including throwing an exception!
   */
  @Contract("null,_,_->null;!null,_,_->!null")
  public static Editor getEditorForInjectedLanguageNoCommit(@Nullable Editor editor, @Nullable PsiFile file, final int offset) {
    if (editor == null || file == null || editor instanceof EditorWindow) return editor;
    PsiFile injectedFile = findInjectedPsiNoCommit(file, offset);
    return getInjectedEditorForInjectedFile(editor, injectedFile);
  }

  public static @NotNull Editor getInjectedEditorForInjectedFile(@NotNull Editor hostEditor, final @Nullable PsiFile injectedFile) {
    return getInjectedEditorForInjectedFile(hostEditor, hostEditor.getCaretModel().getCurrentCaret(), injectedFile);
  }

  /**
   * @param hostCaret if not {@code null}, take into account caret's selection (in case it's not contained completely in injected fragment,
   *                  return host editor)
   */
  public static @NotNull Editor getInjectedEditorForInjectedFile(@NotNull Editor hostEditor,
                                                                 @Nullable Caret hostCaret,
                                                                 final @Nullable PsiFile injectedFile) {
    if (injectedFile == null || hostEditor.isDisposed()) return hostEditor;
    Project project = hostEditor.getProject();
    if (project == null) project = injectedFile.getProject();
    Document document = PsiDocumentManager.getInstance(project).getDocument(injectedFile);
    if (!(document instanceof DocumentWindowImpl documentWindow)) return hostEditor;
    if (hostCaret != null && hostCaret.hasSelection()) {
      int selstart = hostCaret.getSelectionStart();
      if (selstart != -1) {
        int selend = Math.max(selstart, hostCaret.getSelectionEnd());
        if (!documentWindow.containsRange(selstart, selend)) {
          // selection spreads out the injected editor range
          return hostEditor;
        }
      }
    }
    if (!documentWindow.isValid()) {
      return hostEditor; // since the moment we got hold of injectedFile and this moment call, document may have been dirtied
    }
    Editor editor = hostEditor.getUserData(FORCE_INJECTED_EDITOR_KEY);
    if (editor != null) return editor;
    return InjectedEditorWindowTracker.getInstance().createEditor(documentWindow, hostEditor, injectedFile);
  }

  public static Editor openEditorFor(@NotNull PsiFile file, @NotNull Project project) {
    Document document = PsiDocumentManager.getInstance(project).getDocument(file);
    // may return editor injected in current selection in the host editor, not for the file passed as argument
    VirtualFile virtualFile = file.getVirtualFile();
    if (virtualFile == null) {
      return null;
    }
    if (virtualFile instanceof VirtualFileWindow window) {
      virtualFile = window.getDelegate();
    }
    Editor editor = FileEditorManager.getInstance(project).openTextEditor(new OpenFileDescriptor(project, virtualFile, -1), false);
    if (editor == null || editor instanceof EditorWindow || editor.isDisposed()) return editor;
    if (document instanceof DocumentWindowImpl window) {
      return InjectedEditorWindowTracker.getInstance().createEditor(window, editor, file);
    }
    return editor;
  }

  public static @NotNull Editor getTopLevelEditor(@NotNull Editor editor) {
    return InjectedLanguageEditorUtil.getTopLevelEditor(editor);
  }

  public static int hostToInjectedUnescaped(DocumentWindow window, int hostOffset) {
    Place shreds = ((DocumentWindowImpl)window).getShreds();
    Segment hostRangeMarker = shreds.get(0).getHostRangeMarker();
    if (hostRangeMarker == null || hostOffset < hostRangeMarker.getStartOffset()) {
      return shreds.get(0).getPrefix().length();
    }
    StringBuilder chars = new StringBuilder();
    int unescaped = 0;
    for (int i = 0; i < shreds.size(); i++, chars.setLength(0)) {
      PsiLanguageInjectionHost.Shred shred = shreds.get(i);
      int prefixLength = shred.getPrefix().length();
      int suffixLength = shred.getSuffix().length();
      PsiLanguageInjectionHost host = shred.getHost();
      TextRange rangeInsideHost = shred.getRangeInsideHost();
      LiteralTextEscaper<? extends PsiLanguageInjectionHost> escaper = host == null ? null : host.createLiteralTextEscaper();
      unescaped += prefixLength;
      Segment currentRange = shred.getHostRangeMarker();
      if (currentRange == null) continue;
      Segment nextRange = i == shreds.size() - 1 ? null : shreds.get(i + 1).getHostRangeMarker();
      if (nextRange == null || hostOffset < nextRange.getStartOffset()) {
        hostOffset = Math.min(hostOffset, currentRange.getEndOffset());
        int inHost = hostOffset - currentRange.getStartOffset();
        if (escaper != null && escaper.decode(rangeInsideHost, chars)) {
          int found = ObjectUtils.binarySearch(
            0, inHost, index -> Integer.compare(escaper.getOffsetInHost(index, TextRange.create(0, host.getTextLength())), inHost));
          return unescaped + (found >= 0 ? found : -found - 1);
        }
        return unescaped + inHost;
      }
      if (escaper != null && escaper.decode(rangeInsideHost, chars)) {
        unescaped += chars.length();
      }
      else {
        unescaped += currentRange.getEndOffset() - currentRange.getStartOffset();
      }
      unescaped += suffixLength;
    }
    return unescaped - shreds.get(shreds.size() - 1).getSuffix().length();
  }

  public static @Nullable DocumentWindow getDocumentWindow(@NotNull PsiElement element) {
    PsiFile file = element.getContainingFile();
    if (file == null) return null;
    VirtualFile virtualFile = file.getVirtualFile();
    if (virtualFile instanceof VirtualFileWindow window) return window.getDocumentWindow();
    return null;
  }

  public static boolean isHighlightInjectionBackground(@Nullable PsiLanguageInjectionHost host) {
    return !(host instanceof InjectionBackgroundSuppressor);
  }

  public static int getInjectedStart(@NotNull List<? extends PsiLanguageInjectionHost.Shred> places) {
    PsiLanguageInjectionHost.Shred shred = places.get(0);
    PsiLanguageInjectionHost host = shred.getHost();
    assert host != null;
    return shred.getRangeInsideHost().getStartOffset() + host.getTextRange().getStartOffset();
  }

  public static @Nullable PsiElement findElementInInjected(@NotNull PsiLanguageInjectionHost injectionHost, final int offset) {
    final Ref<PsiElement> ref = Ref.create();
    enumerate(injectionHost, (injectedPsi, places) -> ref.set(injectedPsi.findElementAt(offset - getInjectedStart(places))));
    return ref.get();
  }

  /**
   * Does not work with multiple injections on the same host.
   *
   * @deprecated Use {@link MultiHostRegistrar#putInjectedFileUserData(Key, Object)} when registering the injection.
   */
  public static <T> void putInjectedFileUserData(@NotNull PsiElement element,
                                                 @NotNull Language language,
                                                 @NotNull Key<T> key,
                                                 @Nullable T value) {
    PsiFile file = getCachedInjectedFileWithLanguage(element, language);
    if (file != null) {
      file.putUserData(key, value);
    }
  }

  public static @Nullable PsiFile getCachedInjectedFileWithLanguage(@NotNull PsiElement element, @NotNull Language language) {
    if (!element.isValid()) return null;
    PsiFile containingFile = PsiUtilCore.getTemplateLanguageFile(element.getContainingFile());
    if (containingFile == null || !containingFile.isValid()) return null;
    return InjectedLanguageManager.getInstance(containingFile.getProject())
      .getCachedInjectedDocumentsInRange(containingFile, element.getTextRange())
      .stream()
      .map(documentWindow -> PsiDocumentManager.getInstance(containingFile.getProject()).getPsiFile(documentWindow))
      .filter(file -> file != null && file.getLanguage() == LanguageSubstitutors.getInstance()
        .substituteLanguage(language, file.getVirtualFile(), file.getProject()))
      .max(LONGEST_INJECTION_HOST_RANGE_COMPARATOR)
      .orElse(null);
  }

  /**
   * Start injecting the reference in {@code language} in this place.
   * Unlike {@link MultiHostRegistrar#startInjecting(Language)} this method doesn't inject the full blown file in the other language.
   * Instead, it just marks some range as a reference in some language.
   * For example, you can inject file reference into string literal.
   * After that, it won't be highlighted as an injected fragment but still can be subject to e.g. "Goto declaraion" action.
   */
  public static void injectReference(@NotNull MultiHostRegistrar registrar,
                                     @NotNull Language language,
                                     @NotNull String prefix,
                                     @NotNull String suffix,
                                     @NotNull PsiLanguageInjectionHost host,
                                     @NotNull TextRange rangeInsideHost) {
    ((InjectionRegistrarImpl)registrar).injectReference(language, prefix, suffix, host, rangeInsideHost);
  }

  @ApiStatus.Internal
  // null means failed to reparse
  public static BooleanRunnable reparse(@NotNull PsiFile injectedPsiFile,
                                        @NotNull DocumentWindow injectedDocument,
                                        @NotNull PsiFile hostPsiFile,
                                        @NotNull Document hostDocument,
                                        @NotNull FileViewProvider hostViewProvider,
                                        @NotNull ProgressIndicator indicator,
                                        @NotNull ASTNode oldRoot,
                                        @NotNull ASTNode newRoot,
                                        @NotNull PsiDocumentManager documentManager) {
    Language language = injectedPsiFile.getLanguage();
    InjectedFileViewProvider provider = (InjectedFileViewProvider)injectedPsiFile.getViewProvider();
    VirtualFile oldInjectedVFile = provider.getVirtualFile();
    VirtualFile hostVirtualFile = hostViewProvider.getVirtualFile();
    BooleanRunnable runnable = InjectionRegistrarImpl
      .reparse(language, (DocumentWindowImpl)injectedDocument, injectedPsiFile, (VirtualFileWindow)oldInjectedVFile, hostVirtualFile,
               hostPsiFile,
               (DocumentEx)hostDocument,
               indicator, oldRoot, newRoot, documentManager);
    if (runnable == null) {
      InjectedEditorWindowTracker.getInstance().disposeEditorFor(injectedDocument);
    }
    return runnable;
  }

  public static void processTokens(@NotNull PsiFile injectedPsi,
                                   @NotNull List<? extends PsiLanguageInjectionHost.Shred> places,
                                   @NotNull BiConsumer<? super @NotNull TextRange, ? super @NotNull TextAttributesKey @NotNull []> consumer) {
    List<? extends InjectedLanguageUtilBase.TokenInfo> tokens = InjectedLanguageUtil.getHighlightTokens(injectedPsi);
    if (tokens == null) return;

    int shredIndex = -1;
    int injectionHostTextRangeStart = -1;
    for (InjectedLanguageUtil.TokenInfo token : tokens) {
      ProgressManager.checkCanceled();
      TextRange range = token.rangeInsideInjectionHost();
      if (range.getLength() == 0) continue;
      if (shredIndex != token.shredIndex()) {
        shredIndex = token.shredIndex();
        PsiLanguageInjectionHost.Shred shred = places.get(shredIndex);
        PsiLanguageInjectionHost host = shred.getHost();
        if (host == null) return;
        injectionHostTextRangeStart = host.getTextRange().getStartOffset();
      }
      TextRange hostRange = range.shiftRight(injectionHostTextRangeStart);

      consumer.accept(hostRange, token.textAttributesKeys());
    }
  }
}
