// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.codeInsight.daemon.impl;

import com.intellij.codeInsight.multiverse.CodeInsightContext;
import com.intellij.codeInsight.multiverse.CodeInsightContextHighlightingUtil;
import com.intellij.codeInspection.LocalInspectionTool;
import com.intellij.codeInspection.LocalInspectionToolSession;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.codeInspection.ex.LocalInspectionToolWrapper;
import com.intellij.injected.editor.DocumentWindow;
import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.lang.annotation.Annotator;
import com.intellij.lang.annotation.HighlightSeverity;
import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.openapi.Disposable;
import com.intellij.openapi.application.ex.ApplicationManagerEx;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.RangeMarker;
import com.intellij.openapi.editor.colors.EditorColorsUtil;
import com.intellij.openapi.editor.colors.TextAttributesKey;
import com.intellij.openapi.editor.ex.MarkupModelEx;
import com.intellij.openapi.editor.ex.RangeHighlighterEx;
import com.intellij.openapi.editor.impl.DocumentMarkupModel;
import com.intellij.openapi.editor.impl.SweepProcessor;
import com.intellij.openapi.editor.markup.HighlighterTargetArea;
import com.intellij.openapi.editor.markup.TextAttributes;
import com.intellij.openapi.fileEditor.FileDocumentManager;
import com.intellij.openapi.fileEditor.impl.FileDocumentManagerBase;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.progress.ProgressIndicatorProvider;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.progress.util.ProgressWrapper;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectManager;
import com.intellij.openapi.util.*;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.PsiDocumentManagerBase;
import com.intellij.psi.impl.source.tree.injected.InjectedFileViewProvider;
import com.intellij.psi.impl.source.tree.injected.InjectedLanguageManagerImpl;
import com.intellij.psi.scope.PsiScopeProcessor;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.SearchScope;
import com.intellij.util.ArrayUtil;
import com.intellij.util.CommonProcessors;
import com.intellij.util.ConcurrencyUtil;
import com.intellij.util.ExceptionUtil;
import com.intellij.util.concurrency.AppExecutorUtil;
import com.intellij.util.containers.CollectionFactory;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.HashingStrategy;
import com.intellij.util.containers.ReferenceQueueable;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.jetbrains.annotations.*;

import javax.swing.*;
import java.lang.ref.Reference;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@ApiStatus.Internal
public final class HighlightInfoUpdaterImpl extends HighlightInfoUpdater implements Disposable {
  @ApiStatus.Internal
  static final int FILE_LEVEL_FAKE_LAYER = -4094; // the layer the (fake) RangeHighlighter is created for file-level HighlightInfo in
  private static final Logger LOG = Logger.getInstance(HighlightInfoUpdaterImpl.class);
  private static final Object UNKNOWN_ID = "unknownId";
  /**
   * {@link HighlightInfo#group} which means this {@link HighlightInfo} is managed by {@link HighlightInfoUpdaterImpl},
   * i.e. its {@link HighlightInfo#group} is essentially ignored and infos are reused/deleted by {@link #psiElementVisited}
   * instead of {@link BackgroundUpdateHighlightersUtil#setHighlightersInRange} or {@link UpdateHighlightersUtil#setHighlightersInRange}
   */
  @ApiStatus.Internal
  public static final int MANAGED_HIGHLIGHT_INFO_GROUP = -6;

  // containing FileViewProvider -> tool id -> map of (visited PsiElement -> list of HighlightInfos generated by this tool while visiting that PsiElement)
  private static final Key<Map<FileViewProvider, Map<Object, ToolHighlights>>> VISITED_PSI_ELEMENTS = Key.create("VISITED_PSI_ELEMENTS");
  // list of HighlightInfos for psi elements which were gc-ed
  private static final Key<Collection<HighlightInfo>> EVICTED_PSI_ELEMENTS = Key.create("EVICTED_PSI_ELEMENTS"); // list guarded by this

  // true if some complex internal invariants must be checked on every operation;
  private boolean ASSERT_INVARIANTS;

  HighlightInfoUpdaterImpl(Project project) {
    Disposer.register(this, () -> {
      if (FileDocumentManager.getInstance() instanceof FileDocumentManagerBase managerBase) {
        managerBase.forEachCachedDocument(document -> {
          // this complicated to make it work when the project is disposed
          Map<FileViewProvider, Map<Object, ToolHighlights>> map = document.getUserData(VISITED_PSI_ELEMENTS);
          FileViewProvider[] array = (map == null ? Map.<FileViewProvider, Map<Object, ToolHighlights>>of() : map).keySet().toArray(new FileViewProvider[0]);
          Project docProject = array.length == 0 ? null : array[0].getManager().getProject();
          if (docProject == project) {
            document.putUserData(VISITED_PSI_ELEMENTS, null);
          }
        });
      }
    });
  }

  private boolean isAssertInvariants() {
    //return ApplicationManagerEx.getApplicationEx().isUnitTestMode() && !ApplicationManagerEx.isInStressTest();
    return ASSERT_INVARIANTS;
  }

  @TestOnly
  public void runAssertingInvariants(@NotNull Runnable runnable) {
    assert ApplicationManagerEx.getApplicationEx().isUnitTestMode();
    boolean old = ASSERT_INVARIANTS;
    try {
      ASSERT_INVARIANTS = true;
      runnable.run();
    }
    finally {
      ASSERT_INVARIANTS = old;
    }
  }

  private final CollectionFactory.EvictionListener<PsiElement, List<? extends HighlightInfo>, List<? extends HighlightInfo>> psiElementEvictionListener = (__, hash, evicted) -> {
    if (LOG.isTraceEnabled()) {
      LOG.trace("psiElementEvictionListener: {" + hash+"} -> ("+(evicted == null ? 0 : evicted.size())+"): "+evicted);
    }
    if (evicted != null) {
      addEvictedInfos(evicted);
    }
  };

  private @NotNull CollectionFactory.EvictionListener<@NotNull FileViewProvider, Map<Object, ToolHighlights>, Map<Object, ToolHighlights>> psiFileEvictionListener(@NotNull Document document) {
    return (__, hash, oldMap) -> {
      if (LOG.isTraceEnabled()) {
        List<HighlightInfo> infos = ContainerUtil.flatten(ContainerUtil.map(ContainerUtil.notNullize(oldMap).values(), t -> ContainerUtil.flatten(t.elementHighlights.values())));
        LOG.trace("psiFileEvictionListener: {" + hash + "} -> (" + (oldMap == null ? 0 : oldMap.size()) + "): " +
                  "\noldMap:" + oldMap +
                  "\nall infos:(" + infos.size() + "): " + infos);
      }
      for (Project project : ProjectManager.getInstance().getOpenProjects()) {
        // all maps for a FileViewProvider were gc-ed, we need to dispose the dangling range highlighters from the document markup
        MarkupModelEx markupModel = (MarkupModelEx)DocumentMarkupModel.forDocument(document, project, false);
        if (markupModel != null) {
          List<HighlightInfo> allInfos = ContainerUtil.mapNotNull(markupModel.getAllHighlighters(), h -> HighlightInfo.fromRangeHighlighter(h));
          List<HighlightInfo> infos = ContainerUtil.filter(allInfos, h-> h.toolId != null);
          addEvictedInfos(infos);
        }
      }
    };
  }

  private final class ToolHighlights {
    // list of HighlightInfos generated by a tool after visited this PsiElement. By convention, the list is sorted by Segment.BY_START_OFFSET_THEN_END_OFFSET
    final @NotNull Map<PsiElement, List<? extends HighlightInfo>> elementHighlights = CollectionFactory.createConcurrentSoftMap(psiElementEvictionListener);

    @NotNull
    ToolLatencies latencies = new ToolLatencies(0,0,0);

    @Override
    public String toString() {
      return "("+elementHighlights.size() +") "+elementHighlights;
    }
  }

  @ApiStatus.Internal
  public record ToolLatencies(
      long errorLatency, // latency of the first error, in nanoseconds (or 0 if none)
      long warningLatency, // latency of the first warning, in nanoseconds (or 0 if none)
      long otherLatency // latency of the first other info, in nanoseconds (or 0 if none)
  ) {
    int compareLatencies(@NotNull ToolLatencies other) {
      int o = cmp(errorLatency, other.errorLatency);
      if (o != 0) return o;
      o = cmp(warningLatency, other.warningLatency);
      if (o != 0) return o;
      o = cmp(otherLatency, other.otherLatency);
      return o;
    }
    static int cmp(long lat1, long lat2) {
      return Long.compare(lat1 == 0 ? Long.MAX_VALUE : lat1, lat2 == 0 ? Long.MAX_VALUE : lat2);
    }
  }

  @Override
  public void dispose() {
  }

  private synchronized void addEvictedInfos(@NotNull List<? extends HighlightInfo> infos) {
    if (!infos.isEmpty()) {
      if (LOG.isTraceEnabled()) {
        LOG.trace("addEvictedInfos: " + StringUtil.join(infos, i->i+(i.getHighlighter()==null ? "" : "; rh:" + i.getHighlighter().getTextRange()), ", ")+currentProgressInfo());
      }
      for (HighlightInfo info : infos) {
        RangeHighlighterEx highlighter = info.getHighlighter();
        if (highlighter != null) {
          Document hostDocument = highlighter.getDocument();
          Collection<HighlightInfo> evictedInfos = ConcurrencyUtil.computeIfAbsent(hostDocument, EVICTED_PSI_ELEMENTS, ()->new HashSet<>());
          evictedInfos.addAll(infos);
        }
      }
    }
  }

  private @NotNull Map<Object, ToolHighlights> getData(@NotNull PsiFile psiFile) {
    Document hostDocument = PsiDocumentManagerBase.getTopLevelDocument(psiFile.getViewProvider().getDocument());
    return getData(psiFile, hostDocument);
  }

  private @NotNull Map<Object, ToolHighlights> getData(@NotNull PsiFile psiFile, @NotNull Document hostDocument) {
    FileViewProvider viewProvider = psiFile.getViewProvider();
    Map<FileViewProvider, Map<Object, ToolHighlights>> map = getOrCreateHostMap(hostDocument);
    Map<Object, ToolHighlights> result = map.get(viewProvider);
    if (result == null) {
      result = map.computeIfAbsent(viewProvider, __->new ConcurrentHashMap<>());
    }
    return result;
  }

  private @NotNull Map<FileViewProvider, Map<Object, ToolHighlights>> getOrCreateHostMap(@NotNull Document hostDocument) {
    Map<FileViewProvider, Map<Object, ToolHighlights>> data = hostDocument.getUserData(VISITED_PSI_ELEMENTS);
    if (data == null) {
      HashingStrategy<FileViewProvider> strategy = new HashingStrategy<>() {
        @Override
        public int hashCode(FileViewProvider provider) {
          return provider ==null?0:provider.hashCode();
        }

        @Override
        public boolean equals(FileViewProvider o1, FileViewProvider o2) {
          if (o1 == null || o2 == null) {
            return (o1==null)==(o2==null);
          }
          //noinspection removal
          if (o1 instanceof InjectedFileViewProvider viewProvider1) {
            //noinspection removal
            if (o2 instanceof InjectedFileViewProvider viewProvider2) {
              // compare injected files by their offsets because they can be created concurrently in the background
              //noinspection removal
              return viewProvider1.getDocument().equals(viewProvider2.getDocument());
            }
            return false;
          }
          return o1 == o2;
        }
      };
      data = ((UserDataHolderEx)hostDocument).putUserDataIfAbsent(VISITED_PSI_ELEMENTS, CollectionFactory.createConcurrentSoftMap(strategy, psiFileEvictionListener(hostDocument)));
    }
    return data;
  }

  private static void invokeProcessQueueToTriggerEvictedListener(@NotNull Map<? extends PsiElement, ?> map) {
    ((ReferenceQueueable)map).processQueue(); // to call evictionListener if needed
  }

  @Override
  public synchronized void removeInfosForInjectedFilesOtherThan(@NotNull PsiFile hostPsiFile,
                                                                @NotNull TextRange restrictRange,
                                                                @NotNull HighlightingSession highlightingSession,
                                                                @NotNull Collection<? extends FileViewProvider> liveInjectedFiles) {
    InjectedLanguageManager injectedLanguageManager = InjectedLanguageManager.getInstance(hostPsiFile.getProject());
    Document hostDocument = hostPsiFile.getFileDocument();
    Map<FileViewProvider, Map<Object, ToolHighlights>> hostMap = getOrCreateHostMap(hostDocument);
    hostMap.entrySet().removeIf(entry -> {
      FileViewProvider viewProvider = entry.getKey();
      Map<Object, ToolHighlights> toolMap = entry.getValue();
      Document document = viewProvider.getDocument();
      TextRange textRange = TextRange.from(0, document.getTextLength());
      if (document instanceof DocumentWindow w) {
        textRange = w.injectedToHost(textRange);
      }
      boolean shouldRemove = injectedLanguageManager.isInjectedViewProvider(viewProvider) &&
                             !liveInjectedFiles.contains(viewProvider) &&
                             restrictRange.contains(textRange);
      if (shouldRemove) {
        removeAllHighlighterInsideFile(viewProvider, this, highlightingSession, toolMap);
        return true;
      }
      return false;
    });
  }

  // dispose all range highlighters from recycler while removing corresponding (invalid) PSI elements from the data
  synchronized void incinerateAndRemoveFromDataAtomically(@NotNull ManagedHighlighterRecycler recycler) {
    // remove highlighters which were reused or incinerated from the HighlightInfoUpdater's maps
    Collection<InvalidPsi> psiElements = recycler.forAllInGarbageBin();
    HighlightingSession session = recycler.myHighlightingSession;
    if (!psiElements.isEmpty()) {
      if (LOG.isTraceEnabled()) {
        LOG.trace("incinerateAndRemoveFromDataAtomically: psiElements (" + psiElements.size() + "): " + psiElements + " " + session.getProgressIndicator());
      }
    }
    Map<Object, ToolHighlights> data = getData(session.getPsiFile(), session.getDocument());
    removeFromDataAtomically(data, psiElements, session);
    recycler.incinerateAndClear();
  }

  @SuppressWarnings("UsagesOfObsoleteApi")
  static @NotNull String currentProgressInfo() {
    ProgressIndicator indicator = ProgressIndicatorProvider.getGlobalProgressIndicator();
    ProgressIndicator original = ProgressWrapper.unwrap(indicator);
    return "; progress=" + (indicator == original ? "" : "wrapped:")+
           (indicator == null ? "null\n" + ExceptionUtil.getThrowableText(new Throwable()) : indicator);
  }
  private static int compareNaturalNullable(@Nullable String o1, @Nullable String o2) {
    //noinspection StringEquality
    return o1 == o2 ? 0 : o1 == null ? -1 : o2 == null ? 1 : o1.compareTo(o2);
  }

  @VisibleForTesting
  public static final Comparator<HighlightInfo> BY_OFFSETS_AND_HASH_ERRORS_FIRST = (o1, o2) -> {
    if (o1 == o2) return 0;
    int r = Segment.BY_START_OFFSET_THEN_END_OFFSET.compare(o1, o2);
    if (r != 0) return r;
    r = o2.getSeverity().compareTo(o1.getSeverity()); // NB: reversed - errors first
    if (r != 0) return r;
    r = compareNaturalNullable(o1.getDescription(), o2.getDescription());
    if (r != 0) return r;
    // have to compare highlighters because we don't want to remove otherwise equal HighlightInfo except for its (recreated) highlighter
    RangeHighlighterEx h1 = o1.getHighlighter();
    RangeHighlighterEx h2 = o2.getHighlighter();
    return Integer.compare(System.identityHashCode(h1), System.identityHashCode(h2));
  };

  // remove `psis` from `data` in one batch for all infos in the list because there can be a lot of them
  // return true if was removed
  private static boolean removeFromDataAtomically(@NotNull Map<Object, ToolHighlights> data,
                                                  @NotNull @Unmodifiable Collection<? extends InvalidPsi> psis,
                                                  @NotNull HighlightingSession session) {
    if (psis.isEmpty()) {
      return true;
    }
    Map<Object, Map<PsiElement, List<HighlightInfo>>> byPsiElement = new HashMap<>();
    for (InvalidPsi invalidPsi : psis) {
      Object toolId = invalidPsi.info().toolId;
      List<HighlightInfo> infos =
      byPsiElement.computeIfAbsent(toolId, __->new HashMap<>())
                  .computeIfAbsent(invalidPsi.psiElement(), __ -> new ArrayList<>());
      infos.add(invalidPsi.info());
    }
    boolean removed = true;
    for (Map.Entry<Object, Map<PsiElement, List<HighlightInfo>>> entry : byPsiElement.entrySet()) {
      Object toolId = entry.getKey();
      ToolHighlights toolHighlights = data.get(toolId);
      if (toolHighlights == null) {
        removed = false;
        continue;
      }
      Map<PsiElement, List<HighlightInfo>> byPsiMap = entry.getValue();
      for (Map.Entry<PsiElement, List<HighlightInfo>> byPsiEntry : byPsiMap.entrySet()) {
        PsiElement psiElement = byPsiEntry.getKey();
        List<? extends HighlightInfo> oldL = toolHighlights.elementHighlights.get(psiElement);
        List<? extends HighlightInfo> oldInfos = oldL == null ? List.of() : ContainerUtil.sorted(oldL, BY_OFFSETS_AND_HASH_ERRORS_FIRST); // need to-resort in case the range-highlighters invalidated and offsets are skewed
        List<HighlightInfo> toRemove = ContainerUtil.sorted(byPsiEntry.getValue(), BY_OFFSETS_AND_HASH_ERRORS_FIRST);
        List<HighlightInfo> resultInfos = new ArrayList<>();
        ContainerUtil.processSortedListsInOrder(oldInfos, toRemove, BY_OFFSETS_AND_HASH_ERRORS_FIRST, true, (info, result) -> {
          if (result == ContainerUtil.MergeResult.COPIED_FROM_LIST1) {
            resultInfos.add(info);
          }
          // in other cases when the info is either from toRemove or both, skip it
        });
        removed &= oldInfos.size() - toRemove.size() == resultInfos.size();
        if (resultInfos.isEmpty()) {
          toolHighlights.elementHighlights.remove(psiElement);
        }
        else {
          toolHighlights.elementHighlights.put(psiElement, List.copyOf(resultInfos));
        }
        if (LOG.isTraceEnabled()) {
          LOG.trace("removeFromDataAtomically: " + debugPsiInfo(psiElement) +
                    (removed ? "" : "; removed="+removed) +
                    ": old=" + oldInfos.size() + (oldInfos.size() == resultInfos.size() ? "; "+StringUtil.join(oldInfos, "\n   ")+"\n  " : "")
                    + "; new=" + resultInfos.size() + (oldInfos.size() == resultInfos.size() ? "; "+StringUtil.join(resultInfos, "\n   ")+"\n  "+"toRemove=("+toRemove.size()+") "+StringUtil.join(toRemove, "\n   ") : "")
                    + " " + session.getProgressIndicator());
        }
      }
    }
    return removed;
  }

  private synchronized void recycleInvalidPsiElements(@NotNull PsiFile psiFile,
                                                      @NotNull Object requestor,
                                                      @NotNull HighlightingSession session,
                                                      @NotNull ManagedHighlighterRecycler invalidPsiRecycler,
                                                      @NotNull WhatTool toolIdPredicate) {
    disposeEvictedInfos(session, toolIdPredicate);
    collectPsiElements(psiFile, requestor, session, toolIdPredicate,
        psiElement -> psiElement != FAKE_ELEMENT && !psiElement.isValid(), // find invalid PSI
         (info, psiElement) -> {
          // heuristic: when the incremental reparse support is poor, and a lot of PSI is invalidated unnecessarily on each typing,
          //  that PSI has a big chance to be recreated in that exact place later, when a (major) chunk of the file is reparsed, so we do not kill that highlighter, just recycle it to avoid annoying blinking
          // if however, that invalid PSI highlighter wasn't recycled after a short delay, kill it (runWithInvalidPsiRecycler()) to improve responsiveness to outdated infos
          if (LOG.isTraceEnabled()) {
            LOG.trace("recycleInvalidPsiElements (predicate=" + toolIdPredicate + ") " + info.getHighlighter() +
                      "; toolIdPredicate=" + toolIdPredicate +
                      " for invalid " + debugPsiInfo(psiElement) +
                      " from " + requestor +
                      " " + session.getProgressIndicator());
          }
           if (info.getHighlighter() != null) {
             invalidPsiRecycler.recycleHighlighter(psiElement, info);
           }
         }
    );

    if (LOG.isTraceEnabled() && !invalidPsiRecycler.forAllInGarbageBin().isEmpty()) {
      Collection<InvalidPsi> psis = invalidPsiRecycler.forAllInGarbageBin();
      LOG.trace("recycleInvalidPsiElements: found " + psis.size() + " invalid psi elements in " + psiFile.getName() + " for " + toolIdPredicate +
                (psis.isEmpty() ? "" : ":\n"+ StringUtil.join(psis, "\n    ")) +
                " " +session.getProgressIndicator());
    }
  }

  static @NotNull String debugPsiInfo(@NotNull PsiElement psiElement) {
    return ""+psiElement + psiElement.getTextRange()+
           (psiElement.isValid() ? "" : " (invalid)") +
           " {" + System.identityHashCode(psiElement) + "}" + psiElement.getClass();
  }

  private void disposeEvictedInfos(@NotNull HighlightingSession session, @NotNull WhatTool predicate) {
    // first, visit all weak maps to call their processQueue() to induce `psiFileEvictionListener` to be called
    Document document = session.getDocument();
    Map<FileViewProvider, Map<Object, ToolHighlights>> hostMap = getOrCreateHostMap(document);
    ((ReferenceQueueable)hostMap).processQueue();
    hostMap.values()
      .stream()
      .flatMap(m -> m.values().stream())
      .forEach(toolHighlights -> ((ReferenceQueueable)toolHighlights.elementHighlights).processQueue());
    Collection<HighlightInfo> evictedInfos = document.getUserData(EVICTED_PSI_ELEMENTS);
    if (evictedInfos != null) {
      int size = evictedInfos.size();
      if (LOG.isTraceEnabled()) {
        LOG.trace("disposeEvictedInfos: disposing " + size + " entries");
      }
      evictedInfos.removeIf(info -> {
        boolean matches = predicate.matches(info.toolId);
        if (matches) {
          if (LOG.isTraceEnabled() && size < 200) {
            LOG.trace("disposeEvictedInfos: "+info);
          }
          UpdateHighlightersUtil.disposeWithFileLevelIgnoreErrors(info, session);
        }
        return matches;
      });
    }
  }

  @ApiStatus.Internal
  public enum WhatTool {
    INSPECTION, ANNOTATOR_OR_VISITOR;
    private boolean matches(@NotNull Object toolId) {
      if (isInspectionToolId(toolId)) {
        return this == INSPECTION;
      }
      if (isAnnotatorToolId(toolId) || isHighlightVisitorToolId(toolId)) {
        return this == ANNOTATOR_OR_VISITOR;
      }
      if (toolId == InjectedLanguageManagerImpl.INJECTION_BACKGROUND_TOOL_ID || toolId == InjectedLanguageManagerImpl.INJECTION_SYNTAX_TOOL_ID) {
        return false;
      }
      assert false : "unknown tool id type: "+toolId +"("+toolId.getClass()+")";
      return false;
    }
  }
  private synchronized void collectPsiElements(@NotNull PsiFile psiFile,
                                               @NotNull Object requestor,
                                               @NotNull HighlightingSession session,
                                               @NotNull WhatTool toolPredicate,
                                               @NotNull Predicate<? super PsiElement> psiElementPredicate,
                                               @NotNull BiConsumer<? super HighlightInfo, ? super PsiElement> rangeHighlighterConsumer) {
    InjectedLanguageManager injectedLanguageManager = InjectedLanguageManager.getInstance(psiFile.getProject());
    PsiFile hostFile = injectedLanguageManager.getTopLevelFile(psiFile);
    Document hostDocument = hostFile.getFileDocument();
    Map<FileViewProvider, Map<Object, ToolHighlights>> hostMap = getOrCreateHostMap(hostDocument);
    List<Map<Object, ToolHighlights>> maps = new ArrayList<>();
    // for invalid files, remove all highlighters inside immediately, there's no chance they'll ever be reused
    hostMap.entrySet().removeIf(entry -> {
      FileViewProvider viewProvider = entry.getKey();
      PsiFile psi = viewProvider.getStubBindingRoot();
      Map<Object, ToolHighlights> toolMap = entry.getValue();
      if (psi.isValid()) {
        Document topLevelDocument = PsiDocumentManagerBase.getTopLevelDocument(viewProvider.getDocument());
        if (topLevelDocument == hostDocument) {
          maps.add(toolMap);
        }
        return false;
      }
      if (psi == psiFile) {
        return false;
      }
      removeAllHighlighterInsideFile(viewProvider, requestor, session, toolMap);
      return true;
    });


    for (Map<Object, ToolHighlights> map : maps) {
      if (map.isEmpty()) {
        continue;
      }
      for (Map.Entry<Object, ToolHighlights> toolEntry : map.entrySet()) {
        ToolHighlights toolHighlights = toolEntry.getValue();
        Object toolId = toolEntry.getKey();
        invokeProcessQueueToTriggerEvictedListener(toolHighlights.elementHighlights);
        if (!toolPredicate.matches(toolId)) {
          continue;
        }
        for (Map.Entry<PsiElement, List<? extends HighlightInfo>> entry : toolHighlights.elementHighlights.entrySet()) {
          PsiElement psiElement = entry.getKey();
          ProgressManager.checkCanceled();
          if (psiElementPredicate.test(psiElement)) {
            List<? extends HighlightInfo> oldInfos = entry.getValue();
            for (HighlightInfo oldInfo : oldInfos) {
              rangeHighlighterConsumer.accept(oldInfo, psiElement);
            }
          }
        }
      }
    }
  }

  private static void removeAllHighlighterInsideFile(@NotNull FileViewProvider psiFile,
                                                     @NotNull Object requestor,
                                                     @NotNull HighlightingSession session,
                                                     @NotNull @Unmodifiable Map<Object, ToolHighlights> toolMap) {
    int removed = 0;
    for (ToolHighlights highlights : toolMap.values()) {
      for (List<? extends HighlightInfo> list : highlights.elementHighlights.values()) {
        for (HighlightInfo info : list) {
          UpdateHighlightersUtil.disposeWithFileLevelIgnoreErrors(info, session);
          removed++;
        }
      }
    }
    if (LOG.isTraceEnabled()) {
      LOG.trace("removeAllHighlighterInsideFile: removed invalid file: " + psiFile + " (" + removed + " highlighters removed); from " + requestor+" " +session.getProgressIndicator());
    }
  }

  /**
   * Tool {@code toolId} has generated (maybe empty) {@code newInfos} highlights during visiting PsiElement {@code visitedPsiElement}.
   * Remove all highlights that this tool had generated earlier during visiting this psi element, and replace them with {@code newInfos}
   * Do not read below, it's very private and just for me.
   * --
   * - retrieve {@code List<HighlightInfo> oldInfos} from {@code data[toolId, psiElement]}
   * - match the oldInfos with newInfos, obtaining 3 lists:
   *  1) a list of infos from {@code oldInfos} removed from newInfos - their RHs need to be disposed
   *  2) a list of infos from {@code newInfos} absent in oldInfos - new RHs must be created for those
   *  3) a list of infos which exist in both {@code oldInfos} and {@code newInfos} - their RHs from {@code oldInfos} must be reused and stored in {@code newInfos}
   * - store {@code newInfos} with correctly updated RHs back to {@code data[toolId, psiElement]}
   * All this must be in an atomic, PCE-non-cancelable block, maintaining the following invariant: it's guaranteed that upon completion there will be
   * - no dangling RHs are in markup (dangling RH is the one not referenced from data), - to avoid duplicating RHs
   * - no removed and then recreated RHs, - to avoid blinking
   * N.B. Sometimes multiple file editors are submitted for highlighting, some of which may have the same underlying document,
   * e.g., when the editor for the file is opened along with the git log with "preview diff" for the same file.
   * In this case, it's possible that several instances of e.g., LocalInspectionPass can run in parallel,
   * thus making `psiElementVisited` potentially reentrant (i.e., it can be called with the same `toolId` from different threads concurrently),
   * so we need to guard {@link ToolHighlights} against parallel modification.
   * @param toolId one of
   *               {@code String}: the tool is a {@link LocalInspectionTool} with its {@link LocalInspectionTool#getShortName()}==toolId
   *               {@code Class<? extends Annotator>}: the tool is an {@link Annotator} of the corresponding class
   *               {@code Class<? extends HighlightVisitor>}: the tool is a {@link HighlightVisitor} of the corresponding class
   *               {@code Object: Injection background and syntax from InjectedGeneralHighlightingPass#INJECTION_BACKGROUND_ID }
   */
  @Override
  @ApiStatus.Internal
  public void psiElementVisited(@NotNull Object toolId,
                                @NotNull PsiElement visitedPsiElement,
                                @NotNull List<? extends HighlightInfo> newInfos,
                                @NotNull Document hostDocument,
                                @NotNull PsiFile psiFile,
                                @NotNull Project project,
                                @NotNull HighlightingSession session,
                                @NotNull ManagedHighlighterRecycler invalidElementRecycler) {
    if (newInfos.isEmpty()) {
      ToolHighlights toolHighlights0 = getData(psiFile, hostDocument).get(toolId);
      List<? extends HighlightInfo> oldInfos0 = ContainerUtil.notNullize(toolHighlights0 == null ? null : toolHighlights0.elementHighlights.get(visitedPsiElement));
      if (oldInfos0.isEmpty()) {
        return;
      }
    }
    long psiTimeStamp = PsiManager.getInstance(project).getModificationTracker().getModificationCount();
    for (HighlightInfo newInfo : newInfos) {
      if (newInfo.getHighlighter() != null) {
        String message = "Highlighter must not be set, but got: " + newInfo;
        UpdateHighlightersUtil.disposeWithFileLevelIgnoreErrors(newInfo, session);
        LOG.error(message);
      }
      newInfo.updateLazyFixesPsiTimeStamp(psiTimeStamp);
    }
    synchronized (this) {
      assertMarkupConsistentWithData(psiFile, isInspectionToolId(toolId) ? WhatTool.INSPECTION : WhatTool.ANNOTATOR_OR_VISITOR);
      Map<Object, ToolHighlights> data = getData(psiFile, hostDocument);
      ToolHighlights toolHighlights = data.get(toolId);
      List<? extends HighlightInfo> oldInfos = ContainerUtil.notNullize(toolHighlights == null ? null : toolHighlights.elementHighlights.get(visitedPsiElement));
      if (oldInfos.isEmpty() && newInfos.isEmpty()) {
        return;
      }
      // execute in non-cancelable block. It should not throw PCE anyway, but just in case
      ProgressManager.getInstance().executeNonCancelableSection(() -> {
        //assertNoDuplicates(psiFile, getInfosFromMarkup(hostDocument, project), "markup before psiElementVisited ");

        ManagedHighlighterRecycler.runWithRecycler(session, recycler -> {
          for (HighlightInfo oldInfo : oldInfos) {
            RangeHighlighterEx highlighter = oldInfo.getHighlighter();
            if (highlighter != null) {
              recycler.recycleHighlighter(visitedPsiElement, oldInfo);
            }
          }
          List<? extends HighlightInfo> newInfosToStore = assignRangeHighlighters(visitedPsiElement, oldInfos, newInfos, toolId, session, psiFile, hostDocument, invalidElementRecycler, recycler, data);
          ToolHighlights notNullToolHighlights = toolHighlights == null ? data.computeIfAbsent(toolId, __ -> new ToolHighlights()) : toolHighlights;

          if (newInfosToStore.isEmpty()) {
            notNullToolHighlights.elementHighlights.remove(visitedPsiElement);
          }
          else {
            notNullToolHighlights.elementHighlights.put(visitedPsiElement, newInfosToStore);
          }

          if (LOG.isTraceEnabled()) {
            //noinspection removal
            LOG.trace("psiElementVisited: " + debugPsiInfo(visitedPsiElement) +
                      (psiFile.getViewProvider() instanceof InjectedFileViewProvider ? " injected in " + InjectedLanguageManager.getInstance(project).injectedToHost(psiFile, psiFile.getTextRange()) : "") +
                      "; tool:" + toolId + "; infos:" + head(newInfosToStore) + "; oldInfos:" + head(oldInfos) + " " + session.getProgressIndicator());
          }

          assertNoDuplicates(psiFile, newInfosToStore, "psiElementVisited ");
          //assertRangeHighlightersAreConsistentWithData(psiFile, newInfosToStore, hostDocument, toolId, visitedPsiElement);
        });
      });
    }
    //assertNoDuplicates(psiFile, getInfosFromMarkup(hostDocument, project), "markup after psiElementVisited ");
    assertMarkupConsistentWithData(psiFile, isInspectionToolId(toolId) ? WhatTool.INSPECTION : WhatTool.ANNOTATOR_OR_VISITOR);
    Reference.reachabilityFence(visitedPsiElement); // ensure no psi is gced while in the middle of modifying soft-ref maps
  }

  private static @NotNull Object head(@NotNull List<? extends HighlightInfo> infos) {
    return infos.size() < 10 ? infos : "("+infos.size()+"):"+ContainerUtil.getFirstItems(infos, 10)+"...";
  }

  private void assertNoDuplicates(@NotNull PsiFile psiFile, @NotNull Collection<? extends HighlightInfo> infos, @NotNull String cause) {
    if (!isAssertInvariants()) return;
    record HI(TextRange range, String desc, TextAttributes attributes){}
    List<HI> map = ContainerUtil.map(infos, h -> new HI(TextRange.create(h), h.getDescription(), h.getTextAttributes(psiFile, EditorColorsUtil.getGlobalOrDefaultColorScheme())));
    if (new HashSet<HI>(map).size() != infos.size()) {
      // duplicates are still possible when e.g. two range highlighters (with same desc) are merged into one after the doc modifications
      // try to remove invalid PSI elements and retry the check
      List<HighlightInfo> filtered = ContainerUtil.filter(infos, info -> {
        Map.Entry<PsiElement, List<? extends HighlightInfo>> entry = findInData(info, psiFile);
        return entry != null && entry.getKey().isValid();
      });
      List<HI> map2 = ContainerUtil.map(filtered, h -> new HI(TextRange.create(h), h.getDescription(), h.getTextAttributes(psiFile, EditorColorsUtil.getGlobalOrDefaultColorScheme())));
      if (new HashSet<HI>(map2).size() != filtered.size()) {
        List<HighlightInfo> sorted = ContainerUtil.sorted(filtered, UpdateHighlightersUtil.BY_ACTUAL_START_OFFSET_NO_DUPS);
        LOG.error(cause + "Duplicates found: \n" + StringUtil.join(sorted, h-> h.toString() + ": textAttributes=" + h.getTextAttributes(psiFile, EditorColorsUtil.getGlobalOrDefaultColorScheme()), "\n"));
      }
    }
  }
  private void assertRangeHighlightersAreConsistentWithData(@NotNull PsiFile psiFile,
                                                            @NotNull Collection<? extends HighlightInfo> newInfos,
                                                            @NotNull Document hostDocument,@NotNull Object toolId, @NotNull PsiElement visitedPsiElement) {
    if (!isAssertInvariants()) return;
    MarkupModelEx model = (MarkupModelEx)DocumentMarkupModel.forDocument(hostDocument, psiFile.getProject(), true);
    for (HighlightInfo info : newInfos) {
      model.processRangeHighlightersOverlappingWith(info.getStartOffset(), info.getEndOffset(), r -> {
        HighlightInfo i = HighlightInfo.fromRangeHighlighter(r);
        if (i != null && findInData(i, psiFile) == null) {
          LOG.error("Inconsistent RH found: " + r + "; info=" + i+"; During "+toolId+"; "+visitedPsiElement+"; "+newInfos);
        }
        return true;
      });
    }
  }

  private Map.Entry<PsiElement, List<? extends HighlightInfo>> findInData(@NotNull HighlightInfo info, @NotNull PsiFile psiFile) {
    ToolHighlights highlights = getData(psiFile).get(info.toolId);
    if (highlights != null) {
      for (Map.Entry<PsiElement, List<? extends HighlightInfo>> entry : highlights.elementHighlights.entrySet()) {
        if (ContainerUtil.containsIdentity(entry.getValue(), info)) {
          return entry;
        }
      }
    }
    return null;
  }

  @NotNull
  private Collection<HighlightInfo> getInfosFromMarkup(@NotNull PsiFile psiFile, @NotNull WhatTool toolIdPredicate) {
    if (!isAssertInvariants()) return Set.of();
    Project project = psiFile.getProject();
    Document hostDocument;
    //noinspection removal
    boolean isInjected = psiFile.getViewProvider() instanceof InjectedFileViewProvider;
    TextRange hostRange;
    if (isInjected) {
      PsiFile hostPsiFile = InjectedLanguageManager.getInstance(project).getTopLevelFile(psiFile);
      hostRange = InjectedLanguageManager.getInstance(project).injectedToHost(psiFile, psiFile.getTextRange());
      hostDocument = hostPsiFile.getFileDocument();
    }
    else {
      hostDocument = psiFile.getFileDocument();
      hostRange = TextRange.from(0, hostDocument.getTextLength());
    }
    return Arrays.stream(DocumentMarkupModel.forDocument(hostDocument, project, true).getAllHighlighters())
      .map(m -> HighlightInfo.fromRangeHighlighter(m))
      .filter(Objects::nonNull)
      .filter(h->h.toolId != null)
      .filter(h->toolIdPredicate.matches(h.toolId))
      .filter(h -> h.isFromInjection() == isInjected && h.getHighlighter() != null && hostRange.intersects(h.getHighlighter()))
      .collect(Collectors.toList());
  }

  private synchronized void assertMarkupConsistentWithData(@NotNull PsiFile psiFile, @NotNull WhatTool toolIdPredicate) {
    if (!isAssertInvariants()) return;
    Collection<HighlightInfo> fromMarkup = getInfosFromMarkup(psiFile, toolIdPredicate);
    // todo IJPL-339 process top level infos
    Set<HighlightInfo> fromData = new HashSet<>(getAllData(psiFile, toolIdPredicate));

    if (!new HashSet<>(fromMarkup).equals(fromData)) {
      Comparator<Object> toString = Comparator.comparing(o->o.toString());
      List<HighlightInfo> ds = ContainerUtil.sorted(fromData, toString);
      List<HighlightInfo> ms = ContainerUtil.sorted(fromMarkup, toString);
      String fromDataStr = StringUtil.join(ds, "\n");
      String fromMarkupStr = StringUtil.join(ms, "\n");
      LOG.error("data inconsistent with markup: data:\n"
                + fromDataStr + "\n---------------markup:\n"
                + fromMarkupStr+"\n========="
      );
    }
  }

  @NotNull
  private Collection<HighlightInfo> getAllData(@NotNull PsiFile psiFile, @NotNull WhatTool toolIdPredicate) {
    InjectedLanguageManager injectedLanguageManager = InjectedLanguageManager.getInstance(psiFile.getProject());
    PsiFile hostFile = injectedLanguageManager.getTopLevelFile(psiFile);
    Document hostDocument = hostFile.getFileDocument(); // store in the document because DocumentMarkupModel is associated with this document
    Map<FileViewProvider, Map<Object, ToolHighlights>> hostMap = getOrCreateHostMap(hostDocument);
    // consolidate all injected files with this range, because there can be several
    List<HighlightInfo> result = new ArrayList<>();
    TextRange psiFileHostRange = injectedLanguageManager.injectedToHost(psiFile, psiFile.getTextRange());
    boolean isPsiFileInjected = injectedLanguageManager.isInjectedFragment(psiFile);
    Document document = psiFile.getFileDocument();
    for (Map.Entry<FileViewProvider, Map<Object, ToolHighlights>> entry : hostMap.entrySet()) {
      FileViewProvider viewProvider = entry.getKey();
      Document keyDocument = viewProvider.getDocument();
      boolean isFileKeyInjected = injectedLanguageManager.isInjectedViewProvider(viewProvider);
      TextRange keyHostRange = keyDocument instanceof DocumentWindow w
                               ? w.injectedToHost(TextRange.from(0, keyDocument.getTextLength()))
                               : TextRange.from(0, keyDocument.getTextLength());
      if (isFileKeyInjected == isPsiFileInjected && keyHostRange.intersects(psiFileHostRange) && document.equals(keyDocument)) {
        Map<Object, ToolHighlights> map = entry.getValue();
        List<? extends HighlightInfo> list = map.entrySet().stream()
          .filter(e -> toolIdPredicate.matches(e.getKey()))
          .flatMap(e -> e.getValue().elementHighlights.values().stream())
          .flatMap(l -> l.stream())
          .filter(h -> h.getHighlighter() != null && h.getHighlighter().isValid()) // maybe LIP isn't started yet and its recycleInvalidPsi wasn't run
          .filter(h -> keyHostRange.intersects(h)) // maybe LIP isn't started yet and its recycleInvalidPsi wasn't run
          .toList();
        result.addAll(list);
      }
    }
    Collection<HighlightInfo> evicted = hostDocument.getUserData(EVICTED_PSI_ELEMENTS);
    if (evicted != null) {
      result.addAll(evicted);
    }
    return result;
  }

  // remove all highlight infos from `data` generated by tools absent in 'actualToolsRun'
  @ApiStatus.Internal
  public synchronized void removeHighlightsForObsoleteTools(@NotNull HighlightingSession highlightingSession,
                                                            @NotNull List<? extends PsiFile> injectedFragments,
                                                            @NotNull BiPredicate<? super Object, ? super PsiFile> keepToolIdPredicate) {
    for (PsiFile psiFile: ContainerUtil.append(injectedFragments, highlightingSession.getPsiFile())) {
      Map<Object, ToolHighlights> data = getData(psiFile, highlightingSession.getDocument());
      data.entrySet().removeIf(entry -> {
        Object toolId = entry.getKey();
        if (UNKNOWN_ID.equals(toolId)) {
          return false;
        }
        if (keepToolIdPredicate.test(toolId, psiFile)) {
          return false;
        }
        ToolHighlights toolHighlights = entry.getValue();
        for (Map.Entry<PsiElement, List<? extends HighlightInfo>> tentry : toolHighlights.elementHighlights.entrySet()) {
          List<? extends HighlightInfo> infos = tentry.getValue();
          if (LOG.isTraceEnabled()) {
            PsiElement psiElement = tentry.getKey();
            LOG.trace("removeHighlightsForObsoleteTools: " + debugPsiInfo(psiElement) +
                      "; infos(" + infos.size() + "): " + infos
            );
          }
          for (HighlightInfo info : infos) {
            UpdateHighlightersUtil.disposeWithFileLevelIgnoreErrors(info, highlightingSession);
          }
        }
        return true;
      });
    }
  }

  @ApiStatus.Internal
  public static boolean isInspectionToolId(Object toolId) {
    return toolId instanceof String;
  }

  static boolean isAnnotatorToolId(Object toolId) {
    return toolId instanceof Class<?> c && Annotator.class.isAssignableFrom(c);
  }

  static boolean isHighlightVisitorToolId(Object toolId) {
    return toolId instanceof Class<?> c && HighlightVisitor.class.isAssignableFrom(c);
  }

  static boolean isInjectionRelated(Object toolId) {
    return InjectedLanguageManagerImpl.isInjectionRelated(toolId);
  }

  // TODO very dirty method which throws all incrementality away, but we'd need to rewrite too many inspections to get rid of it
  @ApiStatus.Internal
  public synchronized void removeWarningsInsideErrors(@NotNull List<? extends PsiFile> injectedFragments,
                                                      @NotNull Document hostDocument,
                                                      @NotNull HighlightingSession session) {
    ManagedHighlighterRecycler.runWithRecycler(session, recycler -> {
      for (PsiFile psiFile: ContainerUtil.append(injectedFragments, session.getPsiFile())) {
        Map<Object, ToolHighlights> map = getData(psiFile, hostDocument);
        if (map.isEmpty()) {
          continue;
        }
        List<? extends HighlightInfo> sorted = map.entrySet().stream()
          .filter(e -> isInspectionToolId(e.getKey())) // inspections only
          .flatMap(e -> e.getValue().elementHighlights.values().stream())
          .flatMap(l->l.stream())
          .sorted(BY_OFFSETS_AND_HASH_ERRORS_FIRST) // sort by errors first, so that when the warning appears, the overlapping errors are already in the 'overlappingIntervals' queue
          .toList();
        SweepProcessor.Generator<HighlightInfo> generator = processor -> ContainerUtil.process(sorted, processor);
        SeverityRegistrar severityRegistrar = SeverityRegistrar.getSeverityRegistrar(session.getProject());
        SweepProcessor.sweep(generator, (__, info, atStart, overlappingIntervals) -> {
          if (!atStart) {
            return true;
          }
          if (info.isFileLevelAnnotation()) {
            return true;
          }

          // TODO uncomment if duplicates need to be removed automatically
          // Currently they are not, to manifest incorrectly written inspections/annotators earlier
          if (UpdateHighlightersUtil.isWarningCoveredByError(info, severityRegistrar, overlappingIntervals)/* || overlappingIntervals.contains(info)*/) {
            RangeHighlighterEx highlighter = info.getHighlighter();
            if (highlighter != null) {
              ToolHighlights elementHighlights = map.get(info.toolId);
              for (Map.Entry<PsiElement, List<? extends HighlightInfo>> elementEntry : elementHighlights.elementHighlights.entrySet()) {
                List<? extends HighlightInfo> infos = elementEntry.getValue();
                int i = infos.indexOf(info);
                if (i != -1) {
                  PsiElement psiElement = elementEntry.getKey();
                  recycler.recycleHighlighter(psiElement, info);
                  HighlightInfo toRemove = infos.get(i);
                  List<HighlightInfo> listMinusInfo = ContainerUtil.filter(infos, h -> h != toRemove);
                  if (listMinusInfo.isEmpty()) {
                    elementHighlights.elementHighlights.remove(psiElement);
                  }
                  else {
                    elementEntry.setValue(listMinusInfo);
                  }

                  if (LOG.isTraceEnabled()) {
                    LOG.trace("removeWarningsInsideErrors: " + debugPsiInfo(psiElement) +
                              "; removed " + info + "; now infos(" + listMinusInfo.size() + "): " + listMinusInfo
                    );
                  }

                  break;
                }
              }
            }
          }
          return true;
        });
      }
      Collection<InvalidPsi> warns = recycler.forAllInGarbageBin();
      if (LOG.isTraceEnabled() && !warns.isEmpty()) {
        LOG.trace("removeWarningsInsideErrors: found " + warns+" " +session.getProgressIndicator());
      }
    });
  }

  /**
   * after inspections completed, save their latencies (from corresponding {@link InspectionRunner.InspectionContext#holder})
   * to use later in {@link com.intellij.codeInsight.daemon.impl.InspectionProfilerDataHolder#sortByLatencies(PsiFile, List, HighlightInfoUpdaterImpl)}
   */
  @ApiStatus.Internal
  public synchronized void saveLatencies(@NotNull PsiFile psiFile, @NotNull @Unmodifiable Map<Object, ToolLatencies> latencies) {
    if (!psiFile.getViewProvider().isPhysical()) {
      // ignore editor text fields/consoles etc.
      return;
    }
    Map<Object, ToolHighlights> map = getData(psiFile);
    if (map.isEmpty()) return;
    for (Map.Entry<Object, ToolLatencies> entry : latencies.entrySet()) {
      Object toolId = entry.getKey();
      ToolHighlights toolHighlights = map.get(toolId);
      // no point saving latencies if nothing was reported
      if (toolHighlights == null) continue;

      ToolLatencies lats = entry.getValue();
      toolHighlights.latencies = new ToolLatencies(merge(toolHighlights.latencies.errorLatency, lats.errorLatency),
                                                   merge(toolHighlights.latencies.warningLatency, lats.warningLatency),
                                                   merge(toolHighlights.latencies.otherLatency, lats.otherLatency));
    }
  }

  private static long merge(long oldL, long newL) {
    return oldL == 0 || newL == 0 ? oldL+newL : Math.min(oldL, newL);
  }

  @ApiStatus.Internal
  public synchronized int compareLatencies(@NotNull PsiFile psiFile, @NotNull String toolId1, @NotNull String toolId2) {
    Map<Object, ToolHighlights> map = getData(psiFile);
    if (map.isEmpty()) return 0;
    ToolHighlights toolHighlights1 = map.get(toolId1);
    ToolHighlights toolHighlights2 = map.get(toolId2);
    if (toolHighlights1 == null) {
      return toolHighlights2 == null ? 0 : 1;
    }
    if (toolHighlights2 == null) {
      return -1;
    }
    return toolHighlights1.latencies.compareLatencies(toolHighlights2.latencies);
  }

  /**
   * sort `elements` by the number of produced diagnostics:
   *  - put first the elements for which this `toolWrapper` has produced some diagnostics on previous run
   *    - in case of a tie, put elements which generated higher severity diagnostics first
   *  - followed by all other elements
   */
  @ApiStatus.Internal
  public @NotNull @Unmodifiable List<? extends PsiElement> sortByPsiElementFertility(@NotNull PsiFile psiFile,
                                                                                     @NotNull LocalInspectionToolWrapper toolWrapper,
                                                                                     @NotNull List<? extends PsiElement> elements) {
    String toolId = toolWrapper.getShortName();
    Map<Object, ToolHighlights> map = getData(psiFile);
    if (map.isEmpty()) return elements;
    ToolHighlights toolHighlights = map.get(toolId);
    if (toolHighlights == null) return elements;
    Map<PsiElement, List<? extends HighlightInfo>> highlights = toolHighlights.elementHighlights;
    if (highlights.isEmpty()) return elements;
    return ContainerUtil.sorted(elements,
    (e1, e2) -> {
      List<? extends HighlightInfo> infos1 = highlights.get(e1);
      List<? extends HighlightInfo> infos2 = highlights.get(e2);
      if (infos1 == null || infos2 == null) {
        if (infos1 != null) { // put fertile element first
          return -1;
        }
        else if (infos2 != null) {
          return 1;
        }
        else {
          return Integer.compare(System.identityHashCode(e1), System.identityHashCode(e2)); // for consistency
        }
      }
      // put error-generating element first
      return maxSeverity(infos2).compareTo(maxSeverity(infos1));
    });
  }

  private static @NotNull HighlightSeverity maxSeverity(@NotNull List<? extends HighlightInfo> infos) {
    HighlightSeverity max = HighlightSeverity.INFORMATION;
    for (HighlightInfo info : infos) {
      HighlightSeverity severity = info.getSeverity();
      if (severity.compareTo(max) > 0) max = severity;
    }
    return max;
  }

  @ApiStatus.Internal
  public void runWithInvalidPsiRecycler(@NotNull HighlightingSession session,
                                        @NotNull WhatTool toolIdPredicate,
                                        @NotNull Consumer<? super ManagedHighlighterRecycler> invalidPsiRecyclerConsumer) {
    ManagedHighlighterRecycler.runWithRecycler(session, invalidPsiRecycler -> {
      recycleInvalidPsiElements(session.getPsiFile(), this, session, invalidPsiRecycler, toolIdPredicate);
      ScheduledFuture<?> future;
      if (invalidPsiRecycler.forAllInGarbageBin().isEmpty()) {
        future = null;
      }
      else {
        // after some time kill highlighters for invalid elements automatically, because it seems they are not going to be reused
        future = AppExecutorUtil.getAppScheduledExecutorService().schedule(() ->
          ProgressManager.getInstance().executeProcessUnderProgress(() -> {
            // grab RA first, to avoid deadlock when InvalidPsi.toString() tries to obtain RA again from within this monitor
            ApplicationManagerEx.getApplicationEx().tryRunReadAction(() -> {
              // do not incinerate when the session is canceled because even though all RHs here need to be disposed eventually, the new restarted session might have used them to reduce flicker
              if (session.isCanceled() || session.getProgressIndicator().isCanceled()) {
                if (LOG.isTraceEnabled()) {
                  LOG.trace("runWithInvalidPsiRecycler: recycler(" + toolIdPredicate + ") abandoned because the session was canceled: " + invalidPsiRecycler+" "+session.getProgressIndicator());
                }
              }
              else {
                incinerateAndRemoveFromDataAtomically(invalidPsiRecycler);
              }
            });
          }, session.getProgressIndicator())
          , Registry.intValue("highlighting.delay.invalid.psi.info.kill.ms"), TimeUnit.MILLISECONDS);
      }

      try {
        invalidPsiRecyclerConsumer.accept(invalidPsiRecycler);
      }
      finally {
        if (future != null) {
          future.cancel(false);
        }
      }
    });
  }

  /**
   * We associate each {@link HighlightInfo} with the PSI element for which the inspection builder has produced that info.
   * Unfortunately, there are some crazy inspections that produce infos in their {@link LocalInspectionTool#inspectionFinished(LocalInspectionToolSession, ProblemsHolder)} method instead.
   * Which is very slow, because that highlight info won't be displayed until the entire file is visited.
   * For these infos the associated PSI element is assumed to be this {@code FAKE_ELEMENT}
   */
  @ApiStatus.Internal
  public static final PsiElement FAKE_ELEMENT = createFakePsiElement("inspectionFinished");

  static @NotNull PsiElement createFakePsiElement(@NotNull String debugString) {
    return new PsiElement() {
      @Override
      public @NotNull Project getProject() {
        throw createException();
      }

      @Override
      public @NotNull Language getLanguage() {
        throw createException();
      }

      @Override
      public PsiManager getManager() {
        throw createException();
      }

      @Override
      public PsiElement @NotNull [] getChildren() {
        return EMPTY_ARRAY;
      }

      @Override
      public PsiElement getParent() {
        return null;
      }

      @Override
      public @Nullable PsiElement getFirstChild() {
        return null;
      }

      @Override
      public @Nullable PsiElement getLastChild() {
        return null;
      }

      @Override
      public @Nullable PsiElement getNextSibling() {
        return null;
      }

      @Override
      public @Nullable PsiElement getPrevSibling() {
        return null;
      }

      @Override
      public PsiFile getContainingFile() {
        return null;
      }

      @Override
      public TextRange getTextRange() {
        return TextRange.EMPTY_RANGE;
      }

      @Override
      public int getStartOffsetInParent() {
        return -1;
      }

      @Override
      public int getTextLength() {
        return 0;
      }

      @Override
      public PsiElement findElementAt(int offset) {
        return null;
      }

      @Override
      public @Nullable PsiReference findReferenceAt(int offset) {
        return null;
      }

      @Override
      public int getTextOffset() {
        return 0;
      }

      @Override
      public String getText() {
        return "";
      }

      @Override
      public char @NotNull [] textToCharArray() {
        return ArrayUtil.EMPTY_CHAR_ARRAY;
      }

      @Override
      public PsiElement getNavigationElement() {
        return null;
      }

      @Override
      public PsiElement getOriginalElement() {
        return null;
      }

      @Override
      public boolean textMatches(@NotNull CharSequence text) {
        return false;
      }

      @Override
      public boolean textMatches(@NotNull PsiElement element) {
        return false;
      }

      @Override
      public boolean textContains(char c) {
        return false;
      }

      @Override
      public void accept(@NotNull PsiElementVisitor visitor) {

      }

      @Override
      public void acceptChildren(@NotNull PsiElementVisitor visitor) {

      }

      @Override
      public PsiElement copy() {
        return null;
      }

      @Override
      public PsiElement add(@NotNull PsiElement element) {
        throw createException();
      }

      @Override
      public PsiElement addBefore(@NotNull PsiElement element, PsiElement anchor) {
        throw createException();
      }

      @Override
      public PsiElement addAfter(@NotNull PsiElement element, PsiElement anchor) {
        throw createException();
      }

      @Override
      public void checkAdd(@NotNull PsiElement element) {
        throw createException();
      }

      @Override
      public PsiElement addRange(PsiElement first, PsiElement last) {
        throw createException();
      }

      @Override
      public PsiElement addRangeBefore(@NotNull PsiElement first, @NotNull PsiElement last, PsiElement anchor) {
        throw createException();
      }

      @Override
      public PsiElement addRangeAfter(PsiElement first, PsiElement last, PsiElement anchor) {
        throw createException();
      }

      @Override
      public void delete() {
        throw createException();
      }

      @Override
      public void checkDelete() {
        throw createException();
      }

      @Override
      public void deleteChildRange(PsiElement first, PsiElement last) {
        throw createException();
      }

      @Override
      public PsiElement replace(@NotNull PsiElement newElement) {
        throw createException();
      }

      @Override
      public boolean isValid() {
        return true;
      }

      @Override
      public boolean isWritable() {
        return false;
      }

      PsiInvalidElementAccessException createException() {
        return new PsiInvalidElementAccessException(this, toString(), null);
      }

      @Override
      public @Nullable PsiReference getReference() {
        return null;
      }

      @Override
      public PsiReference @NotNull [] getReferences() {
        return PsiReference.EMPTY_ARRAY;
      }

      @Override
      public <T> T getCopyableUserData(@NotNull Key<T> key) {
        throw createException();
      }

      @Override
      public <T> void putCopyableUserData(@NotNull Key<T> key, T value) {
        throw createException();
      }

      @Override
      public boolean processDeclarations(@NotNull PsiScopeProcessor processor,
                                         @NotNull ResolveState state,
                                         PsiElement lastParent,
                                         @NotNull PsiElement place) {
        return false;
      }

      @Override
      public PsiElement getContext() {
        return null;
      }

      @Override
      public boolean isPhysical() {
        return true;
      }

      @Override
      public @NotNull GlobalSearchScope getResolveScope() {
        throw createException();
      }

      @Override
      public @NotNull SearchScope getUseScope() {
        throw createException();
      }

      @Override
      public ASTNode getNode() {
        throw createException();
      }

      @Override
      public <T> T getUserData(@NotNull Key<T> key) {
        throw createException();
      }

      @Override
      public <T> void putUserData(@NotNull Key<T> key, T value) {
        throw createException();
      }

      @Override
      public Icon getIcon(int flags) {
        throw createException();
      }

      @Override
      public boolean isEquivalentTo(final PsiElement another) {
        return this == another;
      }

      @Override
      public String toString() {
        return "FAKE_PSI_ELEMENT: "+debugString;
      }
    };
  }

  /**
   * for each info in `newInfos` retrieve the RH from recycler (and then invalidElementRecycler if not found) or create new RH
   * could be reentrant, be careful to avoid leaking/blinking RHs
   */
  private @NotNull @Unmodifiable List<? extends HighlightInfo> assignRangeHighlighters(@NotNull PsiElement visitedPsiElement,
                                                                                       @NotNull List<? extends HighlightInfo> oldInfos,
                                                                                       @NotNull List<? extends HighlightInfo> newInfos,
                                                                                       @NotNull Object toolId,
                                                                                       @NotNull HighlightingSession session,
                                                                                       @NotNull PsiFile psiFile,
                                                                                       @NotNull Document hostDocument,
                                                                                       @NotNull ManagedHighlighterRecycler invalidElementRecycler,
                                                                                       @NotNull ManagedHighlighterRecycler recycler,
                                                                                       @NotNull Map<Object, ToolHighlights> data) {
    MarkupModelEx markup = (MarkupModelEx)DocumentMarkupModel.forDocument(hostDocument, session.getProject(), true);

    SeverityRegistrar severityRegistrar = SeverityRegistrar.getSeverityRegistrar(session.getProject());
    Long2ObjectMap<RangeMarker> range2markerCache = new Long2ObjectOpenHashMap<>(10);
    // infos which highlighters are recycled from `invalidElementRecycler` and which need to be removed from `data` to avoid its highlighter to be registered in two places
    // this list must be sorted by BY_OFFSETS_AND_HASH
    List<HighlightInfo> newInfosToStore = new ArrayList<>(newInfos.size());
    List<InvalidPsi> toRemove = new ArrayList<>(newInfos.size());
    //noinspection ForLoopReplaceableByForEach
    for (int i = 0; i < newInfos.size(); i++) {
      HighlightInfo newInfo = newInfos.get(i);
      boolean isFileLevel = newInfo.isFileLevelAnnotation();
      long finalInfoRange = isFileLevel
                            ? TextRangeScalarUtil.toScalarRange(0, psiFile.getTextLength())
                            : BackgroundUpdateHighlightersUtil.getRangeToCreateHighlighter(newInfo, hostDocument);
      if (finalInfoRange == -1) {
        continue;
      }
      // workaround for rogue plugins that cache HighlightInfos and then return identical HI for different calls
      newInfo = newInfo.copy(true).createUnconditionally();
      assert toolId.equals(newInfo.toolId) : "HighlightInfo generated by "+toolId + "(" + toolId.getClass() + ") must have consistent toolId, but got:"+ newInfo;
      newInfosToStore.add(newInfo);
      int layer = isFileLevel ? FILE_LEVEL_FAKE_LAYER : UpdateHighlightersUtil.getLayer(newInfo, severityRegistrar);
      int infoStartOffset = TextRangeScalarUtil.startOffset(finalInfoRange);
      int infoEndOffset = TextRangeScalarUtil.endOffset(finalInfoRange);

      InvalidPsi recycled = recycler.pickupHighlighterFromGarbageBin(infoStartOffset, infoEndOffset, layer, newInfo.getDescription());
      String from;
      if (recycled == null) {
        recycled = invalidElementRecycler.pickupHighlighterFromGarbageBin(infoStartOffset, infoEndOffset, layer, newInfo.getDescription());
        from = "invalidElementRecycler";
      }
      else {
        from = "recycler";
      }
      if (recycled != null) {
        if (LOG.isTraceEnabled()) {
          LOG.trace("assignRangeHighlighters: pickedup " + recycled + " from " + from+ " "+session.getProgressIndicator());
        }
        toRemove.add(recycled);
      }
      List<Object> existingInfos = List.of();
      if (recycled == null && LOG.isTraceEnabled()) {
        List<RangeHighlighterEx> dups = new ArrayList<>();
        markup.processRangeHighlightersOverlappingWith(newInfo.getStartOffset(), newInfo.getEndOffset(), new CommonProcessors.CollectProcessor<>(dups));
        String description = newInfo.getDescription();
        TextAttributesKey infoTextAttributesKey = newInfo.forcedTextAttributesKey == null ? newInfo.type.getAttributesKey() : newInfo.forcedTextAttributesKey;
        dups.removeIf(r-> {
          HighlightInfo hi;
          return !r.getTextRange().equalsToRange(infoStartOffset, infoEndOffset) ||
                 !Objects.equals(r.getTextAttributesKey(), infoTextAttributesKey) ||
                 (hi=HighlightInfo.fromRangeHighlighter(r)) == null ||
                 !Objects.equals(hi.getDescription(), description)
            ;
        });

        existingInfos =
          ContainerUtil.map(dups, dup -> {
            HighlightInfo info = HighlightInfo.fromRangeHighlighter(dup);
            return info == null ? "no HI" : info.getHighlighter() != dup ? "inconsistent HI" : "duplicate: "+dup+":"+findInData(info, psiFile);
          });
      }
      RangeHighlighterEx newHighlighter = changeRangeHighlighterAttributes(session, psiFile, markup, newInfo, range2markerCache, finalInfoRange, recycled, isFileLevel, infoStartOffset, infoEndOffset, layer, severityRegistrar);
      if (recycled != null) {
        recycled.info().invalidate();
      }
      else if (LOG.isTraceEnabled() && !existingInfos.isEmpty()) {
        LOG.trace("assignRangeHighlighters duplicates: " + newInfo +
                  "\ndups(" + existingInfos.size() + "): " + existingInfos +
                  "\nnew: " + newHighlighter +
                  "\nrecycler: " + recycler + ":" + recycler.forAllInGarbageBin() +
                  "\ninvalidRecycler:" + invalidElementRecycler + ":" + invalidElementRecycler.forAllInGarbageBin() +
                  "\nold infos(" + oldInfos.size() + "): " + oldInfos +
                  "\nvisitedPsiElement:" + debugPsiInfo(visitedPsiElement) +
                  "\n " + session.getProgressIndicator()
                  );
      }
    }
    removeFromDataAtomically(data, toRemove, session);
    List<HighlightInfo> sorted = ContainerUtil.sorted(newInfosToStore, BY_OFFSETS_AND_HASH_ERRORS_FIRST);
    // this list must be sorted by BY_OFFSETS_AND_HASH
    for (int i = 0; i < sorted.size(); i++) {
      HighlightInfo info = sorted.get(i);
      assert toolId.equals(info.toolId) : info + "; " + toolId + "(" + toolId.getClass() + ")";
      assert info.getHighlighter() != null : info;
      assert info.getHighlighter().isValid() : info;
      HighlightInfo assignedInfo = HighlightInfo.fromRangeHighlighter(info.getHighlighter());
      assert assignedInfo == info : "from RH: " + assignedInfo + "(" + System.identityHashCode(assignedInfo)+ "); but expected: " + info+ "(" + System.identityHashCode(info)+ ")";
      if (i>0) {
        int compare = BY_OFFSETS_AND_HASH_ERRORS_FIRST.compare(sorted.get(i - 1), sorted.get(i));
        assert compare <= 0 : "assignRangeHighlighters returned unsorted list: " + sorted +"; "+compare;
      }
    }
    return sorted;
  }

  private static @NotNull RangeHighlighterEx changeRangeHighlighterAttributes(@NotNull HighlightingSession session,
                                                                              @NotNull PsiFile psiFile,
                                                                              @NotNull MarkupModelEx markup,
                                                                              @NotNull HighlightInfo newInfo,
                                                                              @NotNull Long2ObjectMap<RangeMarker> range2markerCache,
                                                                              long finalInfoRange,
                                                                              @Nullable InvalidPsi recycled,
                                                                              boolean isFileLevel,
                                                                              int infoStartOffset,
                                                                              int infoEndOffset,
                                                                              int newLayer,
                                                                              @NotNull SeverityRegistrar severityRegistrar) {
    CodeInsightContext context = session.getCodeInsightContext();
    TextAttributes infoAttributes = newInfo.getTextAttributes(psiFile, session.getColorsScheme());
    com.intellij.util.Consumer<RangeHighlighterEx> changeAttributes = finalHighlighter ->
      BackgroundUpdateHighlightersUtil.changeAttributes(finalHighlighter, newInfo, session.getColorsScheme(), psiFile, infoAttributes, context);

    RangeHighlighterEx highlighter;
    if (recycled == null) {
      // create new
      if (isFileLevel) {
        highlighter = createOrReuseFakeFileLevelHighlighter(MANAGED_HIGHLIGHT_INFO_GROUP, newInfo, null, markup, session.getProject(), context);
        ((HighlightingSessionImpl)session).addFileLevelHighlight(newInfo, highlighter);
      }
      else {
        //assertNoInfoInMarkup(newInfo, markup, recycler, invalidElementRecycler);
        highlighter = markup.addRangeHighlighterAndChangeAttributes(null, infoStartOffset, infoEndOffset, newLayer,
                                                                    HighlighterTargetArea.EXACT_RANGE, false,
                                                                    changeAttributes);
        newInfo.updateQuickFixFields(session.getDocument(), range2markerCache, finalInfoRange);
      }
    }
    else {
      // recycle
      HighlightInfo oldInfo = recycled.info();
      HighlightSeverity oldSeverity = oldInfo.getSeverity();
      int oldLayer = isFileLevel ? FILE_LEVEL_FAKE_LAYER : UpdateHighlightersUtil.getLayer(oldInfo, severityRegistrar);
      highlighter = oldInfo.getHighlighter();
      if (isFileLevel) {
        highlighter = createOrReuseFakeFileLevelHighlighter(MANAGED_HIGHLIGHT_INFO_GROUP, newInfo, highlighter, markup, session.getProject(), context);
        ((HighlightingSessionImpl)session).replaceFileLevelHighlight(oldInfo, newInfo, highlighter);
      }
      else {
        markup.changeAttributesInBatch(highlighter, changeAttributes);
        newInfo.updateQuickFixFields(session.getDocument(), range2markerCache, finalInfoRange);
      }
      oldInfo.copyComputedLazyFixesTo(newInfo, session.getDocument());
      assert oldInfo.getGroup() == MANAGED_HIGHLIGHT_INFO_GROUP: oldInfo;
      assert newLayer == oldLayer : "Trying to to change the severity/layer of existing range highlighter: "+highlighter+"; oldInfo="+oldInfo+"; newInfo="+newInfo+"; oldSeverity="+oldSeverity+"(oldLayer="+oldLayer+"); newSeverity="+newInfo.getSeverity()+"(newLayer="+newLayer+")";
    }
    range2markerCache.put(finalInfoRange, highlighter);
    if (LOG.isTraceEnabled()) {
      LOG.trace("changeRangeHighlighterAttributes: "+
                (recycled == null ? "(create new RH)" : "(recycled)")
                + newInfo + " "+session.getProgressIndicator());
    }

    return highlighter;
  }

  public static @NotNull RangeHighlighterEx createOrReuseFakeFileLevelHighlighter(int group,
                                                                                  @NotNull HighlightInfo info,
                                                                                  @Nullable RangeHighlighterEx toReuse,
                                                                                  @NotNull MarkupModelEx markupModel,
                                                                                  @NotNull Project project,
                                                                                  @Nullable CodeInsightContext context) {
    Document document = markupModel.getDocument();
    RangeHighlighterEx highlighter;
    if (toReuse != null && toReuse.isValid()) {
      highlighter = toReuse;
      BackgroundUpdateHighlightersUtil.associateInfoAndHighlighter(info, highlighter);
    }
    else {
      highlighter = markupModel.addRangeHighlighterAndChangeAttributes(null, 0, document.getTextLength(), FILE_LEVEL_FAKE_LAYER,
                                                                       HighlighterTargetArea.EXACT_RANGE, false, ex -> {
          BackgroundUpdateHighlightersUtil.associateInfoAndHighlighter(info, ex);
        });
    }
    highlighter.setGreedyToLeft(true);
    highlighter.setGreedyToRight(true);
    // for the condition `existing.equalsByActualOffset(info)` above work correctly,
    // create a fake whole-file highlighter which will track the document size changes
    // and which will make possible to calculate correct `info.getActualEndOffset()`
    info.setGroup(group);
    if (context != null) {
      CodeInsightContextHighlightingUtil.installCodeInsightContext(highlighter, project, context);
    }
    return highlighter;
  }
}
