// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.platform.eel.tcp

import com.intellij.execution.ijent.nio.IjentEphemeralRootAwareFileSystemProvider
import com.intellij.openapi.components.service
import com.intellij.openapi.diagnostic.logger
import com.intellij.platform.eel.annotations.MultiRoutingFileSystemPath
import com.intellij.platform.eel.impl.fs.telemetry.TracingFileSystemProvider
import com.intellij.platform.eel.provider.MultiRoutingFileSystemBackend
import com.intellij.platform.ijent.community.impl.IjentFailSafeFileSystemPosixApi
import com.intellij.platform.ijent.community.impl.nio.IjentNioFileSystemProvider
import java.net.URI
import java.nio.file.*
import java.util.concurrent.ConcurrentHashMap


class TcpEelMrfsBackend : MultiRoutingFileSystemBackend {
  companion object {
    private val LOG = logger<TcpEelMrfsBackend>()
  }

  private val cache = ConcurrentHashMap<String, FileSystem>()
  override fun compute(localFS: FileSystem, sanitizedPath: String): FileSystem? {
    val parsedInternalName = TcpEelPathParser.extractInternalMachineId(sanitizedPath) ?: return null
    val tcpDescriptor = TcpEelRegistry.getInstance().get(parsedInternalName) ?: return null
    val localPath = localFS.getPath(tcpDescriptor.rootPathString)
    if (Files.exists(localPath)) {
      LOG.warn("A file system for a path already exists: $localPath")
    }
    return cache.computeIfAbsent(parsedInternalName) { createFilesystem(it, tcpDescriptor, localPath, localFS) }
  }

  private fun createFilesystem(internalName: String, descriptor: TcpEelDescriptor, localPath: Path, localFS: FileSystem): FileSystem {
    val ijentUri = URI("ijent", "tcp", "/$internalName", null, null)
    val ijentDefaultProvider = TracingFileSystemProvider(IjentNioFileSystemProvider.getInstance())
    val scope = service<TcpEelScopeHolder>().coroutineScope

    try {
      val ijentFs = IjentFailSafeFileSystemPosixApi(scope, descriptor, checkIsIjentInitialized = null)
      ijentDefaultProvider.newFileSystem(ijentUri, IjentNioFileSystemProvider.newFileSystemMap(ijentFs))
    } catch (_: FileSystemAlreadyExistsException) {
      // Nothing.
    }
    LOG.info("New FileSystem initialized for $internalName at $localPath and URI=$ijentUri")
    return IjentEphemeralRootAwareFileSystemProvider(
      root = localPath,
      ijentFsProvider = ijentDefaultProvider,
      originalFsProvider = TracingFileSystemProvider(localFS.provider()),
      useRootDirectoriesFromOriginalFs = false
    ).getFileSystem(ijentUri)
  }

  override fun getCustomRoots(): Collection<@MultiRoutingFileSystemPath String> {
    return cache.keys.map { "${TcpEelConstants.TCP_PROTOCOL_PREFIX}$it" }
  }

  override fun getCustomFileStores(localFS: FileSystem): Collection<FileStore> {
    return cache.values.flatMap { it.fileStores }
  }
}