// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.execution.configurations;

import com.intellij.diagnostic.PluginException;
import com.intellij.execution.BeforeRunTask;
import com.intellij.execution.RunManager;
import com.intellij.execution.RunnerAndConfigurationSettings;
import com.intellij.openapi.components.BaseState;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NlsSafe;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.Icon;

/**
 * Factory for run configuration instances.
 *
 * @see ConfigurationType#getConfigurationFactories()
 * @see <a href="https://plugins.jetbrains.com/docs/intellij/run-configurations.html">Execution / Run Configurations (IntelliJ Platform Docs)</a>
 */
public abstract class ConfigurationFactory {
  public static final ConfigurationFactory[] EMPTY_ARRAY = new ConfigurationFactory[0];

  private final ConfigurationType myType;

  protected ConfigurationFactory(@NotNull ConfigurationType type) {
    myType = type;
  }

  @ApiStatus.Internal
  protected ConfigurationFactory() {
    myType = null;
  }

  /**
   * Creates a new run configuration with the specified name by cloning the specified template.
   *
   * @param name the name for the new run configuration.
   * @param template the template from which the run configuration is copied
   * @return the new run configuration.
   */
  public @NotNull RunConfiguration createConfiguration(@NlsSafe @Nullable String name, @NotNull RunConfiguration template) {
    RunConfiguration newConfiguration = template.clone();
    newConfiguration.setName(name);
    return newConfiguration;
  }

  /**
   * Override this method and return {@code false} to hide the configuration from 'New' popup in 'Edit Configurations' dialog.
   * It still will be possible to create this configuration by clicking on '42 more items' in the 'New' popup.
   *
   * @return {@code true} if it makes sense to create configurations of this type in {@code project}
   */
  public boolean isApplicable(@NotNull Project project) {
    return true;
  }

  /**
   * Creates a new template run configuration within the context of the specified project.
   *
   * @param project the project in which the run configuration will be used
   */
  public abstract @NotNull RunConfiguration createTemplateConfiguration(@NotNull Project project);

  public @NotNull RunConfiguration createTemplateConfiguration(@NotNull Project project, @NotNull RunManager runManager) {
    return createTemplateConfiguration(project);
  }

  /**
   * Returns the id of the run configuration that is used for serialization. For compatibility reason the default implementation calls
   * the method {@link #getName()} and this may cause problems if {@link #getName} returns localized value. So the default implementation
   * <strong>must be overridden</strong> in all inheritors. In existing implementations you need to use the same value which is returned
   * by {@link #getName()} for compatibility but store it directly in the code instead of taking from a message bundle. For new configurations
   * you may use any unique ID; if a new {@link ConfigurationType} has a single {@link ConfigurationFactory}, use {@link SimpleConfigurationType} instead.
   */
  public @NotNull @NonNls String getId() {
    PluginException.reportDeprecatedDefault(
      getClass(), "getId",
      "The default implementation delegates to 'getName' which may be localized, but return value of this method must not depend on current localization.");
    return getName();
  }

  /**
   * The name of the run configuration variant created by this factory.
   */
  public @NotNull @Nls String getName() {
    // null only if SimpleConfigurationType (but method overridden)
    //noinspection ConstantConditions
    return myType.getDisplayName();
  }

  public Icon getIcon(final @NotNull RunConfiguration configuration) {
    return getIcon();
  }

  public Icon getIcon() {
    // null only if SimpleConfigurationType (but method overridden)
    //noinspection ConstantConditions
    return myType.getIcon();
  }

  public @NotNull ConfigurationType getType() {
    // null only if SimpleConfigurationType (but method overridden)
    //noinspection ConstantConditions
    return myType;
  }

  public void configureDefaultSettings(@NotNull RunnerAndConfigurationSettings settings) {
  }

  /**
   * In this method you can configure defaults for the task, which are preferable to be used for your particular configuration type
   */
  @SuppressWarnings("rawtypes")
  public void configureBeforeRunTaskDefaults(Key<? extends BeforeRunTask> providerID, BeforeRunTask task) { }

  /** @deprecated Use {@link RunConfigurationSingletonPolicy} */
  @Deprecated
  public boolean isConfigurationSingletonByDefault() {
    return getSingletonPolicy() != RunConfigurationSingletonPolicy.MULTIPLE_INSTANCE;
  }

  /** @deprecated Use {@link RunConfigurationSingletonPolicy} */
  @Deprecated
  public boolean canConfigurationBeSingleton() {
    return getSingletonPolicy() != RunConfigurationSingletonPolicy.SINGLE_INSTANCE_ONLY;
  }

  public @NotNull RunConfigurationSingletonPolicy getSingletonPolicy() {
    return RunConfigurationSingletonPolicy.SINGLE_INSTANCE;
  }

  public boolean isEditableInDumbMode() {
    return false;
  }

  public @Nullable Class<? extends BaseState> getOptionsClass() {
    return null;
  }
}
