// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.editor.impl;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.colors.FontPreferences;
import com.intellij.util.text.CharArrayUtil;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.intellij.lang.annotations.JdkConstants;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

import java.awt.*;
import java.awt.font.FontRenderContext;
import java.util.*;
import java.util.List;

public final class ComplementaryFontsRegistry {
  private static final Logger LOG = Logger.getInstance(ComplementaryFontsRegistry.class);
  private static final String DEFAULT_FALLBACK_FONT = Font.MONOSPACED;
  private static final Object lock = new Object();
  @SuppressWarnings("unchecked")
  private static final List<String>[] ourFontNames = new List[4]; // per font style
  @SuppressWarnings("unchecked")
  private static final LinkedHashMap<String, FallBackInfo>[] ourUsedFonts = new LinkedHashMap[] { // per font style
    new LinkedHashMap<String, FallBackInfo>(), new LinkedHashMap<String, FallBackInfo>(),
    new LinkedHashMap<String, FallBackInfo>(), new LinkedHashMap<String, FallBackInfo>()
  };
  private static final Map<Font, FallBackInfo> ourMainUsedFonts = new HashMap<>();
  // This is the font that will be used to show placeholder glyphs for characters no installed font can display.
  // Glyph with code 0 will be used as a placeholder from this font.
  private static final FallBackInfo UNDISPLAYABLE_FONT_INFO = new FallBackInfo("JetBrains Mono", Font.PLAIN);
  private static final IntSet[] ourUndisplayableChars = new IntOpenHashSet[] { // per font style
    new IntOpenHashSet(), new IntOpenHashSet(), new IntOpenHashSet(), new IntOpenHashSet()
  };
  private static String ourLastFontFamily = null;
  private static String ourLastRegularSubFamily;
  private static String ourLastBoldSubFamily;
  private static boolean ourLastTypographicNames;
  private static final FallBackInfo[] ourLastFallBackInfo = new FallBackInfo[4]; // per font style

  private ComplementaryFontsRegistry() {
  }

  private static final @NonNls String BOLD_SUFFIX = ".bold";

  private static final @NonNls String ITALIC_SUFFIX = ".italic";

  // This font renders all characters as empty glyphs, so there's no reason to use it for fallback
  private static final String ADOBE_BLANK = "Adobe Blank";

  static {
    List<String> fontNames = new ArrayList<>();
    if (ApplicationManager.getApplication().isUnitTestMode()) {
      fontNames.add("Monospaced");
    } else {
      // This must match the corresponding call in com.intellij.idea.StartupUtil#updateFrameClassAndWindowIconAndPreloadSystemFonts for optimal performance
      String[] families = GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
      for (final String fontName : families) {
        if (!fontName.endsWith(BOLD_SUFFIX) && !fontName.endsWith(ITALIC_SUFFIX) && !fontName.equals(ADOBE_BLANK)) {
          fontNames.add(fontName);
        }
      }
    }
    ourFontNames[0] = fontNames;
    for (int i = 1; i < 4; i++) {
      ourFontNames[i] = new ArrayList<>(fontNames);
    }
  }

  /**
   * If you intend to use font metrics from returned {@link FontInfo} object,
   * pass not-null correct {@link FontRenderContext} to this method.
   */
  public static @NotNull FontInfo getFontAbleToDisplay(@NotNull CharSequence text, int start, int end,
                                              @JdkConstants.FontStyle int style, @NotNull FontPreferences preferences,
                                              FontRenderContext context) {
    assert 0 <= start && start < end && end <= text.length() : "Start: " + start + ", end: " + end + ", length: " + text.length();
    if (end - start == 1) {
      // fast path for BMP code points
      return getFontAbleToDisplay(text.charAt(start), style, preferences, context);
    }
    int firstCodePoint = Character.codePointAt(text, start);
    int secondOffset = Character.offsetByCodePoints(text, start, 1);
    if (secondOffset == end) {
      // fast path for a single SMP code point
      return getFontAbleToDisplay(firstCodePoint, style, preferences, context);
    }
    char[] tmp = CharArrayUtil.fromSequence(text, secondOffset, end);
    return getFontAbleToDisplay(firstCodePoint, tmp, 0, tmp.length, style, preferences, context);
  }

  /**
   * If you intend to use font metrics from returned {@link FontInfo} object,
   * pass not-null correct {@link FontRenderContext} to this method.
   */
  public static @NotNull FontInfo getFontAbleToDisplay(char @NotNull [] text, int start, int end,
                                              @JdkConstants.FontStyle int style, @NotNull FontPreferences preferences,
                                              FontRenderContext context) {
    assert 0 <= start && start < end && end <= text.length : "Start: " + start + ", end: " + end + ", length: " + text.length;
    if (end - start == 1) {
      // fast path for BMP code points
      return getFontAbleToDisplay(text[start], style, preferences, context);
    }
    int firstCodePoint = Character.codePointAt(text, start);
    int secondOffset = Character.offsetByCodePoints(text, start, end - start, start, 1);
    if (secondOffset == end) {
      // fast path for a single SMP code point
      return getFontAbleToDisplay(firstCodePoint, style, preferences, context);
    }
    return getFontAbleToDisplay(firstCodePoint, text, secondOffset, end, style, preferences, context);
  }

  private static FontInfo getFontAbleToDisplay(int codePoint, char @NotNull [] remainingText, int start, int end,
                                               @JdkConstants.FontStyle int style, @NotNull FontPreferences preferences,
                                               FontRenderContext context) {
    boolean tryDefaultFallback = true;
    List<String> fontFamilies = preferences.getEffectiveFontFamilies();
    boolean useLigatures = preferences.useLigatures();
    Set<String> variants = preferences.getCharacterVariants();
    FontInfo result;
    for (int i = 0, len = fontFamilies.size(); i < len; ++i) { // avoid foreach, it instantiates ArrayList$Itr, this traversal happens very often
      final String fontFamily = fontFamilies.get(i);
      result = doGetFontAbleToDisplay(codePoint, preferences.getSize2D(fontFamily), style, fontFamily,
                                      i == 0 ? preferences.getRegularSubFamily() : null, i == 0 ? preferences.getBoldSubFamily() : null,
                                      useLigatures, variants, context, true, true);
      if (result != null && result.getFont().canDisplayUpTo(remainingText, start, end) == -1) {
        return result;
      }
      tryDefaultFallback &= !DEFAULT_FALLBACK_FONT.equals(fontFamily);
    }
    float size = FontPreferences.DEFAULT_FONT_SIZE;
    if (!fontFamilies.isEmpty()) {
      size = preferences.getSize2D(fontFamilies.get(0));
    }
    if (tryDefaultFallback) {
      result = doGetFontAbleToDisplay(codePoint, size, style, DEFAULT_FALLBACK_FONT, null, null, useLigatures, Collections.emptySet(), context, false, false);
      if (result != null && result.getFont().canDisplayUpTo(remainingText, start, end) == -1) {
        return result;
      }
    }
    result = doGetFontAbleToDisplay(codePoint, remainingText, start, end, size, style, useLigatures, Collections.emptySet(), context);
    if (LOG.isTraceEnabled()) {
      LOG.trace("Fallback font: " + result.getFont().getFontName());
    }
    return result;
  }

  /**
   * If you intend to use font metrics from returned {@link FontInfo} object,
   * pass not-null correct {@link FontRenderContext} to this method.
   */
  public static @NotNull FontInfo getFontAbleToDisplay(int codePoint, @JdkConstants.FontStyle int style, @NotNull FontPreferences preferences,
                                              FontRenderContext context) {
    boolean tryDefaultFallback = true;
    List<String> fontFamilies = preferences.getEffectiveFontFamilies();
    boolean useLigatures = preferences.useLigatures();
    Set<String> variants = preferences.getCharacterVariants();
    FontInfo result;
    for (int i = 0, len = fontFamilies.size(); i < len; ++i) { // avoid foreach, it instantiates ArrayList$Itr, this traversal happens very often
      final String fontFamily = fontFamilies.get(i);
      result = doGetFontAbleToDisplay(codePoint, preferences.getSize2D(fontFamily), style, fontFamily,
                                      i == 0 ? preferences.getRegularSubFamily() : null, i == 0 ? preferences.getBoldSubFamily() : null,
                                      useLigatures, variants, context, true, true);
      if (result != null) {
        return result;
      }
      tryDefaultFallback &= !DEFAULT_FALLBACK_FONT.equals(fontFamily);
    }
    float size = FontPreferences.DEFAULT_FONT_SIZE;
    if (!fontFamilies.isEmpty()) {
      size = preferences.getSize2D(fontFamilies.get(0));
    }
    if (tryDefaultFallback) {
      result = doGetFontAbleToDisplay(codePoint, size, style, DEFAULT_FALLBACK_FONT, null, null, useLigatures, Collections.emptySet(), context, false, false);
      if (result != null) {
        return result;
      }
    }
    result = doGetFontAbleToDisplay(codePoint, null, 0, 0, size, style, useLigatures, Collections.emptySet(), context);
    if (LOG.isTraceEnabled()) {
      LOG.trace("Fallback font: " + result.getFont().getFontName());
    }
    return result;
  }

  /**
   * If you intend to use font metrics from returned {@link FontInfo} object,
   * pass not-null correct {@link FontRenderContext} to this method.
   */
  public static @NotNull FontInfo getFontAbleToDisplay(int codePoint, int size, @JdkConstants.FontStyle int style, @NotNull String defaultFontFamily,
                                              FontRenderContext context) {
    FontInfo result = doGetFontAbleToDisplay(codePoint, size, style, defaultFontFamily, null, null, false, context, false, false);
    if (result != null) {
      return result;
    }
    if (!DEFAULT_FALLBACK_FONT.equals(defaultFontFamily)) {
      result = doGetFontAbleToDisplay(codePoint, size, style, DEFAULT_FALLBACK_FONT, null, null, false, context, false, false);
      if (result != null) {
        return result;
      }
    }
    return doGetFontAbleToDisplay(codePoint, null, 0, 0, size, style, false, context);
  }

  private static @Nullable FontInfo doGetFontAbleToDisplay(int codePoint, float size, @JdkConstants.FontStyle int style,
                                                           @NotNull String defaultFontFamily, String regularSubFamily, String boldSubFamily,
                                                           boolean useLigatures, FontRenderContext context, boolean disableFontFallback,
                                                           boolean useTypographicNames) {
    return doGetFontAbleToDisplay(codePoint, size, style, defaultFontFamily, regularSubFamily, boldSubFamily,
                                  useLigatures, Collections.emptySet(), context, disableFontFallback, useTypographicNames);
  }

  private static @Nullable FontInfo doGetFontAbleToDisplay(int codePoint, float size, @JdkConstants.FontStyle int style,
                                                           @NotNull String defaultFontFamily, String regularSubFamily, String boldSubFamily,
                                                           boolean useLigatures, @Unmodifiable @NotNull Set<@NotNull String> variants, FontRenderContext context, boolean disableFontFallback,
                                                           boolean useTypographicNames) {
    if (style < 0 || style > 3) style = Font.PLAIN;
    synchronized (lock) {
      FallBackInfo fallBackInfo = null;
      if (useTypographicNames == ourLastTypographicNames &&
          defaultFontFamily.equals(ourLastFontFamily) &&
          (!useTypographicNames ||
           Objects.equals(regularSubFamily, ourLastRegularSubFamily) && Objects.equals(boldSubFamily, ourLastBoldSubFamily))) {
        fallBackInfo = ourLastFallBackInfo[style];
      }
      else {
        ourLastTypographicNames = useTypographicNames;
        ourLastFontFamily = defaultFontFamily;
        ourLastRegularSubFamily = regularSubFamily;
        ourLastBoldSubFamily = boldSubFamily;
        Arrays.fill(ourLastFallBackInfo, null);
      }
      if (fallBackInfo == null) {
        if (useTypographicNames) {
          Font font = FontFamilyService.getFont(defaultFontFamily, regularSubFamily, boldSubFamily, style);
          fallBackInfo = ourMainUsedFonts.computeIfAbsent(font, FallBackInfo::new);
        }
        else {
          LinkedHashMap<String, FallBackInfo> usedFonts = ourUsedFonts[style];
          fallBackInfo = usedFonts.get(defaultFontFamily);
          if (fallBackInfo == null) {
            fallBackInfo = new FallBackInfo(defaultFontFamily, style);
            usedFonts.put(defaultFontFamily, fallBackInfo);
          }
        }
        ourLastFallBackInfo[style] = fallBackInfo;
      }
      return fallBackInfo.canDisplay(codePoint, disableFontFallback) ? fallBackInfo.getFontInfo(size, useLigatures, variants, context) : null;
    }
  }

  private static @NotNull FontInfo doGetFontAbleToDisplay(int codePoint, char[] remainingText, int start, int end,
                                                          float size, @JdkConstants.FontStyle int style, boolean useLigatures,
                                                          FontRenderContext context) {
    return doGetFontAbleToDisplay(codePoint, remainingText, start, end, size, style, useLigatures,
                                  Collections.emptySet(), context);
  }

  private static @NotNull FontInfo doGetFontAbleToDisplay(int codePoint, char[] remainingText, int start, int end,
                                                          float size, @JdkConstants.FontStyle int style, boolean useLigatures,
                                                          @Unmodifiable @NotNull Set<@NotNull String> variants, FontRenderContext context) {
    if (style < 0 || style > 3) style = Font.PLAIN;
    synchronized (lock) {
      FallBackInfo fallBackInfo = UNDISPLAYABLE_FONT_INFO;
      IntSet undisplayableChars = ourUndisplayableChars[style];
      if (!undisplayableChars.contains(codePoint)) {
        boolean canDisplayFirst = false;
        LinkedHashMap<String, FallBackInfo> usedFonts = ourUsedFonts[style];
        final Collection<FallBackInfo> descriptors = usedFonts.values();
        for (FallBackInfo info : descriptors) {
          if (info.canDisplay(codePoint, false)) {
            canDisplayFirst = true;
            if (remainingText == null || info.myBaseFont.canDisplayUpTo(remainingText, start, end) == -1) {
              fallBackInfo = info;
              break;
            }
          }
        }
        if (fallBackInfo == UNDISPLAYABLE_FONT_INFO) {
          List<String> fontNames = ourFontNames[style];
          for (int i = 0; i < fontNames.size(); i++) {
            String name = fontNames.get(i);
            FallBackInfo info = new FallBackInfo(name, style);
            if (info.canDisplay(codePoint, false)) {
              canDisplayFirst = true;
              if (remainingText == null || info.myBaseFont.canDisplayUpTo(remainingText, start, end) == -1) {
                usedFonts.put(name, info);
                fontNames.remove(i);
                fallBackInfo = info;
                break;
              }
            }
          }
          if (fallBackInfo == UNDISPLAYABLE_FONT_INFO && !canDisplayFirst) {
            undisplayableChars.add(codePoint);
          }
        }
      }
      return fallBackInfo.getFontInfo(size, useLigatures, variants, context);
    }
  }

  private static final class FontKey implements Cloneable {
    private float mySize;
    private boolean myUseLigatures;
    private @NotNull List<@NotNull String> myVariants;
    private FontRenderContext myContext;

    private FontKey(float size, boolean useLigatures, @NotNull Set<@NotNull String> variants, FontRenderContext context) {
      mySize = size;
      myUseLigatures = useLigatures;
      myVariants = variants.isEmpty() ? Collections.emptyList() : new ArrayList<>(variants);
      myContext = context;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      FontKey key = (FontKey)o;

      if (mySize != key.mySize) return false;
      if (myUseLigatures != key.myUseLigatures) return false;
      if (!Objects.equals(myContext, key.myContext)) return false;
      if (!Objects.equals(myVariants, key.myVariants)) return false;

      return true;
    }

    @Override
    public int hashCode() {
      int result = (mySize != 0.0f ? Float.floatToIntBits(mySize) : 0);
      result = 31 * result + (myUseLigatures ? 1 : 0);
      result = 31 * result + (myContext != null ? myContext.hashCode() : 0);
      result = 31 * result + myVariants.hashCode();
      return result;
    }

    @Override
    protected FontKey clone() {
      try {
        return (FontKey)super.clone();
      }
      catch (CloneNotSupportedException e) {
        throw new RuntimeException(e);
      }
    }
  }

  private static final class FallBackInfo {
    private final Font myBaseFont;
    private final Map<FontKey, FontInfo> myFontInfoMap = new HashMap<>();
    private final FontKey myLastFontKey = new FontKey(-1, false, Collections.emptySet(), FontInfo.DEFAULT_CONTEXT);
    private FontInfo myLastFontInfo;

    private FallBackInfo(Font font) {
      myBaseFont = font;
    }

    private FallBackInfo(String familyName, @JdkConstants.FontStyle int style) {
      myBaseFont = new Font(familyName, style, 1);
    }

    private boolean canDisplay(int codePoint, boolean disableFontFallback) {
      return codePoint < 128 || FontInfo.canDisplay(myBaseFont, codePoint, disableFontFallback);
    }

    private FontInfo getFontInfo(float size, boolean useLigatures, @Unmodifiable @NotNull Set<@NotNull String> variants, FontRenderContext fontRenderContext) {
      if (myLastFontKey.mySize == size &&
          myLastFontKey.myUseLigatures == useLigatures &&
          Objects.equals(myLastFontKey.myVariants, variants.isEmpty() ? Collections.emptyList() : new ArrayList<>(variants)) &&
          Objects.equals(myLastFontKey.myContext, fontRenderContext)) {
        return myLastFontInfo;
      }
      myLastFontKey.mySize = size;
      myLastFontKey.myUseLigatures = useLigatures;
      myLastFontKey.myVariants = variants.isEmpty() ? Collections.emptyList() : new ArrayList<>(variants);
      myLastFontKey.myContext = fontRenderContext;
      FontInfo fontInfo = myFontInfoMap.get(myLastFontKey);
      if (fontInfo == null) {
        fontInfo = new FontInfo(myBaseFont, size, useLigatures, variants, fontRenderContext);
        myFontInfoMap.put(myLastFontKey.clone(), fontInfo);
      }
      myLastFontInfo = fontInfo;
      return fontInfo;
    }
  }
}
