// Copyright 2000-2026 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.editor.impl.zombie

import com.intellij.ide.plugins.PluginManagerCore
import com.intellij.idea.AppMode
import com.intellij.openapi.Disposable
import com.intellij.openapi.application.PathManager
import com.intellij.openapi.application.WriteIntentReadAction
import com.intellij.openapi.application.readActionBlocking
import com.intellij.openapi.components.Service
import com.intellij.openapi.components.service
import com.intellij.openapi.components.serviceAsync
import com.intellij.openapi.components.serviceIfCreated
import com.intellij.openapi.diagnostic.ControlFlowException
import com.intellij.openapi.diagnostic.Logger
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.editor.Document
import com.intellij.openapi.editor.Editor
import com.intellij.openapi.editor.EditorKind
import com.intellij.openapi.editor.ex.EditorEx
import com.intellij.openapi.extensions.ExtensionPointName
import com.intellij.openapi.fileEditor.FileDocumentManager
import com.intellij.openapi.project.Project
import com.intellij.openapi.vfs.FileIdAdapter
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.util.concurrency.ThreadingAssertions
import kotlinx.coroutines.*
import org.jetbrains.annotations.TestOnly
import java.nio.file.Path
import java.util.concurrent.CancellationException
import java.util.concurrent.atomic.AtomicReference
import kotlin.time.Duration.Companion.minutes


/**
 * Service managing all necromancers.
 *
 * See [Zombie]
 */
@Service(Service.Level.PROJECT)
class Necropolis(private val project: Project, private val coroutineScope: CoroutineScope) : Disposable {

  companion object {
    private val LOG: Logger = logger<Necropolis>()
    private val NECROMANCER_EP = ExtensionPointName<NecromancerAwaker<Zombie>>("com.intellij.textEditorNecromancerAwaker")

    @JvmStatic
    fun getInstance(project: Project, onlyIfCreated: Boolean = false): Necropolis? {
      return if (isEnabled()) {
        if (onlyIfCreated) {
          project.serviceIfCreated<Necropolis>()
        } else {
          project.service<Necropolis>()
        }
      } else {
        null
      }
    }

    @JvmStatic
    suspend fun getInstanceAsync(project: Project): Necropolis? {
      return if (isEnabled()) project.serviceAsync<Necropolis>() else null
    }

    internal fun necropolisPath(): Path {
      return PathManager.getSystemDir().resolve("editor")
    }

    private fun isEnabled(): Boolean = !AppMode.isRemoteDevHost()
  }

  private val necromancersDeferred: Deferred<List<Necromancer<Zombie>>>
  private val necromancersRef: AtomicReference<List<Necromancer<Zombie>>> = AtomicReference()

  init {
    necromancersDeferred = coroutineScope.async {
      service<NecropolisDestroyer>().cleanGravesIfNeeded()
      val necromancers = createAwakers().map {
        async(Dispatchers.IO + CoroutineName(it.javaClass.name)) {
          it.awake(project, coroutineScope)
        }
      }.awaitAll()
      necromancersRef.set(necromancers)
      necromancers
    }
  }

  suspend fun spawnZombies(
    project: Project,
    file: VirtualFile,
    document: Document,
    editorSupplier: suspend () -> EditorEx,
    highlighterReady: suspend () -> Unit,
  ) {
    require(project == this.project)
    if (!project.isDisposed && !project.isDefault) {
      val fileId = FileIdAdapter.getInstance().getId(file) ?: return
      val (modStamp, documentContent) = readActionBlocking {
        // get consistent modStamp with docContent under RA
        document.modificationStamp to document.immutableCharSequence
      }
      val fingerprint = FingerprintedZombieImpl.captureFingerprint(documentContent)
      val recipe = SpawnRecipe(project, fileId, file, document, modStamp, editorSupplier, highlighterReady)
      coroutineScope {
        for (necromancer in necromancersDeferred.await()) {
          if (necromancer.enoughMana(recipe)) {
            launch(CoroutineName(necromancer.name())) {
              try {
                if (!project.isDisposed && necromancer.shouldSpawnZombie(recipe)) {
                  val zombie = exhumeZombieIfValid(recipe, necromancer, fingerprint)
                  necromancer.spawnZombie(recipe, zombie)
                }
              } catch (e: CancellationException) {
                throw e
              } catch (e: Throwable) {
                LOG.warn(
                  "Exception during editor loading",
                  if (e is ControlFlowException) RuntimeException(e) else e
                )
              }
            }
          }
        }
      }
    }
  }

  fun turnIntoZombiesAndBury(editor: Editor) {
    ThreadingAssertions.assertEventDispatchThread()
    require(editor.project == this.project)
    val necromancers = necromancersRef.get()
    if (necromancers != null) {
      val recipe = createTurningRecipe(editor)
      if (recipe != null) {
        //maybe readaction
        WriteIntentReadAction.run {
          turnIntoZombiesAndBury(necromancers, recipe)
        }
      }
    }
  }

  private fun createTurningRecipe(editor: Editor): TurningRecipe? {
    if (editor.editorKind == EditorKind.MAIN_EDITOR) {
      val document = editor.document
      val file = FileDocumentManager.getInstance().getFile(document)
      if (file != null) {
        val fileId = FileIdAdapter.getInstance().getId(file)
        if (fileId != null) {
          return TurningRecipe(project, fileId, file, document, document.modificationStamp, editor)
        }
      }
    }
    return null
  }

  private fun turnIntoZombiesAndBury(necromancers: List<Necromancer<Zombie>>, recipe: TurningRecipe) {
    val zombies = turnIntoZombies(necromancers, recipe)
    if (LOG.isDebugEnabled) {
      LOG.debug("Turned into zombies for ${recipe.fileId}: ${zombies.map { it.first.name() }}")
    }
    if (zombies.isNotEmpty()) {
      coroutineScope.launch {
        withContext(NonCancellable) {
          withTimeout(1.minutes) {
            buryZombies(zombies, recipe)
          }
        }
      }
    }
  }

  private fun turnIntoZombies(
    necromancers: List<Necromancer<Zombie>>,
    recipe: TurningRecipe,
  ): List<Pair<Necromancer<Zombie>, Zombie>> {
    return necromancers
      .filter { it.enoughMana(recipe) }
      .mapNotNull { necromancer ->
        necromancer.turnIntoZombie(recipe)?.let { zombie ->
          necromancer to zombie
        }
      }.toList()
  }

  private suspend fun CoroutineScope.buryZombies(
    zombies: List<Pair<Necromancer<Zombie>, Zombie>>,
    recipe: TurningRecipe,
  ) {
    val documentContent = readActionBlocking {
      if (recipe.isValid()) {
        recipe.document.immutableCharSequence
      } else {
        LOG.debug("Invalid recipe for ${recipe.fileId}}")
        null
      }
    }
    if (documentContent != null) {
      val fingerprint = FingerprintedZombieImpl.captureFingerprint(documentContent)
      for ((necromancer, zombie) in zombies) {
        launch(CoroutineName(necromancer.name())) {
          if (recipe.isValid() && necromancer.shouldBuryZombie(recipe, zombie)) {
            necromancer.buryZombie(recipe.fileId, FingerprintedZombieImpl(fingerprint, zombie))
            LOG.debug("Buried ${necromancer.name()} for ${recipe.fileId}")
          }
        }
      }
    }
  }

  private suspend fun exhumeZombieIfValid(
    recipe: SpawnRecipe,
    necromancer: Necromancer<Zombie>,
    fingerprint: Long,
  ): Zombie? {
    if (recipe.isValid()) {
      val fingerprinted = necromancer.exhumeZombie(recipe.fileId)
      if (fingerprinted?.fingerprint() == fingerprint) {
        return fingerprinted.zombie()
      }
    }
    return null
  }

  private fun createAwakers(): List<NecromancerAwaker<Zombie>> {
    val allAwakers = NECROMANCER_EP.filterableLazySequence()
      .filter {
        val isCore = it.pluginDescriptor.pluginId == PluginManagerCore.CORE_ID ||
                     it.implementationClassName == "com.jetbrains.rider.daemon.grave.RiderHighlightingNecromancerAwaker"
        if (!isCore) {
          LOG.error("Only core plugin can define ${NECROMANCER_EP.name}: ${it.pluginDescriptor}")
        }
        isCore
      }
      .mapNotNull { it.instance }
      .toList()
    return allAwakers
      .filter { !hasSubclass(it, allAwakers) } // workaround for Rider's overriding of HighlightingNecromancer
      .toList()
  }

  private fun hasSubclass(obj: Any, allObj: List<Any>): Boolean {
    for (o in allObj) {
      if (obj !== o && obj.javaClass.isAssignableFrom(o.javaClass)) {
        return true
      }
    }
    return false
  }

  override fun dispose() {
  }

  private fun getNecromancers(): List<Necromancer<Zombie>> {
    return necromancersRef.get() ?: throw IllegalStateException("necromancers are not initialized yet")
  }

  @TestOnly
  fun necromancerByName(name: String): Necromancer<Zombie> {
    return getNecromancers().find { it.name() == name }!!
  }

  override fun toString(): String {
    val necromancersStr = necromancersRef.get()?.joinToString(", ") { it.name() }
    return "Necropolis(project=${project.name}, necromancers=[$necromancersStr])"
  }
}
