// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.progress.impl

import com.intellij.openapi.components.Service
import com.intellij.openapi.components.serviceAsync
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.diagnostic.trace
import com.intellij.openapi.progress.ProgressIndicatorModel
import com.intellij.openapi.progress.ProgressManager
import com.intellij.openapi.progress.ProgressModel
import com.intellij.openapi.progress.util.ProgressIndicatorBase
import com.intellij.openapi.project.Project
import com.intellij.openapi.project.ProjectManager
import com.intellij.platform.ide.progress.*
import com.intellij.platform.ide.progress.suspender.TaskSuspension
import com.intellij.platform.project.projectId
import fleet.kernel.rete.asValuesFlow
import fleet.kernel.rete.collect
import fleet.kernel.rete.collectLatest
import fleet.kernel.rete.filter
import fleet.kernel.tryWithEntities
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.awaitCancellation
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch

private val LOG = logger<TaskInfoEntityCollector>()

internal class TaskInfoEntityCollector(cs: CoroutineScope) {
  init {
    LOG.trace { "TaskInfoEntityCollector started for application"}
    collectActiveTasks(cs, project = null)
  }
}

@Service(Service.Level.PROJECT)
internal class PerProjectTaskInfoEntityCollector(private val project: Project, private val cs: CoroutineScope) {
  fun startCollectingActiveTasks() {
    LOG.trace { "PerProjectTaskInfoEntityCollector started for $project" }
    collectActiveTasks(cs, project)
  }
}

private fun collectActiveTasks(cs: CoroutineScope, project: Project?) {
  cs.launch {
    val projectOrDefault = project ?: serviceAsync<ProjectManager>().defaultProject
    activeTasks
      .filter { it.projectEntity?.projectId == project?.projectId() }
      .collect { task ->
        showTaskIndicator(cs, projectOrDefault, task)
      }
  }
}

private fun showTaskIndicator(cs: CoroutineScope, project: Project, task: TaskInfoEntity) {
  cs.launch {
    tryWithEntities(task) {
      LOG.trace { "Showing indicator for task: entityId=${task.eid}, title=${task.title}, project=$project" }

      val progressModel = if (isRhizomeProgressModelEnabled) {
        ProgressTaskInfoEntityModel(task, cs)
      }
      else {
        val entityId = task.eid
        val title = task.title
        ProgressIndicatorModel(task.title, task.cancellation, visibleInStatusBar = task.visibleInStatusBar) {
          LOG.trace { "Cancelling task: entityId=$entityId, title=$title" }
          cs.launch {
            TaskManager.cancelTask(task, TaskStatus.Source.USER)
          }
        }
      }

      showIndicator(
        project,
        progressModel,
        task.updates.asValuesFlow()
      )

      collectSuspendableChanges(task, progressModel)
    }
  }
}

private suspend fun collectSuspendableChanges(task: TaskInfoEntity, progressModel: ProgressModel) {
  task.suspensionState.collectLatest {
    markSuspendable(task, progressModel)
  }
}

private suspend fun CoroutineScope.markSuspendable(task: TaskInfoEntity, progressModel: ProgressModel) {
  val suspendableInfo = task.suspension
  if (suspendableInfo !is TaskSuspension.Suspendable) return

  // HACK: tempIndicator is required to avoid runProcess stopping the original indicator when the execution is finished
  val tempIndicator = ProgressIndicatorBase()
  val suspender = ProgressManager.getInstance().runProcess<ProgressSuspender>(
    { ProgressSuspender.markSuspendable(tempIndicator, suspendableInfo.suspendText) }, tempIndicator)

  try {
    val suspenderStateChange = MutableSharedFlow<Unit>(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)

    ProgressSuspenderTracker.getInstance().startTracking(suspender, object : ProgressSuspenderTracker.SuspenderListener {
      override fun onStateChanged(progressSuspender: ProgressSuspender) {
        suspenderStateChange.tryEmit(Unit)
      }
    })

    // Instead of markSuspendable, which has to be called under runProcess, we can use attachToProgress on the already created suspender
    suspender.attachToProgress(progressModel.getProgressIndicator()) //propagate events to original indicator

    launch {
      suspenderStateChange.collectLatest {
        if (suspender.isSuspended) {
          TaskManager.pauseTask(task, suspender.suspendedText, TaskStatus.Source.USER)
        }
        else {
          TaskManager.resumeTask(task, TaskStatus.Source.USER)
        }
      }
    }

    launch {
      // We shouldn't process events generated by TaskInfoEntityCollector to avoid infinite update cycles
      task.statuses
        .filter { it.source != TaskStatus.Source.USER }
        .collect { status ->
          when (status) {
            is TaskStatus.Paused -> suspender.suspendProcess(status.reason)
            is TaskStatus.Running -> suspender.resumeProcess()
            is TaskStatus.Canceled -> { /* do nothing */ }
          }
        }
    }

    awaitCancellation()
  }
  finally {
    ProgressSuspenderTracker.getInstance().stopTracking(suspender)
    suspender.close()
  }
}