// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.build.issue.quickfix

import com.intellij.build.issue.BuildIssueQuickFix
import com.intellij.codeInsight.highlighting.HighlightManager
import com.intellij.codeInsight.highlighting.HighlightUsagesHandler
import com.intellij.find.FindManager
import com.intellij.find.FindModel
import com.intellij.openapi.actionSystem.DataContext
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.editor.colors.EditorColors
import com.intellij.openapi.fileEditor.FileEditorManager
import com.intellij.openapi.fileEditor.OpenFileDescriptor
import com.intellij.openapi.project.Project
import com.intellij.openapi.vfs.VfsUtil
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.util.concurrency.annotations.RequiresEdt
import org.jetbrains.annotations.ApiStatus
import java.nio.file.Path
import java.util.concurrent.CompletableFuture

/**
 * @author Vladislav.Soroka
 */
@ApiStatus.Experimental
class OpenFileQuickFix(private val path: Path, private val search: String?) : BuildIssueQuickFix {
  override val id: String
    get() = path.toString()

  override fun runQuickFix(project: Project, dataContext: DataContext): CompletableFuture<*> {
    val future = CompletableFuture<Any>()
    ApplicationManager.getApplication().invokeLater {
      try {
        showFile(project, path, search)
        future.complete(null)
      }
      catch (e: Exception) {
        future.completeExceptionally(e)
      }
    }
    return future
  }

  companion object {
    @JvmStatic
    fun showFile(project: Project, path: Path, search: String?) {
      ApplicationManager.getApplication().invokeLater {
        val file = VfsUtil.findFileByIoFile(path.toFile(), false) ?: return@invokeLater
        doShowFile(project, file, search)
      }
    }

    @JvmStatic
    fun showFile(project: Project, file: VirtualFile, search: String?) {
      ApplicationManager.getApplication().invokeLater {
        doShowFile(project, file, search)
      }
    }

    @RequiresEdt
    private fun doShowFile(project: Project, file: VirtualFile, search: String?) {
      val editor = FileEditorManager.getInstance(project).openTextEditor(OpenFileDescriptor(project, file), false)
      if (search == null || editor == null) return

      val findModel = FindModel().apply { FindModel.initStringToFind(this, search) }
      val findResult = FindManager.getInstance(project).findString(editor.document.charsSequence, 0, findModel, file)
      val highlightManager = HighlightManager.getInstance(project)
      HighlightUsagesHandler.highlightRanges(highlightManager, editor, EditorColors.SEARCH_RESULT_ATTRIBUTES, false, listOf(findResult))
    }
  }
}