// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.editor.richcopy.view;

import com.intellij.openapi.editor.richcopy.FontMapper;
import com.intellij.openapi.editor.richcopy.model.ColorRegistry;
import com.intellij.openapi.editor.richcopy.model.FontNameRegistry;
import com.intellij.openapi.editor.richcopy.model.MarkupHandler;
import com.intellij.openapi.editor.richcopy.model.SyntaxInfo;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.ui.mac.MacColorSpaceLoader;
import org.jetbrains.annotations.NotNull;

import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.datatransfer.DataFlavor;

public final class RtfTransferableData extends AbstractSyntaxAwareInputStreamTransferableData {
  public static final int PRIORITY = 100;
  public static final DataFlavor FLAVOR = new DataFlavor("text/rtf;class=java.io.InputStream", "RTF text");

  private static final String HEADER_PREFIX = "{\\rtf1\\ansi\\deff0";
  private static final String HEADER_SUFFIX = "}";
  private static final String TAB           = "\\tab\n";
  // using undocumented way to denote line break on Mac (used e.g. by TextEdit) to resolve IDEA-165337
  private static final String NEW_LINE      = SystemInfo.isMac ? "\\\n" : "\\line\n";
  private static final String BOLD          = "\\b";
  private static final String ITALIC        = "\\i";

  public RtfTransferableData(@NotNull SyntaxInfo syntaxInfo) {
    super(syntaxInfo, FLAVOR);
  }

  @Override
  protected void build(final @NotNull StringBuilder holder, final int maxLength) {
    holder.append(HEADER_PREFIX);

    holder.append("{\\colortbl;");
    ColorRegistry colorRegistry = mySyntaxInfo.getColorRegistry();
    for (int id : colorRegistry.getAllIds()) {
      Color color = colorRegistry.dataById(id);
      int[] components = getAdjustedColorComponents(color);
      holder.append(String.format("\\red%d\\green%d\\blue%d;", components[0], components[1], components[2]));
    }
    holder.append("}\n");

    holder.append("{\\fonttbl");
    FontNameRegistry fontNameRegistry = mySyntaxInfo.getFontNameRegistry();
    for (int id : fontNameRegistry.getAllIds()) {
      String fontName = fontNameRegistry.dataById(id);
      holder.append("{\\f").append(id);
      if (FontMapper.isMonospaced(fontName)) {
        holder.append("\\fmodern");
      }
      holder.append(' ').append(fontName).append(";}");
    }
    holder.append("}\n");

    holder.append("\n\\s0\\box")
      .append("\\cbpat").append(mySyntaxInfo.getDefaultBackground())
      .append("\\cb").append(mySyntaxInfo.getDefaultBackground())
      .append("\\cf").append(mySyntaxInfo.getDefaultForeground());
    addFontSize(holder, mySyntaxInfo.getFontSize());
    holder.append('\n');

    mySyntaxInfo.processOutputInfo(new MyVisitor(holder, myRawText, mySyntaxInfo, maxLength));

    holder.append("\\par");
    holder.append(HEADER_SUFFIX);
  }

  private static int[] getAdjustedColorComponents(Color color) {
    ColorSpace genericRgbSpace;
    if (SystemInfo.isMac && (genericRgbSpace = MacColorSpaceLoader.getGenericRgbColorSpace()) != null) {
      // on macOS color components are expected in Apple's 'Generic RGB' color space
      float[] components = genericRgbSpace.fromRGB(color.getRGBColorComponents(null));
      return new int[]{
        colorComponentFloatToInt(components[0]),
        colorComponentFloatToInt(components[1]),
        colorComponentFloatToInt(components[2])
      };
    }
    else {
      return new int[]{color.getRed(), color.getGreen(), color.getBlue()};
    }
  }

  private static int colorComponentFloatToInt(float component) {
    return (int)(component * 255 + 0.5f);
  }

  @Override
  protected @NotNull String getCharset() {
    return "US-ASCII";
  }

  private static void addFontSize(StringBuilder buffer, float fontSize) {
    buffer.append("\\fs").append(Math.round(fontSize * 2));
  }

  @Override
  public int getPriority() {
    return PRIORITY;
  }

  private static final class MyVisitor implements MarkupHandler {
    private final @NotNull StringBuilder myBuffer;
    private final @NotNull String        myRawText;
    private final int myMaxLength;
    private final int myDefaultBackgroundId;
    private final float myFontSize;
    private int myForegroundId = -1;
    private int myFontNameId   = -1;
    private int myFontStyle    = -1;

    MyVisitor(@NotNull StringBuilder buffer, @NotNull String rawText, @NotNull SyntaxInfo syntaxInfo, int maxLength) {
      myBuffer = buffer;
      myRawText = rawText;
      myMaxLength = maxLength;

      myDefaultBackgroundId = syntaxInfo.getDefaultBackground();
      myFontSize = syntaxInfo.getFontSize();
    }

    @Override
    public void handleText(int startOffset, int endOffset) {
      myBuffer.append("\n");
      for (int i = startOffset; i < endOffset; i++) {
        char c = myRawText.charAt(i);
        if (c > 127) {
          // Escape non-ascii symbols.
          myBuffer.append(String.format("\\u%04d?", (int)c));
          continue;
        }

        switch (c) {
          case '\t' -> {
            myBuffer.append(TAB);
            continue;
          }
          case '\n' -> {
            myBuffer.append(NEW_LINE);
            continue;
          }
          case '\\', '{', '}' -> myBuffer.append('\\');
        }
        myBuffer.append(c);
      }
    }

    @Override
    public void handleBackground(int backgroundId) {
      if (backgroundId == myDefaultBackgroundId) {
        myBuffer.append("\\plain"); // we cannot use \chcbpat with default background id, as it doesn't work in MS Word,
                                    // and we cannot use \chcbpat0 as it doesn't work in OpenOffice

        addFontSize(myBuffer, myFontSize);
        if (myFontNameId >= 0) {
          handleFont(myFontNameId);
        }
        if (myForegroundId >= 0) {
          handleForeground(myForegroundId);
        }
        if (myFontStyle >= 0) {
          handleStyle(myFontStyle);
        }
      }
      else {
        myBuffer.append("\\chcbpat").append(backgroundId);
      }
      myBuffer.append("\\cb").append(backgroundId);
      myBuffer.append('\n');
    }

    @Override
    public void handleForeground(int foregroundId) {
      myBuffer.append("\\cf").append(foregroundId).append('\n');
      myForegroundId = foregroundId;
    }

    @Override
    public void handleFont(int fontNameId) {
      myBuffer.append("\\f").append(fontNameId).append('\n');
      myFontNameId = fontNameId;
    }

    @Override
    public void handleStyle(int style) {
      myBuffer.append(ITALIC);
      if ((style & Font.ITALIC) == 0) {
        myBuffer.append('0');
      }
      myBuffer.append(BOLD);
      if ((style & Font.BOLD) == 0) {
        myBuffer.append('0');
      }
      myBuffer.append('\n');
      myFontStyle = style;
    }

    @Override
    public boolean canHandleMore() {
      if (myBuffer.length() > myMaxLength) {
        myBuffer.append("... truncated ...");
        return false;
      }
      return true;
    }
  }
}