// Copyright 2000-2026 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package com.intellij.codeInsight.lookup.impl;

import com.intellij.codeInsight.AutoPopupController;
import com.intellij.codeInsight.completion.CodeCompletionFeatures;
import com.intellij.codeInsight.completion.CompletionPhase;
import com.intellij.codeInsight.completion.CompletionProgressIndicator;
import com.intellij.codeInsight.completion.PrefixMatcher;
import com.intellij.codeInsight.completion.impl.CompletionServiceImpl;
import com.intellij.codeInsight.editorActions.AutoHardWrapHandler;
import com.intellij.codeInsight.editorActions.TypedHandler;
import com.intellij.codeInsight.lookup.CharFilter;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupManager;
import com.intellij.codeInsight.lookup.impl.actions.ChooseItemAction;
import com.intellij.codeInsight.template.impl.editorActions.TypedActionHandlerBase;
import com.intellij.featureStatistics.FeatureUsageTracker;
import com.intellij.ide.DataManager;
import com.intellij.openapi.actionSystem.CommonDataKeys;
import com.intellij.openapi.actionSystem.DataContext;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.editor.EditorModificationUtil;
import com.intellij.openapi.editor.actionSystem.TypedActionHandler;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiFile;
import com.intellij.psi.util.PsiUtilBase;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class LookupTypedHandler extends TypedActionHandlerBase {
  @ApiStatus.Internal
  public static final Key<Character> CANCELLATION_CHAR = Key.create("CANCELLATION_CHAR");

  @ApiStatus.Internal
  public static final Key<Character> COMPLETE_TILL_TYPED_CHAR_OCCURRENCE = Key.create("COMPLETE_TILL_TYPED_CHAR_OCCURRENCE");

  private static final Logger LOG = Logger.getInstance(LookupTypedHandler.class);

  public LookupTypedHandler(@Nullable TypedActionHandler originalHandler) {
    super(originalHandler);
  }

  @Override
  public void execute(@NotNull Editor originalEditor, char charTyped, @NotNull DataContext dataContext) {
    final Project project = CommonDataKeys.PROJECT.getData(dataContext);
    PsiFile file = project == null ? null : PsiUtilBase.getPsiFileInEditor(originalEditor, project);

    if (file == null) {
      if (myOriginalHandler != null) {
        myOriginalHandler.execute(originalEditor, charTyped, dataContext);
      }
      return;
    }

    if (!EditorModificationUtil.checkModificationAllowed(originalEditor)) {
      return;
    }

    CompletionPhase oldPhase = CompletionServiceImpl.getCompletionPhase();
    if (oldPhase instanceof CompletionPhase.CommittingDocuments && oldPhase.indicator != null) {
      oldPhase.indicator.scheduleRestart();
    }

    Editor editor = TypedHandler.injectedEditorIfCharTypedIsSignificant(charTyped, originalEditor, file);
    if (editor != originalEditor) {
      file = PsiDocumentManager.getInstance(project).getPsiFile(editor.getDocument());
    }

    if (originalEditor.isInsertMode() && beforeCharTyped(charTyped, project, originalEditor, editor, file)) {
      return;
    }

    if (myOriginalHandler != null) {
      myOriginalHandler.execute(originalEditor, charTyped, dataContext);
    }
  }

  /**
   * @return true if `charTyped` was consumed by lookup and other typed handlers should not be invoked.
   */
  private static boolean beforeCharTyped(char charTyped,
                                         @NotNull Project project,
                                         @NotNull Editor originalEditor,
                                         @NotNull Editor editor,
                                         @NotNull PsiFile file) {
    LookupImpl lookup = (LookupImpl)LookupManager.getActiveLookup(originalEditor);
    if (lookup == null) {
      return false;
    }

    if (charTyped == ' ' && ChooseItemAction.hasTemplatePrefix(lookup, ' ')) {
      return false;
    }

    CharFilter.Result result = getLookupAction(charTyped, lookup);
    if (lookup.isLookupDisposed()) {
      return false;
    }

    if (result == CharFilter.Result.ADD_TO_PREFIX) {
      addCharToPrefix(charTyped, originalEditor, editor, lookup, project);
      return true;
    }

    if (result == CharFilter.Result.SELECT_ITEM_AND_FINISH_LOOKUP && lookup.isFocused()) {
      if (selectItemAndFinish(charTyped, lookup)) {
        return true;
      }
    }

    lookup.putUserData(CANCELLATION_CHAR, charTyped);
    lookup.hideLookup(false);
    TypedHandler.autoPopupCompletion(editor, charTyped, project, file);
    return false;
  }

  private static boolean selectItemAndFinish(char charTyped, @NotNull LookupImpl lookup) {
    LookupElement item = lookup.getCurrentItem();
    if (item == null) {
      return false;
    }

    if (completeTillTypedCharOccurrence(charTyped, lookup, item)) {
      lookup.putUserData(COMPLETE_TILL_TYPED_CHAR_OCCURRENCE, charTyped);
      return true;
    }

    FeatureUsageTracker.getInstance().triggerFeatureUsed(CodeCompletionFeatures.EDITING_COMPLETION_FINISH_BY_DOT_ETC);
    lookup.finishLookupInWritableFile(charTyped, item);
    return true;
  }

  /**
   * Adds the given character to the lookup prefix, sends corresponding events, and adds the char to the document honoring multiple carets.
   */
  private static void addCharToPrefix(char charTyped,
                                      @NotNull Editor originalEditor,
                                      @NotNull Editor editor,
                                      @NotNull LookupImpl lookup,
                                      @NotNull Project project) {
    Document document = editor.getDocument();
    long modificationStamp = document.getModificationStamp();

    boolean lookupSurvived = lookup.performGuardedChange(() -> {
      lookup.fireBeforeAppendPrefix(charTyped);
      EditorModificationUtil.typeInStringAtCaretHonorMultipleCarets(originalEditor, String.valueOf(charTyped), true);
    });

    if (!lookupSurvived) {
      return;
    }

    lookup.appendPrefix(charTyped);

    if (lookup.isStartCompletionWhenNothingMatches() && lookup.getItems().isEmpty()) {
      CompletionProgressIndicator completion = CompletionServiceImpl.getCurrentCompletionProgressIndicator();
      if (completion != null) {
        completion.scheduleRestart();
      }
      else {
        AutoPopupController.getInstance(project).scheduleAutoPopup(editor);
      }
    }

    originalEditor.getCaretModel().runForEachCaret(caret -> {
      DataContext context = DataManager.getInstance().getDataContext(originalEditor.getContentComponent());
      AutoHardWrapHandler.getInstance().wrapLineIfNecessary(originalEditor, context, modificationStamp);
    });

    CompletionProgressIndicator completion = CompletionServiceImpl.getCurrentCompletionProgressIndicator();
    if (completion != null) {
      completion.prefixUpdated();
    }
  }

  private static boolean completeTillTypedCharOccurrence(char charTyped, LookupImpl lookup, LookupElement item) {
    PrefixMatcher matcher = lookup.itemMatcher(item);
    final String oldPrefix = matcher.getPrefix() + lookup.getAdditionalPrefix();
    PrefixMatcher expanded = matcher.cloneWithPrefix(oldPrefix + charTyped);
    if (expanded.prefixMatches(item)) {
      for (String s : item.getAllLookupStrings()) {
        if (matcher.prefixMatches(s)) {
          int i = -1;
          while (true) {
            i = s.indexOf(charTyped, i + 1);
            if (i < 0) break;
            final String newPrefix = s.substring(0, i + 1);
            if (expanded.prefixMatches(newPrefix)) {
              lookup.replacePrefix(oldPrefix, newPrefix);
              return true;
            }
          }
        }
      }
    }
    return false;
  }

  public static @NotNull CharFilter.Result getLookupAction(char charTyped, @NotNull LookupImpl lookup) {
    CharFilter.Result filtersDecision = getFilterDecision(charTyped, lookup);
    if (filtersDecision != null) {
      return filtersDecision;
    }
    return CharFilter.Result.HIDE_LOOKUP;
  }

  private static @Nullable CharFilter.Result getFilterDecision(char charTyped, @NotNull LookupImpl lookup) {
    lookup.checkValid();
    LookupElement item = lookup.getCurrentItem();
    int prefixLength = item == null ? lookup.getAdditionalPrefix().length() : lookup.itemPattern(item).length();

    for (CharFilter extension : CharFilter.EP_NAME.getExtensionList()) {
      CharFilter.Result result = extension.acceptChar(charTyped, prefixLength, lookup);
      if (result != null) {
        if (LOG.isDebugEnabled()) {
          LOG.debug(extension + " of " + extension.getClass() + " returned " + result);
        }
        return result;
      }
      if (lookup.isLookupDisposed()) {
        throw new AssertionError("Lookup disposed after " + extension);
      }
    }
    return null;
  }
}
