// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.refactoring.move.moveFilesOrDirectories;

import com.intellij.ide.util.DirectoryChooserUtil;
import com.intellij.ide.util.EditorHelper;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.command.CommandProcessor;
import com.intellij.openapi.command.WriteCommandAction;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.EmptyRunnable;
import com.intellij.psi.*;
import com.intellij.refactoring.RefactoringBundle;
import com.intellij.refactoring.RefactoringSettings;
import com.intellij.refactoring.copy.CopyFilesOrDirectoriesHandler;
import com.intellij.refactoring.move.MoveCallback;
import com.intellij.refactoring.move.MoveHandler;
import com.intellij.refactoring.util.CommonRefactoringUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.SmartList;
import com.intellij.util.ui.IoErrorText;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.*;
import java.util.function.Function;

public final class MoveFilesOrDirectoriesUtil {
  private MoveFilesOrDirectoriesUtil() { }

  /**
   * Moves the specified directory to the specified parent directory. Does not process non-code usages!
   *
   * @param aDirectory          the directory to move.
   * @param destDirectory the directory to move {@code dir} into.
   * @throws IncorrectOperationException if the modification is not supported or not possible for some reason.
   */
  public static void doMoveDirectory(@NotNull PsiDirectory aDirectory, @NotNull PsiDirectory destDirectory) throws IncorrectOperationException {
    var manager = aDirectory.getManager();
    doJustMoveDirectory(aDirectory, destDirectory, manager);
    DumbService.getInstance(manager.getProject()).completeJustSubmittedTasks();
  }

  private static void doJustMoveDirectory(PsiDirectory aDirectory, PsiDirectory destDirectory, @Nullable Object requestor) {
    checkMove(aDirectory, destDirectory);
    try {
      aDirectory.getVirtualFile().move(requestor, destDirectory.getVirtualFile());
    }
    catch (IOException e) {
      throw new IncorrectOperationException(e);
    }
  }

  /**
   * Moves the specified file to the specified directory. Does not process non-code usages!
   * The file may be invalidated, need to be refreshed before use, like {@code newDirectory.findFile(file.getName())}.
   *
   * @param file         the file to move.
   * @param newDirectory the directory to move the file into.
   * @throws IncorrectOperationException if the modification is not supported or not possible for some reason.
   */
  public static void doMoveFile(@NotNull PsiFile file, @NotNull PsiDirectory newDirectory) throws IncorrectOperationException {
    // the class is already there, this is true when multiple classes are defined in the same file
    if (!newDirectory.equals(file.getContainingDirectory())) {
      // do actual move
      checkMove(file, newDirectory);

      var vFile = file.getViewProvider().getVirtualFile();

      try {
        vFile.move(file.getManager(), newDirectory.getVirtualFile());
      }
      catch (IOException e) {
        throw new IncorrectOperationException(e);
      }
    }
  }

  /**
   * @param elements should contain PsiDirectories or PsiFiles only
   */
  public static void doMove(
    @NotNull Project project,
    PsiElement @NotNull [] elements,
    PsiElement @NotNull [] targetElement,
    @Nullable MoveCallback moveCallback
  ) {
    doMove(project, elements, targetElement, moveCallback, null);
  }

  /**
   * @param elements should contain PsiDirectories or PsiFiles only if adjustElements == null
   */
  public static void doMove(
    @NotNull Project project,
    PsiElement @NotNull [] elements,
    PsiElement @NotNull [] targetElement,
    @Nullable MoveCallback moveCallback,
    @Nullable Function<? super PsiElement[], ? extends PsiElement[]> adjustElements
  ) {
    if (adjustElements == null) {
      for (var element : elements) {
        if (!(element instanceof PsiFile) && !(element instanceof PsiDirectory)) {
          throw new IllegalArgumentException("unexpected element type: " + element);
        }
      }
    }

    var targetDirectory = resolveToDirectory(project, targetElement[0]);
    if (targetElement[0] != null && targetDirectory == null) return;

    var adjustedElements = adjustElements != null ? adjustElements.apply(elements) : elements;

    var initialTargetDirectory = getInitialTargetDirectory(targetDirectory, elements);

    if (ApplicationManager.getApplication().isUnitTestMode()) {
      Objects.requireNonNull(initialTargetDirectory, "It is null! The target directory, it is null!");
      doMove(project, elements, adjustedElements, initialTargetDirectory, moveCallback, EmptyRunnable.INSTANCE);
    }
    else {
      new MoveFilesOrDirectoriesDialog(project, adjustedElements, initialTargetDirectory) {
        @Override
        protected void performMove(@NotNull PsiDirectory targetDirectory) {
          var doneCallback = (Runnable)this::closeOKAction;
          doMove(project, elements, adjustedElements, targetDirectory, moveCallback, doneCallback);
        }
      }.show();
    }
  }

  private static void doMove(
    Project project,
    PsiElement[] elements,
    PsiElement[] adjustedElements,
    PsiDirectory targetDirectory,
    @Nullable MoveCallback moveCallback,
    Runnable doneCallback
  ) {
    CommandProcessor.getInstance().executeCommand(project, () -> {
      Collection<PsiElement> toCheck = new SmartList<>(targetDirectory);
      for (var e : adjustedElements) {
        toCheck.add(e instanceof PsiFileSystemItem && e.getParent() != null ? e.getParent() : e);
      }
      if (!CommonRefactoringUtil.checkReadOnlyStatus(project, toCheck, false)) {
        return;
      }

      try {
        var choice = elements.length > 1 || elements[0] instanceof PsiDirectory ? new int[]{-1} : null;
        var els = new ArrayList<PsiElement>();
        for (var psiElement : adjustedElements) {
          if (psiElement instanceof PsiFile file) {
            if (CopyFilesOrDirectoriesHandler.checkFileExist(targetDirectory, choice, file, file.getName(), RefactoringBundle.message("command.name.move"))) {
              continue;
            }
          }
          checkMove(psiElement, targetDirectory);
          els.add(psiElement);
        }

        if (els.isEmpty()) {
          doneCallback.run();
        }
        else if (DumbService.isDumb(project)) {
          ApplicationManager.getApplication().invokeAndWait(doneCallback);
          var filePointers = new HashSet<SmartPsiElementPointer<PsiFile>>();
          if (MoveFilesOrDirectoriesDialog.isOpenInEditorProperty()) {
            var manager = SmartPointerManager.getInstance(project);
            for (var element : elements) {
              addFilePointers(filePointers, element, manager);
            }
          }
          WriteCommandAction.runWriteCommandAction(project, RefactoringBundle.message("move.title"), null, () -> {
            try {
              for (var element : elements) {
                if (element instanceof PsiDirectory) {
                  doJustMoveDirectory((PsiDirectory)element, targetDirectory, MoveFilesOrDirectoriesUtil.class);
                }
                else if (element instanceof PsiFile movedFile) {
                  var moving = targetDirectory.findFile(movedFile.getName());
                  if (moving == null) {
                    doMoveFile(movedFile, targetDirectory);
                  }
                }
              }
            }
            finally {
              if (moveCallback != null) {
                moveCallback.refactoringCompleted();
              }
              if (MoveFilesOrDirectoriesDialog.isOpenInEditorProperty()) {
                ApplicationManager.getApplication().invokeLater(
                  () -> EditorHelper.openFilesInEditor(
                    filePointers.stream()
                      .map(SmartPsiElementPointer::getContainingFile).filter(file -> file != null && file.isValid())
                      .toArray(PsiFile[]::new)),
                  project.getDisposed()
                );
              }
            }
          });
        }
        else {
          new MoveFilesOrDirectoriesProcessor(
            project, els.toArray(PsiElement.EMPTY_ARRAY), targetDirectory,
            RefactoringSettings.getInstance().MOVE_SEARCH_FOR_REFERENCES_FOR_FILE,
            false, false, moveCallback, doneCallback
          ).run();
        }
      }
      catch (IncorrectOperationException e) {
        var cause = e.getCause();
        if (cause == null) throw e;
        var message = IoErrorText.message(cause);
        CommonRefactoringUtil.showErrorMessage(RefactoringBundle.message("error.title"), message, "refactoring.moveFile", project);
      }
    }, MoveHandler.getRefactoringName(), null);
  }

  private static void addFilePointers(Set<SmartPsiElementPointer<PsiFile>> pointers, PsiElement element, SmartPointerManager manager) {
    if (element instanceof PsiFile) {
      pointers.add(manager.createSmartPsiElementPointer((PsiFile)element, (PsiFile)element));
    }
    else if (element instanceof PsiDirectory) {
      for (var child : element.getChildren()) {
        addFilePointers(pointers, child, manager);
      }
    }
  }

  public static @Nullable PsiDirectory resolveToDirectory(@NotNull Project project, PsiElement element) {
    if (!(element instanceof PsiDirectoryContainer container)) {
      return (PsiDirectory)element;
    }

    var directories = container.getDirectories();
    return switch (directories.length) {
      case 0 -> null;
      case 1 -> directories[0];
      default -> DirectoryChooserUtil.chooseDirectory(directories, directories[0], project, new HashMap<>());
    };
  }

  private static @Nullable PsiDirectory getCommonDirectory(PsiElement @NotNull [] movedElements) {
    var commonDirectory = (PsiDirectory)null;

    for (var movedElement : movedElements) {
      PsiDirectory containingDirectory;
      if (movedElement instanceof PsiDirectory directory) {
        containingDirectory = directory.getParentDirectory();
      }
      else {
        var containingFile = movedElement.getContainingFile();
        containingDirectory = containingFile == null ? null : containingFile.getContainingDirectory();
      }

      if (containingDirectory != null) {
        if (commonDirectory == null) {
          commonDirectory = containingDirectory;
        }
        else {
          if (commonDirectory != containingDirectory) {
            return null;
          }
        }
      }
    }
    return commonDirectory;
  }

  public static @Nullable PsiDirectory getInitialTargetDirectory(@Nullable PsiDirectory initialTargetElement, PsiElement[] movedElements) {
    var initialTargetDirectory = initialTargetElement;
    if (initialTargetDirectory == null) {
      if (movedElements != null) {
        var commonDirectory = getCommonDirectory(movedElements);
        if (commonDirectory != null) {
          initialTargetDirectory = commonDirectory;
        }
        else {
          initialTargetDirectory = getContainerDirectory(movedElements[0]);
        }
      }
    }
    return initialTargetDirectory;
  }

  private static @Nullable PsiDirectory getContainerDirectory(PsiElement psiElement) {
    if (psiElement instanceof PsiDirectory) {
      return (PsiDirectory)psiElement;
    }
    else if (psiElement != null) {
      var containingFile = psiElement.getContainingFile();
      if (containingFile != null) {
        return containingFile.getContainingDirectory();
      }
    }

    return null;
  }

  /**
   * Checks if it is possible to move the specified PSI element under the specified container,
   * and throws an exception if the move is not possible. Does not actually modify anything.
   *
   * @param element      the element to check the move possibility.
   * @param newContainer the target container element to move into.
   * @throws IncorrectOperationException if the modification is not supported or not possible for some reason.
   */
  public static void checkMove(@NotNull PsiElement element, @NotNull PsiElement newContainer) throws IncorrectOperationException {
    if (element instanceof PsiDirectoryContainer) {
      var dirs = ((PsiDirectoryContainer)element).getDirectories();
      if (dirs.length == 0) {
        throw new IncorrectOperationException();
      }
      else if (dirs.length > 1) {
        throw new IncorrectOperationException("Moving of packages represented by more than one physical directory is not supported.");
      }
      checkMove(dirs[0], newContainer);
      return;
    }

    //element.checkDelete(); //move != delete + add
    newContainer.checkAdd(element);
    checkIfMoveIntoSelf(element, newContainer);
  }

  public static void checkIfMoveIntoSelf(PsiElement element, PsiElement newContainer) throws IncorrectOperationException {
    var container = newContainer;
    while (container != null) {
      if (container == element) {
        if (element instanceof PsiDirectory) {
          if (element == newContainer) {
            throw new IncorrectOperationException("Cannot place directory into itself.");
          }
          else {
            throw new IncorrectOperationException("Cannot place directory into its subdirectory.");
          }
        }
        else {
          throw new IncorrectOperationException();
        }
      }
      container = container.getParent();
    }
  }
}
