// Copyright 2000-2026 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.codeInsight.daemon.impl

import com.intellij.codeInsight.hints.InlayHintsSettings
import com.intellij.codeInsight.hints.declarative.DeclarativeInlayHintsSettings
import com.intellij.codeInsight.hints.declarative.InlayActionPayload
import com.intellij.codeInsight.hints.declarative.PsiPointerInlayActionPayload
import com.intellij.codeInsight.hints.declarative.impl.*
import com.intellij.codeInsight.hints.declarative.impl.inlayRenderer.DeclarativeIndentedBlockInlayRenderer
import com.intellij.codeInsight.hints.declarative.impl.inlayRenderer.DeclarativeInlayRenderer
import com.intellij.codeInsight.hints.declarative.impl.util.TinyTree
import com.intellij.openapi.application.readActionBlocking
import com.intellij.openapi.editor.Editor
import com.intellij.openapi.editor.impl.zombie.*
import com.intellij.openapi.project.Project
import com.intellij.openapi.util.registry.Registry
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.platform.ide.diagnostic.startUpPerformanceReporter.FUSProjectHotStartUpMeasurer
import com.intellij.platform.ide.diagnostic.startUpPerformanceReporter.FUSProjectHotStartUpMeasurer.MarkupType
import kotlinx.coroutines.CoroutineScope


internal class DeclarativeHintsNecromancerAwaker : NecromancerAwaker<DeclarativeHintsZombie> {
  override fun awake(project: Project, coroutineScope: CoroutineScope): Necromancer<DeclarativeHintsZombie> {
    return DeclarativeHintsNecromancer(project, coroutineScope)
  }
}

private class DeclarativeHintsNecromancer(
  project: Project,
  coroutineScope: CoroutineScope,
) : CleaverNecromancer<DeclarativeHintsZombie, InlayData>(
  project,
  coroutineScope,
  "graved-declarative-hints",
  DeclarativeHintsNecromancy,
) {

  override fun enoughMana(recipe: Recipe): Boolean {
    return isDeclarativeEnabled() && isCacheEnabled()
  }

  override fun cutIntoLimbs(recipe: TurningRecipe): List<InlayData> {
    val inlineHints = recipe.editor.getInlayModel().getInlineElementsInRange(
      0,
      recipe.editor.getDocument().textLength,
      DeclarativeInlayRenderer::class.java,
    )
    val eolHints = recipe.editor.getInlayModel().getAfterLineEndElementsInRange(
      0,
      recipe.editor.getDocument().textLength,
      DeclarativeInlayRenderer::class.java,
    )
    val blockHints = recipe.editor.getInlayModel().getBlockElementsInRange(
      0,
      recipe.editor.getDocument().textLength,
      DeclarativeIndentedBlockInlayRenderer::class.java,
    )
    val inlayDataList = ArrayList<InlayData>()
    inlineHints.flatMapTo(inlayDataList) { it.renderer.toInlayData() }
    eolHints.flatMapTo(inlayDataList) { it.renderer.toInlayData() }
    blockHints.flatMapTo(inlayDataList) { it.renderer.toInlayData() }
    return inlayDataList
  }

  override suspend fun spawnZombie(
    recipe: SpawnRecipe,
    limbs: List<InlayData>,
  ): ((Editor) -> Unit)? {
    val settings = DeclarativeInlayHintsSettings.getInstance()
    for (inlayData in limbs) {
      initZombiePointers(recipe.project, recipe.file, inlayData.tree)
    }
    val inlayDataMap = readActionBlocking {
      if (!recipe.isValid()) return@readActionBlocking emptyMap()
      limbs
        .filter { settings.isProviderEnabled(it.providerId) ?: true }
        .groupBy { it.sourceId }
        .mapValues { (_, inlayDataList) -> DeclarativeInlayHintsPass.preprocessCollectedInlayData(inlayDataList, recipe.document) }
    }
    if (inlayDataMap.isEmpty()) {
      return null
    }
    return { editor ->
      inlayDataMap.forEach { (sourceId, preparedInlayData) ->
        DeclarativeInlayHintsPass.applyInlayData(
          editor,
          recipe.project,
          preparedInlayData,
          emptySet(),
          sourceId,
        )
      }
      DeclarativeInlayHintsPassFactory.resetModificationStamp(editor)
      FUSProjectHotStartUpMeasurer.markupRestored(recipe, MarkupType.DECLARATIVE_HINTS)
    }
  }

  private fun initZombiePointers(project: Project, file: VirtualFile, tree: TinyTree<Any?>, index: Byte = 0) {
    val dataPayload = tree.getDataPayload(index)
    if (dataPayload is ActionWithContent) {
      val payload: InlayActionPayload = dataPayload.actionData.payload
      if (payload is PsiPointerInlayActionPayload) {
        val pointer = payload.pointer
        if (pointer is ZombieSmartPointer) {
          pointer.projectSupp = { project }
          pointer.fileSupp = { file }
        }
      }
    }
    tree.processChildren(index) { child ->
      initZombiePointers(project, file, tree, child)
      true
    }
  }

  private fun isCacheEnabled(): Boolean {
    return Registry.`is`("cache.inlay.hints.on.disk", true)
  }

  private fun isDeclarativeEnabled(): Boolean {
    val enabledGlobally = InlayHintsSettings.instance().hintsEnabledGlobally()
    return enabledGlobally && Registry.`is`("inlays.declarative.hints", true)
  }
}
