// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.find.impl.livePreview;

import com.intellij.find.FindManager;
import com.intellij.find.FindModel;
import com.intellij.find.FindResult;
import com.intellij.openapi.Disposable;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.editor.SelectionModel;
import com.intellij.openapi.editor.colors.EditorColorsListener;
import com.intellij.openapi.editor.colors.EditorColorsManager;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.event.DocumentListener;
import com.intellij.openapi.editor.event.SelectionEvent;
import com.intellij.openapi.editor.event.SelectionListener;
import com.intellij.openapi.editor.event.VisibleAreaListener;
import com.intellij.openapi.editor.ex.MarkupModelEx;
import com.intellij.openapi.editor.ex.RangeHighlighterEx;
import com.intellij.openapi.editor.ex.util.EditorUtil;
import com.intellij.openapi.editor.markup.HighlighterTargetArea;
import com.intellij.openapi.editor.markup.RangeHighlighter;
import com.intellij.openapi.editor.markup.TextAttributes;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.ui.popup.Balloon;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.Segment;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.ui.awt.RelativePoint;
import com.intellij.usages.impl.UsagePreviewPanel;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.ui.PositionTracker;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.Point;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

public final class LivePreview implements SearchResults.SearchResultsListener, SelectionListener, DocumentListener, EditorColorsListener {
  private static final Key<RangeHighlighter> IN_SELECTION_KEY = Key.create("LivePreview.IN_SELECTION_KEY");

  private final Disposable myDisposable = Disposer.newDisposable("livePreview");
  private boolean mySuppressedUpdate = false;

  private static final Key<Boolean> MARKER_USED = Key.create("LivePreview.MARKER_USED");
  private static final Key<Boolean> SEARCH_MARKER = Key.create("LivePreview.SEARCH_MARKER");

  public static PrintStream ourTestOutput;
  private String myReplacementPreviewText;
  private static boolean NotFound;

  private final List<RangeHighlighter> myHighlighters = new ArrayList<>();
  private RangeHighlighter myCursorHighlighter;
  private VisibleAreaListener myVisibleAreaListener;
  private Delegate myDelegate;
  private final SearchResults mySearchResults;
  private final LivePreviewPresentation myPresentation;
  private Balloon myReplacementBalloon;

  @Override
  public void selectionChanged(@NotNull SelectionEvent e) {
    updateInSelectionHighlighters();
  }

  public static void processNotFound() {
    NotFound = true;
  }

  public interface Delegate {
    @NlsSafe @Nullable String getStringToReplace(@NotNull Editor editor, @Nullable FindResult findResult)
      throws FindManager.MalformedReplacementStringException;
  }

  @Override
  public void searchResultsUpdated(@NotNull SearchResults sr) {
    if (mySuppressedUpdate) {
      mySuppressedUpdate = false;
      return;
    }
    highlightUsages();
    updateCursorHighlighting();
  }

  private void dumpState() {
    if (ApplicationManager.getApplication().isUnitTestMode() && ourTestOutput != null) {
      dumpEditorMarkupAndSelection(ourTestOutput);
    }
  }

  private void dumpEditorMarkupAndSelection(PrintStream dumpStream) {
    dumpStream.println(mySearchResults.getFindModel());
    if (myReplacementPreviewText != null) {
      dumpStream.println("--");
      dumpStream.println("Replacement Preview: " + myReplacementPreviewText);
    }
    dumpStream.println("--");

    Editor editor = mySearchResults.getEditor();

    RangeHighlighter[] highlighters = editor.getMarkupModel().getAllHighlighters();
    Arrays.sort(highlighters, Segment.BY_START_OFFSET_THEN_END_OFFSET);
    List<Pair<Integer, Character>> ranges = new ArrayList<>();
    for (RangeHighlighter highlighter : highlighters) {
      ranges.add(new Pair<>(highlighter.getStartOffset(), '['));
      ranges.add(new Pair<>(highlighter.getEndOffset(), ']'));
    }

    SelectionModel selectionModel = editor.getSelectionModel();

    if (selectionModel.getSelectionStart() != selectionModel.getSelectionEnd()) {
      ranges.add(new Pair<>(selectionModel.getSelectionStart(), '<'));
      ranges.add(new Pair<>(selectionModel.getSelectionEnd(), '>'));
    }
    ranges.add(new Pair<>(-1, '\n'));
    ranges.add(new Pair<>(editor.getDocument().getTextLength() + 1, '\n'));
    ContainerUtil.sort(ranges, (pair, pair2) -> {
      int res = pair.first - pair2.first;
      if (res == 0) {

        Character c1 = pair.second;
        Character c2 = pair2.second;
        if (c1 == '<' && c2 == '[') {
          return 1;
        }
        else if (c1 == '[' && c2 == '<') {
          return -1;
        }
        return c1.compareTo(c2);
      }
      return res;
    });

    Document document = editor.getDocument();
    for (int i = 0; i < ranges.size()-1; ++i) {
      Pair<Integer, Character> pair = ranges.get(i);
      Pair<Integer, Character> pair1 = ranges.get(i + 1);
      dumpStream.print(pair.second + document.getText(TextRange.create(Math.max(pair.first, 0),
                                                                       Math.min(pair1.first, document.getTextLength()))));
    }
    dumpStream.println("\n--");

    if (NotFound) {
      dumpStream.println("Not Found");
      dumpStream.println("--");
      NotFound = false;
    }

    for (RangeHighlighter highlighter : highlighters) {
      dumpStream.println("highlighter: "+highlighter.getTextRange() + "; layer: "+highlighter.getLayer()+" : " + highlighter.getTextAttributes(editor.getColorsScheme()).getEffectType());
    }
    dumpStream.println("------------");
  }

  private void clearUnusedHighlighters() {
    myHighlighters.removeIf(h -> {
      if (h.getUserData(MARKER_USED) == null) {
        removeHighlighterWithDependent(h);
        return true;
      }
      else {
        h.putUserData(MARKER_USED, null);
        return false;
      }
    });
  }

  private void removeHighlighterWithDependent(@NotNull RangeHighlighter highlighter) {
    removeHighlighter(highlighter);
    RangeHighlighter additionalHighlighter = highlighter.getUserData(IN_SELECTION_KEY);
    if (additionalHighlighter != null) {
      removeHighlighter(additionalHighlighter);
    }
  }

  @Override
  public void cursorMoved() {
    updateInSelectionHighlighters();
    updateCursorHighlighting();
  }

  @Override
  public void updateFinished() {
    dumpState();
  }

  private void updateCursorHighlighting() {
    hideBalloon();

    if (myCursorHighlighter != null) {
      removeHighlighter(myCursorHighlighter);
      myCursorHighlighter = null;
    }

    final FindResult cursor = mySearchResults.getCursor();
    Editor editor = mySearchResults.getEditor();
    if (cursor != null && cursor.getEndOffset() <= editor.getDocument().getTextLength()) {
      myCursorHighlighter = addHighlighter(cursor.getStartOffset(), cursor.getEndOffset(), myPresentation.getCursorAttributes(),
                                           myPresentation.getCursorLayer());
      editor.getScrollingModel().runActionOnScrollingFinished(() -> showReplacementPreview());
    }
  }

  public LivePreview(@NotNull SearchResults searchResults, @NotNull LivePreviewPresentation presentation) {
    mySearchResults = searchResults;
    myPresentation = presentation;
    searchResultsUpdated(searchResults);
    searchResults.addListener(this);
    EditorUtil.addBulkSelectionListener(mySearchResults.getEditor(), this, myDisposable);
    ApplicationManager.getApplication().getMessageBus().connect(myDisposable).subscribe(EditorColorsManager.TOPIC, this);
  }

  public Delegate getDelegate() {
    return myDelegate;
  }

  public void setDelegate(Delegate delegate) {
    myDelegate = delegate;
  }

  @Override
  public void globalSchemeChange(@Nullable EditorColorsScheme scheme) {
    highlightUsages();
    updateCursorHighlighting();
  }

  public void dispose() {
    hideBalloon();

    for (RangeHighlighter h : myHighlighters) {
      removeHighlighterWithDependent(h);
    }
    myHighlighters.clear();

    if (myCursorHighlighter != null) {
      removeHighlighter(myCursorHighlighter);
    }
    myCursorHighlighter = null;

    Disposer.dispose(myDisposable);

    mySearchResults.removeListener(this);
  }

  private void highlightUsages() {
    List<RangeHighlighter> newHighlighters = mySearchResults.getMatchesCount() < mySearchResults.getMatchesLimit()
                                             ? addNewHighlighters() : Collections.emptyList();
    clearUnusedHighlighters();
    myHighlighters.addAll(newHighlighters);
    updateInSelectionHighlighters();
  }

  private List<RangeHighlighter> addNewHighlighters() {
    List<FindResult> occurrences = mySearchResults.getOccurrences();
    List<RangeHighlighter> newHighlighters = new ArrayList<>(occurrences.size());
    for (FindResult range : occurrences) {
      if (range.getEndOffset() > mySearchResults.getEditor().getDocument().getTextLength()) continue;
      TextAttributes attributes = createAttributes(range);
      RangeHighlighter existingHighlighter = findExistingHighlighter(range.getStartOffset(), range.getEndOffset(), attributes);
      if (existingHighlighter == null) {
        RangeHighlighter highlighter = addHighlighter(range.getStartOffset(), range.getEndOffset(),
                                                      attributes, myPresentation.getDefaultLayer());
        if (highlighter != null) {
          highlighter.putUserData(SEARCH_MARKER, Boolean.TRUE);
          newHighlighters.add(highlighter);
        }
      }
      else {
        existingHighlighter.putUserData(MARKER_USED, Boolean.TRUE);
      }
    }
    return newHighlighters;
  }

  private TextAttributes createAttributes(FindResult range) {
    if (mySearchResults.isExcluded(range)) {
      return myPresentation.getExcludedAttributes();
    }
    else if (range.isEmpty()) {
      return myPresentation.getEmptyRangeAttributes();
    }
    else {
      return myPresentation.getDefaultAttributes();
    }
  }

  private RangeHighlighter findExistingHighlighter(int startOffset, int endOffset, TextAttributes attributes) {
    MarkupModelEx markupModel = (MarkupModelEx)mySearchResults.getEditor().getMarkupModel();
    RangeHighlighter[] existing = new RangeHighlighter[1];
    markupModel.processRangeHighlightersOverlappingWith(startOffset, startOffset, highlighter -> {
      if (highlighter.getUserData(SEARCH_MARKER) != null &&
          highlighter.getStartOffset() == startOffset && highlighter.getEndOffset() == endOffset &&
          Objects.equals(highlighter.getTextAttributes(mySearchResults.getEditor().getColorsScheme()), attributes)) {
        existing[0] = highlighter;
        return false;
      }
      return true;
    });
    return existing[0];
  }

  private void updateInSelectionHighlighters() {
    final SelectionModel selectionModel = mySearchResults.getEditor().getSelectionModel();
    int[] starts = selectionModel.getBlockSelectionStarts();
    int[] ends = selectionModel.getBlockSelectionEnds();

    for (RangeHighlighter highlighter : myHighlighters) {
      if (!highlighter.isValid()) continue;
      boolean needsAdditionalHighlighting = false;
      TextRange cursor = mySearchResults.getCursor();
      if (cursor == null ||
          highlighter.getStartOffset() != cursor.getStartOffset() || highlighter.getEndOffset() != cursor.getEndOffset()) {
        for (int i = 0; i < starts.length; ++i) {
          TextRange selectionRange = new TextRange(starts[i], ends[i]);
          needsAdditionalHighlighting = selectionRange.intersects(highlighter.getStartOffset(), highlighter.getEndOffset()) &&
                           selectionRange.getEndOffset() != highlighter.getStartOffset() &&
                           highlighter.getEndOffset() != selectionRange.getStartOffset();
          if (needsAdditionalHighlighting) break;
        }
      }

      RangeHighlighter inSelectionHighlighter = highlighter.getUserData(IN_SELECTION_KEY);
      if (inSelectionHighlighter != null) {
        if (!needsAdditionalHighlighting) {
          removeHighlighter(inSelectionHighlighter);
          highlighter.putUserData(IN_SELECTION_KEY, null);
        }
      } else if (needsAdditionalHighlighting) {
        RangeHighlighter additionalHighlighter = addHighlighter(highlighter.getStartOffset(), highlighter.getEndOffset(),
                                                                myPresentation.getSelectionAttributes(),
                                                                myPresentation.getDefaultLayer());
        highlighter.putUserData(IN_SELECTION_KEY, additionalHighlighter);
      }
    }
  }

  private void showReplacementPreview() {
    hideBalloon();
    if (!mySearchResults.isUpToDate()) return;
    final FindResult cursor = mySearchResults.getCursor();
    final Editor editor = mySearchResults.getEditor();
    final FindModel findModel = mySearchResults.getFindModel();
    if (myDelegate != null && cursor != null && findModel.isReplaceState() && findModel.isRegularExpressions()) {
      String replacementPreviewText;
      try {
        replacementPreviewText = myDelegate.getStringToReplace(editor, cursor);
      }
      catch (FindManager.MalformedReplacementStringException e) {
        return;
      }
      if (replacementPreviewText == null) {
        return;//malformed replacement string
      }
      if (!replacementPreviewText.equals(findModel.getStringToReplace()) ||
          Registry.is("ide.find.show.replacement.hint.for.simple.regexp")) {
        showBalloon(editor, replacementPreviewText);
      }
    }
  }

  private void showBalloon(Editor editor, @NotNull @NlsSafe String replacementPreviewText) {
    if (ApplicationManager.getApplication().isUnitTestMode()) {
      myReplacementPreviewText = replacementPreviewText;
      return;
    }

    myReplacementBalloon = UsagePreviewPanel.createPreviewBalloon(UsagePreviewPanel.createPreviewHtml(replacementPreviewText));
    EditorUtil.disposeWithEditor(editor, myReplacementBalloon);
    myReplacementBalloon.show(new ReplacementBalloonPositionTracker(editor), Balloon.Position.below);
  }

  private void hideBalloon() {
    if (ApplicationManager.getApplication().isUnitTestMode()) {
      myReplacementPreviewText = null;
      return;
    }

    if (myReplacementBalloon != null) {
      myReplacementBalloon.hide();
      myReplacementBalloon = null;
    }

    removeVisibleAreaListener();
  }

  private void removeVisibleAreaListener() {
    if (myVisibleAreaListener != null) {
      mySearchResults.getEditor().getScrollingModel().removeVisibleAreaListener(myVisibleAreaListener);
      myVisibleAreaListener = null;
    }
  }

  private RangeHighlighter addHighlighter(int startOffset, int endOffset, @NotNull TextAttributes attributes, int layer) {
    Project project = mySearchResults.getProject();
    if (project.isDisposed()) return null;
    var markupModel = mySearchResults.getEditor().getMarkupModel();
    var highlighter = markupModel.addRangeHighlighter(startOffset, endOffset, layer, attributes, HighlighterTargetArea.EXACT_RANGE);
    if (highlighter instanceof RangeHighlighterEx ex) ex.setVisibleIfFolded(true);
    return highlighter;
  }

  private void removeHighlighter(@NotNull RangeHighlighter highlighter) {
    Project project = mySearchResults.getProject();
    if (project.isDisposed()) return;
    mySearchResults.getEditor().getMarkupModel().removeHighlighter(highlighter);
  }

  private final class ReplacementBalloonPositionTracker extends PositionTracker<Balloon> {
    private final Editor myEditor;

    ReplacementBalloonPositionTracker(Editor editor) {
      super(editor.getContentComponent());
      myEditor = editor;

    }

    @Override
    public RelativePoint recalculateLocation(@NotNull Balloon balloon) {
      FindResult cursor = mySearchResults.getCursor();
      if (cursor == null) return null;
      final TextRange cur = cursor;
      int startOffset = cur.getStartOffset();
      int endOffset = cur.getEndOffset();

      if (endOffset > myEditor.getDocument().getTextLength()) {
        if (!balloon.isDisposed()) {
          requestBalloonHiding(balloon);
        }
        return null;
      }
      if (!SearchResults.insideVisibleArea(myEditor, cur)) {
        requestBalloonHiding(balloon);

        removeVisibleAreaListener();
        myVisibleAreaListener = e -> {
          if (SearchResults.insideVisibleArea(myEditor, cur)) showReplacementPreview();
        };
        myEditor.getScrollingModel().addVisibleAreaListener(myVisibleAreaListener);
      }

      Point startPoint = myEditor.visualPositionToXY(myEditor.offsetToVisualPosition(startOffset));
      Point endPoint = myEditor.visualPositionToXY(myEditor.offsetToVisualPosition(endOffset));
      Point point = new Point((startPoint.x + endPoint.x)/2, endPoint.y + myEditor.getLineHeight());

      return new RelativePoint(myEditor.getContentComponent(), point);
    }
  }

  private static void requestBalloonHiding(Balloon balloon) {
    ApplicationManager.getApplication().invokeLater(() -> balloon.hide());
  }
}
