// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.ide.util;

import com.intellij.ide.IdeBundle;
import com.intellij.lang.findUsages.LanguageFindUsages;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.ElementDescriptionLocation;
import com.intellij.psi.ElementDescriptionProvider;
import com.intellij.psi.PsiDirectory;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.psi.meta.PsiMetaData;
import com.intellij.psi.meta.PsiMetaOwner;
import com.intellij.psi.meta.PsiPresentableMetaData;
import com.intellij.psi.util.PsiUtilBase;
import org.jetbrains.annotations.NotNull;


public final class DeleteTypeDescriptionLocation extends ElementDescriptionLocation {
  private final boolean myPlural;

  private DeleteTypeDescriptionLocation(final boolean plural) {
    myPlural = plural;
  }

  public static final DeleteTypeDescriptionLocation SINGULAR = new DeleteTypeDescriptionLocation(false);
  public static final DeleteTypeDescriptionLocation PLURAL = new DeleteTypeDescriptionLocation(true);

  private static final ElementDescriptionProvider ourDefaultProvider = new DefaultProvider();

  @Override
  public @NotNull ElementDescriptionProvider getDefaultProvider() {
    return ourDefaultProvider;
  }

  public boolean isPlural() {
    return myPlural;
  }

  private static final class DefaultProvider implements ElementDescriptionProvider {
    @Override
    public String getElementDescription(final @NotNull PsiElement element, final @NotNull ElementDescriptionLocation location) {
      if (location instanceof DeleteTypeDescriptionLocation) {
        final boolean plural = ((DeleteTypeDescriptionLocation)location).isPlural();
        final int count = plural ? 2 : 1;
        if (element instanceof PsiFileSystemItem && PsiUtilBase.isSymLink((PsiFileSystemItem)element)) {
          return IdeBundle.message("prompt.delete.symlink", count);
        }
        if (element instanceof PsiFile) {
          return IdeBundle.message("prompt.delete.file", count);
        }
        if (element instanceof PsiDirectory) {
          return IdeBundle.message("prompt.delete.directory", count);
        }
        PsiMetaData metaData = element instanceof PsiMetaOwner ? ((PsiMetaOwner)element).getMetaData() : null;
        String typeName = metaData instanceof PsiPresentableMetaData ? ((PsiPresentableMetaData)metaData).getTypeName() : null;
        if (typeName == null) {
          typeName = LanguageFindUsages.getType(element);
        }
        return !plural ? typeName : StringUtil.pluralize(typeName);
      }
      return null;
    }
  }
}
