// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.util.indexing.roots;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.platform.workspace.storage.EntityStorage;
import com.intellij.platform.workspace.storage.WorkspaceEntity;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;

/**
 * This extension point together with  {@link com.intellij.util.indexing.roots.builders.IndexableIteratorBuilderHandler} allows to control
 * indexing of Workspace Model entities and might be needed for custom entities in exotic corner cases.
 * {@link Enforced} interface allows enforcing reindexing of additional paths on entity change. Those paths are not directly associated
 * with the entity by its {@link com.intellij.workspaceModel.core.fileIndex.WorkspaceFileIndexContributor}.
 * <p/>
 * {@link IndexableEntityProvider} which is not {@link Enforced} is effectively ignored.
 */
@ApiStatus.Internal
@ApiStatus.Experimental
public interface IndexableEntityProvider<E extends WorkspaceEntity> {
  ExtensionPointName<IndexableEntityProvider<? extends WorkspaceEntity>> EP_NAME =
    new ExtensionPointName<>("com.intellij.indexableEntityProvider");

  @NotNull
  Class<E> getEntityClass();

  /**
   * Provides builders of iterators to index files after {@code entity} was added
   */
  @NotNull
  Collection<? extends IndexableIteratorBuilder> getAddedEntityIteratorBuilders(@NotNull E entity,
                                                                                @NotNull Project project);

  /**
   * Idea behind this marker interface is to mark that something should be reindexed as cheap as possible,
   * with expensive checks and merges made in batch in corresponding
   * {@link com.intellij.util.indexing.roots.builders.IndexableIteratorBuilderHandler#instantiate(Collection, Project, EntityStorage)}
   */
  interface IndexableIteratorBuilder {
  }

  /**
   * Marks providers that should be used to determine the scope of reindexing on Workspace model changes even after switching to
   * {@link com.intellij.workspaceModel.core.fileIndex.WorkspaceFileIndexContributor}
   */
  interface Enforced<E extends WorkspaceEntity> extends IndexableEntityProvider<E> {
  }
}
