// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.execution.util

data class EnvVariables(val envs: Map<String, String> = emptyMap(), val envFiles: List<String> = emptyList()) {
  companion object {
    /**
     * This will parse the environment variable pairs from a text string. The expected text corresponds to the text generated by
     * `com.intellij.execution.configuration.EnvironmentVariablesTextFieldWithBrowseButton.stringifyEnvironment`.
     *
     * The specification for that string follows.
     *
     * The string consists of pairs `key=value`, separated by one or more `;` characters, and the whole string terminated by zero or more
     * `;` characters.
     *
     * Each of `key` and `value` might be either
     * - a sequence of characters excluding `"`, `;` and `=`,
     * - or a sequence of any characters (including `;` and `=`) inside of double quotes `""`.
     *
     * For the latter case, escape sequences are allowed: character `\` might precede other characters, so that they lose their meta meaning
     * and just become characters included into the `key` or `value`. Escape sequences supported:
     * - `\\` that gets converted into `\` character,
     * - `\"` that gets converted into `"` character.
     *
     * Items that don't contain '=' are treated as potential file paths
     * - If an item without '=' can be converted to a valid Path, it's added to the file list
     * - Otherwise, it's treated as an environment variable with an empty value
     *
     * Note that this method has "quirks mode" behavior for incorrect character sequences, such as unpaired quotes, invalid escape sequences
     * etc. This behavior is not a part of the method contract.
     */
    @JvmStatic
    fun parseFromText(content: String?): EnvVariables {
      val parsedEnvs = mutableMapOf<String, String>()
      val parsedFiles = mutableListOf<String>()
      val possibleFiles = mutableListOf<String>()

      if (content == null) {
        return EnvVariables(parsedEnvs, parsedFiles)
      }

      var index = 0
      while (index < content.length) {
        fun readItem(endOnEqualSign: Boolean): String {
          if (index >= content.length) {
            return ""
          }
          val buffer = StringBuilder()

          val firstCharacter = content[index]
          if (firstCharacter == '"') {
            while (index + 1 < content.length) {
              index += 1
              var c = content[index]
              if (c == '\\') {
                index += 1
                if (index >= content.length) {
                  // Escape character is present but there's no escaped character. Add slash and exit.
                  buffer.append('\\')
                  break
                }

                val next = content[index]
                buffer.append(next)
                continue
              }

              if (c == '"') {
                // We found the closing quote.
                index += 1
                if (index >= content.length) break
                val next = content[index]
                if ((endOnEqualSign && next == '=') || next == ';') {
                  break
                }

                // At this point, we found the closing quote, but next to it there are some characters other than = or ;
                // Means we should treat the remaining text literally as the fallback.
                while (index < content.length) {
                  c = content[index]
                  if ((endOnEqualSign && c == '=') || c == ';') break
                  buffer.append(c)
                  index += 1
                }

                break
              }

              buffer.append(c)
            }
          }
          else {
            while (index < content.length) {
              val c = content[index]
              if ((endOnEqualSign && c == '=') || c == ';') break
              buffer.append(c)
              index += 1
            }
          }

          return buffer.toString()
        }

        fun readKey() = readItem(true)
        fun readValue() = readItem(false)

        val key = readKey()
        if (index >= content.length) {
          if (key.isNotBlank()) {
            possibleFiles.add(key)
          }
          possibleFiles.forEach { possibleFile ->
            parsedFiles.add(possibleFile)
          }
          possibleFiles.clear()
          break
        }

        when (content[index]) {
          '=' -> {
            index += 1 // eat '='

            val value = readValue()
            parsedEnvs[key] = value
            possibleFiles.forEach { parsedEnvs.putIfAbsent(it, "") }
            possibleFiles.clear()
          }
          ';' -> {
            if (key.isNotBlank()) {
              possibleFiles.add(key)
            }
          }
          else -> {
            throw RuntimeException("Parse error at $index")
          }
        }
        index += 1
      }

      return EnvVariables(parsedEnvs, parsedFiles)
    }
  }
}