/*
 * Copyright 2000-2009 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.history.core.changes;

import com.intellij.history.ActivityId;
import com.intellij.history.core.Content;
import com.intellij.history.core.DataStreamUtil;
import com.intellij.history.utils.LocalHistoryLog;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.io.DataInputOutputUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public final class ChangeSet {
  private static final int VERSION = 1;
  private final long myId;
  private final long myTimestamp;
  private final List<Change> myChanges;

  private @Nullable @NlsContexts.Label String myName;
  private @Nullable @NonNls ActivityId myActivityId = null;

  private volatile boolean isLocked = false;

  public ChangeSet(long id, long timestamp) {
    myId = id;
    myTimestamp = timestamp;
    myChanges = new ArrayList<>();
  }

  public ChangeSet(DataInput in) throws IOException {
    int version = DataInputOutputUtil.readINT(in);
    myId = DataInputOutputUtil.readLONG(in);
    myName = DataStreamUtil.readStringOrNull(in); //NON-NLS
    myTimestamp = DataInputOutputUtil.readTIME(in);

    if (version >= 1) {
      myActivityId = readActivityId(in);
    }

    int count = DataInputOutputUtil.readINT(in);
    List<Change> changes = new ArrayList<>(count);
    while (count-- > 0) {
      changes.add(DataStreamUtil.readChange(in));
    }
    myChanges = Collections.unmodifiableList(changes);
    isLocked = true;
  }

  public void write(DataOutput out) throws IOException {
    LocalHistoryLog.LOG.assertTrue(isLocked, "Changeset should be locked");
    if (LocalHistoryLog.LOG.isTraceEnabled()) {
      int maximumChanges = Registry.intValue("lvcs.trace.changes.persistence.limit", 100);
      String lastChanges = myChanges.reversed().stream().limit(maximumChanges)
        .map(Object::toString)
        .collect(Collectors.joining("\n"));
      LocalHistoryLog.LOG.trace("Writing changeset. Changes count: " + myChanges.size() + ". 100 latest changes: \n" + lastChanges);
    }
    DataInputOutputUtil.writeINT(out, VERSION);
    DataInputOutputUtil.writeLONG(out, myId);
    DataStreamUtil.writeStringOrNull(out, myName);
    DataInputOutputUtil.writeTIME(out, myTimestamp);

    writeActivityId(out, myActivityId);

    DataInputOutputUtil.writeINT(out, myChanges.size());
    for (Change c : myChanges) {
      DataStreamUtil.writeChange(out, c);
    }
  }

  public void setName(@Nullable @NlsContexts.Label String name) {
    myName = name;
  }

  public @NlsContexts.Label @Nullable String getName() {
    return myName;
  }

  public void setActivityId(@Nullable ActivityId activityId) {
    myActivityId = activityId;
  }

  public @Nullable ActivityId getActivityId() {
    return myActivityId;
  }

  public long getTimestamp() {
    return myTimestamp;
  }

  public void lock() {
    isLocked = true;
  }

  public @NlsContexts.Label @Nullable String getLabel() {
    //noinspection RedundantTypeArguments
    return this.<@NlsContexts.Label @Nullable String>accessChanges(() -> {
      for (Change each : myChanges) {
        if (each instanceof PutLabelChange) {
          return ((PutLabelChange)each).getName();
        }
      }
      return null;
    });
  }

  public int getLabelColor() {
    return accessChanges(() -> {
      for (Change each : myChanges) {
        if (each instanceof PutSystemLabelChange) {
          return ((PutSystemLabelChange)each).getColor();
        }
      }
      return -1;
    });
  }

  public void addChange(final Change c) {
    LocalHistoryLog.LOG.assertTrue(!isLocked, "Changeset is already locked");
    accessChanges((Runnable)() -> myChanges.add(c));
  }

  public List<Change> getChanges() {
    return accessChanges(() -> {
      if (isLocked) return myChanges;
      return Collections.unmodifiableList(new ArrayList<>(myChanges));
    });
  }

  public boolean isEmpty() {
    return accessChanges(() -> myChanges.isEmpty());
  }

  public boolean anyChangeMatches(@NotNull Predicate<Change> predicate) {
    return accessChanges(() -> {
      for (Change c : myChanges) {
        if (predicate.test(c)) return true;
      }
      return false;
    });
  }

  public List<Content> getContentsToPurge() {
    return accessChanges(() -> {
      List<Content> result = new ArrayList<>();
      for (Change c : myChanges) {
        result.addAll(c.getContentsToPurge());
      }
      return result;
    });
  }

  public boolean isContentChangeOnly() {
    return accessChanges(() -> myChanges.size() == 1 && getFirstChange() instanceof ContentChange);
  }

  public boolean isLabelOnly() {
    return accessChanges(() -> myChanges.size() == 1 && getFirstChange() instanceof PutLabelChange);
  }

  public boolean isSystemLabelOnly() {
    return accessChanges(() -> myChanges.size() == 1 && getFirstChange() instanceof PutSystemLabelChange);
  }

  public Change getFirstChange() {
    return accessChanges(() -> myChanges.get(0));
  }

  public Change getLastChange() {
    return accessChanges(() -> myChanges.get(myChanges.size() - 1));
  }

  public List<String> getAffectedPaths() {
    return accessChanges(() -> {
      List<String> result = new SmartList<>();
      for (Change each : myChanges) {
        if (each instanceof StructuralChange) {
          result.add(((StructuralChange)each).getPath());
        }
      }
      return result;
    });
  }

  @Override
  public String toString() {
    return accessChanges(() -> myChanges.toString());
  }

  public long getId() {
    return myId;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    ChangeSet change = (ChangeSet)o;

    if (myId != change.myId) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return Long.hashCode(myId);
  }

  public void accept(ChangeVisitor v) throws ChangeVisitor.StopVisitingException {
    if (isLocked) {
      doAccept(v);
      return;
    }

    synchronized (myChanges) {
      doAccept(v);
    }
  }

  private void doAccept(ChangeVisitor v) throws ChangeVisitor.StopVisitingException {
    v.begin(this);
    for (Change c : ContainerUtil.iterateBackward(myChanges)) {
      c.accept(v);
    }
    v.end(this);
  }

  private <T> T accessChanges(@NotNull Supplier<T> func) {
    if (isLocked) {
      return func.get();
    }

    synchronized (myChanges) {
      return func.get();
    }
  }

  private void accessChanges(final @NotNull Runnable func) {
    accessChanges(() -> {
      func.run();
      return null;
    });
  }

  private static @Nullable ActivityId readActivityId(@NotNull DataInput in) throws IOException {
    String kind = DataStreamUtil.readStringOrNull(in);
    String provider = DataStreamUtil.readStringOrNull(in);
    if (kind == null || provider == null) return null;
    return new ActivityId(provider, kind);
  }

  private static void writeActivityId(@NotNull DataOutput out, @Nullable ActivityId activityId) throws IOException {
    DataStreamUtil.writeStringOrNull(out, activityId != null ? activityId.getKind() : null);
    DataStreamUtil.writeStringOrNull(out, activityId != null ? activityId.getProviderId() : null);
  }
}
