// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.testFramework.common

import org.jetbrains.annotations.ApiStatus
import org.jetbrains.intellij.bazelEnvironment.BazelLabel
import org.jetbrains.intellij.bazelEnvironment.BazelRunfiles
import java.nio.file.Path
import kotlin.io.path.*

@ApiStatus.Experimental
object BazelTestUtil {
  // see https://bazel.build/reference/test-encyclopedia#initial-conditions
  // also https://leimao.github.io/blog/Bazel-Test-Outputs/
  private const val TEST_SRCDIR_ENV_NAME = "TEST_SRCDIR"
  private const val TEST_TMPDIR_ENV_NAME = "TEST_TMPDIR"
  private const val TEST_UNDECLARED_OUTPUTS_DIR_ENV_NAME = "TEST_UNDECLARED_OUTPUTS_DIR"

  @JvmStatic
  val isUnderBazelTest: Boolean =
    System.getenv(TEST_SRCDIR_ENV_NAME) != null &&
    System.getenv(TEST_UNDECLARED_OUTPUTS_DIR_ENV_NAME) != null

  @JvmStatic
  val bazelTestTmpDirPath: Path by lazy {
    val value = System.getenv(TEST_TMPDIR_ENV_NAME)
    if (value == null) {
      error("Not running under `bazel test` because $TEST_TMPDIR_ENV_NAME env is not set. Check isUnderBazelTest first.")
    }
    val path = Path.of(value).absolute()
    if (!path.isDirectory()) {
      error("Bazel test env '$TEST_TMPDIR_ENV_NAME' points to non-directory: $path")
    }
    path
  }

  @JvmStatic
  val bazelUndeclaredTestOutputsPath: Path by lazy {
    val value = System.getenv(TEST_UNDECLARED_OUTPUTS_DIR_ENV_NAME)
                ?: error("Not running under `bazel test` because " +
                         "$TEST_UNDECLARED_OUTPUTS_DIR_ENV_NAME env is not set. " +
                         "Check isUnderBazelTest first.")
    val path = Path.of(value).absolute()
    if (!path.isDirectory()) {
      error("Bazel test env '$TEST_UNDECLARED_OUTPUTS_DIR_ENV_NAME' points to non-directory: $path")
    }
    path
  }

  @JvmStatic
  fun getFileFromBazelRuntime(label: BazelLabel): Path {
    return BazelRunfiles.getFileByLabel(label)
  }

  /**
   * Tests under community root may run in community (OSS) or in the ultimate monorepo context.
   *
   * Under ultimate monorepo Bazel project, workspace for test dependencies is named `community+`,
   * while when run under community Bazel project, it's named `_main`.
   *
   * This function finds `relativePath` under one of them, depending on current project.
   * It fails when the directory can't be found or there is an ambiguity.
   *
   * see https://bazel.build/reference/be/common-definitions#typical-attributes (check `data`)
   *
   * see https://bazel.build/reference/test-encyclopedia#initial-conditions
   */
  @JvmStatic
  fun findRunfilesDirectoryUnderCommunityOrUltimate(relativePath: String): Path {
    return BazelRunfiles.findRunfilesDirectoryUnderCommunityOrUltimate(relativePath)
  }
}
