// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.psi.codeStyle

import com.intellij.openapi.util.TextRange
import com.intellij.util.containers.FList
import org.jetbrains.annotations.ApiStatus
import kotlin.jvm.JvmStatic

class PinyinMatcher internal constructor(override val pattern: String) : MinusculeMatcher() {
  override fun match(name: String): List<TextRange>? {
    val pattern = this.pattern
    val patternLength = pattern.length
    val nameLength = name.length
    val maxOffset = nameLength - patternLength
    var start = 0
    while (start <= maxOffset) {
      val c = name[start]
      if (c.code in BASE_CODE_POINT..<END_CODE_POINT) {
        break
      }
      start++
    }
    var end = start + patternLength
    OUTER@ while (start <= maxOffset && end <= nameLength) {
      for (i in start..<end) {
        val c = name[i]
        if (c.code !in BASE_CODE_POINT..<END_CODE_POINT) {
          start++
          end++
          continue@OUTER
        }
        val code: Int = DATA[c.code - BASE_CODE_POINT].code - BASE_CHAR.code
        if (code < 0 || ENCODING_ARRAY[code].indexOf(pattern[i - start]) == -1) {
          start++
          end++
          continue@OUTER
        }
      }
      return listOf(TextRange.create(start, start + patternLength))
    }
    return null
  }

  @Deprecated("use match(String)", replaceWith = ReplaceWith("match(name)"))
  override fun matchingFragments(name: String): FList<TextRange>? {
    return match(name)?.asReversed()?.let { FList.createFromReversed(it) }
  }

  override fun matchingDegree(name: String, valueStartCaseMatch: Boolean, fragments: List<TextRange>?): Int {
    return if (!fragments.isNullOrEmpty()) {
      val range = fragments.first()
      if (range.startOffset == 0) {
        return 500 + range.length
      }

      val prefix = name.substring(0, range.startOffset)
      if (prefix == "get" || prefix == "is" || prefix == "set") {
        200 + range.length
      }
      else {
        range.length
      }
    }
    else {
      Int.MIN_VALUE
    }
  }

  @Deprecated("use matchingDegree(String, Boolean, List<TextRange>)", replaceWith = ReplaceWith("matchingDegree(name, valueStartCaseMatch, fragments as List<TextRange>?)"))
  override fun matchingDegree(
    name: String,
    valueStartCaseMatch: Boolean,
    fragments: FList<out TextRange>?,
  ): Int {
    return matchingDegree(name, valueStartCaseMatch, fragments as List<TextRange>?)
  }

  companion object {
    @ApiStatus.Internal
    const val BASE_CODE_POINT: Int = 0x3400

    @ApiStatus.Internal
    const val BASE_CHAR: Char = '!'

    /**
     * To generate `ENCODING` and `DATA`, see `PinyinMatcherDataTest` class in tests.
     */
    @ApiStatus.Internal
    val ENCODING: String =
      "a,b,c,d,e,f,g,h,j,k,l,m,n,o,p,q,r,s,t,w,x,y,z,ab,ad,ae,ag,ah,aj,ak,al,an,ap,aq,at,aw,ax,ay,az,bc,bd,b" +
      "e,bf,bg,bh,bj,bk,bl,bm,bn,bp,bq,bs,bt,bx,by,bz,cd,ce,cf,cg,ch,cj,ck,cl,cm,cn,cp,cq,cr,cs,ct,cw,cx,cy," +
      "cz,de,df,dg,dh,dj,dl,dm,dn,dq,dr,ds,dt,dw,dx,dy,dz,eg,eh,ej,ek,el,en,eo,ep,eq,er,es,et,ew,ex,ey,ez,fg" +
      ",fh,fj,fk,fl,fm,fn,fp,fq,fr,fs,ft,fw,fx,fy,fz,gh,gj,gk,gl,gm,gn,gp,gq,gr,gs,gt,gw,gx,gy,gz,hj,hk,hl,h" +
      "m,hn,ho,hp,hq,hr,hs,ht,hw,hx,hy,hz,jk,jl,jm,jn,jp,jq,jr,js,jt,jw,jx,jy,jz,kl,km,kn,ko,kq,kr,ks,kt,kw," +
      "kx,ky,kz,lm,ln,lp,lq,lr,ls,lt,lw,lx,ly,lz,mn,mp,mr,ms,mt,mw,mx,my,mz,nq,nr,ns,nt,nw,nx,ny,nz,oq,os,ow" +
      ",ox,oy,pq,ps,pt,pw,px,py,pz,qr,qs,qt,qw,qx,qy,qz,rs,rt,rw,rx,ry,rz,st,sw,sx,sy,sz,tw,tx,ty,tz,wx,wy,w" +
      "z,xy,xz,yz,abd,aek,aen,aex,aey,agh,agq,agw,agy,ahj,ahk,ahq,ajn,ajx,akz,any,aoy,aqy,ast,asx,awy,bcp,bc" +
      "t,bfh,bfp,bfy,bgh,bhp,bjx,blp,bmp,bps,bpt,bpz,btx,cdj,cdq,cds,cdt,cdx,cdy,cdz,ceh,ceq,cez,cgh,cgl,cgq" +
      ",cgz,chj,chx,chy,chz,cjq,cjr,cjs,cjt,cjw,cjx,cjz,clx,cnt,cnx,cnz,cox,cps,cqs,cqt,cqx,cqy,cqz,cst,csx," +
      "csy,csz,cty,ctz,cwz,cxy,cxz,cyz,dgj,dgk,dgz,dhk,djs,dkq,dlt,dlx,dly,dmt,dnq,dnr,dnt,dnx,dps,dqx,dqz,d" +
      "ry,dst,dsw,dsy,dsz,dtw,dtx,dty,dtz,dwz,dxy,dxz,dyz,egh,egl,egw,ehw,ehz,ejq,ekq,emn,enr,eqw,ewy,exy,fg" +
      "j,fgz,fjn,fmw,fmz,fpw,fsx,ghj,ghk,ghl,ghn,ghq,ghs,ghw,ghx,ghy,gjk,gjl,gjq,gjr,gjx,gkl,gkn,gkq,gkt,gkw" +
      ",glw,glx,gly,gny,gqw,gqx,gqy,gqz,gst,gtx,gty,gwy,gxy,gyz,hjk,hjn,hjq,hjx,hjz,hko,hkq,hkt,hkx,hky,hly," +
      "hms,hmw,hnt,hpt,hpx,hqx,hqy,hrs,hry,hst,hsw,hxy,jkq,jkx,jky,jkz,jlm,jln,jlp,jlx,jnp,jny,jpx,jqs,jqx,j" +
      "qy,jqz,jst,jsx,jsy,jsz,jtz,jxy,jxz,jyz,klw,koq,kqs,kqx,kqy,ksz,kxy,lmp,lmy,lnx,lpx,lqs,lqx,lrs,lsx,ls" +
      "y,ltx,ltz,lxz,mnw,mnx,mnz,mow,mqx,msy,mtx,nqx,nrs,nrw,nrx,nrz,nsz,ntx,nwz,nxy,psx,psy,pyz,qst,qsx,qsy" +
      ",qsz,qtx,qtz,qxy,rst,rsx,rtz,stx,sty,stz,swy,sxy,sxz,syz,twz,txy,tyz,wxy,xyz,achy,acsy,aghk,ajqy,bcpz" +
      ",bdsz,bflp,cdnq,cdqz,cdst,cdtx,cdtz,chqz,cjnt,cjqz,cjsx,cjxz,cqtz,csty,dgkt,dgyz,dlsy,dqsx,dqty,dqtz," +
      "dsty,dstz,dsyz,dtxy,dtxz,dtyz,egqy,ehkw,ejwy,enyz,foqx,ghjk,gkoq,hjkq,hmtw,hswx,hsxy,hwxy,jlqy,jlqz,j" +
      "lxy,jlyz,jsty,jsyz,klxy,ksyz,nsyz,qsyz,stxz,bcjxz,bjlmx,cdstz,djstz,dltyz,dqxyz,ghkwx,cdjstyz,gjklqyz"

    @ApiStatus.Internal
    val DATA: String =
      "03  *46     6         5     #    -  u   55 5++5657\$4/1 ,e6\u0117-#&   7&!47 7\$260\u00D0+-72   5  2)" +
        "\$-2k+6\"   +,6+)6+#   (5(1 7 (536(),+367767%10#4(3220+2  \u00AD/3\u017B6.3 ,\u0113\$6\u00E4)-56  #&+" +
        "+ + )    0 #+', '2  ,,02+ 4*\$73  \"(\$'# 0)+5 -5 \")2 ' 650/\$\u008D+\u00A403\"'#\u00C5+  2\$+(\u0109" +
        "/\$1%06|\u014E\u00C0 ##j +\$+\u00C5257  7|\u00BCll5+ #5) ) K#\") \"+* \u00B8\"\u00C5+5S6+\"%+  #-\u010B" +
        "(6 '65'+3+2 26\u00C57 0\u010157)(+6*&0! *#5#4  (26 )752 2*2(\u009D602(5&/\$&\u00A6!&-\u00E56\u00B9 #'" +
        "-\u00B2+&5/ 2  ++6( 50%  5    -453(3\u00ED-67     42-603()\$\u00B7+\u00C2\u01A1+ 3+      '6\u0109\u00ED" +
        "\u0094#\u00E9\$7%\u01156%2\u00B7*('\u0115,\u0081\u00BC(7  4 \"!70'\$\"\"\u0157+55     p3#\u009F+|E5 0" +
        "75  #)(x65/2   '655\u00B876# +531\u00EE7\u01155))( /+ +2\u00AE60 6(*#-( \"54\u01154605-\"()D!  73) 7\"" +
        ")!7#&    -\u00BC,y\"\"\" #53l  \$4&\$66*\u00B7* 46+5\"#5-+67,75'5 )+6)6 \$65(6\u00AD#337\",\u009F#+ 5" +
        " 50-6 +&7 '*\$(* '3 2+\u00ED&\u00AB0\u012F)*5\u011572#5)\u00EC5666621(2)6 5S0Z \"\$\u010C% \u00D46) 0" +
        "5/#/-)6%\$   \$  \u0100+02#\$5\u0094\u008B2\u0212/\u0100\u011056\"+    6+3+,0765+52\u0197 4&6-#3)2/!*" +
        "g 57  &+55,\u00B9,676\u01A93562&6\$ +#g\u014D+2-!Fe ## 1\u00EB,56-!2,  +5674(,-4\u00E5-07+77(5(\u0100" +
        "7,#' #+72+)5\$)1, *6-6+6662 +6,,766',)4 *\$+ 5#0,,0 49! \"-+)6#7\"(63)!+)4\"75\u00BE6 S\u00C7)45\u00C0" +
        "65#0\$-\u0102)3 2\$+7  k#)(7+-)0\$ )5#6\$b ,(2*7(#5-#+!,256(#0+&\$,+3(,\"00( +\"3707,/75' 0 +#6%\$,40" +
        "2,3 ,6)+*'6\$ ++\u0102!\"6,\$ ! 5&\$)130%0))()7\",\"46!6(\$4-# ++-(+ +&,6 )  7\u00A36 \u00C0\" 1(&'*)" +
        "\u00E2 236l\"*+(5- \u00C7\u00E450F, 2\u0219 \u00C07*46#3+\"',5&+77#,+ 6/7\u00C0+-6 ))|6)(7++7\"l)\$+6" +
        "75##g\u0181+\$+# x#F# 67\u0110 3#67 S0+0+\u0101*26#)) ) \$ )(\$  ( #s\" 5+ \u010B\" \" 51\"%) 7\"%607" +
        ",62   23,)\"1##\$    |x\u00B652\$53 32)746  (|07 5574  37-  561\u00EA#| ,)&/\u0125&!0\u01015&'0'3#6'\"" +
        "&d7 7 (-)#\u0141*++6 6(+('*6\$()')7n0\"\u013D#,(5\u00C0(  #\u01096|\$3,+23#\$#/6))+36 #06+#   6\u00BB" +
        "4,255#\u00C0( 602\u010A\$+ /3l62&g6\u0110(\u00FD6,&( 6l-!&+##,Z\u00B8+!6(7\u00B8%(\u010B-\u00B8+\u0106" +
        "6+,)) \"6\u00B8,#, +)4655#+657++0()4  '7{6)615266\u00B3)\u00B33\u00C20s),6\u01A2S5,K\u0101'2(6''+%\u00C7" +
        "7\"4++  * \u00B767#5\u00B8 0/\u00A8\u0086'0(++#2Y\u0089/*3/33|\"'( \"7\$Z\u0111\u00E9+/)\u00E4\u017F#" +
        "'\"13\"7*+(67)55\u010B2052(,(/ (-2\"a)(+ (\$('6#\u009E))\$),\u00B8,\u0162+/6))0 02 \u00C7\u00BA6\u0170" +
        "5\u010B&#6 \$ 0(3 05-2,3+(#7A+, & 5\"(05  \"0/))*2\u00C0%-0\$\$ \u00A47\u0143\u0102*6\u00CC*(& \$5'(0" +
        "(3\"\u00EA\"+6-#2Z+3+*27Z+, \u00C0*2\u0100\"7652221l++# (  )\"(\$ +))*2\u00A07\$\$76/#    \u008B  6\u009D" +
        "66\u010D'0'  (,&('3\$ 6  0 #,3\"!  5   ,+-\"# &7))2 *\$+    63#,,(4-(()\"5+\u00C7 ,/(6 524#5)g(&2+ \$" +
        "0S6\" (\$6))-+6334(5( 37 (&  3 1) 5  6)( \", '4 ,2\"+0  \u009D7+S1\u00F0+'555 -   5+&#7\"7|7() )   35" +
        "F33\$)!(5)\u0105\"7-3-  6,  -2e6'h1)0,%+k7()\u00CA '(' K24)50\u01CB45)(/2#7 72)27\"   +\"5/\$07Sg\"2(" +
        "(\u00C6#)-'&  x#5\$5*73(#+)+0/7 +l0  3+0++2\u01697/)0  7/)\")' \"7+++2,)|\" + 5/0+7 )25 5 +06-+6 6#\u00C0" +
        "(5*7\u00B3\u00B80(5('#552\u01C2\u01CB2\u01156/7\u00F47+\u00B85\u01D1\u00FD6)5#\u00CE*6\u00AF65\u010B)" +
        "660k+*7    5 #5!\u00A9#H7 \u00DD/\u009F2+35+0\u0113+\$,+4\$*!7(6,7&S \"+++(\u0102\u009F\u00B3l\u00D3#" +
        ")\u00BB)+ \u010901#-7+/  T\u01035+5&0 2731\u010D6,-\" 23(21 \$/)3 1-+\$|7 +\$5')-k\u00E964'76#5)3&/ \u0117" +
        "06\"\"276\" \$+6##5565   3'   +5f)3/755-     ,)\u00E9\u024C(467X,05&63+6 )\u00E26) 2  -4,262\u021D)+6" +
        ",727\"+       0'576-\u00C65657- 2+#,\u00B3\$2   25\"\$7 22\u00B93'   -,5,)#)(\u01302#2-/,    6# *\"+*" +
        "2,2\u00AAS6\u0115-67#&5/4+#\u00C77\"6\u00AF/5 +\u00C05)\u009D  65l(65  +5(6 +\u0109#\")2&\$ \"6  -+6p" +
        " k#')-   +,6 +( 6)#1-  (3 E'\"\"0 2#(#&\u009D%4&3 +(6( 67607)3      3+)75\u00B8 5   #0, 7\"a  6',\")\$" +
        "\u010D 46 )53( \$))*7 5+&(5-,1#\u00C6, )0347+2*\u00D17732\"6/#\u00AC*)!#0\"\"&*337+ '62\u01C0+22 \"1\$" +
        "\"+0&)7(\u00AC+\u011365g\$+)#2/6)((31461626\u00BB35\u009E707\$ 6  626,  054'632-'34)#\u00BF7\u00B0-\u00AA" +
        "'6( 2#(645/1+\u00D92\u02037(K(#7 37-(676+(55#+ 2\u00C0/0'76  1-,\") 5/\",    66 /0\"(   \",+34++ ))7\$" +
        ")\$\"     +\"23z     \$  2\"32#3,)+  7\")75+  \u010B62  %   60+73\u00A56+\"-/+3+4\$+347'5!&S\"326\"0\u00CF" +
        "'3(g)(5\$e\$/\u00C1+2.\$6l1\u00AD3#/g+\u01117+2/2 \$(655++ \u010F' 3   6- ')#\u00A44+++5\"!6)157##76\u0094" +
        "1 52\u00BE\u00C557  2#&)7 '4 02(3Z71,&+\u014F55-\u0115\u00C6+p16+2++'5500#6\"*0!00#/\u00C8\u0129'2/ 7" +
        "(\u00B6(\"  7)'5'l,l3\u0111\u00DC+   |!5'5 3#25)&0 6\u0141)+6+36/++\" ,\"( 6(#(-#+3(\"\"a6\$85 7\$1\"" +
        "#\u00E43\"'6576&6!3,2)')0)65*,5240(173\u00C5\u00C50!\u00E4' 3&)\u013E2Q,2'(\u009D+(\u00C30'(\"'&,!\u00D2" +
        "2+)#\u0255h/f6\$\$ +5\"5,6,004\u01166\u00CCh666\u00C7\u00C7\u00B8,+*66\u00FD'(62#+ 5#\u014C-0,(67\u00AF" +
        "#\"05)*,,\u010DM)\u00C4-\"\u010C\u00D9,+,,5#0)(55-3(6 #\"l\$7)0 5(2* \$66\")7)2\$\$#&,77 \u0101u('0+'" +
        "+1&#6')''2\u00D07-\$+3#6'  \$ ,),)6\u0117775 S('\u00B3+4Y*+l+602 5\u00C00(2#\$+ -3\$\u01E9'\$+#5/+7-\u009D" +
        "6\u00C5+,\" \u0194\"&(6+66)(#z% 6l7060(-()2  /,#+a# #2\$0 +++166\$06-&)6&15  \"2/-\u0111h3316+!(\u00C0" +
        "+\u00A87266#4)36()1,)7)F2+(0#\u013E*-#35)(0\u009E,6 \u01023 !/#3%(0l27-h\u010D+Q+,\u00A9e(,'+\$77  7\"" +
        "b0 +5&'#)\u0086#6(6\u01176x76\u01166>(+6#  2 -,(#65)763*0 #)57h/0 0+61202 & ,(  (0e0\$+S)5\u00D5 67(6" +
        "/70 ')77\u009F-17F\u00C7\u00A96#)\u0162+117-#  6\u00C4\"h)\$4676 2+/7(6\$7K\u00FD+30,6K''\u00D5\u00A4" +
        " \u0254*2#73*05\$l\u00B9g\u010B70)74  5-/\u00B5)723S3041+(27/,)2,2++)+7\$ 2\u009D&,2)35*4&+##+6#6,F76" +
        "\$(4325220+6 ++&++5\u00C5.,51726\"+(0+\"\"(( g/0,\u00C0\",24  +0 (5 26 -()6 235l,5, (x56\"+&+##Z\$*#5" +
        "#,007   ('#\u009E,\"('6,4   \u00E957,) 776+ 7\"16 #%6,0 4)\" \"\u00946+&0&%%736)0)+#\"-65,5& -\u0110\"" +
        "\u00EF4*6.\u010B'3'(3 6\$0)+,,)+&.,)&\"(26-)+\")3+7\u010D6\$##,6+#)2\u00A4-+ 2+7 #))%61#1&+266\$+73/7" +
        ")#\"*j ((,('\$&5,,+'7367++4+)++ \u0094643/ 7' 5-\u014D47\$4\" (5l+6(5(\"1\u010F/+765\u00D6*# -(#+73&(" +
        "702/' (# %\u0082102) 6SF7-#\"#3(7\"74\u00A6\"%74477+4'0+72,|0#/+2+2\u01016&5)\$/'\u02071\u0117'/\u0182" +
        "b#(7,\"\u00D72\")2 \u00A7cp5%5, 571)x72,\"(6j/-lS 0)|+\u00C7\u00B85#i7\u00EF\u00AD#3#\$g0771\")5|)')\$" +
        "#21\u00A9(0l2#4*\u00AD\u00A4\u009F&##|#+5776'\u009D\u01E0\$\u009D\u010B3'5,\$!/4!,\u00FA,67\"641+6#\u00B3" +
        "7/ 3&2 \$#\u01043#3)2(+  #)\u0087'&\$ %\"\u0110\$ \$\"47+0,0 72 673 2\"5)#&3#&&456,/,2,#3*\"550  2(-5" +
        "k\$\u009A6!11,&02(\"\$,\$  * #\$6( '5+)0046+  66607\u01C366\u011377(#&#/*# 30#\u009E\u009Fe07,+\u00DE" +
        "37'+(+\$#6  07#,\$6l1\u00B81&+5)6, \u00AF)2( +776'+B'\u00D9\"2  2#)\u0221\"5)/+57\$#0-7+626(+/  +  2 " +
        "6-2*&6Z1# 30\"24\$\u010D!\u00857*& 6 2\$(573376&% ))#5, \"+)*) #+5&+7()-6\u00C53+7* \u0100\u00C0/&3+," +
        "75(,)7#+176( 6+\u00DD1+-6- 6-605#6,5-\u00AD75633426#7,6-,74+\u0102-\u010D\"l\" )5\u00D0/*\u011676*(5%" +
        "\u00E636)+ +#\u00C0(7&#'\u00C0,#3,5+6226\"#(4\$)'(\"5+##6567(67 *607,  2)'/+ 3)+66#{)545-&' ,7-54+#,)" +
        "-#45-,7**(  \u010C47  6 \$&\"772\u00C5- )5#-172 26 ) +(\"6l764-')\"# \$\u01026*\u00E9(55 * 5\"*#5\".7" +
        " 3l#-'&5\$4)\u00AD72\$+, ',+ 6 '#+\$ J05 \$\$5+ \u00C2524  (6++62)0)6 0 +),l-\"++'#55(\"7\$\$(2#3 \u00E9" +
        "7+&\u00C0'E+51\u010F\u0084527433\u01155/66-(\u00E9\u0115K-6465762,\u0116)\u00F7&5-#,\")-6%7\u00E95+(5" +
        "-()\$(3d#52,5\u01900735))(3,6\u00C02+4#j)6(\u01622'-5 \$#i\u01FE'l#7(+++3(4!22\u00AF3\u00B8):3#43)##6" +
        "55   - 60\u00AC2\u00C1+) ,+ g#50,2##\u00A166 ++272(  % 27))3+y#7+ 77 6#-\"26(6(,,0+(7,57277'5675)^ \"" +
        "+7)4')\u00C0!&'(617\"62\$'25#20  #(\$#4\u01072#)7\u00CC6&6&/737e\u00956(('\u008A)63e3&7( \"7\u00C73)&" +
        "(#6#2Z0#j,#\u0102\u00C7\"\u00C57\u010D07)I ++60\u0141K#32(00K-3##6|##\",\"3,\u00D7&g0\u0111K*)   70le" +
        "\u009F+S|4 \u00C7+h #+3*#)3(&\" \u01004\u00BB0\u00D0(e2# \u0100//4615#\$Z#,5\$\"7##'\$7\$+3607+#7&23*" +
        "! 02\u00AD2\u00D07+)+ j16)*&0(&+(3,3\u00C5070'3+,6+6)/ 5\"\u00B87124#\u00D32/,5 l\u009F7Z2#2\"* 74,+7" +
        "\u00EE+06 65# 1^-6)\"&  #\$)6(5\u00DD0 \"\u009D) 6262\u0224#6262+l6\"2 \u0115++3\$6) 6  )\u00C0##'55!" +
        "\u01B16+ 3#50( \$#\$67 6# +2,, 3S+\u0100#3%h\u01E56)\$)\"+6##/+\u00AB3/h)5& 5%\u00B67\u00E9&)(+\u0111" +
        ",6 #,F,3\u009F  ,1\u00AA0,)/)4),)\u00B8\"'e++0\$66#71 +*\u00A86(/#\$#222562#6(\u0094F672\u00C6\$5'30#" +
        "(,6#0)2,6 /77#6,,)02/3*#2 \u00C0)!)6 6\u00D77\"+ (+453\"77\",5.6 5+7, 6  /5   27 2))\u00C7 -!#4\u009F" +
        "535ulx/5552*74(1i\$(*&#500+ 660(#7 \u00D0\u015D56-7# 2\$7\u00BF56  5\$3'6(6 +*5255  \$+7-#)\"7+(S3l\u0092" +
        "526\u00F0 /(466 #(\" 5l( (l70*1 65 (# \u0100\$40 6+\" \" 6-\"6(#036+\" )&5#(\u010B\"064!(\$'\$\u00B5/" +
        " 7+,,+557/&\u00EC5\$\$6562\$\"(&&7\"&52,\u00EF7|(&4,0+ ,\u019A(60\u0177F \"\$5\$S7/x\u0110 \u009E|35Z" +
        "35l6'+K +\u00C05\u010C\$2(6#\"1\u00804\"&67&'\"'\$(\$3\"0++4\u010D+2t\"37\$K)/5y\u0094x&k(7     &5(#\"" +
        "6 )\"1#66\u009D*6|6)\u009D,S'K7(72'\"\$l#2#1l,/\u010E/&70+\u0101,4\u01ED\$g5\u01BB2'(!&\u0095(/\u01DF" +
        "25,*\u009F !22+\u0102(# +-\u00AD\$),&%#((65/\"#66525\u00DC+6\$(2+/(\u00A9K1\u0113\u010B!6\u010B(656 \"" +
        " (3+ 7 \u01B276w6) \"36 6\u00EB2\"'\",;%2 76)6w34576\u00E92(-6)26\"((6 6\u01F55%\$\u00C7766)#\$ *66)%" +
        "-#05\",2,+002&\"\"\"64(&0,&!!&0036\"/\" 5  6#+6+ -54 +/l77#F%6+&\u0117(#2\"\"7 (+ (7'#+,,,!77(&x3 \$4" +
        "\u00A8)#+\"+)-+\u00C2 '-6*44\u00CCK*+'\u009D(*(*73+\"(/64\u010F2*z!),\u009F*B,0\u00B3 *(\$5-2)\u0094S" +
        "2#-,2#5* \$Sx\u010D4&#\u00942 10#S\",l#*)+ ,,/+7)\u00BC64\u00A0+77(2#54\$\u00B9/(,Z,l)- 61+6 07 \"'7)" +
        "6\u0211\"\"36067),2202,'6''3&563h,+#+2+\u01D1#S+)7)775\u00C7  +  +5'%,7x67'4+6\u00FD7((7#+#745\u0162," +
        "'2  2\u017F#*3-++6560/#5')+,6#+6#036  \$\$( 5 0 6)6&\u00A6(70  )/'\u01152w0#{\"\u022F4#7\"4\"\$\"\u010F" +
        " ')\u009E#66+)&-%%5\u00AD63S\",\u010D\u011706\" 5((6)*%),#\"475x#\u01DE#,*6l (\$1365-2&/+!\u00B86\u00B8" +
        ",6+6\"\$+2)+))3/'54#\$(6+,(2)64 /++6#2(\u010C3,(\u00EB\"/(&\u019C\"+#/#,207\u00AD\"!,5*3345 3+#--+\$)" +
        "0\"2730-6)'|),\$\u00C56\u01127,+\u009D\u011537632504\u0117l\u00A44 7x\u009D2#&333\u00C7(/(77\"+-5/\$(" +
        "3\u00B730)#\")\u00CC+)0+'005)\u010D\u00C5;(\u0153767)#6) 6 )3-6 # ) 7                                " +
        "                                      6|\u01ED025(\u00E47225\u00C0K6,'##7\u014D/220\"6#\$2\u0162\$0+\$" +
        "6+6\"2')'607))&\u009F\u0147#+77\"4\$47q+)\u00FA&\u00C6F@4))\u0111\u00E666747(&+\u020E//0('#g\u00C666\u00DF" +
        ")0655')5(2\$2)-))2,(,+7156&2-'26#7\u00A4\u00DE''++6)+,\u011772\u010D%#\u0223\u00D266(04)22\u00A9'6\u010B" +
        "6\u0102\u008B\u00C034\u009F\$)\u00AC6#\u01CA,65)3+5)60\"65\u0141+\$\u00E411))4621+\$7\u0195/\u014D\"7" +
        ")\u009E\"1#&2+\"#7237&55\u0230(310\u009D\u01B8\"\$+\u010C##2#6,,1&bF\u00C07S44\u00BD)\u009B&#\u00D84K" +
        "\$\u009470/\u0115\$\$46\u00DC\u009F6\u008F!4\u01E0&&5\u00C6/\$&3\u01176((6#624##602#+#554775-Q'--5S5+" +
        "72\u0102g\"2\u01A5/62\u02267x\u009D,\u01817\"0\"\u010D#\u00DB\$7766\u012E\u0141(\"\u01102\u0221{K7\u01A3" +
        "-3-506402*\"/(\u009D+5}6Q&,)\u0082\")(('0\u0243\u00C5#62523*7\u009D+6#\u0245'+62,756!+,%+x##5'761_52\$" +
        "+3)7#0*#--)4()hl7#0+)w323\"-5S365#%0\u010B'*4\u016D6&+7\u0100+6(\u00C6\u01002/2&5+\u0092/\"6\u00FD5\u010B" +
        "56\$_#76+++2)5&()5/)\"\u015F&K6F\"65S^l7\")6l+4+#7''S322,\$x)l5/h(\u00C60\u0182'\u00BC)2#\u00C0&7*)7)" +
        "0-+74+2+(\$7\"4)-#)76720-6#061\u0164#\u01A7\u00C046\"134/6&x4%\u00C5#2*\$7#3\"\u0224(67\u015E66)57#&K" +
        "7l,)655g\$l7.33\"7+)\u0189&#\u009F2\u010D25&61+-6\u009E,S\$3()52\u00C0\u00C0##2\"5\u01076326&\"\u00C5" +
        "\$73'\"#-g+#kl7S2!l67\u01022#)#2(706\$\u010B+\"S)++,05\u00DA6\$752\u00C0)3lS5+#'+0\u013El4)\"(#\u00C3" +
        "\u013D\u00C6)g&5\u00C6%)\u00C7\u0164+\"'525),6)\u00C00/&7!26)-\u025D6\$)5*)#w)27#\"!13Z#+\u0127)0,4-0" +
        "-#+++*\"{\"7726(0e+34+##\u0106\u0109\u00B8++7-h6+-\u00864676\u00D1#755'\$*\$\u00E43#%v\u0082\u00C7h26" +
        "62 \$0\$&,2\$ )+(14-0+\u021F\"'+5(+\u009D3'5\"\u00C0)x7&6)2)6)#),1\u00E96,'\u0174#)l7'),7,'5,,165*)-6" +
        ",2',7)6,*+&5,\"\$3'&R(#)(*6+/&,\$\u010B+0)_2,30'72)+0\$+\$')6#!+#,7#2\$)++-5\$)&&&&)#7&,7&&/&*(*'\$/\u00CC" +
        "5*3D#)\$((7\$\$\$11#K06)*0##4)\$5\u009D4)+6++7'#&#0\$2,+7/\"))S+2\"#)''\$#**\$%72\u0100ge*)'#'*\$)3)r" +
        "+76#\u0210*+\u00C02#')#+3&/#0#7'4\")Z\u00D7\u010C|))S6)(2)\$\u01824'\u00942)'|>e##\u0111\u00BC+/2S*)'" +
        "0)(77+)/+'_'))3())6)7#\u00C7,+7+60\"')4,+)*\u00DC7\$l-)0267,+)++)*645,*)+\u00AD2*)'\",#+66,*5)0+\",a+" +
        "*,\$\u00E55*\u00E465\u01132+Q+/2)0\u00C0_056)&)x)\$5,55++#\u01060\"\u017E6)\"'\u00E4\u00C645''\u0130#" +
        "65/)\u0110'/%/&'\$)'S(\"-g&)67)*)\u00D0(50&'0l*&(6'*(\$\u009F++2\$))5/06\u01196\"\u00EB\u024A250-2724" +
        "(S254(54Sl75\$,-\$_\"2S\u009FSN\u0111\u00CC+6+5'45))4F07,642)0\u00C8#\u01A55\u010B)\u01DE4)%052\u0225" +
        "\u0113%3+7<+6662|7/60F\u010D#,3a2(37_&6#656+)2\$#))!'62+#\u00D7+666\u00E2\u009D\u00D4\u00FC00%+\$5\u00BA" +
        "2#g##!\$6#)62&2'\"&1\u00D92\u02530w\"5\u00C5/2)4\u010B\$1#*\u009F\u01A5+'x*7)\u010DS\$\u009F\u021A\u0142" +
        "260/\u0115\u00B8g\u0110#+\$)+(,\u0115,e'\u010B6\u00C7\u009D)\$Z3,(+357\u00B86+F,.(\u0115)#+3\u00C6&\"" +
        "00\u01A7S\u0094Sx\u0094~(\u0110\u00AD20(\u0102\u010C\u0113\u010Eb-\u01525#v4(\u00B1\u0113'\u0115)+:'\u00E4" +
        "\u01180#4\u009E&)(#2-3,6x.+\"6'404%2)/\u00E4-,+16\$727hj6\u0102/\u0235'l\u00E4\u01A1\u0122-\u01A7/k\u00B8" +
        "2\u00B8,\u0181\u00CC)\u01A173\$//\"&6(7(\u00B6\u009E6&\$74\u00CA'\u01A37\"+\$-325(0%%'7\u0185\u0115+7" +
        ",,7\u00C6\u01DE\u0108\u01A8\u010B75\u019B(*\u019F3\u00C5\u011A55\u01A06+6!/23(\u00B8\u0141\u01A0\u01CF" +
        "76\$\u00A35!\u019D\u00CC6\$5\"(-(5*\$&)-,6(6+/,'\u0083\u014A\u010B\u00D4-7(*55+S77+\",+2&K((\u01B0\u010C" +
        "'66''\"(2l6!\u00C03\u01844+53\u010D0#\u01FC76\$0\$0,,(\$0++27(+2)74K\u00C6\u01A10242#\u0089+0@\u00C76" +
        "t63+\u00FD53(!\u0182-*772\u017E\u018B+:j\u0100\u024244\u0144\u025E'003\$\$\u0113lV#l(\u008B07&+5/2+7\u0101" +
        "'/727-'+6\$\u00C0#\"\$+5)3Z3!)\$*64-267+\u00C5(\u01BB\u01797#4\u00E96676\u00F2,(6#\u00BA(\u01D1j\u01C4" +
        ")\u00ADl4g5(|\u00EE5+62`\u00C06\u025DS\u015A+674,6/\"*56)+*\u0102(36\u00B8\u012B5e262\u009D5\u012C2,#" +
        "(\u01C3x2h12\u00A8)S4\u01002'7)+442#(2 (20-(72-'-\$!03\"!!+\u010D7,223|\u0102)#q\u009F3g#)!5/+'\u009E" +
        "\u025C(('\u015857#\u010B/\$,,(+\$'36\"62),5(,1l/\u0154j)7l*752(\u0092+05\u0160\u013Ex\u00AF5\u00897K#" +
        "(l#\u014C\$6\u00CF#)\u010C50(+5\$(6/)05-+266\$|\u0128|)-\u00B950%7F2\u014D6!6\u00C5\u01AC6\u0094\$'\u00B8" +
        "\$\$52//67-\$\u00AE3\u00B71(\u00B86\$\u00F9l\u00C7)(3#5,#\u0111+(S6\u00EE6\u00B9,(7++(-5,6+++,\$#//5\u00B8" +
        ",5\$*6#61\$+35)#((7\u00EFEl+##+6+-l257)\u02477+--++\u00A8(602-\u01DB(56-\u00FE3x*6)/6#((\u0113%'*#334" +
        "+'01+''3'36'6(/6(6+\u00C060'#460*/666333h+(6\u00B9++\u013E633\u00CE/+*67\u0223\u00AA\u02224'/6\$\u00FD" +
        "0\u02317\$052*#\u0101-,))77\"560\u0092\u00C6\u00CDx&&\"3*\u00B27*\"\u00C5r))\u00D2\$))3+\"4&7/S3*\u00C0" +
        "37\u0110'/|'-3\u00B2)6&F+0,*'5\"Z#+7&(7l++++!\$/,5x)(+##+'6(\$7&'6\$\$'#6\u00D2&'6\$5\u00D2\u00C22\u0087" +
        "\"\$(6*\$*\$*(-!55\u00B9\"\u0094\"66<5#0'GKn\u00C06)\u00C5\$,+)#\u010C\u00C07++S#(\"25')6t\$6\"\u0191" +
        "\u00C02S\u01024)+l*g\$\$3F#h\"\u00D17\$\u010C76/)\u016B02)R**3*-)\$\u00C0'6\u008BS'3+&60*!2\$-3#6(\"+" +
        "\u00A8\$7(S\"6\u017C\u00E4)\u00E9\u011C'`76(%66K_,\u013D|3\u00FA&7))('#\u01D3/+\u00C5(+\$45\u00C5)*6x" +
        "#6*222,x4#x307+\u00FB\"27\$3475'7!'6*\u0164645+(+g/\"#++.\u00C0,,\u011122+#,\")#\u010C|7*6\$#75'0\u00C0" +
        "\$2,#63700+470l5g2K/\u00CFx&0,\$\$l\$2p\$3\$\u00E4&(3\$67)!0)B*\u0110/\"\$)6\u0113\u0134\u017B+)(\$10" +
        "5\u00DC5(!6\$(1)+*+634(++1++1#56+\"421277\u00FD6,*77((*6(5*2,7267\u00AA#)\u0094\"7\"20+&#\u00C555-5*\u0115" +
        "4\u0113\u00E42\$\$604''0,,6(#x733&\u009E\u01136M'\u010CV33\u0110\"6*)\$(*6\u01A7\"%+(x6/)\u0228-&\u01F3" +
        "\u0090\$0*l(\u01DE*\u0162K6)37\"5(&\$5O\u0182E24\u01C9F4!)+\$6)2&(\"+Z\u00E9-x-0\u009E\u0150)-#(5&)\u010C" +
        "1\u00944(7&,\$1\u009B)6\u00AC47/\$)\u00AD7\u00C6\u00C56,\u0094#\u00C67\u00B06-5&\"63&\u0113&46'\$\"-l" +
        "77\$-635\u0117%,,0\"20%(5&7,\",&+7721\u020C6,\u01E7'2\u010B4l)2/1\u0110\$)2)'5()\u00C5'\u00D4))6-7\u00C7" +
        "\u00C65('\u00C5\u00AD6,+/\u008B66#\u010B6\u01E1\u00A80724(\u00A8+61)+/z2+g5,&2\u00E44*a#3-5-6-S\u00EA" +
        ")27(\$7%/35\u00E4\u011364E66264+5)\u01DD\u010D057\$#+:%%+,#S)/#+4\"520(&4\u02053&&\u00C03-\u00C0)()\u00C0" +
        "\$5(\u00A8+\"6lc&)+@+\u00AD66)+\$5((6#23{6\u00E4-f)3564x1,\$\u00E90(40#,&)\$\u01157,(,\u011C65)4,670\u023F" +
        "5\u0175+,12/44#(\u012C)\"5\u0110'l6S56\u010C11#+,/!,'`\u00FD)276\u00C21,\u0115)2-53S+-\"!//6S\u00D2+5" +
        "\u00E667*6-+\$\u00AB6)\u0111#l\u00B0-+,7((!-0,/'4\u00C037,55,++('\$756(5*\u010756#),&&\u01E66()&2\"g2" +
        "0+\u01CB5-\$6#!--,x#)#6/1\u00E9\u00826360-66,\",25-\$+6+62+)\u00E9+\u010062\u00B8\u00B3,++6\u0162+7))" +
        ")*6\u00E77#2&\"755,23\")'-56\u00EF(+\u010B-,#02\u015367\u00EE&7+\u01BE\"1---6+,\u00EF1\u0110'\u014106" +
        "2!\u00C32\u00D016(6)7\u00927\$(7'7\$\u01136\"22#2\u00CC526(6325'#0'577U(65\u009E2#\u0107(,**\"\u0210#" +
        "7)6)6,*0(7)&-\"(,0(62-)-76\"*-0,#\u00BC'0(4+\u010D-\u0212\u00F645*(+)\u00B866\"0a\"&#\$25\$+\$2/K7\u0130" +
        "\u010C*)\u01DF\u021B\u011375\u010D\$\$5)2%%%')2#2g,'#+55*\u011346\u00D5+6\u00E0444'6\$*7)'''\u0111''2" +
        "6#*-)\u0113\u00EA)/#5\"))3\u00C03)4\$2\u010BS)575-754,%+/3&2\u010F3++5#+)5))+)\u010D5#\u0111\u01DA245" +
        "+\u008B((\u00AE\u00AA014<26#2016\"6!\u01154)\u007F)0\\40#00#)0'5!+\$\"776)'*'5/+3\u01E06&5)+3/!\$*60(" +
        "/,!3+#/\$(*5,356\"(O+%\u00935\$+%%'0x6,2!4(\u010D,+)34605+6)7+)%%6\u0115\"00&&-+651\$2#3')'56\u0113+*" +
        "++50*##3+()+\u0101,**l'#66'+++\u00C6\$7'6\$(74\u0115S62)\"7\u00BB\$7'677(64{&06367w76%7\u00D3+6,()\u00C5" +
        "(3g,\u01C1\u00E467+\u00BB\u00EE647#21*060+3\$54#x\u0107)>+42\u00CC74\"\$#06-#)2127)\u00BC*#\$#\$3+77l" +
        "!#00l7\$\$56/+5l\"0)6+7+++\u00C0\$\$7\u00C0\u009E6)6)\u010D65-666-\u0115)\u00C5*5\$!74\u00EE1\$+)\u0115" +
        "661)+\u01A7++\$\u01CE5)#6\u00D064-0##+\$\$-666\u00CA6g\u00AD#7(\u00C5\u021D##+'70\u00C0')4//_00\u00C0" +
        "6\u010C\"77\u00B86)5)\"\u0116)\u00947X2\"\$2&-2&/75(\$47\u00EB\$\u00E4/\u00E03Ka77\"7\$,66/0)12\$7207" +
        "205\"\$'#/7\u00C24\$4#\u00FD\u00C07',\$(75,44K\u00D2\"/\$'/(3,)3(7\u00FD,,\"'l,\"7)#&7\u00B7&l\"\"7,0" +
        "#&,\"Z,#)5+;S-\u00C0\"\"\u00ED\u009D\u00E6(66\u00C0\u0120/37'\u00FB,0S0x#'6\u013A)5+47*6\$/\$6,'#&3/&" +
        "567375|753'*,36\"\"#+S)\u011C\u010D*6322066,2\u01625&)%\u01A0+\u010B++27\"0))\u00C0!*+6+\$+6##32\u0100" +
        ",#4&\u009F**\"\u00FD5++++)6536+36563\$/)(-('-*S60\u00D5&\u00C0\u00EE67\"66%2\u0109%22'\$6(&(43#)\"2x7" +
        "\u00C73&\$,5(#-)76,\u00C0421\u0115)\$7)0/\$\u00C0\"\"2\$)\u009F'&\"*)\"5x'\u00C0(\u00DF'4)))'\$++\u0109" +
        "\u00B57((666666\u010B5436\u00FA6##\"\$\"S6\u00E076#\\Y\u0110)\u0094764#\"\$+&47#4)\$55(6(+(4l75)3#7+#" +
        "\$/\u010B\$)#7l7+65))\u0109\u00C5S(&54S6457/h\$7l\"\$l#)(\u00C6(,+\u0106\"\u00C055\"6+1\$x')11e\u00CF" +
        "x3\u009D\u0101\u0109#74,\u00DC&63\u00E447#4)45)64#\u010D*\u010C\"\u0102\u00FD#Z3\u01820-(5-\u00D055\u00AD" +
        "(\u00CC&\u008E324.##)6\"#,/776\u00BB\"--7S\"\u0111\u009D(\u0233\$+2k\${6x6&)/56-'K5\"605#\"(5jS25367\$" +
        ",762\u00E9\u010B\u010D5+\u009D*,75+\u0110\u009D\$(*\u00A8(\u0100\u00C522\u00AD06((#\u00C065'+(75#2-(%" +
        "*x3'\u0102506/*\$(%536*#-6,\u01106\u00C6S*06\u010C\u0110\u00BB57+3(\u00AC03\"60(5\",6(2\u0100#*446++(" +
        "#66+--%0504,#&\"\$##,+)''5\$0*3+5*'-0\u013C\$'7\u00BC4\u0113))+\u0115((\u00C0x33-4#(\u00AF5h54(\u018A" +
        "\u01037)6\$)##\$\"0#\$'x-65\u0148x(#015#\"\u00AF7\u01090\u014D\"54'046\u0110\"5(\u0092\"6,6\u01C1\u023E" +
        "6%\u013F,\u00CC*6!06-\u009D\u011372+&6**0\u009D622062()')#-5*76#52\"\u01EC65(66+g36#j\u01003(6\u010D," +
        "'2#/,,'#####,\u00EB73!2,'0l)(*+.2\u00C06+243,\u0100\u010D60*Z7+)006/+g6#73#00/\"\"0)7#+/*(0\u0141\u0115" +
        "655Z3\$\$x2)#\u00C5&&++l(05,)+4#)#532/6#5-\$3)2p)4\u00B8\$00)g*56F,6+2)(,+!+6*5#66\$,@,\$\u00C0,\u00DC" +
        ",#7--66\"7**6&+,#(#,+(51#))\u009D26+-,3)''\u00AB'64)\u0100215#4)'_\u0102\u00B907\$e)\$7))7\u00D3)*'\$" +
        "7e'))6)6+(75\u024E#\$0(((%\u010C3,(+\u00942S\$)\u00C2662(&622l70a/S\$1&1737\u0171\u009F\u0113\u00D0\u009D" +
        "2l\u0163\u01A5*4\$071*,26\u0157\u012F#\u0107\u015C0S)6\u0094!\u0116/\u0102\u01EE%\$\u00BAl\u00C06\u00D0" +
        "\"g)4(6)S0w764\u00E9627/x\$\u009F\u021B/&/\"!73*+06(\"\"7S-S/h\u0110776S\u00B3-\u010C\u0179#/),,g\u01E6" +
        "#0,#\u00C6\u013077q#,t+K/SS+-'0)\u0111K3h\u0127\u009E7\u0130\u0212\"\"\$-)*+)06+-\"70(*2)7-\u009E'0\u00FD" +
        "l7*\u014B5g(/\u0117\u00C2-\"#'|\u009F\$\$7\u01DD!-7\u009D)*\$\u00E9x+7+w4)+)7+6435-\$)7)(56!3-l\"'37#" +
        "22*,-)/)63\u010C\u01F64\u00C5#&\u00BB4)\u00C03\u00AD^\u015F7\u0094+\"\u01A1\u0210-\u00C6726\u00C7\"\u00A8" +
        "\u0139767+2\u021A+2\")(\$4\u01130\u00942+,,K\")\$\u00DB!\u00C067\u0155Z\u014D3\"--\u0105\u0162\u01120" +
        "\u00B8#\$2+0'|2t\u0094\$7(\u00C6300S2\u00FF+66\u00BB7+'\u00EE\u00B8\u010C|+#\u010F#\"\u02343\u00CC36l" +
        "l)30*\"\u00F6)+',)7\$,#2'/k-)\u00D3\u00C6\u00C0\u01616*-\u01BE1d4gl5,\u022E-l2\u010D7\u00C65\u00B9\"6" +
        "(67F\u00C56\u00F2\u00CC\u0230)x++#\u00BB))3\u01BD('Z\u010B\u017D\u00C5\u00B9\u00C06#7\"64\"+40#'+7')'" +
        "01\u00B3lk722\"#\u00E9S#2\u00806\$7-\u00DC\u00BC\u010B0\u01AA522),\u008BZ-2x\u00C73SUx+3\u00AD\u00EF4" +
        "\u01CB04e\u01097%5-0#7'4%2*22\"6\"232g2\u01D7##'\"\u00942\u0182e27+##'+++7\u00A9^7#3(l\u00EF,\u00C0'6" +
        "7B7c,#\u00CB#\u00E2\u0111),,\u01FBg\u00CCS)6'0+)6\u009E/S+\$5\u0103'765##2\u0206(\u010B\u00B3'\$7-+l\u00B7" +
        "7\u00BB)#\u0180\u025D#K#)\u0199+\"\u00920\"l7\u021E\u010ES0\$-(5+)#3(\u01DF\u0114\u00BC'6+-+27\u0117/" +
        "#\u00C5\$2#0\u00C0\u01BE)0*w5\u0263S\"!)6%,2,)37Z5+#)6\u00E966-\u00B7)\"-'7\u00C7\u01AB,)5+3\"2+\u00DD" +
        "1\u0111/6+g\u00D97-5\u00BF\u00B3++(6++0070+\u00C56,\u0206g4#5\u0208+),#l+\u00EB7\u00D9\$4x))++-\u0102" +
        "''\u00C05//2*6'6'\u009D\"&7/\$*\u0092\u00E4'(#5,g'p5),#)2\$6g!\"5)\$+-\"l\$u6'2*\u00C6\u0182\u00C035\$" +
        "\u00C0)65,2G0!7\$7&2+\u00F056)2)7\u0181+\"\u00DA5545007\")7+&\"\"+6+4|2+)(\u0166)67)\u00A83\$)l\u0117" +
        "&070l7\$l2577\u00C0+7#\$7\u0198_(\u0113\u010C/\u00DD,S07\u00E4+/S+&&5))-\u016276+2)6607&/&7*264)\u00C5" +
        ")(1\$)7753\u00C55\u00A9+\u009D(\u0110\$5#2*624,,\u0111#46\"F7\")\u00AD2(&(\u009E#5,(&0(6j567&\u01102)" +
        "67K565/7+#(\u00E47,\"5(77\u0109\u010B6&{,\u00EF#4\$!\"!7+5*3l2((5*\u00D1))7)23(6'52566((()45*75\u014F" +
        "2\"7744((#43777/\u00C652-50u)'767F4++#457&5'6\u00C56(5,*66w420,-\u00BC\u00ED!6#62/,)*\u009D4#0(6+!))," +
        "75(,,#-7(S(5#+53)/+356)(\$66+335&27(6\u010B52)\")6!612,56/+#*\$+6(+51+-++23676606'6(~2##2,#l3l\u014D5" +
        "\u00AD60&/\u0107\u0113\u0105/KK+\u0090\u015C3-\u011027++\u00C7,\u01C743l\u00C00674h++,+,\$4,\"\u0117\u010D" +
        "\u010D,\u010D1\"/\$\$\u0172+\u00A4\u00C4)\"5\u0164#21|00\u0113'4#25&47+5##\u0086\u022A\u0111\$7,#6''x" +
        "\u025F02'3) 4+),6,,\u021E6\u00ADK\")\$'6\u0100##Sw(5c\u010D#2\"2)\u0113)'),SS\u00D0!K&6&-\u010B(6|5l6" +
        "+\u00E9%,7\u00A0\u0102#6(227\$++)#20&75\u0116\u009F/\u015F57'5))\u01A5K,\u0115662-\"x\u0171+\"c+(S&7\$" +
        "#6&\",\u00A4011,227\u00C5\u01172\u009E37\u00E9\u01576Z|7l\$'S)!&)\u0130\u0183*-7\"+l\u010D27S\u0097'7" +
        "6_)27+,+1\u01627\"77+\$+2++62\u010B\u00FD070+657+g\u01AB*\"1\u01DE\"1\"7#6#\u01157\u00C77%%6&'*+7+66\u019B" +
        "'\u01172\u019C7+&)('\u0111'\u00A8*1!!\u00C0\u01107*70x22(\u00C5)5\$7\u00E67 6)+7617\$00(')7522l\"\u0110" +
        "*)S\"&13)5++&04)\u009D\u010C+036,\"+/7\$\u00C5h7F\u009E\"\$(771\"'\u00C5(4-)3)5+\u00AD\u010B#&2,\u0113" +
        "+\u01F4\$06\u010B)35\u00AD72#&4)+\"5&37++)\u0218++0\"+g0,0\u00C0#\u011D#|&S\"S(7#7)+/66')\$3'4x#&#l06" +
        "66\u00C062\u0103Za\u01007\u0097*+\u010D77'&&g0-4'+(\u00C06\u0141))#l+\u00CC7l\u008B)S)77\"/\$##,()'5\$" +
        "0\$'\$+70,&'/2632Z6&\u00B74\$)7\u00AC6\u010D566#\u00C7#6(l4/\"6\u019B516+6\"(*)*2&53\u00C7),,#\u00AD(" +
        "++3-66,g\u0115567/{)\u00C06l\u013D6l4,\u017B))\u00BB061(++05/2\u009E3+\u01136#+)2#\"5)6\u0157K-,+1'))" +
        "367S\u010B67,2)6)(\u009D&70,265127#43+1307223*5'0\u009F\u009F2/3'#|-)2\u009E\u0141e#\u01F5(,5''7306\$" +
        ")*7\$+\"72z\u009E+(60'''\u020F(\u02122l/k()(('\u020B6'*''l\u00E4\u00C0|l\u00DD+#la502&3'72+++,+_2*#h\"" +
        "r\u00BC77\u00F17\u00D4,\u00DF\u01153/7250\u009D)#)6056\u008C5\u011707#/2(*77625#\$&',40\u00E9++\u00C0" +
        "51&\u00C7\u016F+62x#h,)Z\u0210))\u0180()-5x-17\u0164l&0\u0107(0'++'#5272)\$)+)#63\u00FD)\u0109+-65(27" +
        "5#')760/{+&(\$)2\"S##7\u019B)\u00C5)\u00C0\u021B)0\$6\"22++g,6\u013E3,03\"()0\u00DF-6'\u00BB6\u00E906" +
        "0,|'#)*/\$#',67\u009D0++2++#a,(.+7'\$6\u00DD&72+)#07+6+7\u0147)%2\u009D-6++)5')\u0100+6\u00C2666\u0106" +
        "-\"l+)22\u0114#\u00A40#)6++l+x)6++7++\"-6+'0l(566\u01CB.5#\u015C0>6)\u00D15\u00B86*+*25\u011B00j#\u0101" +
        "\u00D1\u016E*\u00CC#\u010B'65\u01156056'\u01133\u00C0.(\u010F(5/\u01155#\u010B2\u00AE#6%6#(77#\"40\"\"" +
        "4)0l2#2722+76+\u009Fm%2)7\u00E4,6\u00E4#632\$2552#)/002\u00C06+6\u0131\u010D6#\u00AF\$)\$764#\"3)26\$" +
        ")\u00AD#6\$\$)+\"\$))2.\$76\u01EB6\u010B0\u00CC\u01155\$(('0)6\u01EA(\$65\u00E4'\u00E4,,!)\$6S\"\"//\"" +
        "#,(#/+)72,35%15),(0t23/)\$\u015C\"22,2\u010C23\u016C)32\u023A,\u00D431#/+727,+0\$\u01E4|,),0/-0\$5#&\u0107" +
        "-\"&2\$0\u010065(6!6*06\$\$+066622l\"6\$2+\u00DF\u00FE&\u01A5x\u021E0S7,#(\$(#\u0259#\u00B7&x53,\u0101" +
        "02d\u009D0\u01134521')h43)3\u00C77\u0101,\u01A2\u00B7))\u0104)(3\u0095)\"\"\u0145\u00E47\u00945\u01DE" +
        "(\u00EE\u01B3\u00945506466\u00AD\u0113\u015F#\$6\u0230((0,-,x,\u0196#/\"27//77'+,7&.++#&4(,\u00AF2\u00C7" +
        "7x\u013E3(\u00DE,\u0161&&63l7'\u023062*\u00C6\u009E\u0115\u0101{)70S(Q\"7\u00D3+F'(\u009D&\u00D42(/l&" +
        "l2\u00EF#+/\"\u01F42-)\u010B72+\u00C5\u00C5&/,36)6'\u01355++\$+5/7)6\u00FB)6((7#64(_\u0115+2)%5&\u0160" +
        "\u02127645*7\u00AC++\u011Cx3,+6,0)/45\u0115(\u01152*\u0181\u00C0)%71/557\u009D(,\u009D\u00A8\u00B3\u0136" +
        "\u00B70+6))0))727#&()+#((-5)+047\u0152\u009D\",7{5\"\$((6/\u0168+\u009D+(+'&4+#\u008E663+(\u00C0\u00C5" +
        "#),\u00EA#/\u00C52\u0163*/-(\u00C05w\u00E93k5\u01843%2\u0110\u01E6\u015E3+22+23\$+++446(\$(1)72\u0094" +
        "'5\u010926#7\u010C\u00CA\u0113(3\$(4)2+(37\$\u00C0\$)7550\u009D'\u00B9+h|/(#g0&7+\u0157)3g+7/&23/\u0181" +
        "\u0115-6+\u009D,_+\u00A06l0(6\u00EA2Ml(6+\u019B06\u02333,76\"\"6\u00E41&06*)262+7\$6),/566221(7)-6\u00FF" +
        "3\u00C0\$&7\"\u00F2\u00A8x4\u01131\u0260#4(\u009D6\u009D5,\u01BB,66\u00DD\u00B3,2,7\u009D-'#%S,+00,3#" +
        "4#\u0111,\u00AF,5)/\u01E0)(&566\u02622\u00C5l\u00AD36\"\u00AF2326)x\u00EB\u00C0)06\u021A+(6-S60\u00BA" +
        "+#5#-742,6+*&)5+4\u00A9\"6/)\u00AA3626\u01F26,7#*3x4+\u009F2,726+,\u0130\u00E4+74\u010040722!\u010226" +
        "j61(22\u00E9321!/j#/4#,'\u0182\u00B8(\u01007\$7\u0115&)\u009D'7332\")('6'+762,6#+++6\"365(\"\"l)*\u01F4" +
        "2\$,!+\u00B8(6#&6',06S)6#05)+++)'g+7'0+6#)##\u0111.365,3,\u0109\u00B6++4h'\u013056,\u0212,*\u00DA/277" +
        "l5(\u00AD\u014D6g\u00DC#36)#6(52'657*5+74/\u01E3\u00C0S\u0184()4/)\u0130\u00A8\u010B00\u0115+5\u010B\$" +
        "\u009D,12+7l674\u0109\u0241#\u0115\u00D162\u011172(\"2##2\u0164/+422#)Z(\u0124+#\u009D#(2\u010D/(6+(&" +
        "((5)256++-6+\u00E2l\$(\u024056+g5+\u01FFl6\u00AD#+\$6F\u00B87\u0141,\$\u017B62/)30)7\u00DC-\u00A8)\u0094" +
        "2\u00C22\u00B7#32x27+\"+\u00DF\u02192,-66,)0S\u00C0(\u018Fl4\u022E66\$#(0\u00BC\u00C5>/\u010D4\"'06\"" +
        "*&#,\u00C74\u00D67++66+27\u00C6\$4(5/\u0159\"#,+)S2#'\"(5+0(61+)562y5\u00B757\u00D9+\$&(+2+6\u00DF)+\u00C7" +
        "6&+6)6#\$\u00E9)+&26\u00C7&\u00EA++#060&\u009D)6(6\u01F7\u015E+6+,2\u01C8\$)#6(\"7++-4+\u010056'66(M," +
        "'\$(55((+77)\u01175#777#607K-\u009E4/6+#/*/\u0110\u00D2\u0094,#+\u01B2\u00ADx'57''40\"\$577*7&\"55+l5" +
        ")3S)\u009433\"63277\$42+#(7(2+3\u00C65(0+(65556\u01137624x,7*(+5l\u00B87/6<'#(*)&26( 3)1+5&)j/\u00C37" +
        "34(/+\u010C5&#1(52(\u00B8*\u00C6556(7\u00C0\$5\u00C046+(6(,)3\"&+*(\u00C35#\u00E4(_&D\u00C0#&/01\u0115" +
        "05l666)6'1/\u01B9'2]25-'4#(*7\$5\u01D1&+55())\"67\u00B73\u00FD5\u00ED-#6))6,24\u0117)01,(57\u0113&026" +
        "+7)7'\"(\u0113-2+\"'33#2\u008B\"\u00B955456(\u01C3565-2065,\u00D7\u01E641'60+\u0115\"\"l?)26+\u0130g6" +
        "3)7\u0113(67!4+\u00B4.1),*2#7)55565#(Z6\u0213165\u0100)\$x2\u01E0K\u010B+6+32&&6\u0100+,x6),))kF65+\u00DA" +
        "\u010D#2507+(\u00D35+4657(\$-+1W(5)#x\u010C(+\"1+,\"1++!5\u00AD\u01F8+++)+\u011555+6#16+#)h')#,3+7++#" +
        "677/7/#6!4())&6\"\$6\u011572\u018E\u00F5#*7\$060/\"/#)/\$#67Q\$S\"#66\$6#--/\u00BC,3,+1,&,,'46\u01B4\"" +
        "2)'k'2,|000735,*04'5++/\u00C6'\u0111\"0#\$))\u00C0&/*)*#5\"+),24\u00D4\$30)5+\$+//\"\u010B#4\u00CD#00" +
        "\"&0)#7\u0121\u009D7',6\u009F\u0130(66\u008E76\u00BB-x*7,\"/)\u015D\u010B/5\u0110(+&/-66'5)\$\"*5-\u00B8" +
        "\u01C4\u00C5\u00AF7\u00BB52\u010912x+\$52*7626\"\u00E7\u010B)+56\u01E2\u00AE\"762+(5)/&5S0-\"6+\u01F5" +
        ")0*6\u00A07,#\u024677\u0113,##2+\$+(7'4&4657,4g53\u01DF-\u01A1)(S66,#62S,(62,6962(067#\u009D,)),7#!!(" +
        "#K)\"\"(/+\u0115\u0111\u0115+252\$)\u01AE)\$(\u00AD5\u01BE3\u00E55-\"(\u00E9,+\u00EE\u009E2+3\u010B,1" +
        "\u00B3-\u008150)5\u00E6\u0116\u00D9+62\u01130'\$+\")(\$#')6066,(4&\u00E4)6K\"\"6)\u00E6)4\u0250,\$/4(" +
        "50+\$6;/\$&55\"#')2/#*122\u0110\u00C7),2+\"77)&+)))+'57'6565+270\u00D25\"%565('#5\"/+\$6(4%hx44#),\"3" +
        "5#(\u0100605++5&+65++)+22&4\$S7g,6)#+,\u010A\u01133#7S#(0%*#0\"4+#'6\$\"+0//070&)0\$)(6,,#535\$1,)41(" +
        "5#)7\u0140+05\$\u00B96-(66(12+2126447)6,3+3++'\u0111e#)76!\"\u0109#*7#)6##0+)205!+,7666\u0114+4\$5750" +
        "\$++\u00C02('/)&)+)()!\"#07\$)'3\u00AD(223#6)\u0091S247+5\u00C7514'+\$+71+7\u01003'2++++7+6\u00E55\u00FF" +
        "'\$7\u00B9'\"\$S\u009D\u00BA/\"1+4 \u00B8\u00FF\"/&\$4.  4(+6/#\")## \$,S7/\"7\u00C7 #605+\$.,l\"+74/" +
        "\$466\u009D\$233(#202##122262+\u0094{T&-36)2\u0117\$&-\u023E/\u0182(3\u01027,\"\"+5+#,\u0110&&'3))04K" +
        "3,\$/)4n-+\"7j,,_3'\"\$\u0162+0+\u00946\u0130(\u010C6,)6+2\$#(\$7\u00C0\u0113)##3+)#+\$3+))#/\$\$\u020D" +
        "\u00C0\$22|\u00EE--t\")+'\u00AA)7525-+6#6\")\u00C50\u0115#\"6+7#&7//'\u018D\u015357\$lK73))&\u009E2)5" +
        "7\"-76)0h36)\u0113(\u01096#7\u009D6,\$)533,/52\u0094+7#\$\u00F82+2\u00B656S7'3&,+#)x\u011C#\"\",'\$\u0080" +
        "46#67#\$27l6(&+{#36'4(*\u00C566+2)#5'6\u0113)#\"\u00AE+#2-Z\$\"37\"2#+67a7\u00C7220\$++,066(&\u00D9+0" +
        "++5&x6(F\"5'\u00A8-6466/++2\$P\$++\")#65)7,+(+)\$566063\$+++\"\"'\"&\$K\"Q0\u013477,\$S)\u00B7'#+\u009D" +
        ",))/'!%((\"\u00B7#05!\u01DC(((7#(56/()!5(P/()/'/7)0#07'))7l'|\$,066\"7l/(6(6\"4\u00AD!76)<\u0113*&'\$" +
        "/&0g\$+7,+)5\u00BB\u00F9'!+5l\$!'+,Z'5,\u00E4706\u02025|5\u00F9\u0094&\u017D,,\u010B2\u00B9,/&,Z,,*5*" +
        "267F2(\$7*\u01D32k2,,777,26Z-77#7\u01BEK\u00E0+\u024D3,|\$-,\u010B#*),736\u0113,7,77,\u00C2\u0210((x#" +
        "l)%,55*+)23\u019C4a20+6)7++\u010D)2\u01D0\$\"/,(-+\u0239\u00C2#76(,55('7#),\$(53\u00BB\u009D1\u00E45&" +
        "4,#*,4\u00BB\$\u015F*25\u00B3,,,&776\$*,2#+#,/l',4\u0103S50/7\$22+#\u00C5*653\u00E4+\u01AC\u00C56!(7)" +
        "'7\u00C0,#2c56\u024F,,/,+\u009F)5,(+,+\u00A4,\u00DC#3*7,\u01A5)\u01E627)2\u011527(26)7)#\$!)76\"w\$u)" +
        "7\u009D4|\u00CC\u01AA5&*\$,2\$)+&,%\u00B8*70*)\u0130%6/7\u011527#\$46)&\u0193,7\u00C76\u00C833|77%&,7" +
        "+\"-//+S+/J/27!++36+*#*07\u009F\u009D%-0+\u00A8!\u00A0\u01EF\u00D16x/5+(2500406\u00CC\u0100\u00CC#+(6" +
        "55Q+\u0110##\u0191a\u00AA,062*6#(+#)-27*/6|*#0l0+)\"\$ \$\u00C7(+!2\u0100+\"6\$40+4\$\u00A2\"0#16w\$|" +
        "3\u01D46\"*\"42\u0126\$5\$x-/\u00C5\$3#30\$&500,'-\u010B%#+\u01093S\u019B/\u00D02+#3\u0215\u0100+/\u011F" +
        "6|7*+76'\u0111#0B/++*##6+\"0,\u0102#70#+)2+07\$)5+\$x\u019BS\u010DB\$+))570\$\u00C567\u0123\u00AD7+)#" +
        "6\u00B3\$6)//6/\u0085F*)61,/#\"6,#5\u009F++7++&0/6+++,\"2'+#6\u02122+1262\u01023,565\u015E\u00A47K\$l" +
        "162673&&Q\u00C772,70(l22g#,+6543/7'57\u00C7'7'\u00D9)2'*\$\$(30''7++\")\$7+#\"7(6567(7&645\u010C76,2\$" +
        "\"776)'32,3&5065\u00C02\u015F\$'2+-\u01FD\$+1637+166a50(352137c'\u01175\"-00l&\u00AF6*,7)\"76Q*\"/-+6" +
        "\u00C7/\"+,d-067772)7(\u00C7#3\u010D\u00AD\u009D677)1\$k7(-\u01305'+&\u01162+*')3#\u01092\u01096++'71" +
        "7\"\"\u00C077*+/\u00ADP\u013E\u010D62+56\u0100\")&S-)l75#\$4\u01D9\u00C76\u00C75Y7\$)\u00C0\u01A3''12" +
        "1)*,g,\u0164)\u00BC2)6402(6(0)2\u0106/)Y\u01117+2-2(1-6/\u00C734H(*+S21\u00C7+l\u00C5\u0113)05\u00CC*" +
        "62)6k\u01113+076\"\"6\"4\u00D3\u01D80\$4\u00A86|#63\u00C7#)5#*#4\$**752'*\$7\u0115446{)0\u010163#63\$" +
        "\u00BC+54*#\u00D3*`+##+7#00\$7+0\u00DB#2&0#(0(2&2,\u00CC77+\u00D5\"))7\"2)(\u01112\u00C7\$62)0)/\$+7#" +
        "\$#\u00B7)*)7|\u00BC/6#',74\$))\"2)0726\u01086\u00B7(5\u0149S\")6\$22\u0176\$7-#'+0,\"\u01092&g\"&\$7" +
        "\$+7-&'&)'&2,/3)0+,\"+'0#665\"0*\$\u00C5\u00C00\u021D1&*7x)\$(#7*+\"2\$7#&\u00C63/++)'\u00C5(\u018157" +
        "#2277523_6'*'#*'6 05)\u0130+7S\"\"'ho6)|'^7\u00C0\u010D7+\")+2\u00C0&7'*00)l'\u0113#7\"703+\$5\u00E9)" +
        "5\"255/\u00C7\u010B(\u010C7k#+4\$\u0210))(5,&Z7/&7(0(0,0'*2+\u0115(36#'&17'-05#\u00A0/\$+S\u00BA#2#7e" +
        "++)\"+\u00B5\"|#6/l\u010D/S\$6,\u011172'6(#*#/l)'2\u00BC_+-2#\$2+\$75\",2x+\$\$&),*\$)\$(\u010D+75+27" +
        "0/0/')++6\"\$2\u00D3'0+\"7+2+*66(\u00C733\u00DFh)60377\u013027073++)366++0++067\u00C0+\"\$7+2+6676,|&" +
        "272-(+57-#707S\"4g\u00BB1&\"#67,3(\"+')/,#\u00EF7#23+52(3\u0153g6\u0117+7\"+&%0(+5&+#)+6+)0g\"7+7\u00C0" +
        "\u009D(223\"1,(57()7#+5&-\"'5'30)#73,2&7*),22-5+)*\"(275-3-+7\$-x+\u00E22)\u00C5'7)6)#75{\u019A\u00F4" +
        "\u019A41440-73-&)2\u0236\u012D7\u00E2(7)&61\$)2K2#)7\")&707l(7+5&\u01152\"\u010D0+7g'6&3\u01E5\$#275\u0102" +
        ")\"0,27*)1(\u0115)7#\u009FQ)*(#\u009D'3\u01DE*\u00C50;+5S5&\u01A7x1x6+5)\u0156\u009D\$32)5()7)\u00C5l" +
        "\u00E47+#02\"3\u010B\u00B7\u0100L53\u010F5(5&3\u0216\$*&)(7\u0115)&)517g\$+++\u013E0200704\u010252\u0113" +
        "034'4\"l#'l\u00A0+\u01017\"#+,00h7\u009D7575+)&1,6l&+562055)*51,\u00C0\$l\$\u00AF,6\u0115\"20\"('#,4&" +
        "4\u0221',5+7S\u00AA63'7\u0113##\$567524'56)67-\"\u009D/l)g0265g513&6l'1(7352773(#\"&#+265l+\u00C70,7+" +
        ",/+,5l)\u0160g\u0094+\"6\u010D\u01D6\u01170(5)25152\u00C0l7722+\u00C6&\u00D8#7)1#15((7506\$\u0201\u009D" +
        "\u00C52))(\u0212#5\u01BF\"\u013E6-2\u010C2\u0106)\"0+\u0094570/\u00DD,+57*65+5#)+#6g\u0217\u00C577+\u01F7" +
        "\$+++2)6(75(6)4*)146(#/2'-17+&74&77-25'5&+725777\"&#26)\$\"1)*1\$(('5)+))3'5)5523)3)5+650&#2'24,2\"#3" +
        "+0774+77*5),+3,)6(2\$\$\"5'7(\$+\",6)&17&#'#+6)\"/,+62,25+27)00()7&5'&\u00CC&0\"/57'6605'73#046+3+'44" +
        "'4(++&2&'|),','\"'\u00CF)&2677\u009D767!&\u00D522/,+S&+#4\")7#+)),7+/))+6,0\$,\u011566&\"'6'\u01017\u00A1" +
        "+\u01157\$506\u01151000(\u010D5\u01156\u00CC\u02156')365&2&2++\u00A0-0#\u00B9'6#4&(###&5\"6+6S++70555" +
        "55*0((\u010B2()|#&|2#7)5#/74((\u00AE((!/6+\u00B3!+/0!&6(5\$6++*,7\u0212'''\$\$\u00822\u00E9\u0082-|+3" +
        "7'#\u00AF6S/k2e))(#++)3.+-)/7+)+(6,(\u00866\$\u0115|206#\$\$('7/-\$7#+76\u0220++7-3\u00826\u0111'5+(2" +
        "+/))\"\$'45/#\$-3)#*+*#+4*++#F223*\u0258\u0111\$-0\u00BE33+66722\u010C2\u01097716\u00BC)0*#\u00A4I((#" +
        "15176\$'f'#4#'\u010D\u00B3&K&/\u0094\u00C5&\u01646-\u011D*1'\u0117460S\u0100W5\u0094*)32775g47\$xK\"\"" +
        "03\"\u009D3\u009C&7-/\u0153\u009C/757/)\u0210\u0141S,0(*#656+\$*+)-6/*6'\u009D'\$l\u00C555%:(/\u02097" +
        "\u009Fl37#,5#5,,)5-*270-,-\u00BC4\"4\u00B75)\u00BB6(e\u00D6g3,#2\u00C0x)_\u0221j3/#\u00EB\u0094\$3--/" +
        "'+++lg)+2S\"+/\u019C\u01D2^\$3-)-\u00DC6Fv2l&&)&\u00CC4\$/'7\$4,\u018F\u0182##+.\$!5-\u010C2-671%2x6\u00C0" +
        "4)6)\$\"#&5-,433S\u0215+42327'6\u0135+)/5'+\".#P\u009D/#S)\u00913,5l+\u01BC6+75#+3/-l+#\u016A53\u0094" +
        "7)2\u01C7#15\u010A&6w7\$*-3+\"6)#6)\u00A0+23'0#\"5\u018F\u011375\"5*+6+(7+07+\u00D57#\u01004\u009El+\u009E" +
        "7)-j)'#Q-|7')|\u00C7537)5\u00D2#66l5\u0213)665))5\u00A622)23\u010C233//'\u00B33#2547\$j26&/3&\"\\(&S\"" +
        "+7)#+77\$\"5'#5+\u00B3\u00945*&7&+26+364\",\u00E8)(2\u01D5\",\$\$\"#62##+\$5\u0110\u0164\$\$0+6))\u00B7" +
        ",0++#2\u009D+))26\u0130/66##6\u00BCx\u00C0F\u00E93))/46#,,'0\u01156\u023B5\$\u00B80\u00AF7(275S&h4470" +
        "\u010A401&*\u009E\u00AE\u0116)\u0101e6&S15)((&\u00B7)'760!l,6&1(#66\"604+/%5##72x61+3\u0109\$,0\u00BC" +
        "6\u00AD,\"S',66\u00C0/1\u019B\u00EF-S\"\u015F565\"(67\$\u025B\$-'(/,\u0094\u0103'Q4K7,&\u00C0,,\u0130" +
        "l,7\$#))+#-6560\u00A0,+16\u009E\u0100l#\u0113\u01C37+4q'#_#''\u00AF#\u01DE\u00BE&67l)(6#\u013011#,370" +
        "75(&e'x)5#l)%!0#1)\u0110\u00AD/+)a27\$)1\"#0()\$71(5+65)26,(76\$4+\$&16(\"S6\$\u0237\u010A2#c4\"5'+\u0094" +
        "7,a7\u00C73022#\u0094),\u02005\u00C6#\u010B)63{#\u00A8(\u0115#)4\u010C\u00E9\u010B\u0112+\u0094%,\u00E4" +
        ")-,*++2435(666'#,,#\u0113)\$0\$\u009El'++)4)\u00E9*\u009D/7''&+##2#7,#\"+3\"(\"0)\u010B0\$\u00C2S\$)\u0102" +
        "\u0164'\u00AD'+&\u00C0F4\"76\u00C7)+3\u00C23\$(\u00C0(#3#S#(&+0,/4\$2\u00BA\u00C66304#-73)3+\"6/\"/\$" +
        "66652\u00C0*54\u00C66K+55-#4#\u010B6S,!%+6\u00AD*),777\"15\u010C!0)&+)\u0094&(((63l75\u0103'0\u00C0,(" +
        "2S'|7\u00C04\"4/)(7\u00C5\$6\u00C2#04\u0102*x\u00B8520+766,62\$7'15#**()+\u0245/6\u00B562#267+\u00E9," +
        "+02\u00B9+)5S#\u010B3+\u00B7\$0\u00C7/#6#6*//S-25&67))1#%,\u009D2\u0102,25+5'+14'#23+125\"\u0117'\"7'" +
        "73#/+\$)+2+6+6,x\u010F,3l/!+\u01625/\u0248)h)43#66\u00C7+\"+\u010B\"7+\u00C0,6+\u00C5S'(\$2a22\$,5\u00E4" +
        "\"Zl725\u0113(!,\u00BC#7gS),#-()6\u00C2\u00EE26'#(2*0,l70\$++*!S+4)\u02132\u0130,.#\$\u010F\u00C0\u00C1" +
        "1+6\u00C0#(),6\"60+\u0107(%3&\u00C7\u00C7&1&*2165&)x4\$255+\u011320)62+65\u0113\u00C0(2-+5665(\"(!4((" +
        ")l5\u00DB,6+)#2\u00FD+*6\$\u011135S72\u0256&\$5'5\u010D)\u00AD(2556\"225\$/\u0113-\u0141\u00E41/3\u014D" +
        "7#7%-6'#\u00B80&)\u0115*)\"|,+)l,005+.\u01002+655+6++)\$7\"3\u01F8,(S&g33004+(.'\u0113\"+\u010Dl!+750" +
        "+(\u0111\u008911\u00C0(+23,6/65))5,02)/R\"\u0206656)+\u00DC606+3(6+\u00C06&,\u00AE\u019B+)\$,++(\u0097" +
        "74*7(+),+(+)*+)23+056+,\"((+-P\u025A50_(5#&55+(6(\u00AC)'\"677*\"\u010B2^0\$)0Z25)7\u010C6\u009D7\u009D" +
        "(,'25#26,5K%\"#0441\u0130/66)036\u0102h\u0113\u00C0\u00B90076^,4&&(\u00AB7&1)&#\$S5-\$0\u01177/Z\u00EE" +
        "\u00AD()+&\u01047\u01B2/S57\$\"\"'7\u00C0\u02373+'\$'6+#0,\u00A0#((Q&6\u00C4+765\u00CE)+\u0115/\u00C0" +
        "\u0187264,\u00A4\u00E8'07,+7)-205)0\u010B6)3#\"\u008B(25&2\u010D\u0094574&++\"k\u01156)\${x\$\u0110(4" +
        "72\u0130),006)\u0251,0\u010B5++\$x3*((\u00AA\"&\u00E04x\u011562\u00C00\u0105-\u00C0\u010E+\u00A04\$%\"" +
        "|4#66'#\$+*)\u00B83,5,6)\u00D95(6S\u00E94&6,4&156\u01DF\u00E4\u01BE62l377h&6*,+\$(0x\u00DC\u01ADF\u0101" +
        "-+#16)\"+S2j#\u010064+2\u013311)45(0\u010CS(36\$-\u00B8(\u009D,,64637#20,322##\$!+4|3#/\u00C0/6_++076" +
        ")245477,,\u00AA+,5#+,5#7,5,72037//\u00C00\"+\u0094'5\u00C07(K+))(\u0115\u015F)2\u00E95\u00E4\u016452#" +
        "#\$+5\u022E\u00C6)\$#\u00C0_52\u00A006#+75+775,501),(1\u00B97)/(,&+)+,+#+0-+\$5l+++\u00946/\u0204\u00AA" +
        ")625#0\u00B3\$#,\u00C0)\u010D75(-\u0094-57,%*,5\u00B86*60+525x\u00DC)\u01156(4\$#4\$7(066\"'6\"#62#&'" +
        "\u0094g)-7\$677)\u021451710\u00C0)6\u00E6#!-(1)3+\$SS67\"2\u0111)\u00AD557\u0110/K\$\u00E4&'771\u0094" +
        "6,3\u00BE'5l0S65,\"\"&c\u022C*1))\u00C7Q)%\u00A017'6#+*57\$5**-2\u00C5*#+)2S\u009E667+00)6\"7220+\u00D9" +
        "+*)&hS*36+##Z\$\"+g\u0094/&\u00AA+'F\$\u010F7)\u01010''*03\u0109&l5S\$*x\u01155(6\"\"&\u011036\u00B9\"" +
        "l+/\$\u01133'\$(16\u00E9-)gS3#2-6)0(*+++\u01852+\u00EE\$554\"#)02S5\"\u0094)\u00C7)#)\$7KK55\"1,+!\u0186" +
        "\u009D#2-Za\")\$w\u0164\"+&17\$242\"5\"#++5\u010B+7\$)l2)/6+6556K\u0110&&&\"())\u00C5'\"6\u00A9)/,,,\u00E0" +
        "22\u013B+),3+67)524536+%\$0/),6'07)'\$N+0)4)0++2q')'6',2#+))5\$36')0\u01A40)#\u01A37#)'\$7|2\u00B80'2" +
        "\u00C5''#)(052-\u00C0+7|\"5\u01BD2'7\u00B8#56+)56\u011266\$&\u00CC0)\u00AC)&5\$(#35)\u023E156203)56%K" +
        "66226\u0115)\u00ED-#675&5b2627\u00B8662#(2(5777'77\u0230'&)\$+x6+-/7'6)67\u0241\u018107/\"5e\"\$737#7" +
        "6556(\u009D2e52(Z\u009D'0()('\u011542lx,|,\u0081(65'[\"x\u0140+7*\u00CC463)71#+%*\u019223\$\"#6*002!6" +
        "5#)544'2\"()27\u021A\$(#2)*\u00C0#5245&a36-6\u0182S7##7)0374)0\$7\u0153707+)_\u00B8+\u00EA\"(/6z5/w52" +
        "(\u009D%7x5&/3\u00BC0\u011077\u012B(6C\u00A9-#&l66(5'-\u0100,F453l2)/\u0238(('\u00B9,\u010B,50l5\u0109" +
        "2\"#\u0102\u021D)\u01015\u00B8366+\u0115S#0+,,27,)7\u00C7++\u0162\u00F3'j\u010D!!)7k\u00B8),#\u00B8\u00B7" +
        "Z57257\$7\u00ED+\u01896))7\u00C0\"(\u0102\u00E4\u00C77\u010D037/2573\$25)))\u00DE-6!7/(\u00B86621-0\$" +
        "3(l(\u01276)6)(\$757+24#+\u0110\"76%#4#6#16+#5-(76\$76\$6)\$&1))(31203565)1)(.)6-5%+52 2&)7'(/7 5727\$" +
        "7#07\"66*+2'2)(#72(\$'0'567'5#(57)4604'6(*22%077-\$7&*4622\$#+72326,#\$()5564%65#7!6\$,/5,\$256\"20,)" +
        ",7),370+/)607#\u01AF0(5)((\u00AE\u00C55(+\u009D\$+\$)\u00CCg\u0091\$4\"25&77662#(36x6)\"(%#5(\u00C6*'" +
        ")\u00945M(\u021D7)&5\u01324(\"\$7&67\"#!/-/'t6\$,25(\u00C9\u00AF\u00AF,!,\u01F1-\"6)3,/56\u00C6,#3()\"" +
        "76&#'36(4/(&3\u00A877%72\"7%'/K,x2*K36#,(\u022B++('/7\u009E57)'7)6572\"\"02#77+7##23/'5,)2&3##7)7\$)5" +
        "#6\"7+&#)2F7&'27+6\"#477\"X7\$76/w46)'57\"\$2625+\u00AB7\"7&6'#75\"7(7&3/\"'7'%)\"23'*\$,&(677)(7++77" +
        "')07+2&\$)22#\"7'/\$+&7&727676726'#5g-\u010F5#\u00B5#75377+)&7\u00A4020\u010157\u009E0Z#\u0157#x\u00C0" +
        "h\$57)\u00B90)e#\u009Djl3\$+'2#5|26\u00C0)\u00C0\u0164\u0164\u0151+\u015E\$e73e#\u009D\u014D0)7\u0110" +
        "%77\u00C7/\u0229\u00D0*\$\u009E&#\u00C6\u00947\u020F6S\u01E060h36\u0232+,\"x*3)lS07\u00C0\u01827&S\u01E0" +
        "2S-)+'6)Z\u010B\u00C0\$l\u0102*\u010D'0\u00D15Z+S7)x#)\$0\"5\$)\u00C0)g\u013E#\u00BB-\u0100\"j\u015E," +
        "2+6)#0,3)\u00C0\u0105i\u0230)\u00C0-)-++a()#4\u00C0xSl0\u01DFZ7`7/77\$7kZ\$#67\u025D7#\u013D)\u00A7\u0109" +
        ")K7\$/1\u00E9xl3)\$#\u010030\u0157\$3)-/+l\"#++#h|25*)70|\u01F07+\"7\u00E9\"3)\$#5&#l\u022D\"l#\u009E" +
        ")+30\u01DE/+\$#'7\$\"\"7)#0\$#)4\u0110-++7\u00DD7l#\$4++547+51\u010B-30)#l5*\u00C0+\u00C2'\$&03\$\$'+" +
        "1+!))3*+6,*0++x763\$66_\u01BA')465\$\u01BE&12*23e\$%-0,1*07((\u01A5*Z+\$!7K*6/\"''/7\u0096%\"77{*\u0186" +
        "\u01022/%')\u00C5'\u01C5*e7|72+626\u0113\u010174&07\u00EE+77+7(4#\u009D*6/0'\u00D5/'\"+/+\u00861)6\u00D1" +
        "##g\u02446(2K7&\u0113\"\u00EF64\u00CF'7\u00B86+\u00C5#74(57)7\"+&&+'2*(6)7%6)(+/++++#6'5\$17%+1('*+77" +
        "6(7+60277)70+)7&+-\"('4+#7#&)42/657+7+5'##\u01367\"++#\u0115\"\"S\"5\"\"#\"S#1--l##61\"\"26+x#'06605\u0110" +
        "',07\u01AC376\$6)\u00B86&4\$%\u008E7\$)64+##c3l\u0110)\u00C5#\$%\$/\u0114{732\u01100)(\$6,\")-6)\u0182" +
        "+532\u00CE3\u0094(-\$)55\"650\u0109\u01820\$\$3)3\u023D\u010C\u01173\u009F42k2l\u0210\u0094+2(+'+\"#\u00C7" +
        "S(\$r67)6#*436\$6+\"-{x2\u01782F##3\u00F42766%\"\u009D%5(0\$x\u00B7-6'6#+53w\u01676230,6'7!\u023F##27" +
        "7\$w+l#+710\u010B6\u021C%+\u00C02\"6,52\u00B873%,\"\"+a66++\u008B3w062(6,\u009605*&\u00AD\"&\u00A9-52" +
        "\"6#(\u021D\"4)6(30\"/\"2\"4\$7\u0101+*'72*6\u009D\u00B8\u0100\u00C7)\u00B8(5)5'\u0093+)*7+66#\$j+&\u0113" +
        "&'\u00B5+)')k\"5\"+6SE0/0'73\u00C7/+-ak\"5\$)606\$6,1\"%2)66(*5523,51#7\u00C76456\u01BD6\",#\u0091+67" +
        "(067\u00E405\$\"5\"l47,2+/\$,6g*&,\u00C7*+7#6\u00C06#7+\u00B35&l\u00DE6\$07/76\u009D6)67,|5\$7&6&6330" +
        "xl('2/#7,+##6\u013E75)#63),*2+/\u009C(62-\u0114++F3/7#x77l*3\u00C5\$(5\u01523e#6/*2,e##66!7(3)/#\u00F4" +
        "+7+6)\"\u01E053\u0094-\u010C+)\u0252F-15\u013D6+,,-5)\u00D9,6\"#262\u010C+\u0164\u010C+\u01F9))\u00A4" +
        "6+\$7r/0\"&77\"+&-\$\u010B\u00C0*#7&\u00B9\u00B7\u009D'0,\u00C1\$256#\u010C35-00)\u00FB6\u00C6S&\u015E" +
        "5\$6\u0102&S\u00E9\u0100%)\$'6\u00B3\"\u010B1#c&{60S'(6)\$675'1(3*6+%0\$7-32,\u00A3*+\"\u015F\u009D\"" +
        "/627\"\u0117x\u009E72\u010D\u0111)\u015B\u010C\u021A57S(U2\u0235\u010D\"7#7/3B&7\u00D4\u0101&+6/6\"\"" +
        ")\u00A4\u010B7,\u00ED\u00ED\$),3\u00B562**\")\u01115()#\u0080\u00A0S2,\u02496)7#\u00C53,+)\u010C\u00B8" +
        "17702a5\u00C50/\u01107\u00B8,*\u023C\u01B655)#+)/1,66'!\$62*0+2!\$(1\u010D*05\u00EF\u011573#'\u009D\u0117" +
        "x+,+4\u0116\u00C6\"426kx\"()(_&#47\u0109\u00C5\u00B7\u01133*l+\u00C5\u00C02(+6_7)5\u0088,/+/\u017A#'+" +
        "3\"773+\u00EF)#\u0113)'\$3#\u0091\u009F+/++0\u0094),7S!S\u01B267+>*3\u009F\$\u020A#77\"-7Z\u0221\$\u0101" +
        "\u00C0l)6)',#3\u010F+x'j27+,+(\"'+*&4-\u00C47(\$5/(+&,\u00BB6\u013D\u00BC'5\$347&1)%)l37(6\$56*2(4&7#" +
        "02(*\u00EB,00(3#(\u0115,)\$)2*(5|)\u0102\"7756,/!)0,gxS5+2*+\u0117\u01B7\u00B04132e\u00A02\u0141\"/wS" +
        "2'74\u00C6(3+*2224\u009D7\u0111+663-5)',)2725&4-+2\u00CBl35'5+2!,,+\"4+\$\u015Al6?8*0#0#3,6#&)\"2+5#+" +
        "7\u00C7#+55*2\"70#+()/\u01CD0/+25\u010B#*2+-(/2&00635\u00C6)7+0+\$57\u00C0)7\$66\$)-\u0111/3&#\$2*)&2" +
        "+)(6+7\u015A\u00C0\u010D+\"x\u00B86\$'\u013E\u00C0K\$\"6!75\$7(7)-((0\"6*-\u0106))0#7,++)7\u009F2/+\$" +
        "\u01F8#+\"\"+5*+%+5)+\"\u00C06#\u0106)\u00B8'#-+#0/+7+7\u00EE)37+)'67\$7/+30#22&\$,-6#5'\"3)\$#7-\"'\"" +
        "060)4'*&(\$-\"6070'\"*/\"\"67,3)\$63\"+20,\"253\"-/\$5*+%,66#)6-7\$3+\$6*7753\$5(02(\$',72)6##3!617+/" +
        "4+3+*52+7#''%5#+&5+*)13+0)!07-#,\"0\$**#5'+#7)75)(/3\$)),706*02%#07\$2((!\$,+7&,,7\"'-3)--+'\"6)\"1\"" +
        "3,+\"6)\$75+#)+/+\$+/#0\$(+(7+6#\"+#5l#)!\$0+,7,,22\u00B5,6\"M\"2\u00CC*\"(125\u00C5),52\$\u01BA-7/\u008A" +
        "+L\"1\u0227'''&#\u00B8',2*++32)66#0+h2*646\u008B(64(\"\u01BE064SF46*\u01D2+w0&3-3*(\u00C0#'\$0*h\u00A8" +
        "/\u01C3\u0219(#/x(343,226(\"4#145(),*,7-'43,+*&'(*)6+\$66#54(66#+0(*0(330*(&&+\$504\u00AA\u022266\$:2" +
        "\"/\u009F6\u01077/)&667)#:0\$77\$+:x\u0222S\"K)++#5\$+,\u00C52w\u0112%'6'2)0\u00C5#\u0099\"522|S\$67\u0162" +
        "5\$-65/\u009476\$+F)#lS+\u01815+256|6\u00865#66\u010C\$6+44-\u017E\u017B!()26\u011E6p\u01A664\u00A8:5" +
        "3)7\$!5621+3\$\u00BF\u010D2F\u01E6&-%)\$567(+5\u0171++\u0165\u00B3\u00BA2)-6060\u00C056)\u009D(\u0257" +
        "'\u00C7#6)#(7+6#\$2(42\u009D_76)5#+\u0146-57))6\u01155-&\u010B,4&\u00946+#6++!\"\u00E4\$\$(4\$\u0206)" +
        ",#5\u021F3\u010D/,+07\u00B52&(76-73++\$64+2+5(6,,6+,\"\u00E44*656!\$356++\$)/6S/4&5)\u00D4\u00E4,++\u00B5" +
        "!&\$++F&+\"(((\"0\u00C0)37)#\u00C0)\u00BC\$)3&&*,,,\"6\u00E3(6\u009E\$#\u024B1\$\$410)5-\"62-\u00E47\$" +
        "\"663\")(/6\"6\u01B53)\u0112!!('0\$03,X23\u00FD\u00C5*\"3\u010DS*\u00C0+\u01851\"%00\u01C60,\u00C0)\"" +
        "\$)43'3K5/'\"*3+\u009D\u00C05)))#x(50\$4)+\u00B71&,\u00C0'4\u00FD(#*1624'K3'\u0113'\"\u0112\u00B5\$4\$" +
        "41&(463))555)6762+/\u00B966/!65(\u0115\$\u00CC*52<565%\u0103*\u00CC\u009D6\u00D0K\$\$\$//+#)+\u019E0:" +
        "%4\u00C5*266\u00AD\$X/+\u0098)3(*)+3#\u00C66((\u009E33S+377#\$+\u0109(0*l5062x%%6\u00B7\u00D267655662" +
        "x\$)*++/\u0114,#\u0101(0'5\u01CC(\u01621,\"5/++-06\$0(52554'\$0\"2(6++/)))3(6)*6/(3(66*36%76%-,\$2(+#" +
        "1/0&I'&57S\u00D9K3+\u009E5\u010B)\"24662*2&+5+//+\"\"\"+\"2&5&67\"2)226+/\"\"&&&&\u010C2#)\$23\u00AB2" +
        "5\u011116\u00C6g6&&g6\u0111\u010C7\")3\"#32\"g\$(13)\u00AC\"63#56)%6+5g\"-%\")\$26\u010B6\u00AD'2)7\"" +
        "\u00D1\"6x&7\u00EC'%-\u00B9((3(\u00C7(<\u010F&4'#23\"'5*+2\u010F64,3,\"65)2\u00CF\u01162#\$\u008B\u00C0" +
        "1#63457&(,6,#5+7-2\$)353516#&6)2\"2\$6%15(+)5\"\"\$%6-)'(5'#*'2#6,\"+5),27-2\u00CC\u00A95&\"-\"S3(&)!" +
        "!K5\u01135&/5,6\u0094\u00B3\$x\u0111#57\u010D/\u00C512647\$+6\"\"\u00BB36040)/76/7)7-)/7)+7\u0182&62\"" +
        "332+,/3713\$502)%(\"76\u00BC7\$(+)20,+(353)%#5\u012A+7|2/*3+7*\u00C006&26'64/#/0&(0(63057\u00BB12'3\u00A7" +
        "2\u00C0#7+S35#\$66\u0102\u010B20l2+5+7,F#S#0\"765(\"25+7\$+3#\$\u01DE36+7)663/76\u00DE+35)2)5(aS,635#" +
        "01\"+722&)773-)6\$5,6)(+(/\"+#650)00*772#/7*24!+02\"+##7)25'4446'6>\u00A7\u00C5\"\"l3\$*\u009D\u01DE(" +
        "*'3'/\"*\u00A462+S5SN#*\",++5\$723\"*+''0*0+2\u0137**\$&51,\$*\" 3/7&13\"\"\u00E1&%\u00860'5\u00D2)/7" +
        "223+\"7x/27072)\u013E(+)0+7\"/,2,,25+00-(*-\"+\u00E9\$\$-\u00B8\u00B8\$(\$\$)#66\u01886\u0091\u0100'7" +
        "+\u00A92\u01E8',\u00C00'*(\"\u0138,565+6h04+4'#/\",)5#676\$1)\"\u009D3\$)\u0115~\u018C2(3,)2\"\u0113S" +
        "+4(+\u010D&&-6/,\u009D5\u0100(/+3\"0/&\"_4\u01A5\$\"k'\u00BF-\u01163S&7)'2\$\$3\u00C02(\u0116%!4776+\u00A0" +
        "3\u0110\u0101\"4)*\u02615'(+&*5\$)3,27206-7''2402S(\u0163+22*,#+756270\u01025-#)\$0'7\$+&-6*+=#\u00C0" +
        "++7+,77-(6\$22(3(57+7\u00C7\"\"(07446#1\u0179(+600)\"%6&\u010B\u01A75\u010E(2154\u00C0(4\u00942+,22\u009F" +
        "73x6}600412+!+(,)30)g,50\"))7)\u0100\u01FA67*5\"605\")52(7\u01092&\u009E+5,575&'(*72\u010D''\u0201+#+" +
        "2!1)\u0115(2+++,75%+'+56\$)63+&\"(\"/-+67&\"\"(/3')*4%37(*))5755'++)+23'2())60+07&*#'--\$)227&\$\"#34" +
        "42%0&(0)\"2!03'6/)+\"5\",,645'2+7('+7'\u01736&+\u01E0\"6&|)&1\u01A22&,\"6l(0\u008DK4\u01E026&F\u009E)" +
        "/(7\"667&&4.\$'1+Q&3\u00E4+\"7'6#\u00A45#\$)F'76666\u00B6636\$4%'!7\u011C(5)+7\u0115\u0110(+1,'1\u00C5" +
        "5|#\u00A0(-%+))3\u019A3S4)6\"))\"5))3)3%%*\u009D42a)/+/2&F7\u009400\"\$+0))3660+67*\$*20)66\u00C07+\$" +
        "0h')\u00C067)26#,1!0h(3%),K#36(6S*,(6l\u00E4)\$e&(66542(#351+)\u009D\u00C0\u010B(##666)+3*3\u010963,5" +
        "'3#)37!\u01156.#7+6+\"2764)632)6(\"62(&)+6')553,7\u010C6+3556/\u0162+\u01006)76766S-(,6,\$66+\"+(+266" +
        "\u019E0++-))6,2.6#\"7'\$+656+#065327%'5(7'+(4\"+)'%65340,!*\"/0#'62((%'0#,4664+)6)(667++))6+(76(,'2+)" +
        "+)\u00C5#))6#+6#)S#/l\u00C07),,6+#)+-0+)))\u00D95\u0115),+27+)0+6#,,(#&,,&/00,&\u0115+0,#&&0,,,,(,7-&" +
        "(()'33((*(2+-a((60\$53,,\u00C0\$#6|6560,606+\u013E\$#6\u00C66\u023F!\u0117|#\u0113,|6\$+\u0116&&&,,6#" +
        "0l477,!\"3\"6#3|,-\$7''x&36/#'e63322&&4K\$370226234))(\u00C5632556+\"60(\u00B74\u00AD5~7547--\u01E07)" +
        "\u00C7))0)###(665\"7\u010Bl#6\u00C73++#05\u01ED-)6#6\u0113#\u0101-ll066\u00F4%46#7\$#)\u008B##(6)+\"3" +
        "766#04\u00D4/\u00A8\u00D66++'*\$+\$+'*\u01A50\"'6#()56    7 4   7    2     +       2+  7    +'3,    '" +
        "\$                     !3-  (\"+3\$7+!3,,\") +5"

    private val END_CODE_POINT: Int = BASE_CODE_POINT + DATA.length

    private val ENCODING_ARRAY: Array<String> = ENCODING.split(',').toTypedArray()

    @JvmStatic
    fun create(pattern: String, delegate: MinusculeMatcher): MinusculeMatcher {
      val pattern = pattern.removePrefix("*")
      if (pattern.isEmpty()) return delegate
      for (c in pattern) {
        // Pinyin initials are always lowercase English letters and never include i, u, or v
        if (c !in 'a'..'z' || c == 'i' || c == 'u' || c == 'v') return delegate
      }
      return MatcherWithFallback(delegate, PinyinMatcher(pattern))
    }
  }
}
