// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.vcs;

import com.intellij.execution.ui.ConsoleView;
import com.intellij.execution.ui.ConsoleViewContentType;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

public final class VcsConsoleLine {
  private final @NotNull List<? extends Pair<String, ConsoleViewContentType>> myChunks;

  private VcsConsoleLine(@NotNull List<? extends Pair<String, ConsoleViewContentType>> chunks) {
    myChunks = chunks;
  }

  public void print(@NotNull ConsoleView console) {
    ConsoleViewContentType lastType = ConsoleViewContentType.NORMAL_OUTPUT;
    for (Pair<String, ConsoleViewContentType> chunk : myChunks) {
      console.print(chunk.first, chunk.second);
      lastType = chunk.second;
    }
    console.print("\n", lastType);
  }

  public static @Nullable VcsConsoleLine create(@Nls @Nullable String message, @NotNull ConsoleViewContentType contentType) {
    return create(Collections.singletonList(Pair.create(message, contentType)));
  }

  public static @Nullable VcsConsoleLine create(@NotNull List<? extends Pair<@Nls String, ConsoleViewContentType>> lineChunks) {
    List<Pair<String, ConsoleViewContentType>> chunks = ContainerUtil.filter(lineChunks, it -> !StringUtil.isEmptyOrSpaces(it.first));
    if (chunks.isEmpty()) return null;
    return new VcsConsoleLine(chunks);
  }
}
