// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package com.intellij.openapi.vcs.changes;

import com.intellij.openapi.util.Factory;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.vcs.FilePath;
import com.intellij.openapi.vcs.VcsKey;
import com.intellij.openapi.vfs.VirtualFile;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.JComponent;

/**
 * Builder for the list of changes in the file system. The instances of
 * this class are used to collect changes that happened in the file system.
 *
 * @see ChangeProvider#getChanges(VcsDirtyScope, ChangelistBuilder, com.intellij.openapi.progress.ProgressIndicator, ChangeListManagerGate)
 */
public interface ChangelistBuilder {
  /**
   * Process a change to the file.
   * This method is used to report changes that the version control system knows about.
   */
  void processChange(@NotNull Change change, VcsKey vcsKey);

  /**
   * Put the given change into the given change list.
   * If there is no such change list, an error is logged and change goes to the default list.
   */
  void processChangeInList(@NotNull Change change, @Nullable ChangeList changeList, VcsKey vcsKey);

  /**
   * Put the given change into the change list with the given name.
   * If there is no such change list, it is created.
   * This method allows not referring to ChangeListManager for the LocalChangeList object.
   */
  void processChangeInList(@NotNull Change change, @Nullable @NlsSafe String changeListName, VcsKey vcsKey);

  void removeRegisteredChangeFor(final FilePath path);

  void processUnversionedFile(FilePath filePath);

  /**
   * Process a file that was deleted locally, but version
   * control has not been notified about removal yet.
   *
   * @param file a file to process
   */
  void processLocallyDeletedFile(FilePath file);

  void processLocallyDeletedFile(final LocallyDeletedChange locallyDeletedChange);

  /**
   * Process the file that was modified without explicit checkout
   * (if version control supports such behavior).
   *
   * @param file a file to process
   */
  void processModifiedWithoutCheckout(VirtualFile file);

  void processIgnoredFile(FilePath filePath);

  /**
   * technically locked folder (for Subversion: locked in working copy to keep WC's state consistent)
   */
  void processLockedFolder(VirtualFile file);

  /**
   * Logically locked file: (in repository) in lock-modify-unlock model
   */
  void processLogicallyLockedFolder(VirtualFile file, LogicalLock logicalLock);

  /**
   * Report a file which has been updated to a branch other than that of the files around it
   * ("switched"). Changed files (reported through {@link #processChange}) can also be reported as switched.
   *
   * @param file      the switched file
   * @param branch    the name of the branch to which the file is switched.
   * @param recursive if true, all subdirectories of the file are also marked as switched to that branch
   */
  void processSwitchedFile(VirtualFile file, @NlsSafe String branch, final boolean recursive);

  void processRootSwitch(VirtualFile file, @NlsSafe String branch);

  boolean reportChangesOutsideProject();

  void reportAdditionalInfo(final @NlsContexts.Label String text);

  void reportAdditionalInfo(@NotNull Factory<@Nullable JComponent> infoComponent);
}
