<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = @(Expand-Excluded $exclusions)
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      $exclusions += $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }

  Set-MpPreference -ExclusionPath $exclusions
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIwKwYJKoZIhvcNAQcCoIIwHDCCMBgCAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQgflJmsGiG0C9Eqfqr
# 7muXHG6isRKVoKYgGTf+u3FG3ZygghNYMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owA
# AAAAUdOUfzANBgkqhkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVn
# YWwtdGVybXMxOTA3BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3Ig
# YXV0aG9yaXplZCB1c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3
# MTYxMzQ1WjBpMQswCQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFC
# MEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9u
# IEF1dGhvcml0eSAtIENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAp4GP9xRFtmJD8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAi
# xqhmdU1Ug8leaBur9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToY
# yVAslyctv9oAfWN/7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tI
# urnXwYJ4hWUuf7XJwOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36
# rL9kUdFcm7T1XOdc/zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32
# KvBPpSA9aCXrYZd8D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+
# Xq14Z1tV417rx9jsTG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6
# hD5B3G5WrsYaW/RnaAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxX
# OwlyndfKt5DGzXtFkpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOj
# c2zjs2s3/+adZwGSht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6H
# RGDjqfTCGKPj/c4MhCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEn
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQG
# CCsGAQUFBwMDBggrBgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEF
# BQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAl
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAn
# MCWgI6Ahhh9odHRwOi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQW
# BBSCutY9l86fz3Hokjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHU
# bI2fkBJmqzANBgkqhkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSY
# Xlk4EwmkWZRCXlC/T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8U
# fY8dQzZks2YTXxTMpXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97c
# v+kR3RaDCNMsjX9NqBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbj
# hbhNm0VyiwfxivtJuF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24X
# iA3hxkOY14FhtoWdR+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBoMw
# ggRroAMCAQICEDWvt3udNB9q/I+ERqsxNSswDQYJKoZIhvcNAQENBQAwaTELMAkG
# A1UEBhMCVVMxFjAUBgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1
# c3QgQ29kZSBTaWduaW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBD
# U0JSMTAeFw0yMTA1MDcxOTE5NTJaFw00MDEyMjkyMzU5MDBaMGMxCzAJBgNVBAYT
# AlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNFbnRydXN0IEV4
# dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZDUzIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC+vac5yaV97F1l8fQrqYfQ560axRo7
# GM7hoVGNcvrOWB9cuCRCD0bVMZfQSk3jmzDJupeonP5FNs8ngOd7uG7BJLffp3Tc
# hfKjJFhFzJSUOwLrxI18RvVcZoLjpMHTH6xuDFMjDtQ6+tpC8YNUXzUyVvK1eJtX
# AEgeqWFhJw5zA0O21nCS+9SFmjDGs+aaXkvvHSrYZqsWRv8L2A+miUoCUVdBPEE4
# TpfHUTJtZ45moV5NnzUir9Vqt39AX2g1zGn68QXw6oWm6jKFy8HByoNpRUkG3Als
# ukllGYz5tzcnjGSeNePl7OcHoJ2ocrxvhTosphZOPZzPCOaC9UR9KMC9ia1sL9wv
# eHkR1xxwS92dCExeLvqNvafdY/Z/8FIxhG462NlchUSeYwZp0IZYeImbh7tYHKQo
# bMb+aQqcHqwRYGpeyWllLu1DLWnxeLc7LTXyqk/iH+MBb5BGqtWoDQRXoLSs4229
# nRsogCdGx9qqZ5Xx0Yd7x8gl6YQMj4k20r4z4YXAM9WgPBmLrzjy5ZOAv8bDq3uT
# xD2due5FdsDUaG8wXjy0NvnXRULgEgaA26Uh/OcFeiiNtI5ge/fItUpHrBRml6Ba
# aXIxV2tThM1hunMHFXA7ewH8pz+MLT2HjPsI1+UvF4N+gqtPCfIp4X5Vd2WUBR1Z
# 1Ardk37jFF3iuwIDAQABo4IBKzCCAScwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNV
# HQ4EFgQUzolPglGqFaKEYsoxI2HSYfv4/ngwHwYDVR0jBBgwFoAUgrrWPZfOn89x
# 6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVu
# dHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYI
# KwYBBQUHAwMwRAYDVR0gBD0wOzAwBgRVHSAAMCgwJgYIKwYBBQUHAgEWGmh0dHA6
# Ly93d3cuZW50cnVzdC5uZXQvcnBhMAcGBWeBDAEDMA0GCSqGSIb3DQEBDQUAA4IC
# AQA+AFS4KvOPZq9hFsRYk2T0QYtkVY0bNTOhm5HYq0bKq1+8vn5w5NLXB6iWB9eG
# 0VcKCNDQeE34Kt+yBPOa4dd88MEAesFAod+KjLfLtB0BYfKYmqbduIMFyqksFtyc
# LQ7+p5fkUKmXlcFLKoxR48QC4Gt8NU1TvEuUE4tTGBZepgqnuDTBbLHrLOQxfQws
# xnzhpFboLXbVshN16oMArgB3xm+pPe5jFIFQFvxbTxIZDlsPjepLzgZMiUuuIMyO
# R6Z11mXuLzDoXTSPH4JNXEKm8hRMUCCcCaJ0JFw52IkyhTyvjOVqnuYEOqUT/6od
# zUdLLgIFtGqP64VPge8K232fKY+lwj9SOFJBlTu8PltUMEIjCWPeUI2JNUX6q7gP
# j6Kte3oRk/GPKIR7aHlHauhkKU0f9B0vbR7IlIY801qZemt8qzX3KzlLT7k/FpLv
# KYyNq6wBGsxxRDnLQD4gEs4IcONH/tyA1wgA0Qtq6iG1eWYX6WqqVt3NFXixA8AU
# rT5HGHXtClNLMpPN/4CxkNYT5eRBeCLtQYLlDt+wzpUhReaLAdMZ9/QrVazZVnNj
# EOC4oG3LVJaYoJkhUQaqpgw6K2PLedpDudISzN6PVXrpCfPimtMlwXs2ktrg5VJn
# zePmVAqlndRzo9MvXQZkhQN3f2DYwrx0yZWRaLe3fmEZdTCCBuowggTSoAMCAQIC
# EDGdnUgatvXgkrzF40/3PFswDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMx
# FjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xPDA6BgNVBAMTM0VudHJ1c3QgRXh0ZW5k
# ZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcgQ0EgLSBFVkNTMjAeFw0yMjEwMTEx
# MjM2MzdaFw0yNTEwMTExMjM2MzZaMIGaMQswCQYDVQQGEwJDWjEOMAwGA1UEBxMF
# UHJhaGExEzARBgsrBgEEAYI3PAIBAxMCQ1oxGTAXBgNVBAoTEEpldEJyYWlucyBz
# LnIuby4xHTAbBgNVBA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMREwDwYDVQQFEwgy
# NjUwMjI3NTEZMBcGA1UEAxMQSmV0QnJhaW5zIHMuci5vLjCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAMmZsBOO0sV6f+qwSfowL7CAaqooz6zzYDcPWA/6
# P1BlOeO9NqKEzd4IyZEI1lccipydIwdaO1YHa2kM7/5kertoyJ6ITuMjmcyxnqXI
# AhdCHLbqO/Vq6R+4fqokQ5FvPvsTZ6bbc8sznX3roST/zjlgYA5+HDR2GRZ4sy5d
# tkt/i/MANl3f3rRv1RO0qvy+dU7GOk0CWPfXFIHrj+sszSQ7zgM52bHAP70NlKfj
# VkKdX7U4Ytz3yb1gnfLRQyxYgM/MBKGDI6BrHlUfskryxjV5gH+aWFfBaXw8+NVl
# fYr8lUdlXlzlWmHqa7J792WNCLlE31NevEYaXA+0TbxJKT8bvXWehHxBpGY9Q15z
# WjnaGk1FguXxqmXKkRQfDgBiJ4sCtGW3iVGmOtMiEKmOURSZ/hoUQSrtMf8r/itK
# re6DHoxGbjA9yjzPXZbT1dJk6eOcKZaesYANyu05Kz7S/lRX83N4UbCQEJ4xpcCI
# G+eeE4d2BrapVFMZxobUKZaFtV+SAByQFEumEjEX5hEMyGzfODoa0KnFowf9Fu5u
# lz0dM3cV6/+riIgphUIlsNYgum7swNn6dp13+iyMuaOvPEL0kBlOPufcz7Lq18a6
# o5anOBJpW9jAMf0cCfJ+hwAOj3gcTMbiKCmEo7gEoQiZvnM4DZra7tf+Nblmv730
# RtPHAgMBAAGjggFgMIIBXDAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBQKghd/k2G5
# FxhP/0MIUgmYqUN3xjAfBgNVHSMEGDAWgBTOiU+CUaoVooRiyjEjYdJh+/j+eDBn
# BggrBgEFBQcBAQRbMFkwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3Qu
# bmV0MDIGCCsGAQUFBzAChiZodHRwOi8vYWlhLmVudHJ1c3QubmV0L2V2Y3MyLWNo
# YWluLnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVudHJ1c3QubmV0
# L2V2Y3MyLmNybDAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# SwYDVR0gBEQwQjA3BgpghkgBhvpsCgECMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAHBgVngQwBAzANBgkqhkiG9w0BAQsFAAOCAgEA
# BLLwIeBU0HShBD5L1ZoZN+ZQIpMOIHkQajuxja1zSLwakCX3lA0V3MpxliDDRx3e
# y4YA3OcefN2zV6xUAVNOb6V8zB552SblNVxa4TIxzmSzKI2uifjkikt1ZogDEXsS
# Cflyak2rbBATAmNrEBFAcZBLUGDxExbK2HyZAtG+CR16jj9Qd3zDHmbSciIMlsBq
# pmgmP34/pcjr/LfjXNqa2r0Q+ISbhVgv5N4ESYdBUKh0SAMf5fcUAreP0IpTOO4v
# FkkZSoUkALhGfBQDluukYrIGUh7fjmNVAwTQ1UjLRb8VDQwsIhFMBJkzevbxkqZA
# 3O6JLdisMuRL6/CFkdnftPAeiBJbs2jRQzADDGylIdIMv8V6A/yymtg9kZH9eCNw
# JibhEhTPHsEJX5Unwk8vE7POUqCKoB7+ULkGURTrUtEBBYmShPcbjQH0l6pcb66J
# oCX78Cbzd/Zr9Fm6mLjjlNbZcyjBSGDyuZq6SpKLL90YYCXHNEFKJdFCtnxtnbM4
# ipy3TZi9Xhx5OSfLd3T7/WgFQYuLHacteeGVonYBGXZbQiFKbUophNfYFRF0N5ZE
# cc7Vy3Vm+JSAvX496v6GRR3/C2zLI9ffZBqYLFhMGo0qIIUNLuIBbhX6pzw3mrlj
# VIdeZGNRj3vWqhAp9A7IzS6x3t2RrmSrkV1an7k6Um0xghwrMIIcJwIBATB3MGMx
# CzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNF
# bnRydXN0IEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZD
# UzICEDGdnUgatvXgkrzF40/3PFswCwYJYIZIAWUDBAIBoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCCASvIZmrXDXszfb7WmrzJFa7q8WLtT11OEajTPL3aEljBC
# BgorBgEEAYI3AgEMMTQwMqAwgC4AZABlAGYAZQBuAGQAZQByAC0AZQB4AGMAbAB1
# AHMAaQBvAG4AcwAuAHAAcwAxMA0GCSqGSIb3DQEBAQUABIICAKzNx8ZN/RrriwtZ
# 1UOefGiCkFh6FZbLeQ+VIATZHrvYBA745PTbDn7FWUWw4gWRGaID3emr/nc2Zcgv
# LvgW1tlDh0k6/E4uCzI+sq5H047FjstJOQrpuB+9ZJrakWs6vpkq9W33ulLFFV5N
# uGZBo1F3KlLKRlYcea9B897nScdGJ/nnn1BbW9ze0dSZfaFoo+Bi1EXeAnQ7BdZu
# C9vwh6OXI+kfevujQX43pRPzcRWdKU4TZ0j7lMg/CorK41bMlDuvCE5jOaj2NeS3
# NXzHk0YQ/Ygbf2wpC6pAtDIen3ZboA4q5s5m3pbh4Q8Kz/f6D5uPumSbMT/pV0Sn
# GdVFJeEjvY+tuL5EDHhpb69g/LETVho3xZ4PL33p6pwmQ+zEoElPOnlkZs8EZpFI
# Zl4cqeP93PDbQJxTq9xE9D47gWyfqTLwwU6Kl4Alr+lAAHqtaScFEj5BQdF+FnC1
# zA4Dtxu5MNkuFdq1E1eRm/XIi2T36rYUrx1Zi2F3HWghRrZnA/tnRdsGhwWNuhkC
# eC9CXHyW16cJTGtM/u88JPe0NZvYOoaVbGhSlDSSrINAMTMt/GXNcdlJtWFN/vhn
# oJL1sollrdGb9EgWPHtqYnLcS1JIK5iRlURuWZsNFJ4kyB+/DHaLPEsFnKFzs9TE
# D2Wm3i4g+hfXc8K7U/WblVJp+mwroYIY1jCCGNIGCisGAQQBgjcDAwExghjCMIIY
# vgYJKoZIhvcNAQcCoIIYrzCCGKsCAQMxDzANBglghkgBZQMEAgIFADCB9gYLKoZI
# hvcNAQkQAQSggeYEgeMwgeACAQEGCisGAQQBsjECAQEwLzALBglghkgBZQMEAgEE
# IAKsLw6b64TBAF487IBxcRNSzioZaumzSlxQwGeVjo6NAhUAvh0t03smBlkd9tMQ
# IJSfsr53amMYDzIwMjUwOTI1MTQwMDE0WqB2pHQwcjELMAkGA1UEBhMCR0IxFzAV
# BgNVBAgTDldlc3QgWW9ya3NoaXJlMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQx
# MDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2lnbmVyIFIz
# NqCCEwQwggZiMIIEyqADAgECAhEApCk7bh7d16c0CIetek63JDANBgkqhkiG9w0B
# AQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjAeFw0y
# NTAzMjcwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMHIxCzAJBgNVBAYTAkdCMRcwFQYD
# VQQIEw5XZXN0IFlvcmtzaGlyZTEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMTAw
# LgYDVQQDEydTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIFNpZ25lciBSMzYw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDThJX0bqRTePI9EEt4Egc8
# 3JSBU2dhrJ+wY7JgReuff5KQNhMuzVytzD+iXazATVPMHZpH/kkiMo1/vlAGFrYN
# 2P7g0Q8oPEcR3h0SftFNYxxMh+bj3ZNbbYjwt8f4DsSHPT+xp9zoFuw0HOMdO3sW
# eA1+F8mhg6uS6BJpPwXQjNSHpVTCgd1gOmKWf12HSfSbnjl3kDm0kP3aIUAhsodB
# YZsJA1imWqkAVqwcGfvs6pbfs/0GE4BJ2aOnciKNiIV1wDRZAh7rS/O+uTQcb6JV
# zBVmPP63k5xcZNzGo4DOTV+sM1nVrDycWEYS8bSS0lCSeclkTcPjQah9Xs7xbOBo
# CdmahSfg8Km8ffq8PhdoAXYKOI+wlaJj+PbEuwm6rHcm24jhqQfQyYbOUFTKWFe9
# 01VdyMC4gRwRAq04FH2VTjBdCkhKts5Py7H73obMGrxN1uGgVyZho4FkqXA8/uk6
# nkzPH9QyHIED3c9CGIJ098hU4Ig2xRjhTbengoncXUeo/cfpKXDeUcAKcuKUYRNd
# GDlf8WnwbyqUblj4zj1kQZSnZud5EtmjIdPLKce8UhKl5+EEJXQp1Fkc9y5Ivk4A
# ZacGMCVG0e+wwGsjcAADRO7Wga89r/jJ56IDK773LdIsL3yANVvJKdeeS6OOEiH6
# hpq2yT+jJ/lHa9zEdqFqMwIDAQABo4IBjjCCAYowHwYDVR0jBBgwFoAUX1jtTDF6
# omFCjVKAurNhlxmiMpswHQYDVR0OBBYEFIhhjKEqN2SBKGChmzHQjP0sAs5PMA4G
# A1UdDwEB/wQEAwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUF
# BwMIMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEAjBKBgNVHR8EQzBBMD+gPaA7
# hjlodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBp
# bmdDQVIzNi5jcmwwegYIKwYBBQUHAQEEbjBsMEUGCCsGAQUFBzAChjlodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBpbmdDQVIzNi5j
# cnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3
# DQEBDAUAA4IBgQACgT6khnJRIfllqS49Uorh5ZvMSxNEk4SNsi7qvu+bNdcuknHg
# XIaZyqcVmhrV3PHcmtQKt0blv/8t8DE4bL0+H0m2tgKElpUeu6wOH02BjCIYM6HL
# InbNHLf6R2qHC1SUsJ02MWNqRNIT6GQL0Xm3LW7E6hDZmR8jlYzhZcDdkdw0cHhX
# jbOLsmTeS0SeRJ1WJXEzqt25dbSOaaK7vVmkEVkOHsp16ez49Bc+Ayq/Oh2BAkST
# Fog43ldEKgHEDBbCIyba2E8O5lPNan+BQXOLuLMKYS3ikTcp/Qw63dxyDCfgqXYU
# hxBpXnmeSO/WA4NwdwP35lWNhmjIpNVZvhWoxDL+PxDdpph3+M5DroWGTc1ZuDa1
# iXmOFAK4iwTnlWDg3QNRsRa9cnG3FBBpVHnHOEQj4GMkrOHdNDTbonEeGvZ+4nSZ
# XrwCW4Wv2qyGDBLlKk3kUW1pIScDCpm/chL6aUbnSsrtbepdtbCLiGanKVR/KC1g
# sR0tC6Q0RfWOI4owggYUMIID/KADAgECAhB6I67aU2mWD5HIPlz0x+M/MA0GCSqG
# SIb3DQEBDAUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgUm9vdCBS
# NDYwHhcNMjEwMzIyMDAwMDAwWhcNMzYwMzIxMjM1OTU5WjBVMQswCQYDVQQGEwJH
# QjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1
# YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjCCAaIwDQYJKoZIhvcNAQEBBQADggGP
# ADCCAYoCggGBAM2Y2ENBq26CK+z2M34mNOSJjNPvIhKAVD7vJq+MDoGD46IiM+b8
# 3+3ecLvBhStSVjeYXIjfa3ajoW3cS3ElcJzkyZlBnwDEJuHlzpbN4kMH2qRBVrjr
# GJgSlzzUqcGQBaCxpectRGhhnOSwcjPMI3G0hedv2eNmGiUbD12OeORN0ADzdpsQ
# 4dDi6M4YhoGE9cbY11XxM2AVZn0GiOUC9+XE0wI7CQKfOUfigLDn7i/WeyxZ43XL
# j5GVo7LDBExSLnh+va8WxTlA+uBvq1KO8RSHUQLgzb1gbL9Ihgzxmkdp2ZWNuLc+
# XyEmJNbD2OIIq/fWlwBp6KNL19zpHsODLIsgZ+WZ1AzCs1HEK6VWrxmnKyJJg2Lv
# 23DlEdZlQSGdF+z+Gyn9/CRezKe7WNyxRf4e4bwUtrYE2F5Q+05yDD68clwnweck
# KtxRaF0VzN/w76kOLIaFVhf5sMM/caEZLtOYqYadtn034ykSFaZuIBU9uCSrKRKT
# PJhWvXk4CllgrwIDAQABo4IBXDCCAVgwHwYDVR0jBBgwFoAU9ndq3T/9ARP/FqFs
# ggIv0Ao9FCUwHQYDVR0OBBYEFF9Y7UwxeqJhQo1SgLqzYZcZojKbMA4GA1UdDwEB
# /wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMI
# MBEGA1UdIAQKMAgwBgYEVR0gADBMBgNVHR8ERTBDMEGgP6A9hjtodHRwOi8vY3Js
# LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBpbmdSb290UjQ2LmNy
# bDB8BggrBgEFBQcBAQRwMG4wRwYIKwYBBQUHMAKGO2h0dHA6Ly9jcnQuc2VjdGln
# by5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ1Jvb3RSNDYucDdjMCMGCCsG
# AQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOC
# AgEAEtd7IK0ONVgMnoEdJVj9TC1ndK/HYiYh9lVUacahRoZ2W2hfiEOyQExnHk1j
# kvpIJzAMxmEc6ZvIyHI5UkPCbXKspioYMdbOnBWQUn733qMooBfIghpR/klUqNxx
# 6/fDXqY0hSU1OSkkSivt51UlmJElUICZYBodzD3M/SFjeCP59anwxs6hwj1mfvzG
# +b1coYGnqsSz2wSKr+nDO+Db8qNcTbJZRAiSazr7KyUJGo1c+MScGfG5QHV+bps8
# BX5Oyv9Ct36Y4Il6ajTqV2ifikkVtB3RNBUgwu/mSiSUice/Jp/q8BMk/gN8+0rN
# IE+QqU63JoVMCMPY2752LmESsRVVoypJVt8/N3qQ1c6FibbcRabo3azZkcIdWGVS
# AdoLgAIxEKBeNh9AQO1gQrnh1TA8ldXuJzPSuALOz1Ujb0PCyNVkWk7hkhVHfcvB
# fI8NtgWQupiaAeNHe0pWSGH2opXZYKYG4Lbukg7HpNi/KqJhue2Keak6qH9A8CeE
# OB7Eob0Zf+fU+CCQaL0cJqlmnx9HCDxF+3BLbUufrV64EbTI40zqegPZdA+sXCmb
# cZy6okx/SjwsusWRItFA3DE8MORZeFb6BmzBtqKJ7l939bbKBy2jvxcJI98Va95Q
# 5JnlKor3m0E7xpMeYRriWklUPsetMSf2NvUQa/E5vVyefQIwggaCMIIEaqADAgEC
# AhA2wrC9fBs656Oz3TbLyXVoMA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAc
# BgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0
# IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0yMTAzMjIwMDAwMDBaFw0z
# ODAxMTgyMzU5NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgUm9v
# dCBSNDYwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCIndi5RWedHd3o
# uSaBmlRUwHxJBZvMWhUP2ZQQRLRBQIF3FJmp1OR2LMgIU14g0JIlL6VXWKmdbmKG
# RDILRxEtZdQnOh2qmcxGzjqemIk8et8sE6J+N+Gl1cnZocew8eCAawKLu4TRrCoq
# CAT8uRjDeypoGJrruH/drCio28aqIVEn45NZiZQI7YYBex48eL78lQ0BrHeSmqy1
# uXe9xN04aG0pKG9ki+PC6VEfzutu6Q3IcZZfm00r9YAEp/4aeiLhyaKxLuhKKaAd
# QjRaf/h6U13jQEV1JnUTCm511n5avv4N+jSVwd+Wb8UMOs4netapq5Q/yGyiQOgj
# sP/JRUj0MAT9YrcmXcLgsrAimfWY3MzKm1HCxcquinTqbs1Q0d2VMMQyi9cAgMYC
# 9jKc+3mW62/yVl4jnDcw6ULJsBkOkrcPLUwqj7poS0T2+2JMzPP+jZ1h90/QpZnB
# khdtixMiWDVgh60KmLmzXiqJc6lGwqoUqpq/1HVHm+Pc2B6+wCy/GwCcjw5rmzaj
# LbmqGygEgaj/OLoanEWP6Y52Hflef3XLvYnhEY4kSirMQhtberRvaI+5YsD3XVxH
# GBjlIli5u+NrLedIxsE88WzKXqZjj9Zi5ybJL2WjeXuOTbswB7XjkZbErg7ebeAQ
# UQiS/uRGZ58NHs57ZPUfECcgJC+v2wIDAQABo4IBFjCCARIwHwYDVR0jBBgwFoAU
# U3m/WqorSs9UgOHYm8Cd8rIDZsswHQYDVR0OBBYEFPZ3at0//QET/xahbIICL9AK
# PRQlMA4GA1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MBMGA1UdJQQMMAoG
# CCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9o
# dHRwOi8vY3JsLnVzZXJ0cnVzdC5jb20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlv
# bkF1dGhvcml0eS5jcmwwNQYIKwYBBQUHAQEEKTAnMCUGCCsGAQUFBzABhhlodHRw
# Oi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQAOvmVB7WhE
# uOWhxdQRh+S3OyWM637ayBeR7djxQ8SihTnLf2sABFoB0DFR6JfWS0snf6WDG2gt
# CGflwVvcYXZJJlFfym1Doi+4PfDP8s0cqlDmdfyGOwMtGGzJ4iImyaz3IBae91g5
# 0QyrVbrUoT0mUGQHbRcF57olpfHhQEStz5i6hJvVLFV/ueQ21SM99zG4W2tB1ExG
# L98idX8ChsTwbD/zIExAopoe3l6JrzJtPxj8V9rocAnLP2C8Q5wXVVZcbw4x4ztX
# LsGzqZIiRh5i111TW7HV1AtsQa6vXy633vCAbAOIaKcLAo/IU7sClyZUk62XD0VU
# nHD+YvVNvIGezjM6CRpcWed/ODiptK+evDKPU2K6synimYBaNH49v9Ih24+eYXNt
# I38byt5kIvh+8aW88WThRpv8lUJKaPn37+YHYafob9Rg7LyTrSYpyZoBmwRWSE4W
# 6iPjB7wJjJpH29308ZkpKKdpkiS9WNsf/eeUtvRrtIEiSJHN899L1P4l6zKVsdrU
# u1FX1T/ubSrsxrYJD+3f3aKg6yxdbugot06YwGXXiy5UUGZvOu3lXlxA+fC13dQ5
# OlL2gIb5lmF6Ii8+CQOYDwXM+yd9dbmocQsHjcRPsccUd5E9FiswEqORvz8g3s+j
# R3SFCgXhN4wz7NgAnOgpCdUo4uDyllU9PzGCBJIwggSOAgEBMGowVTELMAkGA1UE
# BhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAxMjU2VjdGln
# byBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYCEQCkKTtuHt3XpzQIh616Trck
# MA0GCWCGSAFlAwQCAgUAoIIB+TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTI1MDkyNTE0MDAxNFowPwYJKoZIhvcNAQkEMTIEMGOX
# UxpsPhV0gPMKnhvL+vlASSMWSzFtB+t56w6QDDbfyxaM3dviRVR7dPij7zqnGzCC
# AXoGCyqGSIb3DQEJEAIMMYIBaTCCAWUwggFhMBYEFDjJFIEQRLTcZj6T1HRLgUGG
# qbWxMIGHBBTGrlTkeIbxfD1VEkiMacNKevnC3TBvMFukWTBXMQswCQYDVQQGEwJH
# QjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1
# YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2AhB6I67aU2mWD5HIPlz0x+M/MIG8
# BBSFPWMtk4KCYXzQkDXEkd6SwULaxzCBozCBjqSBizCBiDELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYD
# VQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBS
# U0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDbCsL18Gzrno7PdNsvJdWgwDQYJ
# KoZIhvcNAQEBBQAEggIAz0PujN7iG0g1QRZaAAOMvMEHkUhGpo/rRdeMag2ZhJn9
# PnZNq6x29QwHzIr8Wux7628kz5O5fH2YfQEV25kP6iXeQO9WJB9fc+aHeIKR15i5
# hfYmA4i9+rbe5obOfcHtSj8D5NZU1I0BqGi+5TmJe69CiKEnUDIOX7pTd+OBkt2e
# +mec4PbtWDgUYpeibh1gkEZRANWy1qtsFm1qvPcOPde7m6Vxcsyc8Ye2ePKdHdsB
# Um2lVEkSEL64wzj7v6KJzx0H3DE2+CFFNzjjEW1zzwQJ2JWYE/mSRVL6HKWQcDvP
# N6lh45vdNpxE1B9pEaogTlOPKYuviqpTZsLc0VOeanl0Nbdf16+RBBt6g4jmSo3/
# +bU1P9Vvcftws9y2j1FM9DmVRE2yiXpUR9HJbPiJshmAUq4dIxhO5EWpggQmV0Ho
# IcNyN31lOQ7XGLstTrXEm1Fncef4bFmHrbtDVcg0ttcgLGDzNcKmJa5W/Laa0uI4
# lprs1FTHxJJcO9QuruTpeSWQeOM131tMEU53u6INtwi17lppstGr2DQy0BhEw5dq
# eEk5pLXi6yoO4p6wi6zhS3kfCn3b/AVT/lIaWDK1SZg4o8IuVPHfiv3XQ7JcrVdY
# Sj2/8JezCYtdm7mxHAjhRHHtCO/fsLqWSKmUr34+SklEsUJcx1do4fFlVkElnRs=
# SIG # End signature block
